/* specter_PCAP.c
 *
 * specter output target for writing pcap-style files (like tcpdump)
 *
 * (C) 2002 by Harald Welte <laforge@gnumonks.org>
 *
 *  11 Jun 2004, Michal Kwiatkowski <ruby@joker.linuxstuff.pl>:
 *      Fixed for specter.
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 
 *  as published by the Free Software Foundation
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <pcap.h>
#include <specter/specter.h>
#include <conffile/conffile.h>
#include "lret.h"

/* we rely on arrays, not pointers, so we need different macro here */
#undef IS_VALID
#define IS_VALID(ret_arr,x)	(ret_arr[x].p->flags & SPECTER_RETF_VALID)


#ifndef PCAP_DEFAULT
#define PCAP_DEFAULT	"/var/log/specter.pcap"
#endif


static config_entry_t my_config[] = {
	{ .key = "sync", .type = CONFIG_TYPE_BOOLEAN, .options = CONFIG_OPT_NONE,
		.u = { .value = 0 } },
	{ .key = "logfile", .type = CONFIG_TYPE_STRING, .options = CONFIG_OPT_NONE,
		.u = { .string = PCAP_DEFAULT } },
};


struct my_data {
	FILE *of;
};


#define LOCAL_RET_NUM 	5
static specter_local_ret_t local_ret[LOCAL_RET_NUM] = {
	{ "raw.pkt", 0 },
	{ "raw.pktlen", 0 },
	{ "ip.totlen", 0 },
	{ "oob.time_sec", 0 },
	{ "oob.time_usec", 0 },
};


static int output_pcap(config_entry_t *ce, void *data)
{
	struct pcap_pkthdr pchdr;
	struct my_data *md = data;

	pchdr.caplen = GET_VALUE(local_ret,1).ui32;
	pchdr.len = GET_VALUE(local_ret,2).ui32;

	if (IS_VALID(local_ret,3) && IS_VALID(local_ret,4)) {
		pchdr.ts.tv_sec = GET_VALUE(local_ret,3).ui32;
		pchdr.ts.tv_usec = GET_VALUE(local_ret,4).ui32;
	} else {
		/* use current system time */
		gettimeofday(&pchdr.ts, NULL);
	}

	if (fwrite(&pchdr, sizeof(pchdr), 1, md->of) != 1) {
		specter_log(SPECTER_ERROR, "Error during write: %s.\n",
			  strerror(errno));
		return -1;
	}
	if (fwrite(GET_VALUE(local_ret,0).ptr, pchdr.caplen, 1, md->of) != 1) {
		specter_log(SPECTER_ERROR, "Error during write: %s.\n",
			  strerror(errno));
		return -1;
	}

	if (GET_CE(ce,0)->u.value)
		fflush(md->of);

	return 0;
}

/* stolen from libpcap savefile.c */
#define LINKTYPE_RAW            101
#define TCPDUMP_MAGIC	0xa1b2c3d4

static int write_pcap_header(FILE *of)
{
	struct pcap_file_header pcfh;
	int ret;

	pcfh.magic = TCPDUMP_MAGIC;
	pcfh.version_major = PCAP_VERSION_MAJOR;
	pcfh.version_minor = PCAP_VERSION_MINOR;
	pcfh.thiszone = timezone;
	pcfh.sigfigs = 0;
	pcfh.snaplen = 64 * 1024; /* we don't know the length in advance */
	pcfh.linktype = LINKTYPE_RAW;

	ret =  fwrite(&pcfh, sizeof(pcfh), 1, of);
	fflush(of);

	return ret;
}


static int append_create_outfile(char *filename, FILE **fd)
{
	struct stat st_dummy;

	if (stat(filename, &st_dummy)) {
		*fd = fopen(filename, "w");
		if (!*fd) {
			specter_log(SPECTER_FATAL, "Can't open \"%s\": %s.\n",
				  filename, strerror(errno));
			return -1;
		}
		if (!write_pcap_header(*fd)) {
			specter_log(SPECTER_FATAL, "Can't write pcap header: %s.\n",
				  strerror(errno));
			return -1;
		}
	} else {
		*fd = fopen(filename, "a");
		if (!*fd) {
			specter_log(SPECTER_FATAL, "Can't open \"%s\": %s.\n", 
				filename, strerror(errno));
			return -1;
		}		
	}

	return 0;
}


static void *init_pcap(config_entry_t *ce)
{
	struct my_data *data;

	if (fill_local_ret(local_ret, LOCAL_RET_NUM) == -1)
		return NULL;

	if ((data = malloc(sizeof(struct my_data))) == NULL) {
		specter_log(SPECTER_FATAL, "Couldn't allocate data: %s.\n",
				strerror(errno));
		return NULL;
	}

	if (append_create_outfile(GET_CE(ce,1)->u.string, &data->of) == -1)
		return NULL;

	return data;
}

static void fini_pcap(config_entry_t *ce, void *data)
{
	struct my_data *md = data;

	fclose(md->of);

	free(data);
}

static int sighup_handler(config_entry_t *ce, void *data, int signal)
{
	struct my_data *md = data;

	switch (signal) {
	case SIGHUP:
		specter_log(SPECTER_INFO, "Reopening capture file.\n");
		fclose(md->of);
		if (append_create_outfile(GET_CE(ce,1)->u.string, &md->of) == -1)
			return -1;
		break;
	default:
		break;
	}

	return 0;
}


static specter_output_t pcap_op = {
	.name = "pcap",
	.ce_base = my_config,
	.ce_num = 2,
	.init = &init_pcap,
	.fini = &fini_pcap,
	.output = &output_pcap,
	.signal = &sighup_handler
};


void _init(void)
{
	if (register_output(&pcap_op, 0) == -1) {
		specter_log(SPECTER_FATAL, "Couldn't register.\n");
		exit(EXIT_FAILURE);
	}
}

