-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with Ada.Real_Time;

with GNAT.OS_Lib;

with SPARK.Expect;
with Work_Manager;
with Log_Files;

---------------------------------------------------------------------------
--  Workers package specification                                        --
--                                                                       --
--  This package provides the Start_Analysis and Run_Analysis procedures to      --
--  run multiple simplifications in parallel.  The number of Workers is  --
--  set by the -p option in the command line.                            --
---------------------------------------------------------------------------
package Workers
is
   Path            : GNAT.OS_Lib.String_Access;
   Spadesimp_Exe   : GNAT.OS_Lib.String_Access;
   ZombieScope_Exe : GNAT.OS_Lib.String_Access;
   Victor_Exe      : GNAT.OS_Lib.String_Access;

   Spadesimp_Command   : constant String := "spadesimp";
   ZombieScope_Command : constant String := "zombiescope";
   Victor_Command      : constant String := "victor";

   -- Sets Path, Spadesimp_Exe, ZombieScope_Exe, Victor_Exe and Alt-Ergo
   -- depending on setting of -x flag, -z flag, and PATH environment
   procedure Locate_Binaries;

   type Worker_Set (Size : Positive) is limited private;

   procedure Initialize
     (Work_Set  : in out Worker_Set;
      S_Options : in     GNAT.OS_Lib.Argument_List_Access;
      Z_Options : in     GNAT.OS_Lib.Argument_List_Access;
      V_Options : in     GNAT.OS_Lib.Argument_List_Access);

   procedure Start_Analysis (The_Job  : in     Work_Manager.Job_Index;
                             Work_Set : in out Worker_Set);

   function Workers_Available (Work_Set : Worker_Set) return Natural;

   procedure Run_Analysis (Work_Set : in out Worker_Set);

private

   type Work_Data is
      record
         Job_ID       : Work_Manager.Job_Index;
         Start_Time   : Ada.Real_Time.Time;
         End_Time     : Ada.Real_Time.Time;
         Elapsed_Time : Duration;
         OK           : Boolean;
         WhyFailed    : Work_Manager.ErrorString;
         OP           : Positive;
      end record;

   type Work_Set_Data is array (Positive range <>) of Work_Data;

   type Worker_Set (Size : Positive) is
      record
         Worker_Count  : Positive;
         Working_Count : Natural;
         Files         : Work_Set_Data (1 .. Size);
         Procs         : SPARK.Expect.Multiprocess_Regexp_Array (1 .. Size);
         Logs          : Log_Files.Log_File_Set (1 .. Size);
      end record;

end Workers;

