-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SPARK.Ada.Text_IO.Not_SPARK;

package body SPARK.Ada.Text_IO is
   --# hide SPARK.Ada.Text_IO;

   pragma Warnings ("Y"); -- Turn off warnings for Ada 2005 features

   ---------------------
   -- File Management --
   ---------------------

   procedure Create
     (File :    out File_Type;
      Mode : in     File_Mode;
      Name : in     String;
      Form : in     String) is
   begin
      Standard.Ada.Text_IO.Create
        (File => Standard.Ada.Text_IO.File_Type (File),
         Mode => SPARK.Ada.Text_IO.Not_SPARK.To_File_Mode (From => Mode),
         Name => Name,
         Form => Form);
   end Create;

   procedure Open
     (File :    out File_Type;
      Mode : in     File_Mode;
      Name : in     String;
      Form : in     String) is
   begin
      Standard.Ada.Text_IO.Open
        (File => Standard.Ada.Text_IO.File_Type (File),
         Mode => SPARK.Ada.Text_IO.Not_SPARK.To_File_Mode (From => Mode),
         Name => Name,
         Form => Form);
   end Open;

   procedure Close (File : in out File_Type) is
   begin
      Standard.Ada.Text_IO.Close
        (File => Standard.Ada.Text_IO.File_Type (File));
   end Close;

   procedure Delete (File : in out File_Type) is
   begin
      Standard.Ada.Text_IO.Delete
        (File => Standard.Ada.Text_IO.File_Type (File));
   end Delete;

   procedure Reset_Mode (File : in out File_Type;
                         Mode : in     File_Mode) is
   begin
      Standard.Ada.Text_IO.Reset
        (File => Standard.Ada.Text_IO.File_Type (File),
         Mode => SPARK.Ada.Text_IO.Not_SPARK.To_File_Mode (From => Mode));
   end Reset_Mode;

   procedure Reset (File : in out File_Type) is
   begin
      Standard.Ada.Text_IO.Reset
        (File => Standard.Ada.Text_IO.File_Type (File));
   end Reset;

   function Mode (File : File_Type) return File_Mode is

      function To_File_Mode
        (From : Standard.Ada.Text_IO.File_Mode) return File_Mode;

      function To_File_Mode
        (From : Standard.Ada.Text_IO.File_Mode) return File_Mode is
      begin
         case From is
            when Standard.Ada.Text_IO.In_File =>
               return In_File;
            when Standard.Ada.Text_IO.Out_File =>
               return Out_File;
            when Standard.Ada.Text_IO.Append_File =>
               return Append_File;
         end case;
      end To_File_Mode;

   begin
      return To_File_Mode
        (From => Standard.Ada.Text_IO.Mode
           (File => Standard.Ada.Text_IO.File_Type (File)));
   end Mode;

   function Is_Open (File : File_Type) return Boolean is
   begin
      return Standard.Ada.Text_IO.Is_Open
        (File => Standard.Ada.Text_IO.File_Type (File));
   end Is_Open;

   ------------------------------------------------------
   -- Control of default input, output and error files --
   ------------------------------------------------------

   function Standard_Input return File_Type is
      function Get_Standard_Input return Standard.Ada.Text_IO.File_Type
        renames Standard.Ada.Text_IO.Standard_Input;
   begin
      return File_Type (Get_Standard_Input);
   end Standard_Input;

   function Standard_Output return File_Type is
      function Get_Standard_Output return Standard.Ada.Text_IO.File_Type
        renames Standard.Ada.Text_IO.Standard_Output;
   begin
      return File_Type (Get_Standard_Output);
   end Standard_Output;

   function Standard_Error return File_Type is
      function Get_Standard_Error return Standard.Ada.Text_IO.File_Type
        renames Standard.Ada.Text_IO.Standard_Error;
   begin
      return File_Type (Get_Standard_Error);
   end Standard_Error;

   --------------------
   -- Buffer control --
   --------------------

   procedure Flush_File (File : in out File_Type) is
   begin
      Standard.Ada.Text_IO.Flush
        (File => Standard.Ada.Text_IO.File_Type (File));
   end Flush_File;

   procedure Flush_Output is
   begin
      Standard.Ada.Text_IO.Flush
        (File => Standard.Ada.Text_IO.Standard_Output);
   end Flush_Output;

   procedure Flush_Error is
   begin
      Standard.Ada.Text_IO.Flush
        (File => Standard.Ada.Text_IO.Standard_Error);
   end Flush_Error;

   --------------------------------------------
   -- Specification of line and page lengths --
   --------------------------------------------

   procedure Set_Line_Length_File (File : in out File_Type;
                                   To   : in     Count) is
   begin
      Standard.Ada.Text_IO.Set_Line_Length
        (File => Standard.Ada.Text_IO.File_Type (File),
         To   => Standard.Ada.Text_IO.Count (To));
   end Set_Line_Length_File;

   procedure Set_Line_Length_Output (To : in Count) is
   begin
      Standard.Ada.Text_IO.Set_Line_Length
        (File => Standard.Ada.Text_IO.Standard_Output,
         To   => Standard.Ada.Text_IO.Count (To));
   end Set_Line_Length_Output;

   procedure Set_Line_Length_Error (To : in Count) is
   begin
      Standard.Ada.Text_IO.Set_Line_Length
        (File => Standard.Ada.Text_IO.Standard_Error,
         To   => Standard.Ada.Text_IO.Count (To));
   end Set_Line_Length_Error;

   procedure Set_Page_Length_File (File : in out File_Type;
                                   To   : in     Count) is
   begin
      Standard.Ada.Text_IO.Set_Page_Length
        (File => Standard.Ada.Text_IO.File_Type (File),
         To   => Standard.Ada.Text_IO.Count (To));
   end Set_Page_Length_File;

   procedure Set_Page_Length_Output (To : in Count) is
   begin
      Standard.Ada.Text_IO.Set_Page_Length
        (File => Standard.Ada.Text_IO.Standard_Output,
         To   => Standard.Ada.Text_IO.Count (To));
   end Set_Page_Length_Output;

   procedure Set_Page_Length_Error (To : in Count) is
   begin
      Standard.Ada.Text_IO.Set_Page_Length
        (File => Standard.Ada.Text_IO.Standard_Error,
         To   => Standard.Ada.Text_IO.Count (To));
   end Set_Page_Length_Error;

   function Line_Length_File (File : File_Type) return Count is
   begin
      return Count
        (Standard.Ada.Text_IO.Line_Length
           (File => Standard.Ada.Text_IO.File_Type (File)));
   end Line_Length_File;

   function Line_Length_Output return Count is
   begin
      return Count
        (Standard.Ada.Text_IO.Line_Length
           (File => Standard.Ada.Text_IO.Standard_Output));
   end Line_Length_Output;

   function Line_Length_Error return Count is
   begin
      return Count
        (Standard.Ada.Text_IO.Line_Length
           (File => Standard.Ada.Text_IO.Standard_Error));
   end Line_Length_Error;

   function Page_Length_File (File : File_Type) return Count is
   begin
      return Count
        (Standard.Ada.Text_IO.Page_Length
           (File => Standard.Ada.Text_IO.File_Type (File)));
   end Page_Length_File;

   function Page_Length_Output return Count is
   begin
      return Count
        (Standard.Ada.Text_IO.Page_Length
           (File => Standard.Ada.Text_IO.Standard_Output));
   end Page_Length_Output;

   function Page_Length_Error return Count is
   begin
      return Count
        (Standard.Ada.Text_IO.Page_Length
           (File => Standard.Ada.Text_IO.Standard_Error));
   end Page_Length_Error;

   ------------------------------------
   -- Column, Line, and Page Control --
   ------------------------------------

   procedure New_Line_File (File    : in out File_Type;
                            Spacing : in     Positive_Count) is
   begin
      Standard.Ada.Text_IO.New_Line
        (File    => Standard.Ada.Text_IO.File_Type (File),
         Spacing => Standard.Ada.Text_IO.Positive_Count (Spacing));
   end New_Line_File;

   procedure New_Line_Output (Spacing : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.New_Line
        (File    => Standard.Ada.Text_IO.Standard_Output,
         Spacing => Standard.Ada.Text_IO.Positive_Count (Spacing));
   end New_Line_Output;

   procedure New_Line_Error (Spacing : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.New_Line
        (File    => Standard.Ada.Text_IO.Standard_Error,
         Spacing => Standard.Ada.Text_IO.Positive_Count (Spacing));
   end New_Line_Error;

   procedure Skip_Line_File (File    : in out File_Type;
                             Spacing : in     Positive_Count) is
   begin
      Standard.Ada.Text_IO.Skip_Line
        (File    => Standard.Ada.Text_IO.File_Type (File),
         Spacing => Standard.Ada.Text_IO.Positive_Count (Spacing));
   end Skip_Line_File;

   procedure Skip_Line_Input (Spacing : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.Skip_Line
        (File    => Standard.Ada.Text_IO.Standard_Input,
         Spacing => Standard.Ada.Text_IO.Positive_Count (Spacing));
   end Skip_Line_Input;

   function End_Of_Line_File (File : File_Type) return Boolean is
   begin
      return Standard.Ada.Text_IO.End_Of_Line
        (File => Standard.Ada.Text_IO.File_Type (File));
   end End_Of_Line_File;

   function End_Of_Line_Input return Boolean is
   begin
      return Standard.Ada.Text_IO.End_Of_Line
        (File => Standard.Ada.Text_IO.Standard_Input);
   end End_Of_Line_Input;

   procedure New_Page_File (File : in out File_Type) is
   begin
      Standard.Ada.Text_IO.New_Page
        (File => Standard.Ada.Text_IO.File_Type (File));
   end New_Page_File;

   procedure New_Page_Output is
   begin
      Standard.Ada.Text_IO.New_Page
        (File => Standard.Ada.Text_IO.Standard_Output);
   end New_Page_Output;

   procedure New_Page_Error is
   begin
      Standard.Ada.Text_IO.New_Page
        (File => Standard.Ada.Text_IO.Standard_Error);
   end New_Page_Error;

   procedure Skip_Page_File (File : in out File_Type) is
   begin
      Standard.Ada.Text_IO.Skip_Page
        (File => Standard.Ada.Text_IO.File_Type (File));
   end Skip_Page_File;

   procedure Skip_Page_Input is
   begin
      Standard.Ada.Text_IO.Skip_Page
        (File => Standard.Ada.Text_IO.Standard_Input);
   end Skip_Page_Input;

   function End_Of_Page_File (File : File_Type) return Boolean is
   begin
      return Standard.Ada.Text_IO.End_Of_Page
        (File => Standard.Ada.Text_IO.File_Type (File));
   end End_Of_Page_File;

   function End_Of_Page_Input return Boolean is
   begin
      return Standard.Ada.Text_IO.End_Of_Page
        (File => Standard.Ada.Text_IO.Standard_Input);
   end End_Of_Page_Input;

   function End_Of_File_File (File : File_Type) return Boolean is
   begin
      return Standard.Ada.Text_IO.End_Of_File
        (File => Standard.Ada.Text_IO.File_Type (File));
   end End_Of_File_File;

   function End_Of_File_Input return Boolean is
   begin
      return Standard.Ada.Text_IO.End_Of_File
        (File => Standard.Ada.Text_IO.Standard_Input);
   end End_Of_File_Input;

   procedure Set_Col_File (File : in out File_Type;
                           To   : in     Positive_Count) is
   begin
      Standard.Ada.Text_IO.Set_Col
        (File => Standard.Ada.Text_IO.File_Type (File),
         To   => Standard.Ada.Text_IO.Positive_Count (To));
   end Set_Col_File;

   procedure Set_Col_Input (To : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.Set_Col
        (File => Standard.Ada.Text_IO.Standard_Input,
         To   => Standard.Ada.Text_IO.Positive_Count (To));
   end Set_Col_Input;

   procedure Set_Col_Output (To : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.Set_Col
        (File => Standard.Ada.Text_IO.Standard_Output,
         To   => Standard.Ada.Text_IO.Positive_Count (To));
   end Set_Col_Output;

   procedure Set_Col_Error (To : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.Set_Col
        (File => Standard.Ada.Text_IO.Standard_Error,
         To   => Standard.Ada.Text_IO.Positive_Count (To));
   end Set_Col_Error;

   procedure Set_Line_File (File : in out File_Type;
                            To   : in     Positive_Count) is
   begin
      Standard.Ada.Text_IO.Set_Line
        (File => Standard.Ada.Text_IO.File_Type (File),
         To   => Standard.Ada.Text_IO.Positive_Count (To));
   end Set_Line_File;

   procedure Set_Line_Input (To : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.Set_Line
        (File => Standard.Ada.Text_IO.Standard_Input,
         To   => Standard.Ada.Text_IO.Positive_Count (To));
   end Set_Line_Input;

   procedure Set_Line_Output (To : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.Set_Line
        (File => Standard.Ada.Text_IO.Standard_Output,
         To   => Standard.Ada.Text_IO.Positive_Count (To));
   end Set_Line_Output;

   procedure Set_Line_Error (To : in Positive_Count) is
   begin
      Standard.Ada.Text_IO.Set_Line
        (File => Standard.Ada.Text_IO.Standard_Error,
         To   => Standard.Ada.Text_IO.Positive_Count (To));
   end Set_Line_Error;

   function Col_File (File : File_Type) return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Col
           (File => Standard.Ada.Text_IO.File_Type (File)));
   end Col_File;

   function Col_Input return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Col
           (File => Standard.Ada.Text_IO.Standard_Input));
   end Col_Input;

   function Col_Output return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Col
           (File => Standard.Ada.Text_IO.Standard_Output));
   end Col_Output;

   function Col_Error return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Col
           (File => Standard.Ada.Text_IO.Standard_Error));
   end Col_Error;

   function Line_File (File : File_Type) return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Line
           (File => Standard.Ada.Text_IO.File_Type (File)));
   end Line_File;

   function Line_Input return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Line
           (File => Standard.Ada.Text_IO.Standard_Input));
   end Line_Input;

   function Line_Output return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Line
           (File => Standard.Ada.Text_IO.Standard_Output));
   end Line_Output;

   function Line_Error return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Line
           (File => Standard.Ada.Text_IO.Standard_Error));
   end Line_Error;

   function Page_File (File : File_Type) return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Page
           (File => Standard.Ada.Text_IO.File_Type (File)));
   end Page_File;

   function Page_Input return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Page
           (File => Standard.Ada.Text_IO.Standard_Input));
   end Page_Input;

   function Page_Output return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Page
           (File => Standard.Ada.Text_IO.Standard_Output));
   end Page_Output;

   function Page_Error return Positive_Count is
   begin
      return Positive_Count
        (Standard.Ada.Text_IO.Page
           (File => Standard.Ada.Text_IO.Standard_Error));
   end Page_Error;

   ----------------------------
   -- Character Input-Output --
   ----------------------------

   procedure Get_File (File : in out File_Type;
                       Item :    out Character) is
   begin
      Standard.Ada.Text_IO.Get (File => Standard.Ada.Text_IO.File_Type (File),
                                Item => Item);
   end Get_File;

   procedure Get_Input (Item : out Character) is
   begin
      Standard.Ada.Text_IO.Get (File => Standard.Ada.Text_IO.Standard_Input,
                                Item => Item);
   end Get_Input;

   procedure Put_File (File : in out File_Type;
                       Item : in     Character) is
   begin
      Standard.Ada.Text_IO.Put (File => Standard.Ada.Text_IO.File_Type (File),
                                Item => Item);
   end Put_File;

   procedure Put_Output (Item : in Character) is
   begin
      Standard.Ada.Text_IO.Put (File => Standard.Ada.Text_IO.Standard_Output,
                                Item => Item);
   end Put_Output;

   procedure Put_Error (Item : in Character) is
   begin
      Standard.Ada.Text_IO.Put (File => Standard.Ada.Text_IO.Standard_Error,
                                Item => Item);
   end Put_Error;

   procedure Look_Ahead_File
     (File        : in     File_Type;
      Item        :    out Character;
      End_Of_Line :    out Boolean) is
   begin
      Standard.Ada.Text_IO.Look_Ahead
        (File        => Standard.Ada.Text_IO.File_Type (File),
         Item        => Item,
         End_Of_Line => End_Of_Line);
   end Look_Ahead_File;

   procedure Look_Ahead_Input
     (Item        : out Character;
      End_Of_Line : out Boolean) is
   begin
      Standard.Ada.Text_IO.Look_Ahead
        (File        => Standard.Ada.Text_IO.Standard_Input,
         Item        => Item,
         End_Of_Line => End_Of_Line);
   end Look_Ahead_Input;

   procedure Get_Immediate_File
     (File : in     File_Type;
      Item :    out Character) is
   begin
      Standard.Ada.Text_IO.Get_Immediate
        (File => Standard.Ada.Text_IO.File_Type (File),
         Item => Item);
   end Get_Immediate_File;

   procedure Get_Immediate_Input
     (Item : out Character) is
   begin
      Standard.Ada.Text_IO.Get_Immediate
        (File => Standard.Ada.Text_IO.Standard_Input,
         Item => Item);
   end Get_Immediate_Input;

   procedure Get_Immediate_Available_File
     (File      : in     File_Type;
      Item      :    out Character;
      Available :    out Boolean) is
   begin
      Standard.Ada.Text_IO.Get_Immediate
        (File      => Standard.Ada.Text_IO.File_Type (File),
         Item      => Item,
         Available => Available);
   end Get_Immediate_Available_File;

   procedure Get_Immediate_Available_Input
     (Item      : out Character;
      Available : out Boolean) is
   begin
      Standard.Ada.Text_IO.Get_Immediate
        (File      => Standard.Ada.Text_IO.Standard_Input,
         Item      => Item,
         Available => Available);
   end Get_Immediate_Available_Input;

   -------------------------
   -- String Input-Output --
   -------------------------

   procedure Get_String_File (File : in out File_Type;
                              Item :    out String) is
   begin
      Standard.Ada.Text_IO.Get (File => Standard.Ada.Text_IO.File_Type (File),
                                Item => Item);
   end Get_String_File;

   procedure Get_String_Input (Item : out String) is
   begin
      Standard.Ada.Text_IO.Get (File => Standard.Ada.Text_IO.Standard_Input,
                                Item => Item);
   end Get_String_Input;

   procedure Put_String_File (File : in out File_Type;
                              Item : in     String) is
   begin
      Standard.Ada.Text_IO.Put (File => Standard.Ada.Text_IO.File_Type (File),
                                Item => Item);
   end Put_String_File;

   procedure Put_String_Output (Item : in String) is
   begin
      Standard.Ada.Text_IO.Put (File => Standard.Ada.Text_IO.Standard_Output,
                                Item => Item);
   end Put_String_Output;

   procedure Put_String_Error (Item : in String) is
   begin
      Standard.Ada.Text_IO.Put (File => Standard.Ada.Text_IO.Standard_Error,
                                Item => Item);
   end Put_String_Error;

   procedure Procedure_Get_Line_File
     (File     : in out File_Type;
      Item     :    out String;
      Arg_Last :    out Natural) is
   begin
      Standard.Ada.Text_IO.Get_Line
        (File => Standard.Ada.Text_IO.File_Type (File),
         Item => Item,
         Last => Arg_Last);
   end Procedure_Get_Line_File;

   procedure Procedure_Get_Line_Input
     (Item     : out String;
      Arg_Last : out Natural) is
   begin
      Standard.Ada.Text_IO.Get_Line
        (File => Standard.Ada.Text_IO.Standard_Input,
         Item => Item,
         Last => Arg_Last);
   end Procedure_Get_Line_Input;

   procedure Put_Line_File
     (File : in out File_Type;
      Item : in     String) is
   begin
      Standard.Ada.Text_IO.Put_Line
        (File => Standard.Ada.Text_IO.File_Type (File),
         Item => Item);
   end Put_Line_File;

   procedure Put_Line_Output
     (Item : in String) is
   begin
      Standard.Ada.Text_IO.Put_Line
        (File => Standard.Ada.Text_IO.Standard_Output,
         Item => Item);
   end Put_Line_Output;

   procedure Put_Line_Error
     (Item : in String) is
   begin
      Standard.Ada.Text_IO.Put_Line
        (File => Standard.Ada.Text_IO.Standard_Error,
         Item => Item);
   end Put_Line_Error;

end SPARK.Ada.Text_IO;
