-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure Wf_Procedure_Specification
  (Node        : in     STree.SyntaxNode;
   Hidden      : in     Boolean;
   Scope       : in out Dictionary.Scopes;
   Subprog_Sym :    out Dictionary.Symbol;
   First_Seen  :    out Boolean) is
   Ident_Node                       : STree.SyntaxNode;
   Ident_Str                        : LexTokenManager.Lex_String;
   First_Sym_Found                  : Dictionary.Symbol;
   Grand_Parent, Great_Grand_Parent : SPSymbols.SPSymbol;
   Adding_Proper_Body               : Boolean;

   -----------------------------------------------

   function In_Package_Body (Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.IsLocalScope (Scope) and then Dictionary.IsPackage (Dictionary.GetRegion (Scope));
   end In_Package_Body;

   -----------------------------------------------

   function In_Protected_Body (Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return Dictionary.IsLocalScope (Scope)
        and then Dictionary.IsType (Dictionary.GetRegion (Scope))
        and then Dictionary.TypeIsProtected (Dictionary.GetRegion (Scope));
   end In_Protected_Body;

   -----------------------------------------------

   procedure Do_Add
     (Add_Subprog, Add_Body, Hidden : in     Boolean;
      Ident_Str                     : in     LexTokenManager.Lex_String;
      Node                          : in     STree.SyntaxNode;
      First_Seen                    : in out Boolean;
      Scope                         : in out Dictionary.Scopes;
      Subprog_Sym                   : in out Dictionary.Symbol)
   --# global in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict   from *,
   --#                                Add_Body,
   --#                                Add_Subprog,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Hidden,
   --#                                Ident_Str,
   --#                                Node,
   --#                                Scope,
   --#                                STree.Table,
   --#                                Subprog_Sym &
   --#         First_Seen        from *,
   --#                                Add_Subprog &
   --#         Scope             from *,
   --#                                Add_Body,
   --#                                Add_Subprog,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Dictionary.Dict,
   --#                                Node,
   --#                                STree.Table,
   --#                                Subprog_Sym &
   --#         SPARK_IO.File_Sys from *,
   --#                                Add_Body,
   --#                                Add_Subprog,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Dictionary.Dict,
   --#                                Hidden,
   --#                                Ident_Str,
   --#                                LexTokenManager.State,
   --#                                Node,
   --#                                Scope,
   --#                                STree.Table,
   --#                                Subprog_Sym &
   --#         Subprog_Sym       from *,
   --#                                Add_Subprog,
   --#                                ContextManager.Ops.Unit_Stack,
   --#                                Dictionary.Dict,
   --#                                Node,
   --#                                STree.Table;
   is
   begin
      if Add_Subprog then
         Dictionary.AddSubprogram
           (Name          => Ident_Str,
            Comp_Unit     => ContextManager.Ops.Current_Unit,
            Specification => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                  End_Position   => Node_Position (Node => Node)),
            Scope         => Scope,
            Context       => Dictionary.ProgramContext,
            Subprogram    => Subprog_Sym);
      else
         First_Seen := False;
      end if;

      if Add_Body then
         Dictionary.AddBody
           (CompilationUnit => Subprog_Sym,
            Comp_Unit       => ContextManager.Ops.Current_Unit,
            TheBody         => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                    End_Position   => Node_Position (Node => Node)),
            Hidden          => Hidden);
         Scope := Dictionary.LocalScope (Subprog_Sym);
      else
         Dictionary.AddBodyStub
           (CompilationUnit => Subprog_Sym,
            Comp_Unit       => ContextManager.Ops.Current_Unit,
            BodyStub        => Dictionary.Location'(Start_Position => Node_Position (Node => Node),
                                                    End_Position   => Node_Position (Node => Node)));
      end if;
   end Do_Add;

   -----------------------------------------------

   procedure Check_For_Child (Ident_Node : in STree.SyntaxNode;
                              Scope      : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
   begin
      if Dictionary.IsPackage (Dictionary.GetRegion (Scope))
        and then not Dictionary.IsEmbeddedPackage (Dictionary.GetRegion (Scope))
        and then Dictionary.LookupSelectedItem
        (Prefix   => Dictionary.GetRegion (Scope),
         Selector => Node_Lex_String (Node => Ident_Node),
         Scope    => Dictionary.GlobalScope,
         Context  => Dictionary.ProofContext) /=
        Dictionary.NullSymbol then
         -- name exists as child
         ErrorHandler.Semantic_Error
           (Err_Num   => 10,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Node_Lex_String (Node => Ident_Node));
      end if;
   end Check_For_Child;

   -----------------------------------------------

   function Declared_In_Same_Or_Related_Scope (Sym   : Dictionary.Symbol;
                                               Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
      -- return true if Sym is declared in Scope or in the visible/private scope of the region
      -- associate with Scope
   begin
      return Dictionary.GetScope (Sym) = Scope
        or else Dictionary.GetScope (Sym) = Dictionary.VisibleScope (Dictionary.GetRegion (Scope))
        or else Dictionary.GetScope (Sym) = Dictionary.PrivateScope (Dictionary.GetRegion (Scope));
   end Declared_In_Same_Or_Related_Scope;

begin -- Wf_Procedure_Specification

   -- ASSUME Node = procedure_specification
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.procedure_specification,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = procedure_specification in Wf_Procedure_Specification");
   Ident_Node := Child_Node (Current_Node => Child_Node (Current_Node => Node));
   -- ASSUME Ident_Node = identifier
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Ident_Node = identifier in Wf_Procedure_Specification");
   Ident_Str          := Node_Lex_String (Node => Ident_Node);
   Grand_Parent       := Syntax_Node_Type (Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node)));
   Great_Grand_Parent :=
     Syntax_Node_Type (Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node))));
   Adding_Proper_Body := Great_Grand_Parent = SPSymbols.abody or else In_Protected_Body (Scope => Scope);
   -- in prot bod we can't be adding a stub
   First_Seen  := True; -- default value in case all checks below fail
   Subprog_Sym := Dictionary.LookupItem (Name              => Ident_Str,
                                         Scope             => Scope,
                                         Context           => Dictionary.ProofContext,
                                         Full_Package_Name => False);

   if Subprog_Sym = Dictionary.NullSymbol then
      if Syntax_Node_Type (Node => Parent_Node (Current_Node => Node)) = SPSymbols.body_stub then
         Check_For_Child (Ident_Node => Ident_Node,
                          Scope      => Scope);
         Do_Add
           (Add_Subprog => True,
            Add_Body    => False,
            Hidden      => Hidden,
            Ident_Str   => Ident_Str,
            Node        => Node,
            First_Seen  => First_Seen,
            Scope       => Scope,
            Subprog_Sym => Subprog_Sym);
      elsif Grand_Parent = SPSymbols.main_program_declaration then
         Do_Add
           (Add_Subprog => True,
            Add_Body    => True,
            Hidden      => Hidden,
            Ident_Str   => Ident_Str,
            Node        => Node,
            First_Seen  => First_Seen,
            Scope       => Scope,
            Subprog_Sym => Subprog_Sym);
      elsif Great_Grand_Parent /= SPSymbols.subunit then
         Do_Add
           (Add_Subprog => True,
            Add_Body    => True,
            Hidden      => Hidden,
            Ident_Str   => Ident_Str,
            Node        => Node,
            First_Seen  => First_Seen,
            Scope       => Scope,
            Subprog_Sym => Subprog_Sym);
      else -- no stub for subunit
         Subprog_Sym := Dictionary.NullSymbol;
         ErrorHandler.Semantic_Error
           (Err_Num   => 15,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => Ident_Node),
            Id_Str    => Ident_Str);
      end if;

   else -- symbol found so further checks needed
      if Great_Grand_Parent = SPSymbols.subunit then
         if Dictionary.IsProcedure (Subprog_Sym)
           and then Dictionary.HasBodyStub (Subprog_Sym)
           and then not Dictionary.HasBody (Subprog_Sym) then
            STree.Set_Node_Lex_String (Sym  => Subprog_Sym,
                                       Node => Ident_Node);
            Do_Add
              (Add_Subprog => False,
               Add_Body    => True,
               Hidden      => Hidden,
               Ident_Str   => Ident_Str,
               Node        => Node,
               First_Seen  => First_Seen,
               Scope       => Scope,
               Subprog_Sym => Subprog_Sym);
         else
            Subprog_Sym := Dictionary.NullSymbol;
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => Ident_Str);
         end if;
      elsif (In_Package_Body (Scope => Scope)
               or else -- a place where completion of declaration may be allowed
               In_Protected_Body (Scope => Scope))
        and then -- another place where completion of declaration may be allowed

        --  check that we are in a place where the the declaration can
        --  be legally completed (i.e. if subprog declared in a
        --  package spec it can only be completed in the package body
        --  (ditto protected type/body)
        Declared_In_Same_Or_Related_Scope (Sym   => Subprog_Sym,
                                           Scope => Scope) then

         First_Sym_Found := Subprog_Sym;
         Subprog_Sym     :=
           Dictionary.LookupImmediateScope
           (Name    => Ident_Str,
            Scope   => Dictionary.VisibleScope (Dictionary.GetRegion (Scope)),
            Context => Dictionary.ProgramContext);
         -- Above looked for declaration in spec vis part, if not found, try again in private part
         if Subprog_Sym = Dictionary.NullSymbol and then Dictionary.IsPackage (Dictionary.GetRegion (Scope)) then
            Subprog_Sym :=
              Dictionary.LookupImmediateScope
              (Name    => Ident_Str,
               Scope   => Dictionary.PrivateScope (Dictionary.GetRegion (Scope)),
               Context => Dictionary.ProgramContext);
         end if;

         if Subprog_Sym = Dictionary.NullSymbol then -- something definitely wrong
            if not Dictionary.IsSubprogram (First_Sym_Found) then
               --name in use for something other than a subprogram
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => Ident_Str);
               -- add anyway to prevent scope problems later
               Do_Add
                 (Add_Subprog => True,
                  Add_Body    => Adding_Proper_Body,
                  Hidden      => Hidden,
                  Ident_Str   => Ident_Str,
                  Node        => Node,
                  First_Seen  => First_Seen,
                  Scope       => Scope,
                  Subprog_Sym => Subprog_Sym);
            else -- it is a subprogram which must be a duplicate
               ErrorHandler.Semantic_Error
                 (Err_Num   => 13,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => Ident_Str);
               if Adding_Proper_Body then
                  if Dictionary.HasBody (First_Sym_Found) then
                     -- add complete duplicate procedure to dict
                     Do_Add
                       (Add_Subprog => True,
                        Add_Body    => True,
                        Hidden      => Hidden,
                        Ident_Str   => Ident_Str,
                        Node        => Node,
                        First_Seen  => First_Seen,
                        Scope       => Scope,
                        Subprog_Sym => Subprog_Sym);
                  else
                     -- add body to duplicate subprogram stub in dict
                     Subprog_Sym := First_Sym_Found;
                     Do_Add
                       (Add_Subprog => False,
                        Add_Body    => True,
                        Hidden      => Hidden,
                        Ident_Str   => Ident_Str,
                        Node        => Node,
                        First_Seen  => First_Seen,
                        Scope       => Scope,
                        Subprog_Sym => Subprog_Sym);
                  end if;
               end if;
            end if;
         else -- Subprog_Sym was found in package's visible part
            if not Dictionary.IsProcedure (First_Sym_Found) then
               -- name in use for something other than a procedure
               ErrorHandler.Semantic_Error
                 (Err_Num   => 10,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => Ident_Str);
               -- add anyway to prevent scope problems later
               Do_Add
                 (Add_Subprog => True,
                  Add_Body    => Adding_Proper_Body,
                  Hidden      => Hidden,
                  Ident_Str   => Ident_Str,
                  Node        => Node,
                  First_Seen  => First_Seen,
                  Scope       => Scope,
                  Subprog_Sym => Subprog_Sym);
            else -- it is a procedure which may be a duplicate
               if Dictionary.HasBody (Subprog_Sym) then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 13,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
                  if Adding_Proper_Body then
                     -- add complete duplicate procedure to dict
                     Do_Add
                       (Add_Subprog => True,
                        Add_Body    => True,
                        Hidden      => Hidden,
                        Ident_Str   => Ident_Str,
                        Node        => Node,
                        First_Seen  => First_Seen,
                        Scope       => Scope,
                        Subprog_Sym => Subprog_Sym);
                  end if;
               elsif Dictionary.HasBodyStub (Subprog_Sym) then
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 13,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => Ident_Str);
                  if Adding_Proper_Body then
                     -- add body to duplicate procedure stub in dict
                     Do_Add
                       (Add_Subprog => False,
                        Add_Body    => True,
                        Hidden      => Hidden,
                        Ident_Str   => Ident_Str,
                        Node        => Node,
                        First_Seen  => First_Seen,
                        Scope       => Scope,
                        Subprog_Sym => Subprog_Sym);
                  end if;
               else -- the non-error case of pre-declaration of procedure
                  STree.Set_Node_Lex_String (Sym  => Subprog_Sym,
                                             Node => Ident_Node);
                  Do_Add
                    (Add_Subprog => False,
                     Add_Body    => Adding_Proper_Body,
                     Hidden      => Hidden,
                     Ident_Str   => Ident_Str,
                     Node        => Node,
                     First_Seen  => First_Seen,
                     Scope       => Scope,
                     Subprog_Sym => Subprog_Sym);
               end if;
            end if;
         end if;
      else -- not in a package so duplicate is definitely error
         if Dictionary.IsSubprogram (Subprog_Sym)
           and then not Dictionary.IsImplicitProofFunction (Subprog_Sym)
           and then Dictionary.HasBody (Subprog_Sym) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 13,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => Ident_Str);
         else
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => Ident_Str);
         end if;
         if Adding_Proper_Body then
            Do_Add
              (Add_Subprog => True,
               Add_Body    => True,
               Hidden      => Hidden,
               Ident_Str   => Ident_Str,
               Node        => Node,
               First_Seen  => First_Seen,
               Scope       => Scope,
               Subprog_Sym => Subprog_Sym);
         else
            Subprog_Sym := Dictionary.NullSymbol;
         end if;
      end if;
   end if;

end Wf_Procedure_Specification;
