-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure wf_positional_argument_association
  (Node          : in out STree.SyntaxNode;
   Scope         : in     Dictionary.Scopes;
   EStack        : in out ExpStack.ExpStackType;
   IsAnnotation  : in     Boolean;
   RefVar        : in     SeqAlgebra.Seq;
   ComponentData : in out ComponentManager.ComponentData) is
   NAME_ARG_LOOKUP : constant Annotation_Symbol_Table :=
     Annotation_Symbol_Table'(False => SPSymbols.name_argument_list,
                              True  => SPSymbols.annotation_name_argument_list);

   EXP_LOOKUP : constant Annotation_Symbol_Table :=
     Annotation_Symbol_Table'(False => SPSymbols.expression,
                              True  => SPSymbols.annotation_expression);

   ExpResult, TypeInfo                  : Exp_Record;
   ExpectedType, VCGtype, Sym, ParamSym : Dictionary.Symbol;
   TypesAreConvertable                  : Boolean;
   ExpValue, UnusedValue                : Maths.Value;
   ErrorFound                           : Boolean := False;

   --------------------------------------------------------------

   procedure ChainUpToNameList (Node : in out STree.SyntaxNode)
   --# global in IsAnnotation;
   --#        in STree.Table;
   --# derives Node from *,
   --#                   IsAnnotation,
   --#                   STree.Table;
   is
   begin
      while Syntax_Node_Type (Node => Node) /= NAME_ARG_LOOKUP (IsAnnotation) loop
         Node := Parent_Node (Current_Node => Node);
      end loop;
   end ChainUpToNameList;

   --------------------------------------------------------------

   function ExpressionLocation (Node         : STree.SyntaxNode;
                                IsAnnotation : Boolean) return STree.SyntaxNode
   --# global in STree.Table;
   is
      ExpLoc : STree.SyntaxNode;

   begin
      if Syntax_Node_Type (Node => Child_Node (Node)) = EXP_LOOKUP (IsAnnotation) then
         ExpLoc := Node;
      else
         ExpLoc := Next_Sibling (Child_Node (Node));
      end if;
      return ExpLoc;
   end ExpressionLocation;

   --------------------------------------------------------------

   procedure CheckTypesAreConvertable (Node           : in     STree.SyntaxNode;
                                       Target, Source : in     Dictionary.Symbol;
                                       Ok             :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     Scope;
   --#        in     STree.Table;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         Source,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Target &
   --#         Ok                         from Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         Source,
   --#                                         Target;
   --#
   is
      OkLocal   : Boolean;
      Undefined : Boolean := False;

      ---------------------------------------

      function DimensionsMatch (Target, Source : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.GetNumberOfDimensions (Target) = Dictionary.GetNumberOfDimensions (Source);
      end DimensionsMatch;

      ---------------------------------------

      function IndexesAreConvertible (Target, Source : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
         TgtIt, SrcIt : Dictionary.Iterator;
         Ok           : Boolean;

         function Convertible (Src, Tgt : Dictionary.Symbol) return Boolean
         --# global in Dictionary.Dict;
         is
         begin
            return (Dictionary.TypeIsNumeric (Src) and then Dictionary.TypeIsNumeric (Tgt))
              or else (Dictionary.GetRootType (Src) = Dictionary.GetRootType (Tgt));

         end Convertible;

      begin --IndexesAreConvertible
         Ok    := True;
         TgtIt := Dictionary.FirstArrayIndex (Target);
         SrcIt := Dictionary.FirstArrayIndex (Source);
         while not Dictionary.IsNullIterator (TgtIt) loop

            if not Convertible (Dictionary.CurrentSymbol (SrcIt), Dictionary.CurrentSymbol (TgtIt)) then
               Ok := False;
               exit;
            end if;
            TgtIt := Dictionary.NextSymbol (TgtIt);
            SrcIt := Dictionary.NextSymbol (SrcIt);
         end loop;
         return Ok;
      end IndexesAreConvertible;

      ---------------------------------------

      function ComponentsSameType (Target, Source : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      is
      begin
         return Dictionary.GetRootType (Dictionary.GetArrayComponent (Target)) =
           Dictionary.GetRootType (Dictionary.GetArrayComponent (Source));
      end ComponentsSameType;

      ---------------------------------------

      function ComponentsConstraintsMatch (Target, Source : Dictionary.Symbol) return Boolean
      --# global in Dictionary.Dict;
      --#        in LexTokenManager.State;
      is
         TgtComponent, SrcComponent : Dictionary.Symbol;
         Result                     : Boolean;

         function ScalarBoundsMatch (SrcSym, TgtSym : Dictionary.Symbol) return Boolean
         --# global in Dictionary.Dict;
         --#        in LexTokenManager.State;
         is
         begin
            return LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => Dictionary.GetScalarAttributeValue (False, LexTokenManager.First_Token, SrcSym),
               Lex_Str2 => Dictionary.GetScalarAttributeValue (False, LexTokenManager.First_Token, TgtSym)) =
              LexTokenManager.Str_Eq
              and then LexTokenManager.Lex_String_Case_Insensitive_Compare
              (Lex_Str1 => Dictionary.GetScalarAttributeValue (False, LexTokenManager.Last_Token, SrcSym),
               Lex_Str2 => Dictionary.GetScalarAttributeValue (False, LexTokenManager.Last_Token, TgtSym)) =
              LexTokenManager.Str_Eq;
         end ScalarBoundsMatch;

      begin --ComponentsConstraintsMatch
         TgtComponent := Dictionary.GetArrayComponent (Target);
         SrcComponent := Dictionary.GetArrayComponent (Source);
         if Dictionary.TypeIsScalar (TgtComponent) then
            Result := ScalarBoundsMatch (TgtComponent, SrcComponent);
         elsif Dictionary.TypeIsArray (TgtComponent) then
            Result := IndexesMatch (TgtComponent, SrcComponent);
         elsif Dictionary.TypeIsRecord (TgtComponent) then
            Result := True;
         else
            Result := False; --unexpected case, above should trap all components
         end if;
         return Result;
      end ComponentsConstraintsMatch;

      ---------------------------------------

   begin --CheckTypesAreConvertable
         -- UnknownTypes considered convertable to stop error propagation
      if Dictionary.IsUnknownTypeMark (Target) or Dictionary.IsUnknownTypeMark (Source) then
         OkLocal   := True;
         Undefined := True;

      elsif (not IsAnnotation)
        and then (Dictionary.IsPrivateType (Source, Scope) or else Dictionary.IsPrivateType (Target, Scope))
        and then Target /= Source then
         OkLocal := False;

      elsif Dictionary.TypeIsNumeric (Target) and then Dictionary.TypeIsNumeric (Source) then
         OkLocal := True;
      elsif Dictionary.TypeIsArray (Target) and then Dictionary.TypeIsArray (Source) then
         OkLocal := True;
         if not DimensionsMatch (Target, Source) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 423,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
            OkLocal := False;
         elsif not IndexesAreConvertible (Target, Source) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 420,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
            OkLocal := False;
         elsif not IndexesMatch (Target, Source) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 418,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
            OkLocal := False;
         elsif not ComponentsSameType (Target, Source) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 421,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
            OkLocal := False;
         elsif not ComponentsConstraintsMatch (Target, Source) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 422,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
            OkLocal := False;
         end if;

         -- allow unnecessary conversions, warning will be produced by later if clause
      elsif Dictionary.GetRootType (Target) = Dictionary.GetRootType (Source) then
         OkLocal := True;
      else
         OkLocal := False;
      end if;

      --if legal (other than undefined case, check if necessary)
      if OkLocal and then not Undefined and then Dictionary.GetRootType (Target) = Dictionary.GetRootType (Source) then
         ErrorHandler.Semantic_Warning
           (Err_Num  => 309,
            Position => Node_Position
              (Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node)))),
            Id_Str   => LexTokenManager.Null_String);
      end if;
      Ok := OkLocal;
   end CheckTypesAreConvertable;

   --------------------------------------------------------------

   procedure DoTaggedTypeConversion (Node   : in     STree.SyntaxNode;
                                     Target : in out Exp_Record;
                                     Source : in     Exp_Record)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     RefVar;
   --#        in     STree.Table;
   --#        in out ComponentData;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out TheHeap;
   --# derives ComponentData,
   --#         Dictionary.Dict            from ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         Source,
   --#                                         Target,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Source,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         Target &
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         RefVar,
   --#                                         Source,
   --#                                         Target,
   --#                                         TheHeap &
   --#         Target                     from *,
   --#                                         CommandLineData.Content,
   --#                                         ComponentData,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         IsAnnotation,
   --#                                         Source,
   --#                                         TheHeap;
   is

      procedure RaiseError (ErrNo : in Natural)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in     LexTokenManager.State;
      --#        in     Node;
      --#        in     STree.Table;
      --#        in out ErrorHandler.Error_Context;
      --#        in out SPARK_IO.File_Sys;
      --#           out Target;
      --# derives ErrorHandler.Error_Context,
      --#         SPARK_IO.File_Sys          from CommandLineData.Content,
      --#                                         Dictionary.Dict,
      --#                                         ErrNo,
      --#                                         ErrorHandler.Error_Context,
      --#                                         LexTokenManager.State,
      --#                                         Node,
      --#                                         SPARK_IO.File_Sys,
      --#                                         STree.Table &
      --#         Target                     from Dictionary.Dict;
      is
      begin
         Target := UnknownTypeRecord;
         ErrorHandler.Semantic_Error
           (Err_Num   => ErrNo,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position
              (Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node)))),
            Id_Str    => LexTokenManager.Null_String);
      end RaiseError;

   begin -- DoTaggedTypeConversion
         -- On entry we know Target.TypeSymbol is tagged.  If Source.TypeSymbol is not then we have some grossly
         -- malformed type conversion
      if not Dictionary.TypeIsTagged (Source.Type_Symbol) then
         RaiseError (32);

      elsif not Dictionary.IsAnExtensionOf (Target.Type_Symbol, Source.Type_Symbol) then
         RaiseError (831);

      else
         -- We have two tagged types and the target is an ancestor of the source; basically ok
         if Source.Is_AVariable or Source.Is_Constant then
            -- we have an object to convert
            if IsAnnotation or else Source.Is_Constant then
               -- In an annotation, or for a constant, all we need to is change
               -- the result type to that expected.
               Target.Sort        := Type_Result;
               Target.Is_Constant := Source.Is_Constant;

               Target.Is_Static := Source.Is_Static and CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83;

               Target.Other_Symbol          := Dictionary.NullSymbol;
               Target.Variable_Symbol       := Source.Variable_Symbol;
               Target.Is_AVariable          := Source.Is_AVariable;
               Target.Is_An_Entire_Variable := False;

            else
               -- In a normal expression with a variable and we need to convert
               -- appropriate record subcomponent symbols.

               -- We can't replace X with X.Inherit unless we add X's subcomponents first
               AddRecordSubComponents
                 (RecordVarSym  => Source.Variable_Symbol,
                  RecordTypeSym => Dictionary.GetType (Source.Variable_Symbol),
                  ComponentData => ComponentData);
               -- Set up Exp_Record
               Target.Variable_Symbol := ConvertTaggedActual (Source.Variable_Symbol, Target.Type_Symbol);
               Target.Sort            := Type_Result;
               Target.Is_Constant     := Source.Is_Constant;

               Target.Is_Static := Source.Is_Static and CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83;

               Target.Other_Symbol          := Dictionary.NullSymbol;
               Target.Is_AVariable          := Source.Is_AVariable;
               Target.Is_An_Entire_Variable := False;

               -- Substitute reference variables to show we only used a subcomponent of Source.
               -- We have to look for the source variable because there may be other items in
               -- the RefVar list if, for example, the type conversion forms part of a larger
               -- expression such as a function call.
               SeqAlgebra.RemoveMember (TheHeap, RefVar, Natural (Dictionary.SymbolRef (Source.Variable_Symbol)));
               SeqAlgebra.AddMember (TheHeap, RefVar, Natural (Dictionary.SymbolRef (Target.Variable_Symbol)));

            end if;

         else -- not an object
            RaiseError (832);
         end if;
      end if;
   end DoTaggedTypeConversion;

   --------------------------------------------------------------

   function ConvertValue (Target : Dictionary.Symbol;
                          Exp    : Exp_Record) return Maths.Value
   --# global in CommandLineData.Content;
   --#        in Dictionary.Dict;
   is
      Val : Maths.Value;
   begin
      Val := Exp.Value;
      if not Maths.HasNoValue (Val) then
         if Dictionary.IsUnknownTypeMark (Target) then
            Val := Maths.NoValue;
         elsif Dictionary.TypeIsReal (Target) then
            Maths.ConvertToReal (Val);
         elsif Dictionary.TypeIsInteger (Target) and then Dictionary.TypeIsReal (Exp.Type_Symbol) then
            case CommandLineData.Content.Language_Profile is
               when CommandLineData.SPARK83 =>

                  Val := Maths.NoValue; -- can't do real to integer safely

               when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

                  Val := Maths.Ada95RealToInteger (Val);
            end case;
         end if;
      end if;
      return Val;
   end ConvertValue;

   --------------------------------------------------------------

   procedure RangeCheck (ARange : in Boolean;
                         Node   : in STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorFound;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                 from *,
   --#                                         ARange &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from ARange,
   --#                                         CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
   is
   begin
      if ARange then
         ErrorFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 341,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ExpressionLocation (Node, IsAnnotation)),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end RangeCheck;

   --------------------------------------------------------------

   function GetExpectedArrayIndexType
     (VarSym    : Dictionary.Symbol;
      TypeSym   : Dictionary.Symbol;
      Dimension : Positive)
     return      Dictionary.Symbol
   --# global in Dictionary.Dict;
   is
      Result : Dictionary.Symbol;
   begin
      -- This function determines what type to plant in the syntax tree so that the VCG can check
      -- that array accesses are in bounds.  FOr a constrained object it is edy - we plant the
      -- appropriate index type for the dimension being accessed.  For indexing into unconstrained
      -- objects we plant a symbol of a special kind (ParameterConstraintSymbol) associated with
      -- the array object (rather than its type); this special symbol represents "the index as
      -- constrained by 'something' at this point".  Typically we will no know the actual bounds
      -- of the constraint represented by this symbol.

      if Dictionary.IsUnconstrainedArrayType (TypeSym) then
         -- For unconstrained arrays, obtain the implcitly declared constraint symbol for the array object
         Result := Dictionary.GetSubprogramParameterConstraint (VarSym, Dimension);
      else
         -- For constrained arrays then obtain appropriate index for the array type; this is what the VCG needs
         Result := Dictionary.GetArrayIndex (TypeSym, Dimension);
      end if;
      return Result;
   end GetExpectedArrayIndexType;

   --------------------------------------------------------------

begin --wf_positional_argument_association
   ExpStack.Pop (ExpResult, EStack);
   ExpStack.Pop (TypeInfo, EStack);
   Sym := TypeInfo.Other_Symbol;

   case TypeInfo.Sort is
      when Is_Type_Mark =>
         -- seed syntax tree with type for run-time check
         STree.AddNodeSymbol (Node, ExpResult.Type_Symbol);

         if Dictionary.IsUnconstrainedArrayType (TypeInfo.Type_Symbol) then
            ErrorHandler.Semantic_Error
              (Err_Num   => 39,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position
                 (Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node)))),
               Id_Str    => LexTokenManager.Null_String);
            TypeInfo := UnknownTypeRecord;

            -- special handling for type conversion of string literals.
         elsif Dictionary.IsPredefinedStringType (ExpResult.Type_Symbol) and then ExpResult.Range_RHS /= Maths.NoValue then
            ErrorHandler.Semantic_Error
              (Err_Num   => 425,
               Reference => 22,
               Position  => Node_Position (Node => Node),
               Id_Str    => LexTokenManager.Null_String);
            TypeInfo := UnknownTypeRecord;

         elsif Dictionary.TypeIsTagged (TypeInfo.Type_Symbol) then
            DoTaggedTypeConversion (Node   => Node,
                                    Target => TypeInfo,
                                    Source => ExpResult);

         else -- some "normal" conversion case
            if ExpResult.Is_ARange then
               -- Type conversion of a range is illegal.  This also
               -- catches the illegal case of type-conversion of a
               -- subtype mark, such as Integer (Natural)
               TypeInfo := UnknownTypeRecord;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 114,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Parent_Node (Current_Node => Node)),
                  Id_Str    => LexTokenManager.Null_String);

            else
               CheckTypesAreConvertable (Node, TypeInfo.Type_Symbol, ExpResult.Type_Symbol,
                                         -- to get
                                         TypesAreConvertable);
               if TypesAreConvertable then
                  ConstraintCheck (ExpResult.Value, ExpValue, IsAnnotation, TypeInfo.Type_Symbol, Node_Position (Node => Node));
                  ExpResult.Value      := ExpValue;
                  TypeInfo.Sort        := Type_Result;
                  TypeInfo.Is_Constant := ExpResult.Is_Constant;

                  TypeInfo.Is_Static := ExpResult.Is_Static and
                    CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83;

                  TypeInfo.Other_Symbol          := Dictionary.NullSymbol;
                  TypeInfo.Value                 := ConvertValue (TypeInfo.Type_Symbol, ExpResult);
                  TypeInfo.Variable_Symbol       := ExpResult.Variable_Symbol;
                  TypeInfo.Is_AVariable          := False;
                  TypeInfo.Is_An_Entire_Variable := False;
               else
                  TypeInfo := UnknownTypeRecord;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 32,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position
                       (Node =>
                          Parent_Node (Current_Node => Parent_Node (Current_Node => Parent_Node (Current_Node => Node)))),
                     Id_Str    => LexTokenManager.Null_String);
               end if;
            end if;
         end if;

      when Is_Function =>
         if TypeInfo.Param_Count = Dictionary.GetNumberOfSubprogramParameters (Sym) then
            TypeInfo := UnknownSymbolRecord;
            ErrorHandler.Semantic_Error
              (Err_Num   => 3,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Dictionary.GetSimpleName (Sym));
            ChainUpToNameList (Node);
         else
            TypeInfo.Param_Count := TypeInfo.Param_Count + 1;
            ParamSym             := Dictionary.GetSubprogramParameter (Sym, TypeInfo.Param_Count);
            ExpectedType         := Dictionary.GetType (ParamSym);
            -- Seed syntax tree with expected type for run-time check;
            -- but, don't do this for instantiation of unchecked_conversion
            -- because we don't want any RTCs for association of those parameters
            -- (provided the function parameter subtype and actual subtype match)
            if not (Dictionary.IsAnUncheckedConversion (Sym) and then ExpResult.Type_Symbol = ExpectedType) then
               STree.AddNodeSymbol (Node, ExpectedType);
            end if;

            -- There is a special case involving functions an stream variables.  We allow a stream
            -- variable to be a parameter to an Unchecked_Conversion but need to ensure that
            -- the function inherits the restrictions associated with referencing a stream
            -- (e.g. cannot be used in general expression).  We can do this here by checking
            -- the StreamSymbol of the parameter expression (there will only be one if we are
            -- talking about an unchecked conversion) and if it is non-null then setting the
            -- stream symbol of the function result record (now an object) to the function symbol.
            -- Note that this clause will only be executed for an unchecked conversion because
            -- a parameter which is a stream would hav ebeen rejected at wf_primary in all other
            -- cases
            if ExpResult.Stream_Symbol /= Dictionary.NullSymbol then
               TypeInfo.Stream_Symbol := Sym;
            end if;

            RangeCheck (ExpResult.Is_ARange, Node);

            -- function is deemed constant if it is predefined and all its parameters
            -- are constant.
            TypeInfo.Is_Constant := TypeInfo.Is_Constant and ExpResult.Is_Constant;

            if (TypeInfo.Tagged_Parameter_Symbol = ExpResult.Type_Symbol
                  or else (TypeInfo.Tagged_Parameter_Symbol = Dictionary.NullSymbol
                             and then Dictionary.CompatibleTypes (Scope, ExpectedType, -- always defined here
                                                                  ExpResult.Type_Symbol))
                  or else (not Dictionary.IsAnExtensionOf (ExpResult.Type_Symbol, TypeInfo.Tagged_Parameter_Symbol)
                             and then Dictionary.CompatibleTypes (Scope, ExpectedType, -- always defined here
                                                                  ExpResult.Type_Symbol))) then
               TaggedActualMustBeObjectCheck
                 (NodePos         => Node_Position (Node => Node),
                  FormalType      => ExpectedType,
                  ActualType      => ExpResult.Type_Symbol,
                  ControllingType => Dictionary.GetSubprogramControllingType (Sym),
                  IsAVariable     => ExpResult.Is_AVariable,
                  IsAConstant     => ExpResult.Is_Constant,
                  ErrorFound      => ErrorFound);
               -- Following call will deal with scalar value constraint checking
               --# accept Flow, 10, UnusedValue, "Expected ineffective assignment";
               ConstraintCheck
                 (ExpResult.Value,
                  UnusedValue,
                  IsAnnotation,
                  ExpectedType,
                  Node_Position (Node => ExpressionLocation (Node, IsAnnotation)));
               --# end accept;
               -- Check array bounds etc.
               if Dictionary.TypeIsArray (Dictionary.GetType (ParamSym))
                 and then not Dictionary.IsUnconstrainedArrayType (ExpectedType) then
                  -- Formal is a constrained subtype of an unconstrained array

                  if Dictionary.IsUnconstrainedArrayType (ExpResult.Type_Symbol) then
                     -- Actual is unconstrained.  In SPARK95 or 2005, this is OK if
                     -- the actual is a static String expression, but illegal
                     -- otherwise.
                     if CommandLineData.Content.Language_Profile /= CommandLineData.SPARK83
                       and then Dictionary.IsPredefinedStringType (ExpResult.Type_Symbol) then
                        -- Formal must be a constrained String subtype, so we need
                        -- to check the upper bound of the actual against the expected
                        -- upper bound of the formal.
                        if ExpResult.Range_RHS = Maths.NoValue then
                           -- Actual is not static, so must be illegal
                           ErrorHandler.Semantic_Error
                             (Err_Num   => 39,
                              Reference => ErrorHandler.No_Reference,
                              Position  => Node_Position (Node => ExpressionLocation (Node, IsAnnotation)),
                              Id_Str    => LexTokenManager.Null_String);
                        else
                           -- Actual is static, so check upper-bound against that expected
                           if ExpResult.Range_RHS /=
                             Maths.ValueRep
                             (Dictionary.GetScalarAttributeValue
                                (False,
                                 LexTokenManager.Last_Token,
                                 Dictionary.GetType (ParamSym))) then
                              ErrorHandler.Semantic_Error
                                (Err_Num   => 418,
                                 Reference => ErrorHandler.No_Reference,
                                 Position  => Node_Position (Node => ExpressionLocation (Node, IsAnnotation)),
                                 Id_Str    => LexTokenManager.Null_String);

                           end if;
                        end if;

                     else
                        -- SPARK83 or not a String type, so illegal
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 39,
                           Reference => ErrorHandler.No_Reference,
                           Position  => Node_Position (Node => ExpressionLocation (Node, IsAnnotation)),
                           Id_Str    => LexTokenManager.Null_String);
                     end if;

                  elsif Illegal_Unconstrained (ExpResult.Type_Symbol, ExpectedType) then
                     -- Although both formal and actual are constrained their bounds don't match
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 418,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => ExpressionLocation (Node, IsAnnotation)),
                        Id_Str    => LexTokenManager.Null_String);
                  end if;
               end if;

               -- To help the VCG with generating checks involving unconstrained formal parameters, we
               -- seed the syntax tree with a constraining type mark.  The positional_argument_association
               -- node is already used for RTC purposes, so we seed the expression node instead.
               if not IsAnnotation then
                  PlantConstrainingType
                    (ExpResult.Type_Symbol,
                     ExpResult.Range_RHS,
                     STree.ExpressionFromPositionalArgumentAssociation (Node));
               end if;
            else
               ErrorFound := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 38,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ExpressionLocation (Node, IsAnnotation)),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         end if;

      when Is_Object =>
         if TypeInfo.Param_Count = Dictionary.GetNumberOfDimensions (TypeInfo.Type_Symbol) then
            TypeInfo := UnknownSymbolRecord;
            ErrorHandler.Semantic_Error
              (Err_Num   => 93,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Node),
               Id_Str    => Dictionary.GetSimpleName (Sym));
            ChainUpToNameList (Node);
         else
            TypeInfo.Param_Count := TypeInfo.Param_Count + 1;

            -- Skip over any array subtypes if necessary.  If TypeInfo.TypeSymbol
            -- denotes an unconstrained array type, then it is left alone.
            TypeInfo.Type_Symbol := Dictionary.GetFirstConstrainedSubtype (TypeInfo.Type_Symbol);

            -- ExpectedType is used to ensure that indexing expression is well-typed
            ExpectedType := Dictionary.GetArrayIndex (TypeInfo.Type_Symbol, TypeInfo.Param_Count);

            -- VCGtype is used to tell VCG what indexing type to expect.  Same as Expected type for a
            -- constrained array but different for unconstrained.  See comment in function GetExpectedArrayIndexType
            VCGtype := GetExpectedArrayIndexType (TypeInfo.Other_Symbol, TypeInfo.Type_Symbol, TypeInfo.Param_Count);

            -- seed syntax tree with expected type for run-time check
            -- Debug.PrintSym ("Planting array index type ", VCGtype);
            STree.AddNodeSymbol (Node, VCGtype);

            RangeCheck (ExpResult.Is_ARange, Node);
            TypeInfo.Is_Constant := TypeInfo.Is_Constant and ExpResult.Is_Constant;
            if Dictionary.CompatibleTypes (Scope, ExpectedType, ExpResult.Type_Symbol) then
               --# accept Flow, 10, UnusedValue, "Expected ineffective assignment";
               ConstraintCheck
                 (ExpResult.Value,
                  UnusedValue,
                  IsAnnotation,
                  ExpectedType,
                  Node_Position (Node => ExpressionLocation (Node, IsAnnotation)));
               --# end accept;
            else
               ErrorFound := True;
               ErrorHandler.Semantic_Error
                 (Err_Num   => 38,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => ExpressionLocation (Node, IsAnnotation)),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         end if;

      when others =>
         null;
   end case;

   TypeInfo.Errors_In_Expression := ErrorFound or TypeInfo.Errors_In_Expression or ExpResult.Errors_In_Expression;

   ExpStack.Push (TypeInfo, EStack);

   --# accept Flow, 33, UnusedValue, "Expected to be neither referenced nor exported";
end wf_positional_argument_association;
