-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure Wf_Full_Type_Declaration (Node  : in STree.SyntaxNode;
                                    Scope : in Dictionary.Scopes) is
   Ident_Node : STree.SyntaxNode;
   IdentStr   : LexTokenManager.Lex_String;
   Sym        : Dictionary.Symbol;

   TypeDeclaredAsLimited : Boolean;
   TypeDeclaredAsTagged  : Boolean;
   Error_Found           : Boolean;

   -------------------------------

   function IsPrivateTypeResolution (Sym   : Dictionary.Symbol;
                                     Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return (not Dictionary.IsDeclared (Sym)
                and then not Dictionary.IsVisibleScope (Scope)
                and then Dictionary.IsType (Sym)
                and then Dictionary.TypeIsPrivate (Sym)
                and then (Dictionary.GetRegion (Scope) = Dictionary.GetRegion (Dictionary.GetScope (Sym))));
   end IsPrivateTypeResolution;

   -------------------------------

   procedure EmptyTypeCheck (DecLoc       : in     LexTokenManager.Token_Position;
                             Lower, Upper : in out LexTokenManager.Lex_String)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Lower,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Upper &
   --#         Lower,
   --#         Upper                      from LexTokenManager.State,
   --#                                         Lower,
   --#                                         Upper;
   is
      Unused       : Maths.ErrorCode;
      MathsResult  : Maths.Value;
      RangeIsEmpty : Boolean;
   begin
      if LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Lower,
                                                              Lex_Str2 => LexTokenManager.Null_String) /=
        LexTokenManager.Str_Eq
        and then LexTokenManager.Lex_String_Case_Insensitive_Compare (Lex_Str1 => Upper,
                                                                      Lex_Str2 => LexTokenManager.Null_String) /=
        LexTokenManager.Str_Eq then
         --# accept Flow, 10, Unused, "Expected ineffective assignment";
         Maths.Lesser (Maths.ValueRep (Upper), Maths.ValueRep (Lower),
                       --to get
                       MathsResult, Unused);  --not used because it can only be ok or type mismatch
         Maths.ValueToBool (MathsResult,
                            --to get
                            RangeIsEmpty, Unused);
         if RangeIsEmpty then
            Lower := LexTokenManager.Null_String;
            Upper := LexTokenManager.Null_String;
            ErrorHandler.Semantic_Error
              (Err_Num   => 416,
               Reference => ErrorHandler.No_Reference,
               Position  => DecLoc,
               Id_Str    => LexTokenManager.Null_String);
         end if;
      end if;
      --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
   end EmptyTypeCheck;

   -------------------------------

   procedure wf_integer
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      DecLoc     : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -------------------------------

   procedure wf_modular
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      DecLoc     : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -------------------------------

   procedure wf_enum
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      DecLoc     : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     STree.Table;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives Dictionary.Dict,
   --#         LexTokenManager.State      from ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Scope,
   --#                                         STree.Table;
      is separate;

   -------------------------------

   procedure Wf_Record
     (Node                        : in STree.SyntaxNode;
      Scope                       : in Dictionary.Scopes;
      Ident_Node                  : in STree.SyntaxNode;
      Dec_Loc                     : in LexTokenManager.Token_Position;
      Extends                     : in Dictionary.Symbol;
      Private_Type_Being_Resolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Extends,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Extends,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Extends,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   -------------------------------

   procedure wf_real
     (Node       : in STree.SyntaxNode;
      Scope      : in Dictionary.Scopes;
      Ident_Node : in STree.SyntaxNode;
      DecLoc     : in LexTokenManager.Token_Position)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --#           out AggregateStack.State;
   --# derives AggregateStack.State,
   --#         LexTokenManager.State,
   --#         STree.Table,
   --#         TheHeap                    from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict            from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Statistics.TableUsage      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -------------------------------

   procedure Wf_Type_Extension
     (Node                        : in STree.SyntaxNode;
      Scope                       : in Dictionary.Scopes;
      Ident_Node                  : in STree.SyntaxNode;
      Dec_Loc                     : in LexTokenManager.Token_Position;
      Private_Type_Being_Resolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in     LexTokenManager.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out STree.Table;
   --# derives Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         STree.Table &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         SLI.State                  from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dec_Loc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Private_Type_Being_Resolved,
   --#                                         Scope,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table;
      is separate;

   -------------------------------

   procedure Wf_Protected_Type_Declaration (Node  : in STree.SyntaxNode;
                                            Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out GlobalComponentData;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         GlobalComponentData,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         GlobalComponentData,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -------------------------------

   procedure Wf_Task_Type_Declaration (Node  : in STree.SyntaxNode;
                                       Scope : in Dictionary.Scopes)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Dictionary.Dict,
   --#         LexTokenManager.State,
   --#         Statistics.TableUsage,
   --#         STree.Table,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap;
      is separate;

   -------------------------------

   procedure wf_type_definition
     (Node                     : in STree.SyntaxNode;
      Scope                    : in Dictionary.Scopes;
      Ident_Node               : in STree.SyntaxNode;
      DecLoc                   : in LexTokenManager.Token_Position;
      PrivateTypeBeingResolved : in Dictionary.Symbol)
   --# global in     CommandLineData.Content;
   --#        in     ContextManager.Ops.File_Heap;
   --#        in     ContextManager.Ops.Unit_Heap;
   --#        in     ContextManager.Ops.Unit_Stack;
   --#        in out AggregateStack.State;
   --#        in out Dictionary.Dict;
   --#        in out ErrorHandler.Error_Context;
   --#        in out LexTokenManager.State;
   --#        in out SLI.State;
   --#        in out SPARK_IO.File_Sys;
   --#        in out Statistics.TableUsage;
   --#        in out STree.Table;
   --#        in out TheHeap;
   --# derives AggregateStack.State,
   --#         Statistics.TableUsage,
   --#         TheHeap                    from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         Dictionary.Dict,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         Dictionary.Dict,
   --#         STree.Table                from CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         PrivateTypeBeingResolved,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         ErrorHandler.Error_Context,
   --#         SLI.State,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         ContextManager.Ops.File_Heap,
   --#                                         ContextManager.Ops.Unit_Heap,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         Ident_Node,
   --#                                         LexTokenManager.State,
   --#                                         Node,
   --#                                         PrivateTypeBeingResolved,
   --#                                         Scope,
   --#                                         SLI.State,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table,
   --#                                         TheHeap &
   --#         LexTokenManager.State      from *,
   --#                                         CommandLineData.Content,
   --#                                         ContextManager.Ops.Unit_Stack,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         Ident_Node,
   --#                                         Node,
   --#                                         Scope,
   --#                                         STree.Table,
   --#                                         TheHeap;
   is
      Type_Node             : STree.SyntaxNode;
      UnusedArrayTypeSymbol : Dictionary.Symbol;
      UnusedErrorsFound     : Boolean;
   begin
      -- ASSUME Node = type_definition
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Node) = SPSymbols.type_definition,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Node = type_definition in Wf_Type_Definition");
      Type_Node := Child_Node (Current_Node => Node);
      -- ASSUME Ident_Node = identifier
      SystemErrors.RT_Assert
        (C       => Syntax_Node_Type (Node => Ident_Node) = SPSymbols.identifier,
         Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Ident_Node = identifier in Wf_Type_Definition");
      -- ASSUME Type_Node = enumeration_type_definition OR integer_type_definition OR real_type_definition OR
      --                    array_type_definition OR record_type_definition OR modular_type_definition OR type_extension
      case Syntax_Node_Type (Node => Type_Node) is
         when SPSymbols.enumeration_type_definition =>
            -- ASSUME Type_Node = enumeration_type_definition
            wf_enum (Node       => Type_Node,
                     Scope      => Scope,
                     Ident_Node => Ident_Node,
                     DecLoc     => DecLoc);
         when SPSymbols.real_type_definition =>
            -- ASSUME Type_Node = real_type_definition
            wf_real (Node       => Type_Node,
                     Scope      => Scope,
                     Ident_Node => Ident_Node,
                     DecLoc     => DecLoc);
         when SPSymbols.record_type_definition =>
            -- ASSUME Type_Node = record_type_definition
            Wf_Record
              (Node                        => Type_Node,
               Scope                       => Scope,
               Ident_Node                  => Ident_Node,
               Dec_Loc                     => DecLoc,
               Extends                     => Dictionary.NullSymbol,
               Private_Type_Being_Resolved => PrivateTypeBeingResolved);
         when SPSymbols.integer_type_definition =>
            -- ASSUME Type_Node = integer_type_definition
            wf_integer (Node       => Type_Node,
                        Scope      => Scope,
                        Ident_Node => Ident_Node,
                        DecLoc     => DecLoc);
         when SPSymbols.modular_type_definition =>
            -- ASSUME Type_Node = modular_type_definition
            wf_modular (Node       => Type_Node,
                        Scope      => Scope,
                        Ident_Node => Ident_Node,
                        DecLoc     => DecLoc);
         when SPSymbols.array_type_definition =>
            -- ASSUME Type_Node = array_type_definition
            --# accept Flow, 10, UnusedErrorsFound, "Expected ineffective assignment" &
            --#        Flow, 10, UnusedArrayTypeSymbol, "Expected ineffective assignment";
            Wf_Array_Type_Definition
              (Node         => Type_Node,
               Scope        => Scope,
               Ident_Node   => Ident_Node,
               Dec_Loc      => DecLoc,
               Is_Generic   => False,
               Errors_Found => UnusedErrorsFound,
               The_Array    => UnusedArrayTypeSymbol);
            --# end accept;
         when SPSymbols.type_extension =>
            -- ASSUME Type_Node = type_extension
            Wf_Type_Extension
              (Node                        => Type_Node,
               Scope                       => Scope,
               Ident_Node                  => Ident_Node,
               Dec_Loc                     => DecLoc,
               Private_Type_Being_Resolved => PrivateTypeBeingResolved);
         when others =>
            SystemErrors.Fatal_Error
              (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
               Msg     => "Expect Type_Node = enumeration_type_definition OR integer_type_definition OR real_type_definition OR array_type_definition OR"
                 &
                 " record_type_definition OR modular_type_definition OR type_extension in Wf_Type_Definition");
      end case;
      --# accept Flow, 33, UnusedErrorsFound, "Expected to be neither referenced nor exported" &
      --#        Flow, 33, UnusedArrayTypeSymbol, "Expected to be neither referenced nor exported";
   end wf_type_definition;

   -------------------------------

   function ValidScopeForTaskOrProtectedTypeDeclaration (Scope : Dictionary.Scopes) return Boolean
   --# global in Dictionary.Dict;
   is
   begin
      return (Dictionary.IsVisibleScope (Scope) or else Dictionary.IsPrivateScope (Scope))
        and then Dictionary.IsPackage (Dictionary.GetRegion (Scope));
   end ValidScopeForTaskOrProtectedTypeDeclaration;

begin -- Wf_Full_Type_Declaration

   --ASSUME Node = full_type_declaration

   Ident_Node := Child_Node (Current_Node => Node);
   -- ASSUME Ident_Node = identifier OR task_type_declaration OR protected_type_declaration
   case Syntax_Node_Type (Node => Ident_Node) is
      when SPSymbols.identifier =>
         -- ASSUME Ident_Node = identifier
         IdentStr := Node_Lex_String (Node => Ident_Node);
         Sym      := Dictionary.LookupItem (Name              => IdentStr,
                                            Scope             => Scope,
                                            Context           => Dictionary.ProofContext,
                                            Full_Package_Name => False);

         if Sym = Dictionary.NullSymbol
           or else (Dictionary.IsTypeMark (Sym)
                      and then Dictionary.TypeIsAnnounced (Sym)
                      and then not Dictionary.IsDeclared (Sym))
           or else IsPrivateTypeResolution (Sym, Scope) then
            if Sym = Dictionary.NullSymbol then
               --initial type declaration
               wf_type_definition
                 (Node                     => Next_Sibling (Ident_Node),
                  Scope                    => Scope,
                  Ident_Node               => Ident_Node,
                  DecLoc                   => Node_Position (Node => Node),
                  PrivateTypeBeingResolved => Sym);
            else
               TypeDeclaredAsLimited := Dictionary.IsLimitedPrivateType (Sym);
               TypeDeclaredAsTagged  := Dictionary.TypeIsTagged (Sym);
               Error_Found           := False;
               wf_type_definition
                 (Node                     => Next_Sibling (Ident_Node),
                  Scope                    => Scope,
                  Ident_Node               => Ident_Node,
                  DecLoc                   => Node_Position (Node => Node),
                  PrivateTypeBeingResolved => Sym);
               if Dictionary.IsUnconstrainedArrayType (Sym) then
                  if Dictionary.TypeIsAnnounced (Sym) then
                     Error_Found := True;
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 311,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => Ident_Node),
                        Id_Str    => IdentStr);
                  else --private type
                     Error_Found := True;
                     ErrorHandler.Semantic_Error
                       (Err_Num   => 331,
                        Reference => ErrorHandler.No_Reference,
                        Position  => Node_Position (Node => Ident_Node),
                        Id_Str    => IdentStr);
                  end if;
               end if;

               if not TypeDeclaredAsLimited and then Dictionary.TypeIsLimited (Sym, Dictionary.GlobalScope) then
                  Error_Found := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 332,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => IdentStr);
               end if;

               -- initial declaration tagged but completion is not
               if TypeDeclaredAsTagged and then not Dictionary.TypeIsTagged (Sym) then
                  Error_Found := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 821,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => IdentStr);
               end if;

               -- reverse case; this could be made legal but is not allowed for now
               if not TypeDeclaredAsTagged and then Dictionary.TypeIsTagged (Sym) then
                  Error_Found := True;
                  ErrorHandler.Semantic_Error
                    (Err_Num   => 830,
                     Reference => ErrorHandler.No_Reference,
                     Position  => Node_Position (Node => Ident_Node),
                     Id_Str    => IdentStr);
               end if;

               if not Error_Found then
                  STree.Set_Node_Lex_String (Sym  => Sym,
                                             Node => Ident_Node);
               end if;

            end if;
         else
            ErrorHandler.Semantic_Error
              (Err_Num   => 10,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => IdentStr);
         end if;

      when SPSymbols.task_type_declaration =>
         -- ASSUME Ident_Node = task_type_declaration
         if CommandLineData.Ravenscar_Selected then
            if ValidScopeForTaskOrProtectedTypeDeclaration (Scope) then
               Wf_Task_Type_Declaration (Node  => Ident_Node,
                                         Scope => Scope);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 987,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         else -- declaration not allowed
            ErrorHandler.Semantic_Error
              (Err_Num   => 850,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => LexTokenManager.Null_String);
         end if;

      when SPSymbols.protected_type_declaration =>
         -- ASSUME Ident_Node = protected_type_declaration
         if CommandLineData.Ravenscar_Selected then
            if ValidScopeForTaskOrProtectedTypeDeclaration (Scope) then
               Wf_Protected_Type_Declaration (Node  => Ident_Node,
                                              Scope => Scope);
            else
               ErrorHandler.Semantic_Error
                 (Err_Num   => 987,
                  Reference => ErrorHandler.No_Reference,
                  Position  => Node_Position (Node => Ident_Node),
                  Id_Str    => LexTokenManager.Null_String);
            end if;
         else -- declaration not allowed
            ErrorHandler.Semantic_Error
              (Err_Num   => 850,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Ident_Node),
               Id_Str    => LexTokenManager.Null_String);
         end if;

      when others =>
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Ident_Node = identifier OR task_type_declaration OR protected_type_declaration in Wf_Full_Type_Declaration");
   end case;
end Wf_Full_Type_Declaration;
