-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with SLI;

separate (Sem.CompUnit.Wf_Full_Type_Declaration)
procedure wf_integer
  (Node       : in STree.SyntaxNode;
   Scope      : in Dictionary.Scopes;
   Ident_Node : in STree.SyntaxNode;
   DecLoc     : in LexTokenManager.Token_Position) is
   ExpNode                   : STree.SyntaxNode;
   LeftExpType, RightExpType : Exp_Record;
   UnwantedSeq               : SeqAlgebra.Seq;
   Lower, Upper              : LexTokenManager.Lex_String; -- StoreVals of type's bounds
   UnusedComponentData       : ComponentManager.ComponentData;
   Type_Symbol               : Dictionary.Symbol;

   -- Checks that Lower .. Upper are legal wrt System.Min_Int and System.Max_Int
   procedure CheckAgainstRootInteger
   --# global in     CommandLineData.Content;
   --#        in     DecLoc;
   --#        in     Dictionary.Dict;
   --#        in     LexTokenManager.State;
   --#        in     Lower;
   --#        in     Upper;
   --#        in out ErrorHandler.Error_Context;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         DecLoc,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         LexTokenManager.State,
   --#                                         Lower,
   --#                                         SPARK_IO.File_Sys,
   --#                                         Upper;
   is
      SystemSym  : Dictionary.Symbol;
      Min_IntSym : Dictionary.Symbol;
      Min_IntVal : LexTokenManager.Lex_String;
      Max_IntSym : Dictionary.Symbol;
      Max_IntVal : LexTokenManager.Lex_String;
      Result     : Maths.Value;
      Unused     : Maths.ErrorCode;
      RangeOK    : Boolean;
   begin
      -- We only check in 95 or 2005 modes, since System may not be
      -- specified in the target configuration file in SPARK83 mode.
      case CommandLineData.Content.Language_Profile is
         when CommandLineData.SPARK83 =>
            null;
         when CommandLineData.SPARK95 | CommandLineData.SPARK2005 =>

            SystemSym :=
              Dictionary.LookupItem
              (Name              => LexTokenManager.System_Token,
               Scope             => Dictionary.GlobalScope,
               Context           => Dictionary.ProgramContext,
               Full_Package_Name => False);

            -- The user may or may not have bothered to supply
            -- package System, so...
            if SystemSym /= Dictionary.NullSymbol then

               -- Find System.Min_Int and check Lower against it.
               Min_IntSym :=
                 Dictionary.LookupSelectedItem
                 (Prefix   => SystemSym,
                  Selector => LexTokenManager.Min_Int_Token,
                  Scope    => Dictionary.GetScope (SystemSym),
                  Context  => Dictionary.ProgramContext);

               -- Even if the user has supplied a package System, they might
               -- not have declared Min_Int, so again we have to guard...
               if Min_IntSym /= Dictionary.NullSymbol then

                  Min_IntVal := Dictionary.GetValue (Min_IntSym);

                  if LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Lower,
                     Lex_Str2 => LexTokenManager.Null_String) /=
                    LexTokenManager.Str_Eq
                    and then LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Min_IntVal,
                     Lex_Str2 => LexTokenManager.Null_String) /=
                    LexTokenManager.Str_Eq then

                     --# accept Flow, 10, Unused, "Expected ineffective assignment";
                     Maths.GreaterOrEqual (Maths.ValueRep (Lower), Maths.ValueRep (Min_IntVal), Result, Unused);
                     --# end accept;

                     --# accept Flow, 10, Unused, "Expected ineffective assignment";
                     Maths.ValueToBool (Result, RangeOK, Unused);
                     --# end accept;

                     if not RangeOK then
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 781,
                           Reference => ErrorHandler.No_Reference,
                           Position  => DecLoc,
                           Id_Str    => LexTokenManager.Null_String);
                     end if;
                  end if;

               end if;

               --# assert True; -- for RTC generation

               -- Find System.Max_Int and check Upper against it.
               Max_IntSym :=
                 Dictionary.LookupSelectedItem
                 (Prefix   => SystemSym,
                  Selector => LexTokenManager.Max_Int_Token,
                  Scope    => Dictionary.GetScope (SystemSym),
                  Context  => Dictionary.ProgramContext);

               -- Even if the user has supplied a package System, they might
               -- not have declared Max_Int, so again we have to guard...
               if Max_IntSym /= Dictionary.NullSymbol then

                  Max_IntVal := Dictionary.GetValue (Max_IntSym);

                  if LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Lower,
                     Lex_Str2 => LexTokenManager.Null_String) /=
                    LexTokenManager.Str_Eq
                    and then LexTokenManager.Lex_String_Case_Insensitive_Compare
                    (Lex_Str1 => Max_IntVal,
                     Lex_Str2 => LexTokenManager.Null_String) /=
                    LexTokenManager.Str_Eq then

                     --# accept Flow, 10, Unused, "Expected ineffective assignment";
                     Maths.LesserOrEqual (Maths.ValueRep (Upper), Maths.ValueRep (Max_IntVal), Result, Unused);
                     --# end accept;

                     --# accept Flow, 10, Unused, "Expected ineffective assignment";
                     Maths.ValueToBool (Result, RangeOK, Unused);
                     --# end accept;

                     if not RangeOK then
                        ErrorHandler.Semantic_Error
                          (Err_Num   => 782,
                           Reference => ErrorHandler.No_Reference,
                           Position  => DecLoc,
                           Id_Str    => LexTokenManager.Null_String);
                     end if;
                  end if;

               end if;

            end if;
      end case;
      --# accept Flow, 33, Unused, "Expected to be neither referenced nor exported";
   end CheckAgainstRootInteger;

begin
   --assume Node is integer_type_definition
   ExpNode := Child_Node (Child_Node (Child_Node (Node)));
   SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
   ComponentManager.Initialise (UnusedComponentData);
   --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
   WalkExpression
     (Exp_Node                => ExpNode,
      Scope                   => Scope,
      Type_Context            => Dictionary.GetUnknownTypeMark,
      Context_Requires_Static => True,
      Result                  => LeftExpType,
      Ref_Var                 => UnwantedSeq,
      Component_Data          => UnusedComponentData);
   --# end accept;
   SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
   Maths.StorageRep (LeftExpType.Value, Lower);
   if Syntax_Node_Type (Node => ExpNode) = SPSymbols.attribute then
      if LeftExpType.Is_ARange then
         Maths.StorageRep (LeftExpType.Range_RHS, Upper);
         ErrorHandler.Semantic_Error
           (Err_Num   => 45,
            Reference => 1,
            Position  => Node_Position (Node => ExpNode),
            Id_Str    => LexTokenManager.Null_String);
      else
         Lower := LexTokenManager.Null_String; --no value in error case
         Upper := LexTokenManager.Null_String; --no value in error case
         ErrorHandler.Semantic_Error
           (Err_Num   => 98,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ExpNode),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   else --not an attribute
      if not (Dictionary.TypeIsInteger (LeftExpType.Type_Symbol) or
                Dictionary.TypeIsModular (LeftExpType.Type_Symbol) or
                Dictionary.IsUnknownTypeMark (LeftExpType.Type_Symbol)) then
         Lower := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.Semantic_Error
           (Err_Num   => 38,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ExpNode),
            Id_Str    => LexTokenManager.Null_String);
      elsif LeftExpType.Is_ARange then
         Lower := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.Semantic_Error
           (Err_Num   => 114,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ExpNode),
            Id_Str    => LexTokenManager.Null_String);
      end if;

      ExpNode := Next_Sibling (ExpNode);
      SeqAlgebra.CreateSeq (TheHeap, UnwantedSeq);
      ComponentManager.Initialise (UnusedComponentData);
      --# accept Flow, 10, UnusedComponentData, "Expected ineffective assignment";
      WalkExpression
        (Exp_Node                => ExpNode,
         Scope                   => Scope,
         Type_Context            => Dictionary.GetUnknownTypeMark,
         Context_Requires_Static => True,
         Result                  => RightExpType,
         Ref_Var                 => UnwantedSeq,
         Component_Data          => UnusedComponentData);
      --# end accept;
      SeqAlgebra.DisposeOfSeq (TheHeap, UnwantedSeq);
      Maths.StorageRep (RightExpType.Value, Upper);

      if not (Dictionary.TypeIsInteger (RightExpType.Type_Symbol) or
                Dictionary.TypeIsModular (RightExpType.Type_Symbol) or
                Dictionary.IsUnknownTypeMark (RightExpType.Type_Symbol)) then
         Upper := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.Semantic_Error
           (Err_Num   => 38,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ExpNode),
            Id_Str    => LexTokenManager.Null_String);
      elsif RightExpType.Is_ARange then
         Lower := LexTokenManager.Null_String; -- no value in error case
         ErrorHandler.Semantic_Error
           (Err_Num   => 114,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => ExpNode),
            Id_Str    => LexTokenManager.Null_String);
      end if;

      if not (LeftExpType.Is_Static and RightExpType.Is_Static) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 45,
            Reference => 1,
            Position  => Node_Position (Node => ExpNode),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end if;

   EmptyTypeCheck (DecLoc, Lower, Upper);
   CheckAgainstRootInteger;
   Dictionary.AddIntegerType
     (Name        => Node_Lex_String (Node => Ident_Node),
      Comp_Unit   => ContextManager.Ops.Current_Unit,
      Declaration => Dictionary.Location'(Start_Position => DecLoc,
                                          End_Position   => DecLoc),
      Lower       => Lower,
      Upper       => Upper,
      Scope       => Scope,
      Context     => Dictionary.ProgramContext,
      Type_Symbol => Type_Symbol);
   if ErrorHandler.Generate_SLI then
      SLI.Generate_Xref_Symbol
        (Comp_Unit      => ContextManager.Ops.Current_Unit,
         Parse_Tree     => Ident_Node,
         Symbol         => Type_Symbol,
         Is_Declaration => True);
   end if;
   Heap.ReportUsage (TheHeap);
end wf_integer;
