-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit.WalkStatements)
procedure wf_exit (Node          : in     STree.SyntaxNode;
                   TheLoop       : in     Dictionary.Symbol;
                   ConditionNode :    out STree.SyntaxNode) is
   IfNode, Local_Node : STree.SyntaxNode;
   Exit_Label         : STree.SyntaxNode;
   Condition          : STree.SyntaxNode;
begin
   -- ASSUME Node = exit_statement
   SystemErrors.RT_Assert
     (C       => Syntax_Node_Type (Node => Node) = SPSymbols.exit_statement,
      Sys_Err => SystemErrors.Invalid_Syntax_Tree,
      Msg     => "Expect Node = exit_statement in Wf_Exit");

   -- The procedure checks that the conditions
   -- of Section 5.7 of the SPARK Definition apply to the exit statement.

   Local_Node := Next_Sibling (Child_Node (Node));
   -- ASSUME Local_Node = condition OR simple_name OR NULL
   if Local_Node = STree.NullNode then
      -- ASSUME Local_Node = NULL
      -- A simple exit statement - no label identifier and no exit condition
      Condition  := STree.NullNode;
      Exit_Label := STree.NullNode;
   elsif Syntax_Node_Type (Node => Local_Node) = SPSymbols.simple_name then
      -- ASSUME Local_Node = simple_name
      -- Exit has a label name
      Condition  := Next_Sibling (Local_Node); -- get the exit condition
      Exit_Label := Child_Node (Local_Node); -- get the label identifier
   elsif Syntax_Node_Type (Node => Local_Node) = SPSymbols.condition then
      -- ASSUME Local_Node = condition
      -- Must be an exit with a condition but no label
      Condition  := Local_Node;
      Exit_Label := STree.NullNode;
   else
      Condition  := STree.NullNode;
      Exit_Label := STree.NullNode;
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Local_Node = condition OR simple_name OR NULL in Wf_Exit");
   end if;
   -- ASSUME Condition = condition OR NULL
   -- ASSUME Exit_Label = identifier OR NULL

   ConditionNode := Condition;

   if Exit_Label /= STree.NullNode then
      if Syntax_Node_Type (Node => Exit_Label) = SPSymbols.identifier then
         -- ASSUME Exit_Label = identifier
         -- Exit names a loop label.  It must match the label attached to the
         -- most closely enclosing loop statement.
         if (not Dictionary.LoopHasName (TheLoop))
           or else LexTokenManager.Lex_String_Case_Insensitive_Compare
           (Lex_Str1 => Dictionary.GetSimpleName (TheLoop),
            Lex_Str2 => Node_Lex_String (Node => Exit_Label)) /=
           LexTokenManager.Str_Eq then
            -- Enclosing loop does not have a label, or labels
            -- are present, but do not match
            ErrorHandler.Semantic_Error
              (Err_Num   => 724,
               Reference => ErrorHandler.No_Reference,
               Position  => Node_Position (Node => Exit_Label),
               Id_Str    => LexTokenManager.Null_String);
         end if;
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Exit_Label = identifier OR NULL in Wf_Exit");
      end if;
   end if;

   -- determine whether exit statement contains a when clause;
   if Condition /= STree.NullNode then
      if Syntax_Node_Type (Node => Condition) = SPSymbols.condition then
         -- ASSUME Condition = condition
         -- exit statement contains a when clause, therefore check that condition (2)
         -- of SPARK Definition Section 5.7 applies, i.e. check that closest-
         -- containing compound statement is a loop statement;
         Local_Node := ParentOfSequence (Node);
         case Syntax_Node_Type (Node => Local_Node) is
            when SPSymbols.loop_statement =>
               null;
            when others =>
               ErrorHandler.Control_Flow_Error (Err_Type => ErrorHandler.Misplaced_Exit,
                                                Position => Node_Position (Node => Node));
         end case;
      else
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Condition = condition OR NULL in Wf_Exit");
      end if;
   else
      -- exit statement is in an if_statement, therefore check that condition (3)
      -- of SPARK Definition Section 5.7 applies:
      -- check that exit-statement is last in its sequence of statements;
      if IsLastInSequence (Node) then
         -- check that closest containing compound statement is an if_statement;
         Local_Node := ParentOfSequence (Node);
         case Syntax_Node_Type (Node => Local_Node) is
            when SPSymbols.if_statement =>
               -- check remainder of condition (3);
               IfNode := Local_Node;
               -- advance to condition node;
               Local_Node := Child_Node (Local_Node);
               -- advance to sequence_of_statements node;
               Local_Node := Next_Sibling (Local_Node);
               -- advance to elsif node;
               Local_Node := Next_Sibling (Local_Node);
               -- check that elsif_part is null;
               if Child_Node (Local_Node) = STree.NullNode then
                  -- advance to else node;
                  Local_Node := Next_Sibling (Local_Node);
                  -- check that else_part is null;
                  if Child_Node (Local_Node) = STree.NullNode then
                     -- check that closest-containing compound statement is a loop statement;
                     Local_Node := ParentOfSequence (IfNode);
                     case Syntax_Node_Type (Node => Local_Node) is
                        when SPSymbols.loop_statement =>
                           null;
                        when others =>
                           ErrorHandler.Control_Flow_Error
                             (Err_Type => ErrorHandler.Misplaced_Exit,
                              Position => Node_Position (Node => Node));
                     end case;
                  else
                     ErrorHandler.Control_Flow_Error
                       (Err_Type => ErrorHandler.Misplaced_Exit,
                        Position => Node_Position (Node => Node));
                  end if;
               else
                  ErrorHandler.Control_Flow_Error
                    (Err_Type => ErrorHandler.Misplaced_Exit,
                     Position => Node_Position (Node => Node));
               end if;
            when others =>
               ErrorHandler.Control_Flow_Error (Err_Type => ErrorHandler.Misplaced_Exit,
                                                Position => Node_Position (Node => Node));
         end case;
      else
         ErrorHandler.Control_Flow_Error (Err_Type => ErrorHandler.Misplaced_Exit,
                                          Position => Node_Position (Node => Node));
      end if;
   end if;
end wf_exit;
