-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure up_wf_quantifier (Node   : in     STree.SyntaxNode;
                            Estack : in out ExpStack.ExpStackType;
                            Scope  : in out Dictionary.Scopes) is
   ErrorsFound, RangeFound      : Boolean;
   RangeNode                    : STree.SyntaxNode;
   PredicateResult, RangeResult : Exp_Record;
   QuantifierType               : Dictionary.Symbol;
begin --up_wf_quantifier

   -- Node is quantified_expression
   -- top of Estack is predicate data
   -- 2nd tos Estack is range info iff a range node exists

   RangeNode  := Next_Sibling (Next_Sibling (Next_Sibling (Child_Node (Node))));
   RangeFound := Syntax_Node_Type (Node => RangeNode) = SPSymbols.annotation_arange;

   ExpStack.Pop (PredicateResult, Estack); --result of predicate node
   ErrorsFound := PredicateResult.Errors_In_Expression;
   if not Dictionary.IsBooleanTypeMark (PredicateResult.Type_Symbol) then --error case
      ErrorsFound := True;
      ErrorHandler.Semantic_Error
        (Err_Num   => 326,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => Last_Sibling_Of (Start_Node => Child_Node (Node))),
         Id_Str    => LexTokenManager.Null_String);
   end if;

   --if the arange node exists there will be other stuff on the stack here
   if RangeFound then
      ExpStack.Pop (RangeResult, Estack);
      ErrorsFound := ErrorsFound or RangeResult.Errors_In_Expression;

      QuantifierType := Dictionary.GetType (Dictionary.GetRegion (Scope));

      -- If the quantifier is over Boolean or a subtype of Boolean, then
      -- an explicit range is illegal.
      if Dictionary.TypeIsBoolean (QuantifierType) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 412,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => RangeNode),
            Id_Str    => LexTokenManager.Null_String);

      elsif not RangeResult.Is_ARange then
         ErrorsFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 98,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => RangeNode),
            Id_Str    => LexTokenManager.Null_String);
      elsif not Dictionary.CompatibleTypes (Scope, QuantifierType, RangeResult.Type_Symbol) then
         ErrorsFound := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 106,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => RangeNode),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   end if;

   -- turn result into a Boolean type result
   ExpStack.Push
     (Exp_Record'(Type_Symbol             => Dictionary.GetPredefinedBooleanType,
                  Other_Symbol            => Dictionary.NullSymbol,
                  Stream_Symbol           => Dictionary.NullSymbol,
                  Tagged_Parameter_Symbol => Dictionary.NullSymbol,
                  Variable_Symbol         => Dictionary.NullSymbol,
                  Param_Count             => 0,
                  Param_List              => Lists.Null_List,
                  Sort                    => Type_Result,
                  Arg_List_Found          => False,
                  Is_AVariable            => False,
                  Is_An_Entire_Variable   => False,
                  Errors_In_Expression    => ErrorsFound,
                  Has_Operators           => False,
                  Is_Constant             => False,
                  Is_Static               => False,
                  Is_ARange               => False,
                  String_Value            => LexTokenManager.Null_String,
                  Value                   => Maths.NoValue,
                  Range_RHS               => Maths.NoValue),
      Estack);
   -- leave local scope of quantifier
   Scope := Dictionary.GetEnclosingScope (Scope);
end up_wf_quantifier;
