-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure up_wf_named_association_rep
  (Node         : in     STree.SyntaxNode;
   Scope        : in     Dictionary.Scopes;
   EStack       : in out ExpStack.ExpStackType;
   HeapParam    : in out Lists.List_Heap;
   IsAnnotation : in     Boolean) is
   NameExp, FieldName, ExpResult : Exp_Record;
   DoingRecord                   : Boolean;
   ExpectedType                  : Dictionary.Symbol;
   ErrorFound                    : Boolean := False;

   ---------------------------------------------------------

   function DoingEmbeddedAggregate (Node : STree.SyntaxNode) return Boolean
   --# global in IsAnnotation;
   --#        in STree.Table;
   is
      ExpNode : STree.SyntaxNode;
   begin
      ExpNode := Next_Sibling (Child_Node (Node));
      if ((not IsAnnotation) and then Syntax_Node_Type (Node => ExpNode) /= SPSymbols.aggregate_or_expression) or
        (IsAnnotation and then Syntax_Node_Type (Node => ExpNode) /= SPSymbols.annotation_aggregate_or_expression) then
         ExpNode := Next_Sibling (ExpNode);
      end if;
      return ((not IsAnnotation) and then Syntax_Node_Type (Node => Child_Node (ExpNode)) = SPSymbols.aggregate) or
        (IsAnnotation and then Syntax_Node_Type (Node => Child_Node (ExpNode)) = SPSymbols.annotation_aggregate);
   end DoingEmbeddedAggregate;

   ---------------------------------------------------------

   procedure CheckRecordCompleteness (NameExp : in out Exp_Record;
                                      Node    : in     STree.SyntaxNode)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in     IsAnnotation;
   --#        in     LexTokenManager.State;
   --#        in     STree.Table;
   --#        in out ErrorFound;
   --#        in out ErrorHandler.Error_Context;
   --#        in out HeapParam;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorFound                 from *,
   --#                                         Dictionary.Dict,
   --#                                         HeapParam,
   --#                                         LexTokenManager.State,
   --#                                         NameExp &
   --#         ErrorHandler.Error_Context,
   --#         SPARK_IO.File_Sys          from CommandLineData.Content,
   --#                                         Dictionary.Dict,
   --#                                         ErrorHandler.Error_Context,
   --#                                         HeapParam,
   --#                                         IsAnnotation,
   --#                                         LexTokenManager.State,
   --#                                         NameExp,
   --#                                         Node,
   --#                                         SPARK_IO.File_Sys,
   --#                                         STree.Table &
   --#         HeapParam                  from *,
   --#                                         LexTokenManager.State,
   --#                                         NameExp &
   --#         NameExp                    from *;
   is
      It        : Dictionary.Iterator;
      FieldStr  : LexTokenManager.Lex_String;
      ErrorPos  : LexTokenManager.Token_Position;
      ErrorNode : STree.SyntaxNode;
      Ptr       : Lists.List;

   begin
      ErrorNode := Next_Sibling (Child_Node (Node));
      if ((not IsAnnotation) and then Syntax_Node_Type (Node => ErrorNode) /= SPSymbols.aggregate_or_expression) or
        (IsAnnotation and then Syntax_Node_Type (Node => ErrorNode) /= SPSymbols.annotation_aggregate_or_expression) then
         ErrorNode := Next_Sibling (ErrorNode);
      end if;
      ErrorNode := Child_Node (ErrorNode);
      ErrorPos  := Node_Position (Node => ErrorNode);

      if Dictionary.TypeIsExtendedTagged (NameExp.Type_Symbol) then
         It := Dictionary.FirstExtendedRecordComponent (NameExp.Type_Symbol);
      else
         It := Dictionary.FirstRecordComponent (NameExp.Type_Symbol);
      end if;

      while not Dictionary.IsNullIterator (It) loop
         FieldStr := Dictionary.GetSimpleName (Dictionary.CurrentSymbol (It));
         if not Lists.Is_Member (Heap     => HeapParam,
                                 The_List => NameExp.Param_List,
                                 Str      => FieldStr) then
            ErrorFound := True;
            ErrorHandler.Semantic_Error
              (Err_Num   => 104,
               Reference => ErrorHandler.No_Reference,
               Position  => ErrorPos,
               Id_Str    => FieldStr);
         end if;
         It := Dictionary.NextSymbol (It);
      end loop;
      Ptr := NameExp.Param_List;
      DisposeOfNameList (Ptr, HeapParam);
      NameExp.Param_List := Ptr;
   end CheckRecordCompleteness;

   ---------------------------------------------------------

   function ExpressionLocation (Node : STree.SyntaxNode) return STree.SyntaxNode
   --# global in IsAnnotation;
   --#        in STree.Table;
   is
      ExpNode : STree.SyntaxNode;

   begin
      ExpNode := Next_Sibling (Child_Node (Node));
      if ((not IsAnnotation) and then Syntax_Node_Type (Node => ExpNode) /= SPSymbols.aggregate_or_expression) or
        (IsAnnotation and then Syntax_Node_Type (Node => ExpNode) /= SPSymbols.annotation_aggregate_or_expression) then
         ExpNode := Next_Sibling (ExpNode);
      end if;
      return Child_Node (ExpNode);
   end ExpressionLocation;

   --------------------------------------------------------------

begin --up_wf_named_association_rep
   if not DoingEmbeddedAggregate (Node) then
      ExpStack.Pop (ExpResult, EStack);
      ExpStack.Pop (NameExp, EStack);
      if NameExp.Sort = Is_Parameter_Name then
         DoingRecord := True;
         FieldName   := NameExp;
         ExpStack.Pop (NameExp, EStack);
      else
         DoingRecord := False;
         FieldName   := UnknownTypeRecord;  -- actually ineffective but removes spurious errs
      end if;

      if Dictionary.IsUnknownTypeMark (NameExp.Type_Symbol) then
         --all we have been doing in this case is checking internal
         --consistency of expression.  We can't actually do anything
         --with the result because the aggregate type is unknown.
         null;
      else -- we are dealing with an array or record
         if DoingRecord then
            if FieldName.Other_Symbol = Dictionary.NullSymbol then
               null;

            else
               ExpectedType := Dictionary.GetType (FieldName.Other_Symbol);
               STree.AddNodeSymbol (Node, ExpectedType);
               AssignmentCheck (Node_Position (Node => ExpressionLocation (Node)), Scope, ExpectedType, ExpResult);
               NameExp.Is_Constant := NameExp.Is_Constant and ExpResult.Is_Constant;
            end if;
            if Next_Sibling (Node) = STree.NullNode then
               --this is the last named association so we need to check that
               --all fields have been given a value
               CheckRecordCompleteness (NameExp, Node);
            end if;

         else --must be array
            ExpectedType := Dictionary.GetArrayComponent (NameExp.Type_Symbol);
            STree.AddNodeSymbol (Node, ExpectedType);
            AssignmentCheck (Node_Position (Node => ExpressionLocation (Node)), Scope, ExpectedType, ExpResult);
            NameExp.Is_Constant := NameExp.Is_Constant and ExpResult.Is_Constant;
         end if;
      end if;
      NameExp.Errors_In_Expression := ErrorFound or NameExp.Errors_In_Expression or ExpResult.Errors_In_Expression;
      ExpStack.Push (NameExp, EStack);
   end if;
end up_wf_named_association_rep;
