-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.CompUnit)
procedure CheckNoOverloadingFromTaggedOps
  (SpecNode      : in STree.SyntaxNode;
   SubprogSym    : in Dictionary.Symbol;
   Scope         : in Dictionary.Scopes;
   Abstraction   : in Dictionary.Abstractions;
   Is_Overriding : in Boolean) is
   RootSubprogSym            : Dictionary.Symbol;
   ActualTaggedParameterType : Dictionary.Symbol;
   RootOpKind                : Dictionary.KindsOfOp;

   function SuccessfullyOverrides
     (RootSubprog, SecondSubprog, ActualTaggedParameterType : Dictionary.Symbol)
     return                                                  Boolean
   --# global in Dictionary.Dict;
   --#        in LexTokenManager.State;
      is separate;

   ------------------------------------------------------------------------

   -- given a node which is a subprogram specification, locate and return the lex string
   -- representing the subprogram name
   function Get_Subprogram_Ident (Node : STree.SyntaxNode) return LexTokenManager.Lex_String
   --# global in STree.Table;
   is
      Result : LexTokenManager.Lex_String;
   begin
      -- ASSUME Node = procedure_specification OR function_specification OR proof_function_declaration OR entry_specification
      if Syntax_Node_Type (Node => Node) = SPSymbols.procedure_specification
        or else Syntax_Node_Type (Node => Node) = SPSymbols.function_specification then
         -- ASSUME Node = procedure_specification OR function_specification
         Result := Node_Lex_String (Node => Child_Node (Current_Node => Child_Node (Current_Node => Node)));
      elsif Syntax_Node_Type (Node => Node) = SPSymbols.proof_function_declaration
        or else Syntax_Node_Type (Node => Node) = SPSymbols.entry_specification then
         -- ASSUME Node = proof_function_declaration OR entry_specification
         Result :=
           Node_Lex_String
           (Node => Child_Node (Current_Node => Child_Node (Current_Node => Child_Node (Current_Node => Node))));
      else
         Result := LexTokenManager.Null_String;
         SystemErrors.Fatal_Error
           (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
            Msg     => "Expect Node = procedure_specification OR function_specification OR" &
              " proof_function_declaration OR entry_specification in Get_Subprogram_Ident");
      end if;
      return Result;
   end Get_Subprogram_Ident;

   ------------------------------------------------------------------------

begin -- CheckNoOverloadingFromTaggedOps

   -- if a potentially inheritable subprogram of the same name exists then
   -- the new declaration is only legal if it successfully overrides it
   Dictionary.SearchForInheritedOperations
     (Name             => Get_Subprogram_Ident (Node => SpecNode),
      Scope            => Scope,
      Prefix           => Dictionary.NullSymbol,
      Context          => Dictionary.ProofContext,
      OpSym            => RootSubprogSym,
      KindOfOp         => RootOpKind,
      ActualTaggedType => ActualTaggedParameterType);
   if (RootSubprogSym /= Dictionary.NullSymbol) and then (RootOpKind /= Dictionary.NotASubprogram) then
      -- An inheritable subprogram has been found.
      -- This declaration is only legal if it overrides it
      if not SuccessfullyOverrides (RootSubprogSym, SubprogSym, ActualTaggedParameterType) then
         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 829,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => SpecNode),
            Sym       => RootSubprogSym,
            Scope     => Scope);
         if not Dictionary.IsProofFunction (SubprogSym) then
            Dictionary.SetSubprogramSignatureNotWellformed (Abstraction, SubprogSym);
         end if;
      elsif CommandLineData.Content.Language_Profile = CommandLineData.SPARK2005
        and then not Is_Overriding
        and then (RootOpKind /= Dictionary.NotASubprogram) then

         -- An inherited sub-program but its declarations contradicts the
         -- its overriding indicator.

         ErrorHandler.Semantic_Error_Sym
           (Err_Num   => 844,
            Reference => ErrorHandler.No_Reference,
            Position  => Node_Position (Node => SpecNode),
            Sym       => SubprogSym,
            Scope     => Scope);
      end if;
   elsif CommandLineData.Content.Language_Profile = CommandLineData.SPARK2005
     and then (RootSubprogSym = Dictionary.NullSymbol)
     and then Is_Overriding then

      ErrorHandler.Semantic_Error_Sym
        (Err_Num   => 845,
         Reference => ErrorHandler.No_Reference,
         Position  => Node_Position (Node => SpecNode),
         Sym       => SubprogSym,
         Scope     => Scope);
   end if;
end CheckNoOverloadingFromTaggedOps;
