-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Dictionary)
procedure AddInheritsReference
  (CompilationUnit  : in     Symbol;
   ThePackage       : in     Symbol;
   Explicit         : in     Boolean;
   Comp_Unit        : in     ContextManager.UnitDescriptors;
   PackageReference : in     Location;
   AlreadyPresent   :    out Boolean) is
   AlreadyExplicitlyPresent, NeedToAdd : Boolean;

   --------------------------------------------------------------------------------

   procedure AddPackageInheritsReference
     (InheritingPackage : in Symbol;
      InheritedPackage  : in Symbol;
      Comp_Unit         : in ContextManager.UnitDescriptors;
      Declaration       : in Location)
   --# global in     Explicit;
   --#        in out Dict;
   --# derives Dict from *,
   --#                   Comp_Unit,
   --#                   Declaration,
   --#                   Explicit,
   --#                   InheritedPackage,
   --#                   InheritingPackage;
   is
      ContextClause : Symbol;
   begin
      RawDict.CreateContextClause
        (Scope         => VisibleScope (InheritingPackage),
         ThePackage    => InheritedPackage,
         Explicit      => Explicit,
         Comp_Unit     => Comp_Unit,
         Loc           => Declaration.Start_Position,
         ContextClause => ContextClause);
      RawDict.SetNextContextClause (ContextClause, RawDict.GetPackageInheritClauses (InheritingPackage));
      RawDict.SetPackageInheritClauses (InheritingPackage, ContextClause);
   end AddPackageInheritsReference;

   --------------------------------------------------------------------------------

   procedure AddMainProgramInheritsReference
     (ThePackage  : in Symbol;
      Comp_Unit   : in ContextManager.UnitDescriptors;
      Declaration : in Location)
   --# global in     Explicit;
   --#        in out Dict;
   --# derives Dict from *,
   --#                   Comp_Unit,
   --#                   Declaration,
   --#                   Explicit,
   --#                   ThePackage;
   is
      Current, Next : Symbol;
   begin
      RawDict.CreateContextClause
        (Scope         => LocalScope (GetMainProgram),
         ThePackage    => ThePackage,
         Explicit      => Explicit,
         Comp_Unit     => Comp_Unit,
         Loc           => Declaration.Start_Position,
         ContextClause => Current);
      Next := Dict.Main.InheritClauses;
      RawDict.SetNextContextClause (Current, Next);
      Dict.Main.InheritClauses := Current;
   end AddMainProgramInheritsReference;

   --------------------------------------------------------------------------------

   procedure CheckIfAlreadyPresent
     (ThePackage, CompilationUnit : in     Symbol;
      AddingExplicit              : in     Boolean;
      AlreadyExplicitlyPresent    :    out Boolean;
      NeedToAdd                   :    out Boolean)
   --# global in out Dict;
   --# derives AlreadyExplicitlyPresent,
   --#         Dict                     from AddingExplicit,
   --#                                       CompilationUnit,
   --#                                       Dict,
   --#                                       ThePackage &
   --#         NeedToAdd                from CompilationUnit,
   --#                                       Dict,
   --#                                       ThePackage;
   is
      CurrentContextClause : Symbol;
   begin
      AlreadyExplicitlyPresent := False;
      NeedToAdd                := True;

      if IsPackage (CompilationUnit) then
         CurrentContextClause := RawDict.GetPackageInheritClauses (CompilationUnit);
      elsif IsMainProgram (CompilationUnit) then
         CurrentContextClause := Dict.Main.InheritClauses;
      else
         CurrentContextClause := NullSymbol;
      end if;

      loop
         exit when CurrentContextClause = NullSymbol;
         if RawDict.GetContextClausePackage (CurrentContextClause) = ThePackage then
            NeedToAdd := False;
            if AddingExplicit then
               if RawDict.GetContextClauseExplicit (CurrentContextClause) then
                  AlreadyExplicitlyPresent := True;
               else
                  RawDict.SetContextClauseExplicit (CurrentContextClause);
               end if;
            end if;
            exit;
         end if;

         CurrentContextClause := RawDict.GetNextContextClause (CurrentContextClause);
      end loop;

   end CheckIfAlreadyPresent;

   --------------------------------------------------------------------------------

begin --AddInheritsReference

   CheckIfAlreadyPresent (ThePackage, CompilationUnit, Explicit,
                          --to get
                          AlreadyExplicitlyPresent, NeedToAdd);

   if NeedToAdd then
      case RawDict.GetSymbolDiscriminant (CompilationUnit) is
         when PackageSymbol =>
            AddPackageInheritsReference
              (InheritingPackage => CompilationUnit,
               InheritedPackage  => ThePackage,
               Comp_Unit         => Comp_Unit,
               Declaration       => PackageReference);
         when others =>
            AddMainProgramInheritsReference (ThePackage  => ThePackage,
                                             Comp_Unit   => Comp_Unit,
                                             Declaration => PackageReference);
      end case;

      AddOtherReference (ThePackage, CompilationUnit, PackageReference);
   end if;

   AlreadyPresent := AlreadyExplicitlyPresent;

end AddInheritsReference;
