<?php
/**
 * Forwards_Driver_mdaemon:: implements the Forwards_Driver API for the
 * Mdaemon mail servers.
 *
 * $Horde: forwards/lib/Driver/mdaemon.php,v 1.17.2.2 2007/01/02 13:54:06 jan Exp $
 *
 * Copyright 2001-2007 Mike Cochrane <mike@graftonhall.co.nz>
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Mike Cochrane <mike@graftonhall.co.nz>
 * @since   Forwards 2.1
 * @package Forwards
 */
class Forwards_Driver_mdaemon extends Forwards_Driver {

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $user        The username of the user.
     * @param string $realm       The realm of the user.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     *
     * @return boolean  True on success, false otherwise.
     */
    function enableForwarding($user, $realm = 'default', $password = '',
                              $target, $keeplocal = false)
    {
        if (!is_dir($this->_params[$realm]['location'])) {
           $this->_error = _("Mdaemon path not found");
           return false;
        }

        // Update forward target list.
        if ($fp = fopen($this->_params[$realm]['location'] . '/forward.dat', 'rb')) {
            $current = fread ($fp, filesize($this->_params[$realm]['location'] . '/forward.dat'));
            fclose($fp);
            if ($fp = fopen($this->_params[$realm]['location'] . '/forward.dat', 'wb')) {
                fwrite($fp, "[$user@$realm]\nAddress=$target\n$current");
                fclose($fp);
            } else {
                return false;
            }
        } else {
            return false;
        }

        // Create lock file.
        $fp = fopen($this->_params[$realm]['location'] . '/edituser.lck', 'w');
        if (!$fp) {
            return false;
        } else {
            fclose($fp);
        }

        // Get current details.
        $current_details = $this->_getUserDetails($user, $realm);
        if ($current_details === false) {
            return false;
        }

        // Set forwarding flag.
        $new_details = substr_replace($current_details, 'Y', 216, 1);

        // Set retain copy flag.
        if ($keeplocal) {
            $new_details = substr_replace($new_details, 'Y', 219, 1);
        } else {
            $new_details = substr_replace($new_details, 'N', 219, 1);
        }

        // Create semaphore file.
        $fp = fopen($this->_params[$realm]['location'] . '/edituser.sem', 'wb');
        if (!$fp) {
            $this->_error = _("Not able to create semaphore file");
            return false;
        } else {
            fwrite($fp, "$user@$realm, " . $new_details);
            fclose($fp);
        }

        // Remove lock file.
        @unlink($this->_params[$realm]['location'] . '/edituser.lck');

        return true;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function disableForwarding($user, $realm = 'default', $password = '')
    {
        if (!is_dir($this->_params[$realm]['location'])) {
           $this->_error = _("Mdaemon path not found");
           return false;
        }

        // Create lock file.
        $fp = fopen($this->_params[$realm]['location'] . '/edituser.lck', 'w');
        if (!$fp) {
            return false;
        } else {
            fclose($fp);
        }

        // Get current details.
        $current_details = $this->_getUserDetails($user, $realm);
        if ($current_details === false) {
            return false;
        }

        // Set forwarding flag.
        $new_details = substr_replace($current_details, 'N', 216, 1);

        // Create semaphore file.
        $fp = fopen($this->_params[$realm]['location'] . '/edituser.sem', 'wb');
        if (!$fp) {
           $this->_error = _("Not able to create semaphore file");
           return false;
        } else {
            fwrite($fp, "$user@$realm, " . $new_details);
            fclose($fp);
        }

        // Remove lock file.
        @unlink($this->_params[$realm]['location'] . '/edituser.lck');

        return true;
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  'Y' if forwarding is enabled, or false.
     */
    function isEnabledForwarding($user, $realm = 'default', $password = '')
    {
        // Get current details.
        $current_details = $this->_getUserDetails($user, $realm);
        if ($current_details === false) {
            return false;
        }

        // Check forwarding flag.
        if (substr($current_details, 216, 1) == 'Y') {
            return 'Y';
        } else {
            return false;
        }
    }

    /**
     * Checks if user is keeping a local copy of forwarded mail.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True if user is keeping a local copy of mail,
     *                  otherwise false.
     */
    function isKeepLocal($user, $realm = 'default', $password = '')
    {
        // Get current details.
        $current_details = $this->_getUserDetails($user, $realm);
        if ($current_details === false) {
            return false;
        }

        // Check retain copy flag.
        if (substr($current_details, 219, 1) == 'Y') {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Retrieves current target of mail redirection
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return string  The current forwarding mail address, or false.
     */
    function currentTarget($user, $realm = 'default', $password = '')
    {
        $searchString = "[$user@$realm]";

        $fp = fopen($this->_params[$realm]['location'] . '/forward.dat', 'rb');
        if (!$fp) {
            return false;
        }

        while (!feof($fp)) {
            $line = fgets($fp, 4096);
            if (substr($line,0,strlen($searchString)) == $searchString) {
                $line = fgets($fp, 4096);
                fclose($fp);
                return substr($line,8);
            }
        }
        fclose($fp);

        return false;
    }

    /**
     * Retrieves user details from userlist.dat
     *
     * @param string $user   The username of the user.
     * @param string $realm  The realm for the user.
     *
     * @return string  Line from userlist.dat, or false.
     */
    function _getUserDetails($user, $realm = 'default')
    {
        $searchString = str_pad($realm, 45) . str_pad($user, 30);

        $fp = fopen($this->_params[$realm]['location'] . '/userlist.dat', 'rb');
        if (!$fp) {
            return false;
        }

        while (!feof ($fp)) {
            $line = fgets($fp, 4096);
            if (substr($line,0,strlen($searchString)) == $searchString) {
                return $line;
            }
        }

        fclose($fp);
        return false;
    }

}
