/*
 * -------------------------------------------------------------------------
 *
 * Alsa sound backend for softgun 
 *
 * (C) 2009 Jochen Karrer
 *
 * State:
 * 	Working with Uzebox emulator. Problems with the rare samplerate
 *	of 15700 Hz on some PC's. Has a feedback loop for controlling
 *	the speed of the CPU. 
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ------------------------------------------------------------------------
 */

#ifdef __linux__
#include <alsa/asoundlib.h>
#include <poll.h>
#include <pthread.h>
#include "fio.h"
#include "sound.h"
#include "sgstring.h"
#include "signode.h"
#include "cycletimer.h"

#if 0
#define dbgprintf(x...) fprintf(stderr,x)
#else
#define dbgprintf(x...)
#endif

#define BUFFERSIZE 12000 
typedef struct AlsaSound {
	SoundDevice sdev;
	CycleTimer buffer_check_timer;
	int bytes_per_frame;
	int alsa_sound_fmt;
	unsigned int samplerate;
	snd_pcm_t *handle;
	snd_pcm_hw_params_t *hw;
	snd_pcm_sw_params_t *sw;
	char buffer[BUFFERSIZE];
	uint64_t buffer_wp;
	uint64_t buffer_rp;
	unsigned int buffer_size;
	int poll_fd;
	int outFh_active;
	FIO_FileHandler outFh;
	pthread_t write_thread;
	pthread_mutex_t write_mutex;
	int speed_up;
	int speed_down;
} AlsaSound;

/*
 *************************************************************************
 * SetSoundFormat
 * 	Set samplerate, sample format and number of channels
 *************************************************************************
 */
static int 
AlsaSound_SetSoundFormat(SoundDevice *sdev,SoundFormat *fmt) 
{
	AlsaSound *asdev = sdev->owner;
	int dir;
	int rc;
	switch (fmt->sg_snd_format) {
		case SG_SND_PCM_FORMAT_S16_LE:
			asdev->alsa_sound_fmt = SND_PCM_FORMAT_S16_LE;
			asdev->bytes_per_frame = fmt->channels * 2;
			break;

		case SG_SND_PCM_FORMAT_U16_LE:
			asdev->alsa_sound_fmt = SND_PCM_FORMAT_U16_LE;
			asdev->bytes_per_frame = fmt->channels * 2;
			break;

		case SG_SND_PCM_FORMAT_S8:
			asdev->alsa_sound_fmt = SND_PCM_FORMAT_S8;
			asdev->bytes_per_frame = fmt->channels;
			break;

		case SG_SND_PCM_FORMAT_U8:
			asdev->alsa_sound_fmt = SND_PCM_FORMAT_U8;
			asdev->bytes_per_frame = fmt->channels;
			break;

		default:
			fprintf(stderr,"Unknown sound format %d\n",fmt->sg_snd_format);
			return -1;
	}
	snd_pcm_hw_params_alloca(&asdev->hw); 
	snd_pcm_hw_params_any(asdev->handle, asdev->hw);

	/* Interleaved mode */
	snd_pcm_hw_params_set_access(asdev->handle, asdev->hw,
		      SND_PCM_ACCESS_RW_INTERLEAVED);
	snd_pcm_hw_params_set_format(asdev->handle, asdev->hw,asdev->alsa_sound_fmt);
	rc = snd_pcm_hw_params_set_channels(asdev->handle, asdev->hw, fmt->channels);
	asdev->samplerate = fmt->samplerate;
	rc = snd_pcm_hw_params_set_rate_near(asdev->handle, asdev->hw, &asdev->samplerate, &dir);
	rc = snd_pcm_hw_params(asdev->handle, asdev->hw);
	if (rc < 0) {
		fprintf(stderr,
		    "unable to set hw parameters: %s\n",
		    snd_strerror(rc));
	    	exit(1);
	}
	/* Reset the buffers */
	asdev->buffer_rp = asdev->buffer_wp = 0;

	snd_pcm_sw_params_alloca(&asdev->sw); 
	snd_pcm_sw_params_current (asdev->handle,asdev->sw); 
	rc = snd_pcm_sw_params_set_avail_min (asdev->handle, asdev->sw, 1024);
	snd_pcm_sw_params_set_start_threshold (asdev->handle, asdev->sw, 1024);
	snd_pcm_sw_params (asdev->handle, asdev->sw);
	return 0;
}

/*
 ****************************************************************
 * write_samples
 * 	The thread writing samples to the sound device
 ****************************************************************
 */

void *
write_samples(void *clientData)
{
	AlsaSound *asdev = (AlsaSound *) clientData;	
	int rc;
	int frames;
	unsigned int wp;
	unsigned int rp;
	while(1) {
		pthread_mutex_lock(&asdev->write_mutex);
		while(asdev->buffer_wp != asdev->buffer_rp) {
			wp = asdev->buffer_wp % asdev->buffer_size;
			rp = asdev->buffer_rp % asdev->buffer_size;
			
			if((asdev->buffer_wp > asdev->buffer_rp) && (wp <= rp)) {
				frames = (asdev->buffer_size - rp) / asdev->bytes_per_frame;
				if(frames > 1024) {
					frames = 1024;
				}
				rc = snd_pcm_writei(asdev->handle, asdev->buffer + rp, frames);
			} else {
				frames = (wp - rp) / asdev->bytes_per_frame;
				if(frames > 1024) {
					frames = 1024;
				}
				rc = snd_pcm_writei(asdev->handle, asdev->buffer + rp, frames);
			}
			if (rc == -EPIPE) {
			      snd_pcm_prepare(asdev->handle);
			      asdev->buffer_rp = asdev->buffer_wp = 0;
			} else if (rc < 0) {
				fprintf(stderr, "error from writei: %s\n",
					snd_strerror(rc));
				asdev->buffer_rp = asdev->buffer_wp = 0;
			} else {
				asdev->buffer_rp = asdev->buffer_rp + frames * asdev->bytes_per_frame;
			}
		}
	}
	return NULL;
}

static void
alsa_check_buffer_fill(void *clientData) {
	uint32_t count;
	AlsaSound *asdev = (AlsaSound *)clientData;
	SoundDevice *sdev = &asdev->sdev;
	count = asdev->buffer_wp - asdev->buffer_rp;
	if((count > (3 * BUFFERSIZE / 4)) && !asdev->speed_down) {
		asdev->speed_down = 1;
		SigNode_Set(sdev->speedDown,SIG_HIGH);
		dbgprintf("Speed down\n");
	} else if((count < (BUFFERSIZE / 2)) && asdev->speed_down) {
		SigNode_Set(sdev->speedDown,SIG_LOW);
		dbgprintf("Speed Ok\n");
		asdev->speed_down = 0;
	} else if((count > (BUFFERSIZE / 2)) && asdev->speed_up) {
		SigNode_Set(sdev->speedUp,SIG_LOW);
		dbgprintf("Speed Ok\n");
		asdev->speed_up = 0;
	} else if((count < (BUFFERSIZE / 4)) && !asdev->speed_up) {
		asdev->speed_up = 1;
		SigNode_Set(sdev->speedUp,SIG_HIGH);
		dbgprintf("Speed up\n");
	}
	if(count > (BUFFERSIZE / 4)) {
		pthread_mutex_unlock(&asdev->write_mutex);
	}
	CycleTimer_Mod(&asdev->buffer_check_timer,CycleTimerRate_Get() >> 1);
}

/*
 *******************************************************************************
 * PlaySamples
 *	Take the samples and put them to the circular buffer for the
 *	thread witch sends them to the sound device
 *******************************************************************************
 */
static int 
AlsaSound_PlaySamples(SoundDevice *sdev,void *data,uint32_t len)
{
	AlsaSound *asdev = sdev->owner;
	uint32_t count;
	int i;
	count = asdev->buffer_wp - asdev->buffer_rp;
	if((count + len) >= sizeof(asdev->buffer)) {
		return len;
	}
	for(i=0;i<len;i++) {
		asdev->buffer[(asdev->buffer_wp) % asdev->buffer_size] = ((uint8_t *)data)[i];
		asdev->buffer_wp = asdev->buffer_wp + 1;
	}
	return len;
}

/*
 *************************************************************************
 * AlsaSound_New
 * 	Constructor for the ALSA implementation of the abstract 
 * 	base class "SoundDevice" 
 *
 **************************************************************************
 */
SoundDevice *
AlsaSound_New(const char *name) 
{
	int rc;
	AlsaSound *asdev = sg_new(AlsaSound);
	SoundDevice * sdev = &asdev->sdev;
	sdev->setSoundFormat = AlsaSound_SetSoundFormat;
	sdev->playSamples = AlsaSound_PlaySamples;
	sdev->owner = asdev;
	asdev->buffer_size = sizeof(asdev->buffer);
	asdev->outFh_active = 0;
	  /* Open PCM device for playback. */
	rc = snd_pcm_open(&asdev->handle, "default",
                    SND_PCM_STREAM_PLAYBACK, 0);
	if (rc < 0) {
    		fprintf(stderr, "unable to open pcm device: %s\n",
		snd_strerror(rc));
		sleep(1);
		sg_free(asdev);
		return NULL;
  	}
	pthread_mutex_init(&asdev->write_mutex,NULL);
	
	pthread_create(&asdev->write_thread,NULL,write_samples,(void*)asdev);
	CycleTimer_Init(&asdev->buffer_check_timer,alsa_check_buffer_fill,asdev);
	CycleTimer_Mod(&asdev->buffer_check_timer,CycleTimerRate_Get() >> 1);
	sdev->speedUp = SigNode_New("%s.speed_up",name);
        sdev->speedDown = SigNode_New("%s.speed_down",name);
        if(!sdev->speedUp || ! sdev->speedDown) {
                fprintf(stderr,"Can not create sound speed control lines\n");
                exit(1);
        }
	fprintf(stderr,"Created ALSA sound device \"%s\"\n",name);
	
	return sdev;
}
#else /* ifdef __linux */

#include "nullsound.h"
SoundDevice * 
AlsaSound_New(const char *name) 
{
	return NullSound_New(name);;
}

#endif
