#ifndef _KINO_THREAD_
#define _KINO_THREAD_

#include <pthread.h>
#include <time.h>
#include "Diagnostics.h"

/** Threader provides a general abstract class for threaded functionality. 
	Classes that need their own threads need only extend this and implement 
	the Thread method. 

	The Thread method should periodically call ThreadIsRunning to determine if 
	a stop has been requested by the user of the class.
*/

class Threader :
	virtual public DiagnosticsId
{
	private:
		pthread_t thread;
		bool running;

	public:
		/** Constructor of the class.
		*/

		Threader( ) : 
			running( false )
		{
		}

		/** Log id for this class.
		*/

		virtual string LogId( )
		{
			return "Unknown Threader";
		}

		/** Start the thread.
		*/

		void ThreadStart( ) 
		{
			Diagnostics::Log( *this, "Thread starting." );

			if ( pthread_create( &thread, (const pthread_attr_t *)NULL, BootStrap, this ) == 0 )
				running = true;
			else
				throw "Unable to start thread";
		}

		/** Stop the thread.
		*/

		void ThreadStop( ) 
		{
			Diagnostics::Log( *this, "Thread stopping." );
			running = false;
			pthread_join( thread, (void **)NULL ); 
			Diagnostics::Log( *this, "Thread stopped." );
		}

		/** Cancel the thread.
		*/

		void ThreadCancel( )
		{
			Diagnostics::Log( *this, "Thread cancellation." );
			running = false;
			pthread_cancel( thread );
			Diagnostics::Log( *this, "Thread cancelled." );
		}

		/** Determine if the thread is still running. 
		*/

		bool ThreadIsRunning( )
		{
			return running;
		}

		/** Optional execution method - runs until the thread terminates.
		*/

		void ThreadExecute( )
		{
			if ( !ThreadIsRunning( ) )
				ThreadStart( );
			pthread_join( thread, (void **)NULL ); 
			Diagnostics::Log( *this, "Thread terminated." );
		}

		/** Utilitiy function for sleep functionality.
		*/

		static void ThreadSleep( time_t sec, long nsec )
		{
			struct timespec t = { sec, nsec };
			nanosleep( &t, (struct timespec *)NULL );
		}

	protected:

		/** Thread implementation.
		*/

		virtual void Thread( ) = 0;

	private: 

		/** Static boostrap method for the thread itself.
		*/

		static void *BootStrap( void *ptr ) 
		{ 
			Threader *threader = static_cast < Threader * > ( ptr );
			threader->ThreadWrapper( ); 
			return NULL;
		}

		/** Wrapper which ensures the correct running state.
		*/

		void ThreadWrapper( )
		{
			running = true;
			try
			{
				Thread( );
			}
			catch( string err )
			{
				Diagnostics::Log( *this, err, 0 );
			}
			catch( char *err )
			{
				Diagnostics::Log( *this, err, 0 );
			}

			running = false;
		}
};

#endif
