/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to manage budgets.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgbudgetpluginwidget.h"
#include "skgbudgetdelegate.h"
#include "skgmainpanel.h"
#include "skgobjectmodel.h"
#include "skgbudgetobject.h"
#include "skgbudgetruleobject.h"
#include "skgcategoryobject.h"
#include "skgtransactionmng.h"
#include "skgtraces.h"
#include "skgdocument.h"

#include <QDomDocument>
#include <QKeyEvent>

SKGBudgetPluginWidget::SKGBudgetPluginWidget(SKGDocument* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEIN(1, "SKGBudgetPluginWidget::SKGBudgetPluginWidget");
    if (!iDocument) return;

    ui.setupUi(this);

    ui.kPeriodLbl->setText(i18n("%1:", iDocument->getDisplay("t_period")));
    ui.kYearLbl->setText(i18n("%1:", iDocument->getDisplay("i_year")));
    ui.kMonthLbl->setText(i18n("%1:", iDocument->getDisplay("i_month")));
    ui.kAmountLabel->setText(i18n("%1:", iDocument->getDisplay("f_value")));
    ui.kCategoryLabel->setText(i18n("%1:", iDocument->getDisplay("t_category")));
    ui.kYearCheck->setText(i18n("%1:", iDocument->getDisplay("i_year")));
    ui.kMonthCheck->setText(i18n("%1:", iDocument->getDisplay("i_month")));
    ui.kCategoryCheck->setText(i18n("%1:", iDocument->getDisplay("t_category")));
    ui.kPeriodLbl2->setText(i18n("%1:", iDocument->getDisplay("t_period")));
    ui.kAmountLabel2->setText(i18n("%1:", iDocument->getDisplay("f_value")));
    ui.kCategoryTransferCheck->setText(i18n("%1:", iDocument->getDisplay("t_category")));

    ui.kYear->setValue(QDate::currentDate().year());
    ui.kYearAuto->setValue(QDate::currentDate().year());
    ui.kYearAutoBase->setValue(QDate::currentDate().year());
    ui.kMonth->setValue(QDate::currentDate().month());
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Monthly"));
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Yearly"));
    ui.kPeriod->addItem(i18nc("Noun, how to define a budget period", "Individual"));


    ui.kView->getShowWidget()->addItem("all", i18nc("Noun, budget items to display", "All"), "",
                                       "",
                                       "current;currentYear;currentMonth;previousYear;previousMonth",
                                       "none"
                                      );
    ui.kView->getShowWidget()->addItem("none", i18nc("Noun, budget items to display", "None"), "",
                                       "1=0",
                                       "",  //Check when checked
                                       "all;current;currentYear;currentMonth;previousYear;previousMonth",       //Uncheck when checked
                                       "",       //Check when unchecked
                                       "all"     //Uncheck when unchecked
                                      );
    ui.kView->getShowWidget()->addSeparator();
    ui.kView->getShowWidget()->addItem("current", i18nc("Noun, budget items to display", "Current"), "",
                                       "t_PERIOD>=STRFTIME('%Y-%m', date('now')) OR t_PERIOD=STRFTIME('%Y', date('now'))",
                                       "currentMonth",  //Check when checked
                                       "none",       //Uncheck when checked
                                       "",       //Check when unchecked
                                       "all"     //Uncheck when unchecked
                                      );
    ui.kView->getShowWidget()->addItem("currentYear", i18nc("Noun, budget items to display", "Current year"), "view-calendar-month",
                                       "t_PERIOD LIKE STRFTIME('%Y', date('now'))||'%'",
                                       "",
                                       "none",
                                       "",
                                       "all"
                                      );
    ui.kView->getShowWidget()->addItem("currentMonth", i18nc("Noun, budget items to display", "Current month"), "view-calendar-week",
                                       "t_PERIOD=STRFTIME('%Y-%m', date('now'))",
                                       "",
                                       "none",
                                       "",
                                       "all;current"
                                      );
    ui.kView->getShowWidget()->addSeparator();
    ui.kView->getShowWidget()->addItem("previousYear", i18nc("Noun, budget items to display", "Previous year"), "view-calendar-month",
                                       "t_PERIOD LIKE STRFTIME('%Y', date('now','-1 Year'))||'%'",
                                       "",
                                       "none",
                                       "",
                                       "all"
                                      );
    ui.kView->getShowWidget()->addItem("previousMonth", i18nc("Noun, budget items to display", "Previous month"), "view-calendar-week",
                                       "t_PERIOD=STRFTIME('%Y-%m', date('now','-1 Month'))",
                                       "",
                                       "none",
                                       "",
                                       "all"
                                      );
    ui.kView->getShowWidget()->setDefaultState("current");

    m_timer.setSingleShot(true);
    connect(&m_timer, SIGNAL(timeout()), this, SLOT(refreshInfoZone()), Qt::QueuedConnection);

    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "All"), SKGBudgetRuleObject::ALL);
    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "Negative"), SKGBudgetRuleObject::NEGATIVE);
    ui.kConditionCmb->addItem(i18nc("Noun, condition item to a apply a transfer of budget", "Positive"), SKGBudgetRuleObject::POSITIVE);

    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Next"), SKGBudgetRuleObject::NEXT);
    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Current"), SKGBudgetRuleObject::CURRENT);
    ui.kModeCmb->addItem(i18nc("Noun, mode item to a apply a transfer of budget", "Current year"), SKGBudgetRuleObject::YEAR);

    ui.kView->getView()->setItemDelegate(new SKGBudgetDelegate(ui.kView->getView(), getDocument()));

    SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
    if (doc) {
        ui.kUnitCmb->addItem("%");
        ui.kUnitCmb->addItem(doc->getPrimaryUnit().Symbol);
        ui.kUnit->setText(doc->getPrimaryUnit().Symbol);

        //Bind operation view
        m_objectModel = new SKGObjectModel(doc, "v_budget_display", "1=0", this, "", false);
        ui.kView->setModel(m_objectModel);
    }

    //Add registered global action in contextual menu
    if (SKGMainPanel::getMainPanel()) {
        ui.kView->getView()->insertGlobalAction();
        ui.kView->getView()->insertGlobalAction("open");
        ui.kView->getView()->insertGlobalAction();
        ui.kView->getView()->insertGlobalAction("edit_delete");
    }
    connect(ui.kView->getView(), SIGNAL(doubleClicked(QModelIndex)), SKGMainPanel::getMainPanel()->getGlobalAction("open"), SLOT(trigger()));
    connect(ui.kView->getView(), SIGNAL(selectionChangedDelayed()), this, SLOT(onSelectionChanged()));

    //Add Standard KDE Icons to buttons to Operations
    ui.kModifyBtn->setIcon(KIcon("dialog-ok-apply"));
    ui.kAddBtn->setIcon(KIcon("list-add"));
    ui.kProcessBtn->setIcon(KIcon("system-run"));

    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGManualSection);
        list.push_back(ui.SKGEditionButtonsSection);
        ui.kWidgetSelector->addButton(KIcon("user-properties"), i18n("Manual"), i18n("Display the edit panel for standard budget"), list);
    }
    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGAutoSection);
        list.push_back(ui.SKGEditionButtonsSection);
        ui.kWidgetSelector->addButton(KIcon("games-solve"), i18n("Auto"), i18n("Display the edit panel for automatic budgets"), list);
    }
    {
        SKGWidgetSelector::SKGListQWidget list;
        list.push_back(ui.SKGRuleSection);
        list.push_back(ui.SKGEditionButtonsSection);
        ui.kWidgetSelector->addButton(KIcon("view-financial-transfer"), i18n("Rules"), i18n("Display the edit panel for rules"), list);
    }
    connect(ui.kWidgetSelector, SIGNAL(selectedModeChanged(int)), this, SLOT(onBtnModeClicked(int)));

    SKGMainPanel::fillWithDistinctValue(ui.kCategoryEdit, getDocument(), "category", "t_fullname", "");
    SKGMainPanel::fillWithDistinctValue(ui.kCategoryRule, getDocument(), "category", "t_fullname", "");
    SKGMainPanel::fillWithDistinctValue(ui.kCategoryTransfer, getDocument(), "category", "t_fullname", "");

    ui.kWidgetSelector->setSelectedMode(0);

    //Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter(this);

    //Refresh
    connect((const QObject*) getDocument(), SIGNAL(tableModified(QString,int)), this, SLOT(dataModified(QString,int)), Qt::QueuedConnection);
    dataModified("", 0);
}

SKGBudgetPluginWidget::~SKGBudgetPluginWidget()
{
    SKGTRACEIN(1, "SKGBudgetPluginWidget::~SKGBudgetPluginWidget");
    m_objectModel = NULL;
}

bool SKGBudgetPluginWidget::eventFilter(QObject* object, QEvent* event)
{
    if (event && event->type() == QEvent::KeyPress) {
        QKeyEvent* keyEvent = (QKeyEvent*) event;
        if ((keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter) && object == this) {
            if (QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAddBtn->isEnabled()) {
                ui.kAddBtn->click();
            } else if (QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kModifyBtn->isEnabled()) {
                ui.kModifyBtn->click();
            }
        }
    }

    return false;
}

QString SKGBudgetPluginWidget::getState()
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    root.setAttribute("currentPage", SKGServices::intToString(ui.kWidgetSelector->getSelectedMode()));
    if (m_objectModel && m_objectModel->getRealTable() == "budget") {
        root.setAttribute("view", ui.kView->getState());
        root.setAttribute("viewRule", m_viewRule);
    } else {
        root.setAttribute("view", m_viewBudget);
        root.setAttribute("viewRule", ui.kView->getState());
    }

    return doc.toString();
}

void SKGBudgetPluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    QString currentPage = root.attribute("currentPage");
    if (currentPage.isEmpty()) currentPage = '0';

    ui.kWidgetSelector->setSelectedMode(SKGServices::stringToInt(currentPage));

    m_viewBudget = root.attribute("view");
    m_viewRule = root.attribute("viewRule");
    if (m_objectModel && m_objectModel->getRealTable() == "budget") {
        ui.kView->setState(m_viewBudget);
    } else {
        ui.kView->setState(m_viewRule);
    }
}

QString SKGBudgetPluginWidget::getDefaultStateAttribute()
{
    return "SKGBUDGET_DEFAULT_PARAMETERS";
}

QWidget* SKGBudgetPluginWidget::mainWidget() const
{
    return ui.kView->getView();
}

void SKGBudgetPluginWidget::refresh()
{
    SKGTRACEIN(1, "SKGBudgetPluginWidget::refresh");

    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled(db != NULL);
    if (db != NULL) {
        //Refresh yours widgets here
    }
}

void SKGBudgetPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::dataModified");
    Q_UNUSED(iIdTransaction);

    //Refresh widgets
    if (iTableName == "budget" || iTableName.isEmpty()) {
        //Refresh info area
        m_timer.start(300);
    }

    if (iTableName == "category" || iTableName.isEmpty()) {
        //Set type category
        SKGMainPanel::fillWithDistinctValue(ui.kCategoryEdit, getDocument(), "category", "t_fullname", "");
        SKGMainPanel::fillWithDistinctValue(ui.kCategoryRule, getDocument(), "category", "t_fullname", "");
        SKGMainPanel::fillWithDistinctValue(ui.kCategoryTransfer, getDocument(), "category", "t_fullname", "");
    }
}
void SKGBudgetPluginWidget::onBtnModeClicked(int mode)
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::onBtnModeClicked");
    if (!m_objectModel) return;

    if (mode == 2 && m_objectModel->getTable() != "v_budgetrule_display") {
        ui.kView->getShowWidget()->setEnabled(false);
        m_viewBudget = ui.kView->getState();
        m_objectModel->setFilter("");
        m_objectModel->setTable("v_budgetrule_display");
        ui.kView->setState(m_viewRule);
    } else if (mode != 2 && m_objectModel->getTable() != "v_budget_display") {
        ui.kView->getShowWidget()->setEnabled(true);
        m_viewRule = ui.kView->getState();
        m_objectModel->setTable("v_budget_display");
        ui.kView->setState(m_viewBudget);
    }

    onCreatorModified();
}

void SKGBudgetPluginWidget::onProcessRules()
{
    SKGError err;
    _SKGTRACEINRC(10, "SKGBudgetPluginWidget::onProcessRules", err);

    {
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Process budget rules")  , err);
        err = SKGBudgetRuleObject::processAllRules(static_cast<SKGDocumentBank*>(getDocument()));
    }
    //status bar
    if (!err) {
        err = SKGError(0, i18nc("Successful message after an user action", "Budget rules processed"));
    } else err.addError(ERR_FAIL, i18nc("Error message", "Budget rules failed"));

    //Display error
    SKGMainPanel::displayErrorMessage(err);
}

void SKGBudgetPluginWidget::onAddClicked()
{
    SKGError err;
    _SKGTRACEINRC(10, "SKGBudgetPluginWidget::onAddClicked", err);

    if (ui.kWidgetSelector->getSelectedMode() == 2) {
        //Creation of a rule
        QStringList uniqueIDs;
        {
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget rule creation")  , err);
            SKGBudgetRuleObject budgetRule(getDocument());
            err = updateBudgetRule(budgetRule);
            uniqueIDs.push_back(budgetRule.getUniqueID());
        }
        //status bar
        if (!err) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget rule created"));
            ui.kView->getView()->selectObjects(uniqueIDs);
        } else err.addError(ERR_FAIL, i18nc("Error message", "Budget rule creation failed"));
    } else {
        //Creation of a budget
        QStringList uniqueIDs;
        {
            SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget creation")  , err, 2);
            if (ui.kWidgetSelector->getSelectedMode() == 0) {
                //Manual creation
                int mode = ui.kPeriod->currentIndex();
                if (mode == 0) { //Monthly
                    for (int m = 1; !err && m <= 12; ++m) {
                        SKGBudgetObject budget(getDocument());
                        err = updateBudget(budget, m);
                        uniqueIDs.push_back(budget.getUniqueID());
                    }
                } else if (mode == 1) { //Yearly
                    SKGBudgetObject budget(getDocument());

                    err = updateBudget(budget, 0);
                    uniqueIDs.push_back(budget.getUniqueID());
                } else { //Individual
                    SKGBudgetObject budget(getDocument());
                    err = updateBudget(budget);
                    uniqueIDs.push_back(budget.getUniqueID());
                }
            } else {
                //Automatic creation
                if (ui.kAutoBudgetCheck->isChecked()) err = SKGBudgetObject::createAutomaticBudget((SKGDocumentBank*) getDocument(),
                            ui.kYearAuto->value(),
                            ui.kYearAutoBase->value(),
                            ui.kRemovePrevious->isChecked());
                if (!err) err = getDocument()->stepForward(1);

                if (!err) err = SKGBudgetObject::balanceBudget((SKGDocumentBank*) getDocument(),
                                    ui.kYearAuto->value(), (ui.kBalancingMonthly->isChecked() ? 0 : -1),
                                    ui.kBalancingAnnual->isChecked());
                if (!err) err = getDocument()->stepForward(2);
            }
        }

        //status bar
        if (!err) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget created"));
            ui.kView->getView()->selectObjects(uniqueIDs);
        } else err.addError(ERR_FAIL, i18nc("Error message", "Budget creation failed"));
    }

    //Display error
    SKGMainPanel::displayErrorMessage(err);
}

void SKGBudgetPluginWidget::onUpdateClicked()
{
    SKGError err;
    SKGTRACEINRC(10, "SKGBudgetPluginWidget::onUpdateClicked", err);
    //Get Selection
    SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();
    if (ui.kWidgetSelector->getSelectedMode() == 2) {
        {
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget rule update"), err);
            SKGBudgetRuleObject rule = selection.at(0);
            err = updateBudgetRule(rule);
        }

        //status bar
        if (!err) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget rule updated"));
        } else err.addError(ERR_FAIL, i18nc("Error message",  "Budget rule update failed"));
    } else {
        {
            int nb = selection.count();
            SKGBEGINPROGRESSTRANSACTION(*getDocument(), i18nc("Noun, name of the user action", "Budget update"), err, nb);
            for (int i = 0; !err && i < nb; ++i) {
                SKGBudgetObject budget = selection.at(i);
                int mode = ui.kPeriod->currentIndex();
                if (mode == 1) { //Yearly
                    err = updateBudget(budget, 0);
                } else { //Individual
                    err = updateBudget(budget);
                }

                if (!err) err = getDocument()->stepForward(i + 1);
            }
        }

        //status bar
        if (!err) {
            err = SKGError(0, i18nc("Successful message after an user action", "Budget updated"));
        } else err.addError(ERR_FAIL, i18nc("Error message",  "Budget update failed"));
    }


    //Display error
    SKGMainPanel::displayErrorMessage(err);
}

SKGError SKGBudgetPluginWidget::updateBudget(SKGBudgetObject& iBudget, int iMonth)
{
    SKGError err;
    if (!err && ui.kYear->isEnabled()) err = iBudget.setYear(ui.kYear->value());
    if (!err && ui.kMonth->isEnabled()) err = iBudget.setMonth(iMonth != -1 ? iMonth : ui.kMonth->value());

    SKGCategoryObject cat;
    QString catName = ui.kCategoryEdit->text().trimmed();
    if (!err) err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catName, cat, true);
    if (!err) err = iBudget.setCategory(cat);
    if (!err) err = iBudget.enableSubCategoriesInclusion(ui.kIncludingSubCategories->isChecked());

    double val = ui.kAmountEdit->value();
    //Is the sign forced ?
    if (ui.kAmountEdit->sign() == 0) {
        //No
        SKGObjectBase cat2(cat.getDocument(), "v_category_display", cat.getID());

        //Are we able to find to sign with the category ?
        if (cat2.getAttribute("t_TYPEEXPENSE") == "-") val = -val;
    }
    if (!err)  err = iBudget.setBudgetedAmount(val);

    if (!err) err = iBudget.save();
    return err;
}

SKGError SKGBudgetPluginWidget::updateBudgetRule(SKGBudgetRuleObject& iRule)
{
    SKGError err;
    SKGCategoryObject cat;
    QString catName = ui.kCategoryRule->text().trimmed();
    if (!err) err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catName, cat, true);
    SKGCategoryObject catchange;
    QString catchangeName = ui.kCategoryTransfer->text().trimmed();
    if (!err) err = SKGCategoryObject::createPathCategory(static_cast<SKGDocumentBank*>(getDocument()), catchangeName, catchange, true);
    if (!err) err = iRule.enableCategoryCondition(ui.kCategoryCheck->isChecked());
    if (!err) err = iRule.setBudgetCategory(cat);
    if (!err) err = iRule.enableYearCondition(ui.kYearCheck->isChecked());
    if (!err) err = iRule.setBudgetYear(ui.kYearRule->value());
    if (!err) err = iRule.enableMonthCondition(ui.kMonthCheck->isChecked());
    if (!err) err = iRule.setBudgetMonth(ui.kMonthRule->value());
    if (!err) {
        bool absolute = (ui.kUnitCmb->currentIndex() == 1);
        double val = ui.kAmountEdit2->value();
        if (!absolute) val = qMin(qMax(double(0), val), double(100));
        err = iRule.setQuantity(val, absolute);
    }
    if (!err) err = iRule.setCondition((SKGBudgetRuleObject::Condition) ui.kConditionCmb->itemData(ui.kConditionCmb->currentIndex()).toInt());
    if (!err) err = iRule.enableCategoryChange(ui.kCategoryTransferCheck->isChecked());
    if (!err) err = iRule.setTransfer((SKGBudgetRuleObject::Mode) ui.kModeCmb->itemData(ui.kModeCmb->currentIndex()).toInt(), catchange);
    if (!err) err = iRule.save();
    return err;
}
void SKGBudgetPluginWidget::open(const SKGBudgetObject& iBudget) //TODO: use open of operations
{
    _SKGTRACEIN(10, "SKGBudgetPluginWidget::open");

    //Build where clause and title
    QString y = SKGServices::intToString(iBudget.getYear());
    QString m = SKGServices::intToString(iBudget.getMonth());
    if (m.length() == 1) m = '0' % m;

    QString wc = "i_IDCATEGORY IN (SELECT bc.id_category FROM budgetcategory bc WHERE bc.id=" % SKGServices::intToString(iBudget.getID()) % ") AND "
                 "STRFTIME('%Y', d_date)='" % y % '\'';
    if (m != "00") wc += " AND STRFTIME('%m', d_date)='" % m % '\'';
    QString title = i18nc("Noun, a list of items", "Operations assigned to budget");

    //Open
    if (QApplication::keyboardModifiers() &Qt::ControlModifier && QApplication::keyboardModifiers() &Qt::ShiftModifier) {
        //Call debug plugin
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);
        root.setAttribute("sqlOrder", "SELECT * from v_operation_consolidated WHERE " % wc);

        SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Debug plugin"), -1, doc.toString());
    } else {
        //Call operation plugin
        QDomDocument doc("SKGML");
        doc.setContent(iBudget.getDocument()->getParameter("SKGOPERATION_CONSOLIDATED_DEFAULT_PARAMETERS"));
        QDomElement root = doc.documentElement();
        if (root.isNull()) {
            root = doc.createElement("parameters");
            doc.appendChild(root);
        }

        root.setAttribute("operationTable", "v_operation_consolidated");
        root.setAttribute("operationWhereClause", wc);
        root.setAttribute("title", title);
        root.setAttribute("title_icon", "view-income-categories");
        root.setAttribute("currentPage", "-1");

        SKGMainPanel::getMainPanel()->openPage(SKGMainPanel::getMainPanel()->getPluginByName("Skrooge operation plugin"), -1, doc.toString());
    }
}

void SKGBudgetPluginWidget::onCreatorModified()
{
    bool test = !ui.kAmountEdit->text().isEmpty() && !ui.kYear->text().isEmpty();
    ui.kAddBtn->setEnabled(test || ui.kWidgetSelector->getSelectedMode() != 0);
    ui.kModifyBtn->setEnabled((test && ui.kPeriod->currentIndex() != 0 && ui.kWidgetSelector->getSelectedMode() == 0 && getNbSelectedObjects())
                              || (ui.kWidgetSelector->getSelectedMode() == 2 && getNbSelectedObjects() == 1));

    bool monthCondition = (ui.kPeriod->currentIndex() == 2 || ui.kWidgetSelector->getSelectedMode() == 2);
    ui.kMonthLbl->setVisible(monthCondition);
    ui.kMonth->setVisible(monthCondition);
}

void SKGBudgetPluginWidget::onSelectionChanged()
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::onSelectionChanged");
    if (!m_objectModel) return;

    SKGObjectBase::SKGListSKGObjectBase objs = getSelectedObjects();
    int nb = objs.count();
    int mode = ui.kWidgetSelector->getSelectedMode();
    if (nb) {
        if (m_objectModel->getRealTable() == "budget") {
            SKGBudgetObject budget = objs.at(0);
            ui.kYear->setValue(budget.getYear());
            ui.kMonth->setValue(budget.getMonth());
            ui.kAmountEdit->setValue(budget.getBudgetedAmount());
            ui.kCategoryEdit->setText(budget.getAttribute("t_CATEGORY"));
            ui.kPeriod->setCurrentIndex(budget.getMonth() == 0 ? 1 : 2); //Set yearly or individual
            ui.kIncludingSubCategories->setChecked(budget.isSubCategoriesInclusionEnabled());

            if (mode > 0) ui.kWidgetSelector->setSelectedMode(0);
        } else {
            SKGBudgetRuleObject rule = objs.at(0);
            ui.kYearCheck->setChecked(rule.isYearConditionEnabled());
            ui.kYearRule->setValue(rule.getBudgetYear());
            ui.kMonthCheck->setChecked(rule.isMonthConditionEnabled());
            ui.kMonthRule->setValue(rule.getBudgetMonth());
            ui.kCategoryCheck->setChecked(rule.isCategoryConditionEnabled());
            ui.kCategoryRule->setText(rule.getAttribute("t_CATEGORYCONDITION"));
            ui.kCategoryTransferCheck->setChecked(rule.isCategoryChangeEnabled());
            ui.kCategoryTransfer->setText(rule.getAttribute("t_CATEGORY"));
            ui.kUnitCmb->setCurrentIndex(rule.isAbolute() ? 1 : 0);
            ui.kAmountEdit2->setValue(rule.getQuantity());
            ui.kModeCmb->setCurrentIndex(ui.kModeCmb->findData(rule.getTransferMode()));
            ui.kConditionCmb->setCurrentIndex(ui.kConditionCmb->findData(rule.getCondition()));
        }
    }

    ui.kPeriod->setEnabled(nb <= 1);
    ui.kYear->setEnabled(nb <= 1);
    ui.kMonth->setEnabled(nb <= 1);

    onCreatorModified();
    refreshInfoZone();
    Q_EMIT selectionChanged();
}

void SKGBudgetPluginWidget::activateEditor()
{
    if (ui.kWidgetSelector->getSelectedMode() == -1) ui.kWidgetSelector->setSelectedMode(0);
    ui.kAmountEdit->setFocus();
}

bool SKGBudgetPluginWidget::isEditor()
{
    return true;
}

void SKGBudgetPluginWidget::refreshInfoZone()
{
    SKGTRACEIN(10, "SKGBudgetPluginWidget::refreshInfoZone");
    KLocale* locale = KGlobal::locale();
    SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
    if (doc && locale && ui.kWidgetSelector->getSelectedMode() != 2) {
        SKGServices::SKGUnitInfo primary = doc->getPrimaryUnit();
        SKGServices::SKGUnitInfo secondary = doc->getSecondaryUnit();
        //Refresh info area with selection
        SKGObjectBase::SKGListSKGObjectBase selection = getSelectedObjects();

        double budgeted = 0;
        double modified = 0;
        int nb = selection.count();
        for (int i = 0; i < nb; ++i) {
            SKGBudgetObject budget = selection.at(i);
            budgeted += budget.getBudgetedAmount();
            modified += budget.getBudgetedModifiedAmount();
        }

        QString budgetedS = doc->formatMoney(budgeted, primary);
        QString modifiedS = doc->formatMoney(modified, primary);
        QString v = (budgetedS == modifiedS ? budgetedS : modifiedS % " <s><small>" % budgetedS % "</small></s>");
        if (nb) {
            ui.kInfo->setText(i18np("Selection: %1 budget for %2", "Selection: %1 budgets for %2", nb, v));
            if (!secondary.Symbol.isEmpty() && secondary.Value) {
                QString budgetedS = doc->formatMoney(budgeted, secondary);
                QString modifiedS = doc->formatMoney(modified, secondary);
                v = (budgetedS == modifiedS ? budgetedS : modifiedS % " <s><small>" % budgetedS % "</small></s>");
            }
            ui.kInfo->setToolTip(i18np("Selection: %1 budget for %2", "Selection: %1 budgets for %2", nb, v));
        } else {
            ui.kInfo->setText(i18nc("Noun", "Selection: none"));
            ui.kInfo->setToolTip(i18nc("Noun", "Selection: none"));
        }
    }
}
#include "skgbudgetpluginwidget.moc"


