/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
* This file defines classes SKGImportExportManager.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgimportexportmanager.h"
#include "skgtraces.h"
#include "skgservices.h"
#include "skgbankincludes.h"
#include "skgobjectbase.h"
#include "skgruleobject.h"
#include "skgpayeeobject.h"
#include "skgrecurrentoperationobject.h"

#include <klocale.h>
#include <ksavefile.h>

#include <QRegExp>
#include <QFileInfo>
#include <QTextCodec>
#include <QCryptographicHash>
#include <kservicetypetrader.h>
#include "skgimportplugin.h"

SKGImportExportManager::SKGImportExportManager(SKGDocumentBank* iDocument,
        const QString& iFileName)
    : QObject(), m_document(iDocument), m_fileName(iFileName),
      m_csvSeparator(';'), m_csvHeaderIndex(-1),
      m_defaultAccount(NULL), m_defaultUnit(NULL)
{
    setAutomaticValidation(true);
    setAutomaticApplyRules(false);

    m_csvMapper["date"] = "^date";
    m_csvMapper["account"] = "^account";
    m_csvMapper["number"] = "^number|^num?ro";
    m_csvMapper["mode"] = "^mode|^type";
    m_csvMapper["payee"] = "^payee|^tiers";
    m_csvMapper["comment"] = "^comment|^libell?|^d?tail|^info";
    m_csvMapper["status"] = "^status|^pointage";
    m_csvMapper["bookmarked"] = "^bookmarked";
    m_csvMapper["category"] = "^cat\\w*gor\\w*";
    m_csvMapper["amount"] = "^value|^amount|^valeur|^montant";
    m_csvMapper["quantity"] = "^quantity";
    m_csvMapper["unit"] = "^unit";
    m_csvMapper["sign"] = "^sign|^sens";
    m_csvMapper["debit"] = "^-|^debit|^withdrawal";
    m_csvMapper["idgroup"] = "^idgroup";
    m_csvMapper["idtransaction"] = "^idtransaction";
    m_csvMapper["property"] = "";
}

SKGImportExportManager::~SKGImportExportManager()
{
    setDefaultAccount(NULL);
    setDefaultUnit(NULL);
    m_document = NULL;
    m_defaultAccount = NULL;
    m_defaultUnit = NULL;
}


SKGError SKGImportExportManager::setDefaultAccount(SKGAccountObject* iAccount)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGImportExportManager::setDefaultAccount", err);
    delete m_defaultAccount;
    m_defaultAccount = NULL;
    if(iAccount) m_defaultAccount = new SKGAccountObject(*iAccount);
    return err;
}

SKGError SKGImportExportManager::getDefaultAccount(SKGAccountObject& oAccount)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGImportExportManager::getDefaultAccount", err);
    if(m_defaultAccount == NULL && m_document) {
        QFileInfo fInfo(m_fileName);
        QString name = fInfo.baseName();
        name.replace('_', ' ');

        //Searching if an account exist
        QString whereClause = "t_name='" % name % '\'';
        foreach(const QString & val, name.split(' ')) {
            if(!whereClause.isEmpty()) whereClause += " OR ";
            whereClause += "t_number='" % val % '\'';
        }
        if(!whereClause.isEmpty()) {
            SKGObjectBase::SKGListSKGObjectBase listAccount;
            err = m_document->getObjects("v_account", whereClause, listAccount);
            if(!err) {
                if(listAccount.count() == 1) {
                    //Yes ! Only one account found
                    SKGAccountObject account = listAccount.at(0);
                    m_defaultAccount = new SKGAccountObject(account);
                    err = m_document->sendMessage(i18nc("An information message",  "Using account '%1' for import", account.getName()));
                } else if(listAccount.count() > 1) {
                    err = m_document->sendMessage(i18nc("An information message",  "More than one possible account found."));
                }
            }
        }

        //If better account not found, then we must create one
        if(m_defaultAccount == NULL) {
            SKGAccountObject account;
            SKGBankObject bank(m_document);
            if(!err) err = bank.setName(name);
            if(!err && bank.load().isFailed()) err = bank.save(false);     //Save only
            if(!err) err = bank.addAccount(account);
            if(!err) err = account.setName(name);
            if(!err && account.load().isFailed()) err = account.save(false);     //Save only
            if(!err) m_defaultAccount = new SKGAccountObject(account);
            if(!err) err = m_document->sendMessage(i18nc("An information message",  "Default account '%1' created for import", name));
        }
    }

    if(m_defaultAccount != NULL)  oAccount = *m_defaultAccount;

    return err;
}

SKGError SKGImportExportManager::setDefaultUnit(SKGUnitObject* iUnit)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGImportExportManager::setDefaultUnit", err);
    delete m_defaultUnit;
    m_defaultUnit = NULL;
    if(iUnit) m_defaultUnit = new SKGUnitObject(*iUnit);
    return err;
}

SKGError SKGImportExportManager::getDefaultUnit(SKGUnitObject& oUnit, const QDate* iDate)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGImportExportManager::getDefaultUnit", err);
    if(m_document && (m_defaultUnit == NULL || iDate)) {
        if(m_defaultUnit == NULL) {
            delete m_defaultUnit;
            m_defaultUnit = NULL;
        }

        //Do we have to found the best unit for a date ?
        QString wc = "t_type IN ('1', '2', 'C')";
        if(iDate) {
            //Yes
            wc += " AND d_MINDATE<'" % SKGServices::dateToSqlString(QDateTime(*iDate)) % '\'';
        }

        //Check if a unit exist
        SKGObjectBase::SKGListSKGObjectBase listUnits;
        err = m_document->getObjects("v_unit", wc % " ORDER BY ABS(f_CURRENTAMOUNT-1) ASC" , listUnits);
        if(!err) {
            if(listUnits.count() == 0) {
                //Not found, we have to create one
                QDateTime now = QDateTime::currentDateTime();

                SKGUnitObject unit(m_document);
                QString name = i18nc("Noun",  "Unit for import");
                err = unit.setName(name);
                if(unit.load().isFailed()) {
                    if(!err) err = unit.setSymbol(name);
                    if(!err) err = unit.save(false);     //Save only

                    SKGUnitValueObject unitval;
                    if(!err) err = unit.addUnitValue(unitval);
                    if(!err) err = unitval.setQuantity(1);
                    if(!err) err = unitval.setDate(QDate(1970, 1, 1));
                    if(!err) err = unitval.save(false, false);     //Save only without reload

                    if(!err) err = m_document->sendMessage(i18nc("An information message",  "Default unit '%1' created for import", name));
                }

                if(!err) m_defaultUnit = new SKGUnitObject(unit);
            } else {
                //Found, we can use it
                m_defaultUnit = new SKGUnitObject((SKGUnitObject) listUnits.at(0));
            }
        }
    }

    if(m_defaultUnit != NULL) {
        oUnit = *m_defaultUnit;
    }

    return err;
}

void SKGImportExportManager::setAutomaticValidation(bool iValidation)
{
    m_automaticValidationOfImportedOperation = iValidation;
}

void SKGImportExportManager::setAutomaticApplyRules(bool iValidation)
{
    m_automaticApplyRulesOfImportedOperation = iValidation;
}

SKGError SKGImportExportManager::finalizedImportedOperations()
{
    SKGError err;
    SKGTRACEINRC(2, "SKGImportExportManager::finalizedImportedOperations", err);
    if(m_document) {
        //Count the number of operations imported but already existing
        QString wc = "t_imported='T' AND v_operation.t_import_id<>'' AND exists (SELECT 1 from v_operation op2 WHERE op2.id!=v_operation.id AND op2.t_imported!='T' AND op2.rd_account_id=v_operation.rd_account_id AND op2.t_import_id=v_operation.t_import_id AND op2.f_CURRENTAMOUNT=v_operation.f_CURRENTAMOUNT)";
        int nb;
        err = m_document->getNbObjects("v_operation", wc, nb);
        if(!err && nb) {
            err = m_document->sendMessage(i18np("one operation not imported because it already exists", "%1 operations not imported because they already exists", nb));

            //Remove operations imported for nothing
            if(!err) err = m_document->executeSqliteOrder("DELETE from operation WHERE id IN (SELECT id from v_operation WHERE " % wc % ')');
        }

        //Apply rules
        if(!err && m_automaticApplyRulesOfImportedOperation) {
            //Get rules
            SKGObjectBase::SKGListSKGObjectBase rules;
            err = m_document->getObjects("v_rule", "1=1 ORDER BY f_sortorder", rules);

            int nb = rules.count();
            err = m_document->beginTransaction("#INTERNAL#", nb);
            for(int i = 0; !err && i < nb; ++i) {
                SKGRuleObject rule = rules.at(i);
                err = rule.execute(SKGRuleObject::IMPORTING);
                if(!err) err = m_document->stepForward(i + 1);
            }
            if(!err) err = m_document->endTransaction(true);
            else m_document->endTransaction(false);
        }

        //Change imported status
        if(!err) err = m_document->executeSqliteOrder(QString("UPDATE operation SET t_imported='") %
                           (m_automaticValidationOfImportedOperation ? "Y" : "P") %
                           "' WHERE t_imported='T'");
    }
    return err;
}

QString SKGImportExportManager::getImportMimeTypeFilter()
{
    QString output;
    QString all;
    KService::List offers = KServiceTypeTrader::self()->query(QLatin1String("SKG IMPORT/Plugin"));
    int nb = offers.count();
    for(int i = 0; i < nb; ++i) {
        KService::Ptr service = offers.at(i);
        KPluginLoader loader(service->library());
        KPluginFactory *factory = loader.factory();
        if(factory) {
            SKGImportPlugin *pluginInterface = factory->create<SKGImportPlugin> (0);
            if(pluginInterface && pluginInterface->isImportPossible()) {
                QString mime = pluginInterface->getMimeTypeFilter();
                if(!mime.isEmpty()) {
                    if(!output.isEmpty()) output += '\n';
                    output += mime;

                    if(!all.isEmpty()) all += ' ';
                    all += SKGServices::splitCSVLine(mime, '|').at(0);
                }
            }
        }
    }
    return all % '|' % i18nc("A file format", "All supported formats") % '\n' % output;
}

QString SKGImportExportManager::getExportMimeTypeFilter()
{
    QString output;
    QString all;
    KService::List offers = KServiceTypeTrader::self()->query(QLatin1String("SKG IMPORT/Plugin"));
    int nb = offers.count();
    for(int i = 0; i < nb; ++i) {
        KService::Ptr service = offers.at(i);
        KPluginLoader loader(service->library());
        KPluginFactory *factory = loader.factory();
        if(factory) {
            SKGImportPlugin *pluginInterface = factory->create<SKGImportPlugin> (0);
            if(pluginInterface && pluginInterface->isExportPossible()) {
                QString mime = pluginInterface->getMimeTypeFilter();
                if(!mime.isEmpty()) {
                    if(!output.isEmpty()) output += '\n';
                    output += mime;

                    if(!all.isEmpty()) all += ' ';
                    all += SKGServices::splitCSVLine(mime, '|').at(0);
                }
            }
        }
    }
    return all % '|' % i18nc("A file format", "All supported formats") % '\n' % output;
}

SKGError SKGImportExportManager::importFile()
{
    SKGError err;
    SKGTRACEINRC(2, "SKGImportExportManager::importFile", err);
    if(m_document) {
        err = m_document->executeSqliteOrder("ANALYZE");
        if(!err) {
            //Search plugins
            bool fileTreated = false;
            KService::List offers = KServiceTypeTrader::self()->query(QLatin1String("SKG IMPORT/Plugin"));
            int nb = offers.count();
            SKGTRACEL(1) << nb << " plugins found" << endl;
            for(int i = 0; !err && i < nb; ++i) {
                KService::Ptr service = offers.at(i);
                QString id = service->property("X-Krunner-ID", QVariant::String).toString();
                KPluginLoader loader(service->library());
                KPluginFactory *factory = loader.factory();
                if(factory) {
                    SKGImportPlugin *pluginInterface = factory->create<SKGImportPlugin> (this);
                    if(pluginInterface && pluginInterface->isImportPossible()) {
                        //Import
                        fileTreated = true;
                        err = pluginInterface->importFile();
                    }
                } else {
                    err = m_document->sendMessage(i18nc("An information message",  "Loading plugin %1 failed because the factory could not be found in %2", id, service->library()));
                }
            }

            if(!err && !fileTreated) {
                err.setReturnCode(ERR_NOTIMPL);
                err.setMessage(i18nc("Error message",  "This import mode is not yet implemented"));
            }
        }

        if(!err) err = finalizedImportedOperations();
    }

    return err;
}

SKGError SKGImportExportManager::exportFile()
{
    SKGError err;
    SKGTRACEINRC(2, "SKGImportExportManager::exportFile", err);
    if(m_document) {
        err = m_document->executeSqliteOrder("ANALYZE");
        if(!err) {
            //Search plugins
            bool fileTreated = false;
            KService::List offers = KServiceTypeTrader::self()->query(QLatin1String("SKG IMPORT/Plugin"));
            int nb = offers.count();
            SKGTRACEL(1) << nb << " plugins found" << endl;
            for(int i = 0; !err && i < nb; ++i) {
                KService::Ptr service = offers.at(i);
                QString id = service->property("X-Krunner-ID", QVariant::String).toString();
                KPluginLoader loader(service->library());
                KPluginFactory *factory = loader.factory();
                if(factory) {
                    SKGImportPlugin *pluginInterface = factory->create<SKGImportPlugin> (this);
                    if(pluginInterface && pluginInterface->isExportPossible()) {
                        //Export
                        fileTreated = true;
                        err = pluginInterface->exportFile();
                    }
                } else {
                    err = m_document->sendMessage(i18nc("An information message",  "Loading plugin %1 failed because the factory could not be found in %2", id, service->library()));
                }
            }

            if(!err && !fileTreated) {
                err.setReturnCode(ERR_NOTIMPL);
                err.setMessage(i18nc("Error message",  "This export mode is not yet implemented"));
            }
        }
    }

    return err;
}

QStringList SKGImportExportManager::getCSVMappingFromLine(const QString& iLine)
{
    QStringList output;
    QString line = iLine.trimmed();

    //Split first line
    QStringList csvAttributes = SKGServices::splitCSVLine(line, ';', true, &m_csvSeparator);
    int nb = csvAttributes.count();
    for(int i = 0; i < nb; ++i) {
        QString att = csvAttributes[i].toLower();

        //Search if this csv column is mapping a std attribute
        QMapIterator<QString, QString> csvMapperIterator(m_csvMapper);
        bool found = false;
        while(!found && csvMapperIterator.hasNext()) {
            csvMapperIterator.next();

            QString key = csvMapperIterator.key();
            if(key != "debit" &&
                    key != "property" &&
                    !csvMapperIterator.value().isEmpty() &&
                    QRegExp(csvMapperIterator.value(), Qt::CaseInsensitive).indexIn(att) != -1 &&
                    (!output.contains(key) || key == "comment" || key == "category" || key == "amount")) {
                output.push_back(key);
                found = true;
            }
        }

        //Search if this csv column must be added as a property
        if(!found &&
                !m_csvMapper["property"].isEmpty() &&
                QRegExp(m_csvMapper["property"], Qt::CaseInsensitive).indexIn(att) != -1 &&
                !output.contains(att)) {
            output.push_back(att);
            found = true;
        }

        if(!found) output.push_back("");       //To ignore this column
    }
    return output;
}

QMap<QString, QString> SKGImportExportManager::getCSVMappingRules()
{
    return m_csvMapper;
}

SKGError SKGImportExportManager::setCSVMappingRules(const QMap<QString, QString>& iCSVMappingRules)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGImportExportManager::setCSVMappingRules", err);
    m_csvMapper = iCSVMappingRules;
    return err;
}

SKGError SKGImportExportManager::setCSVMapping(const QStringList* iCSVMapping)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGImportExportManager::setCSVMapping", err);

    m_csvMapping.clear();

    if(iCSVMapping == NULL) {
        //Automatic build
        //Open file
        QFile file(m_fileName);
        if(!file.open(QIODevice::ReadOnly)) {
            err.setReturnCode(ERR_INVALIDARG);
            err.setMessage(i18nc("Error message",  "Open file '%1' failed", m_fileName));
        } else {
            QTextStream stream(&file);
            if(!m_codec.isEmpty()) stream.setCodec(m_codec.toAscii().constData());

            //Ignore useless lines
            int headerIndex = getCSVHeaderIndex();
            for(int i = 1; i < headerIndex; ++i)
                stream.readLine();

            //Get mapping
            if(!stream.atEnd()) {
                m_csvMapping = getCSVMappingFromLine(stream.readLine());
            } else {
                err.setReturnCode(ERR_INVALIDARG);
            }

            //close file
            file.close();
        }
    } else {
        //Manual build
        m_csvMapping = *iCSVMapping;
    }

    if(!err) {
        //Check if mandatory attributes have been found
        if(!m_csvMapping.contains("date") || !m_csvMapping.contains("amount"))
            err = SKGError(ERR_FAIL, i18nc("Error message",  "Columns date and amount not found. Set import parameters in settings."));
    }

    return err;
}

QStringList SKGImportExportManager::getCSVMapping() const
{
    SKGTRACEIN(10, "SKGImportExportManager::getCSVMapping");
    return m_csvMapping;
}

SKGError SKGImportExportManager::setCSVHeaderIndex(int iIndex)
{
    SKGError err;
    SKGTRACEINRC(10, "SKGImportExportManager::setCSVHeaderIndex", err);

    if(iIndex == -1) {
        //Automatic build
        //Open file
        QFile file(m_fileName);
        if(!file.open(QIODevice::ReadOnly)) {
            err.setReturnCode(ERR_INVALIDARG);
            err.setMessage(i18nc("Error message",  "Open file '%1' failed", m_fileName));
        } else {
            QTextStream stream(&file);
            if(!m_codec.isEmpty()) stream.setCodec(m_codec.toAscii().constData());

            int i = 1;
            m_csvHeaderIndex = -1;
            while(!stream.atEnd() && m_csvHeaderIndex == -1) {
                //Read line
                QStringList map = getCSVMappingFromLine(stream.readLine());
                if(map.contains("date") && map.contains("amount")) m_csvHeaderIndex = i;

                ++i;
            }

            //close file
            file.close();
        }
    } else {
        //Manual build
        m_csvHeaderIndex = iIndex;
    }

    return err;
}

int SKGImportExportManager::getCSVHeaderIndex()
{
    SKGTRACEIN(10, "SKGImportExportManager::getCSVHeaderIndex");
    if(m_csvHeaderIndex == -1) setCSVHeaderIndex(-1);
    return m_csvHeaderIndex;
}

QChar SKGImportExportManager::getCSVSeparator()
{
    SKGTRACEIN(10, "SKGImportExportManager::getCSVSeparator");
    return m_csvSeparator;
}

SKGError SKGImportExportManager::importCSVUnit()
{
    SKGError err;
    SKGTRACEINRC(2, "SKGImportExportManager::importCSVUnit", err);
    SKGTRACEL(10) << "Input filename=" << m_fileName << endl;

    if(m_document) {
        //Begin transaction
        err = m_document->beginTransaction("#INTERNAL#", 3);
        if(!err) {
            //File name is the name of the unit
            QFileInfo fInfo(m_fileName);
            QString unitName = fInfo.baseName();

            //Default mapping
            if(getCSVMapping().count() == 0) {
                err = setCSVMapping(NULL);
                if(!err) err = m_document->sendMessage(i18nc("An information message",  "Use automatic search of the columns"));
            }

            //Step 1 done
            if(!err) err = m_document->stepForward(1);

            //Open file
            if(!err) {
                QFile file(m_fileName);
                if(!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
                    err.setReturnCode(ERR_INVALIDARG);
                    err.setMessage(i18nc("Error message",  "Open file '%1' failed", m_fileName));
                } else {
                    QTextStream stream(&file);
                    if(!m_codec.isEmpty()) stream.setCodec(m_codec.toAscii().constData());

                    //Ignore useless lines
                    int headerIndex = getCSVHeaderIndex();
                    for(int i = 1; i <= headerIndex; ++i)
                        stream.readLine();

                    //Get data column
                    QStringList dates;
                    QStringList lines;
                    int posdate = m_csvMapping.indexOf("date");
                    if(posdate != -1) {
                        while(!stream.atEnd()) {
                            //Read line
                            QString line = stream.readLine().trimmed();
                            if(!line.isEmpty()) {
                                lines.push_back(line);

                                //Get date
                                QStringList field = SKGServices::splitCSVLine(line);
                                if(posdate < field.count()) dates.push_back(field.at(posdate));
                            }
                        }
                    }

                    //close file
                    file.close();

                    //Select dateformat
                    QString dateFormat = SKGServices::getDateFormat(dates);
                    if(dateFormat.isEmpty()) {
                        err.setReturnCode(ERR_FAIL);
                        err.setMessage(i18nc("Error message",  "Date format not supported"));
                    }
                    if(!err)
                        err = m_document->sendMessage(i18nc("An information message",  "Import of '%1' with code '%2' and date format '%3'", m_fileName, m_codec, dateFormat));

                    //Step 2 done
                    if(!err) err = m_document->stepForward(2);

                    //Treat all lines
                    if(!err) {
                        int nb = lines.size();
                        err = m_document->beginTransaction("#INTERNAL#", nb);

                        //Save last mapping used in a settings
                        QString mappingDesc;
                        int nbMap = m_csvMapping.count();
                        for(int i = 0; i < nbMap; ++i) {
                            if(i) mappingDesc += '|';
                            mappingDesc += m_csvMapping.at(i);
                        }
                        if(!err) err = m_document->setParameter("SKG_LAST_CSV_UNIT_MAPPING_USED", mappingDesc);

                        int posdate = m_csvMapping.indexOf("date");
                        int posvalue = m_csvMapping.indexOf("amount");
                        if(posdate != -1 && posvalue != -1) {
                            for(int i = 0; !err && i < nb; ++i) {
                                QStringList atts = SKGServices::splitCSVLine(lines.at(i));
                                err = m_document->addOrModifyUnitValue(unitName,
                                                                       SKGServices::stringToTime(SKGServices::dateToSqlString(atts.at(posdate), dateFormat)).date(),
                                                                       SKGServices::stringToDouble(atts.at(posvalue)));

                                if(!err) err = m_document->stepForward(i + 1);
                            }
                        }

                        if(!err) err = m_document->endTransaction(true);
                        else  m_document->endTransaction(false);

                        //Lines treated
                        if(!err) err = m_document->stepForward(3);
                    }
                }
            }
        }
        if(!err) err = m_document->endTransaction(true);
        else  m_document->endTransaction(false);
    }

    return err;
}

SKGError SKGImportExportManager::cleanBankImport()
{
    SKGError err;
    SKGTRACEINRC(2, "SKGImportExportManager::cleanBankImport", err);

    //Begin transaction
    if(m_document) {
        err = m_document->beginTransaction("#INTERNAL#", 3);
        if(!err) {
            //Step 1 Clean operations without mode and with comment with double space
            SKGObjectBase::SKGListSKGObjectBase operations;
            if(!err) err = m_document->getObjects("operation", "t_imported!='N' and t_mode='' and t_comment like '%  %'", operations);

            int nb = operations.count();
            for(int i = 0; !err && i < nb ; ++i) {
                SKGOperationObject op = operations[i];

                //Comment is like this: <TYPE>  <INFO>
                //Example: RETRAIT DAB             20/01/08 11H44 013330 LCL GAILLAC 000497
                QRegExp rx("(.+) {2,}(.+)");
                QString comment = op.getComment();
                if(rx.indexIn(comment) != -1) {
                    //Get parameters
                    QString mode = rx.cap(1);
                    QString info = rx.cap(2);

                    //Modify
                    err = op.setComment(info.trimmed());
                    if(!err)  err = op.setMode(mode.trimmed());
                    if(!err)  err = op.save(true, false);     //No reload
                }
            }

            //Step 1 done
            if(!err) err = m_document->stepForward(1);

            //Step 2 Clean operations without mode and with comment
            if(!err) err = m_document->getObjects("operation", "t_imported!='N' and t_mode='' and t_comment!=''", operations);

            nb = operations.count();
            for(int i = 0; !err && i < nb ; ++i) {
                SKGOperationObject op = operations[i];

                //Comment is like this: <TYPE> <INFO>
                //Example: RETRAIT DAB 14-05-16607-482390
                QRegExp rx("(\\S+) +(.+)");
                QString comment = op.getComment();
                if(rx.indexIn(comment) != -1) {
                    //Get parameters
                    QString mode = rx.cap(1);
                    QString info = rx.cap(2);

                    //Modify
                    err = op.setComment(info.trimmed());
                    if(!err)  err = op.setMode(mode.trimmed());
                    if(!err)  err = op.save(true, false);     //No reload
                }
            }

            //Step 2 done
            if(!err) err = m_document->stepForward(2);

            //Step 3 Clean cheque without number
            if(!err) err = m_document->getObjects("operation", "t_imported!='N' and i_number=0 and lower(t_mode)='cheque'", operations);

            nb = operations.count();
            for(int i = 0; !err && i < nb ; ++i) {
                SKGOperationObject op = operations[i];

                //Comment is like this: <TYPE>  <INFO>
                //Example: RETRAIT DAB             20/01/08 11H44 013330 LCL GAILLAC 000497
                QRegExp rx("(\\d+)");
                QString comment = op.getComment();
                if(rx.indexIn(comment) != -1) {
                    //Get parameters
                    int number = SKGServices::stringToInt(rx.cap(1));

                    //Modify
                    err = op.setNumber(number);
                    if(!err)  err = op.save(true, false);     //No reload
                }
            }

            //Step 3 done
            if(!err) err = m_document->stepForward(3);
        }

        if(!err) err = m_document->endTransaction(true);
        else  m_document->endTransaction(false);
    }

    return err;
}

SKGError SKGImportExportManager::anonymize()
{
    SKGError err;
    SKGTRACEINRC(2, "SKGImportExportManager::anonymize", err);
    if(m_document) {
        if(m_document->isFileModified()) {
            err = SKGError(ERR_ABORT, i18nc("An information message",  "The document must be saved to be anonymized."));
        } else {
            {
                SKGBEGINTRANSACTION(*m_document, "##INTERNAL##", err);
                if(!err) {
                    QStringList sqlOrders;
                    sqlOrders << "UPDATE bank SET t_bank_number='', t_name='bank_'||id"
                              << "UPDATE account SET t_number='', t_agency_number='', t_agency_address='', t_comment='', t_name='account_'||id"
                              << "UPDATE category SET t_name='category_'||id"
                              << "UPDATE payee SET t_address='', t_name='payee_'||id"
                              << "UPDATE refund SET t_comment='', t_name='tracker_'||id"
                              << "UPDATE operation SET t_comment=''"
                              << "UPDATE suboperation SET t_comment='', f_value=f_value%1234.56"
                              << "DELETE FROM parameters WHERE t_name='SKG_PASSWORD'"
                              << "DELETE FROM parameters WHERE t_name NOT LIKE 'SKG_%'"
                              ;
                    err = m_document->executeSqliteOrders(sqlOrders);
                }
            }

            if(!err) err = m_document->removeAllTransactions();
        }
    }
    return err;
}

SKGError SKGImportExportManager::findAndGroupTransfers(int& oNbOperationsMerged)
{
    SKGError err;
    SKGTRACEINRC(2, "SKGImportExportManager::findAndGroupTransfers", err);

    oNbOperationsMerged = 0;

    //Begin transaction
    if(m_document) {
        err = m_document->beginTransaction("#INTERNAL#", 3);
        if(!err) {
            //Look for operations with
            //  Same units
            //  Same dates
            //  Null i_group_id
            //  Different accounts
            //  Oposite amounts
            SKGStringListList listTmp;
            //+A.i_group_id=0 AND +B.i_group_id=0 is for avoiding to use bad index
            err = m_document->executeSelectSqliteOrder(
                      "SELECT A.id, B.id FROM v_operation_tmp1 A, v_operation_tmp1 B WHERE A.id<=B.id AND A.rc_unit_id=B.rc_unit_id AND A.d_date=B.d_date AND A.rd_account_id!=B.rd_account_id AND A.f_QUANTITY=-B.f_QUANTITY AND +A.i_group_id=0 AND +B.i_group_id=0 AND A.f_QUANTITY!=0",
                      listTmp);
            //Step 1 done
            if(!err) err = m_document->stepForward(1);

            SKGStringListList listTmp2;
            //+A.i_group_id=0 AND +B.i_group_id=0 is for avoiding to use bad index
            if(!err) err = m_document->executeSelectSqliteOrder(
                                   "SELECT A.id, B.id FROM v_operation A, v_operation_tmp1 B, parameters P WHERE P.t_uuid_parent=B.id||'-operation' AND A.rc_unit_id!=B.rc_unit_id AND P.t_name='SKG_OP_ORIGINAL_AMOUNT' AND A.d_date=B.d_date AND A.rd_account_id!=B.rd_account_id AND A.f_CURRENTAMOUNT=-P.t_value AND +A.i_group_id=0 AND +B.i_group_id=0 AND A.f_CURRENTAMOUNT!=0",
                                   listTmp2);

            //Step 2 done
            if(!err) err = m_document->stepForward(2);
            listTmp2.removeAt(0);    //Remove header
            listTmp += listTmp2;

            //Group
            {
                oNbOperationsMerged = listTmp.count();
                if(!err) err = m_document->beginTransaction("#INTERNAL#", oNbOperationsMerged - 1);
                for(int i = 1; !err && i < oNbOperationsMerged ; ++i) {  //First line ignored because of it's header
                    SKGOperationObject op1(m_document, SKGServices::stringToInt(listTmp.at(i).at(0)));
                    SKGOperationObject op2(m_document, SKGServices::stringToInt(listTmp.at(i).at(1)));

                    if(!op1.isInGroup() && !op2.isInGroup()) {
                        err = op2.setGroupOperation(op1);
                        if(!err) err = op2.save(true, false);     //No reload
                    }
                    if(!err) err = m_document->stepForward(i);
                }
                if(!err) err = m_document->endTransaction(true);
                else  m_document->endTransaction(false);
            }
            oNbOperationsMerged = (oNbOperationsMerged - 1) * 2;

            //Step 3 done
            if(!err) err = m_document->stepForward(3);

        }
        if(!err) err = m_document->endTransaction(true);
        else  m_document->endTransaction(false);
    }

    return err;
}

#include "skgimportexportmanager.moc"
