/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a plugin for file operation.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgfileplugin.h"
#include "skgfile_settings.h"

#include "skgtraces.h"
#include "skgerror.h"
#include "skgmainpanel.h"
#include "skgtransactionmng.h"

#include <kactioncollection.h>
#include <kstandardaction.h>
#include <krecentfilesaction.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <ksplashscreen.h>
#include <kcmdlineargs.h>
#include <kaboutdata.h>
#include <knewpassworddialog.h>
#include <kpassworddialog.h>
#include <kwallet.h>

#include <QtGui/QWidget>
#include <QFile>

/**
 * This plugin factory.
 */
K_PLUGIN_FACTORY ( SKGFilePluginFactory, registerPlugin<SKGFilePlugin>(); )
/**
 * This plugin export.
 */
K_EXPORT_PLUGIN ( SKGFilePluginFactory ( "skg_file", "skg_file" ) )

SKGFilePlugin::SKGFilePlugin ( QObject* iParent, const QVariantList& /*iArg*/ ) : SKGInterfacePlugin ( iParent )
{
    SKGTRACEIN ( 10, "SKGFilePlugin::SKGFilePlugin" );

    //Set save on close mode
    SKGMainPanel::getMainPanel()->setSaveOnClose ( skgfile_settings::saveonclose() );
}

SKGFilePlugin::~SKGFilePlugin()
{
    SKGTRACEIN ( 10, "SKGFilePlugin::~SKGFilePlugin" );
    if ( recentFiles ) recentFiles->saveEntries ( KConfigGroup ( KGlobal::config(), "RecentFiles" ) );

    currentDocument=NULL;
    recentFiles=NULL;
    saveAction=NULL;
    saveAsAction=NULL;
    passwordAction=NULL;
}

bool SKGFilePlugin::setupActions ( SKGDocument* iDocument, const QStringList& iArgument )
{
    SKGTRACEIN ( 10, "SKGFilePlugin::setupActions" );

    currentDocument=iDocument;
    if (currentDocument==NULL) return false;

    setComponentData ( SKGFilePluginFactory::componentData() );
    setXMLFile ( "skg_file.rc" );

    //Menu
    KAction* opennew=KStandardAction::openNew ( this, SLOT ( actionNew() ), actionCollection() );
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->registedGlobalAction ( "file_new", opennew );

    KAction* open=KStandardAction::open ( this, SLOT ( actionOpen() ), actionCollection() );
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->registedGlobalAction ( "file_open", open );

    saveAction=KStandardAction::save ( this, SLOT ( actionSave() ), actionCollection() );
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->registedGlobalAction ( "file_save", saveAction );

    saveAsAction=KStandardAction::saveAs ( this, SLOT ( actionSaveAs() ), actionCollection() );
    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->registedGlobalAction ( "file_save_as", saveAsAction );

    passwordAction = new KAction ( KIcon ( "document-encrypt" ),  i18nc ("Verb", "Change password..." ), this );
    connect ( passwordAction, SIGNAL ( triggered ( bool ) ), this, SLOT ( actionChangePassword() ) );
    actionCollection()->addAction ( QLatin1String ( "file_change_password" ), passwordAction );
    passwordAction->setShortcut ( Qt::CTRL+Qt::Key_K );

    if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->registedGlobalAction ( "file_change_password", passwordAction );

    //Recent file
    recentFiles=KStandardAction::openRecent ( this,SLOT ( actionOpen ( const KUrl ) ), actionCollection() );
    if ( recentFiles ) recentFiles->loadEntries ( KConfigGroup ( KGlobal::config(), "RecentFiles" ) );

    //Get last argument
    connect ( this, SIGNAL ( loadFile ( const KUrl ) ), this, SLOT ( actionOpen ( const KUrl ) ), Qt::QueuedConnection );

    int nbArg=iArgument.count();
    int openMode=1; //0=no open, 1=open last opened if settings set, 2=new document
    if ( nbArg )
    {
        openMode=2;
        QString filename=iArgument.at ( nbArg-1 );
        QString extension=QFileInfo ( filename ).suffix().toUpper();
        QString extensionDocument=currentDocument->getFileExtension().toUpper();
        if ( QFile ( filename ).exists() && extension==extensionDocument )
        {
            processArguments ( iArgument );
            openMode=0;
        }
    }

    if ( openMode )
    {
        //Read Setting
        bool openlastfile = skgfile_settings::openlastfile();
        if ( openMode==1 && openlastfile )
        {
            SKGTRACEIN ( 10, "SKGFilePlugin::openlastfile" );
            QString lastOpenedFile=skgfile_settings::lastfilepath();
            if ( !lastOpenedFile.isEmpty() && QFile ( lastOpenedFile ).exists() )
            {
                if ( SKGMainPanel::getMainPanel() )
                {
                    KSplashScreen* splashScreen=SKGMainPanel::getMainPanel()->splashScreen();
                    if ( splashScreen ) splashScreen->showMessage ( i18nc("Splash screen message",  "Opening file %1...", lastOpenedFile ), Qt::AlignLeft, QColor ( 221, 130, 8 ) ); // krazy:exclude=qmethods
                }
                Q_EMIT loadFile ( lastOpenedFile );
            }
            else openMode=2;
        }

        if ( openMode==2 )
        {
            actionNew();
        }
    }

    //To be sure that the document has the right parameters
    savePreferences();

    return true;
}

void SKGFilePlugin::processArguments ( const QStringList& iArgument )
{
    SKGTRACEIN ( 10, "SKGFilePlugin::processArguments" );
    int nbArg=iArgument.count();
    if ( nbArg && currentDocument )
    {
        QString filename=iArgument.at ( nbArg-1 );
        QString extension=QFileInfo ( filename ).suffix().toUpper();
        QString extensionDocument=currentDocument->getFileExtension().toUpper();
        if ( QFile ( filename ).exists() && extension==extensionDocument )
        {
            if ( SKGMainPanel::getMainPanel() )
            {
                KSplashScreen* splashScreen=SKGMainPanel::getMainPanel()->splashScreen();
                if ( splashScreen ) splashScreen->showMessage ( i18nc("Splash screen message", "Opening file %1...", filename ), Qt::AlignLeft, QColor ( 221, 130, 8 ) ); // krazy:exclude=qmethods
            }
            Q_EMIT loadFile ( filename );
        }
    }
}

QWidget* SKGFilePlugin::getPreferenceWidget()
{
    SKGTRACEIN ( 10, "SKGFilePlugin::getPreferenceWidget" );
    QWidget* widget=new QWidget();
    ui.setupUi ( widget );

    ui.kcfg_prefix->addItem ( "" );
    ui.kcfg_prefix->addItem ( "." );

    ui.kcfg_suffix->addItem ( ".old" );
    ui.kcfg_suffix->addItem ( ".back" );
    ui.kcfg_suffix->addItem ( ".<DATE>.back" );
    ui.kcfg_suffix->addItem ( ".<DATE>.old" );
    ui.kcfg_suffix->addItem ( "~" );

    return widget;
}

KConfigSkeleton* SKGFilePlugin::getPreferenceSkeleton()
{
    return skgfile_settings::self();
}

SKGError SKGFilePlugin::savePreferences() const
{
    SKGError err;
    if ( currentDocument )
    {
        //Read Setting
        QString prefix;
        QString suffix;
        if ( skgfile_settings::backup_enabled() )
        {
            prefix = skgfile_settings::prefix();
            suffix = skgfile_settings::suffix();
        }

        //Save setting in document
        currentDocument->setBackupParameters ( prefix, suffix );

        //Set save on close mode
        SKGMainPanel::getMainPanel()->setSaveOnClose ( skgfile_settings::saveonclose() );
    }
    return err;
}

void SKGFilePlugin::refresh()
{
    SKGTRACEIN ( 10, "SKGFilePlugin::refresh" );

    //Refresh action status
    if ( currentDocument )
    {
        bool test= ( currentDocument->getDatabase() !=NULL );
        if ( saveAction ) saveAction->setEnabled ( currentDocument->isFileModified() );
        if ( saveAsAction ) saveAsAction->setEnabled ( test );
        if ( passwordAction ) passwordAction->setEnabled ( test );
    }
}

QString SKGFilePlugin::title() const
{
    return i18nc ( "Noun, a file as in a text file","File" );
}

QString SKGFilePlugin::icon() const
{
    return "document-save";
}

QString SKGFilePlugin::toolTip () const
{
    return i18nc("Noun", "File management" );
}

QStringList SKGFilePlugin::tips() const
{
    QStringList output;
    output.push_back(i18nc("Description of a tips", "<p>... the last opened file can be open automatically when the application is launched.</p>" ) );
    output.push_back(i18nc("Description of a tips", "<p>... you can secure your document with a password.</p>" ) );
    return output;
}

int SKGFilePlugin::getOrder() const
{
    //Must be one of the first
    return 1;
}

void SKGFilePlugin::close()
{
    SKGTRACEIN ( 10, "SKGFilePlugin::close" );
}

void SKGFilePlugin::actionNew()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGFilePlugin::actionNew",err );
    if ( SKGMainPanel::getMainPanel() && currentDocument && SKGMainPanel::getMainPanel()->queryFileClose() )
    {
        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );

        SKGMainPanel::getMainPanel()->closeAllPages();

        err=currentDocument->initialize();

        KLocale* locale=KGlobal::locale();
        if ( err.isSucceeded() ) err=currentDocument->setLanguage ( locale->language() );
        QApplication::restoreOverrideCursor();

        //status bar
        if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "Document successfully created." ) );
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Document creation failed." ) );

        //Display error
        if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGFilePlugin::actionSave()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGFilePlugin::actionSave",err );
    if ( currentDocument && SKGMainPanel::getMainPanel() )
    {
        if ( currentDocument->getCurrentFileName().length() ==0 ) actionSaveAs();
        else
        {
            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            err=currentDocument->save();
            QApplication::restoreOverrideCursor();

            //Refresh
            SKGMainPanel::getMainPanel()->refresh();

            //status bar
            if ( err.isSucceeded() ) err=SKGError(0, i18nc("Successful message after an user action", "File successfully saved." ) );
            else err.addError ( ERR_FAIL, i18nc("Error message",  "Cannot save file" ) );

            //Display error
            SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
        }
    }
}

void SKGFilePlugin::actionSaveAs()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGFilePlugin::actionSaveAs",err );
    if ( currentDocument && SKGMainPanel::getMainPanel() )
    {
        QString fileName=SKGMainPanel::getSaveFileName ( KUrl ( "kfiledialog:///"+currentDocument->objectName() ),
                         "*."+currentDocument->getFileExtension()+'|'+i18nc ( "File format", "%1 document", KCmdLineArgs::aboutData()->programName() ) ,
                         SKGMainPanel::getMainPanel() );
        if ( fileName.isEmpty() ) return;

        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
        err=currentDocument->saveAs ( fileName, true );
        QApplication::restoreOverrideCursor();

        //Refresh
        SKGMainPanel::getMainPanel()->refresh();

        //status bar
        if ( err.isSucceeded() )
        {
            err=SKGError(0, i18nc("Successful message after an user action", "File '%1' saved.", fileName ) );
            //Add in recentFiles
            if ( recentFiles )
            {
                recentFiles->addUrl ( fileName );
                recentFiles->saveEntries ( KConfigGroup ( KGlobal::config(), "RecentFiles" ) );
            }
            //Set as last open file in kcfg
            KSharedConfigPtr config=KSharedConfig::openConfig ();
            KConfigGroup pref=config->group ( "File" );
            pref.writePathEntry ( "lastfilepath",fileName );

        }
        else err.addError ( ERR_FAIL, i18nc("Error message",  "Failed to save '%1'.", fileName ) );

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

void SKGFilePlugin::actionOpen ( const KUrl & iUrl )
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGFilePlugin::actionOpen",err );
    if ( SKGMainPanel::getMainPanel() && currentDocument && SKGMainPanel::getMainPanel()->queryFileClose() )
    {
        bool useKWallet=skgfile_settings::storeInKdeWallet();
        QString pwd;
        QString programName=KCmdLineArgs::aboutData()->programName();
        QString fileName=iUrl.pathOrUrl();
        if ( !fileName.length() ) fileName=KFileDialog::getOpenUrl ( KUrl ( "kfiledialog:///"+currentDocument->objectName() ),
                                               "*."+currentDocument->getFileExtension()+'|'+i18nc ( "File format", "%1 document", programName ) ,
                                               SKGMainPanel::getMainPanel() ).pathOrUrl();
        if ( fileName.length() )
        {
            //Check if temporary file exists
            bool restoreTmpFile=false;
            QString tmpFile=SKGDocument::getTemporaryFile ( fileName );
            if ( QFile ( tmpFile ).exists() )
            {
                int conf=KMessageBox::questionYesNo ( SKGMainPanel::getMainPanel(),
                                                      i18nc("Question", "The temporary file of your document already exists. Do you want to restore it?" ),
                                                      i18nc("Question", "Restore temporary file" ),
                                                      KStandardGuiItem::yes(),
                                                      KStandardGuiItem::no() );
                if ( conf==KMessageBox::Yes ) restoreTmpFile=true;
            }

            //Open
            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            SKGMainPanel::getMainPanel()->closeAllPages();
            err=currentDocument->load ( fileName, "", restoreTmpFile );
            QApplication::restoreOverrideCursor();

            if ( err.isFailed() && err.getReturnCode()!=ERR_WRITEACCESS )
            {
                currentDocument->close();

                //Open failed
                //Password must be asked
                QString additionalMessage;
                do
                {
                    //Reset error
                    err=SKGError ( 0, "" );
                    pwd="";

                    //Get password
                    if (useKWallet)
                    {
                        //Use KWallet
                        KWallet::Wallet* w=KWallet::Wallet::openWallet("kdewallet", SKGMainPanel::getMainPanel()->winId());
                        if (w)
                        {
                            //Change folder
                            if (!w->hasFolder(programName)) w->createFolder(programName);
                            w->setFolder(programName);

                            //Read password
                            w->readPassword(fileName, pwd);
                            if (pwd.isEmpty()) useKWallet=false;

                            delete w;
                            w=NULL;
                        }
                    }

                    if (!useKWallet)
                    {
                        //Use password dialog
                        QPointer<KPasswordDialog> dlg=new KPasswordDialog(SKGMainPanel::getMainPanel());
                        dlg->setPrompt(additionalMessage+i18nc("Question", "This file seems to be protected.\nPlease enter the password." ));
                        if ( dlg->exec() ==QDialog::Accepted ) pwd=dlg->password();
                        delete dlg;
                    }

                    //Load file
                    if (!pwd.isEmpty())
                    {
                        KSplashScreen* splashScreen=SKGMainPanel::getMainPanel()->splashScreen();
                        if ( splashScreen ) splashScreen->hide();

                        QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
                        err=currentDocument->load ( fileName, pwd, restoreTmpFile );
                        if ( err.isFailed() )
                        {
                            additionalMessage=i18nc ("Noun", "<b>Wrong password.</b>\n" );
                            useKWallet=false;
                        }
                        QApplication::restoreOverrideCursor();

                        if ( splashScreen ) splashScreen->show();
                    }
                }
                while ( err.isFailed() );
            }

            //status bar
            if ( err.isSucceeded() )
            {
                err=SKGError(0, i18nc("Successful message after an user action", "File '%1' opened.", fileName ) );
                //Add in recentFiles
                if ( recentFiles )
                {
                    recentFiles->addUrl ( fileName );
                    recentFiles->saveEntries ( KConfigGroup ( KGlobal::config(), "RecentFiles" ) );
                }
                //Set as last open file in kcfg
                KSharedConfigPtr config=KSharedConfig::openConfig ();
                KConfigGroup pref=config->group ( "File" );
                pref.writePathEntry ( "lastfilepath",fileName );

                //Store password if KDE wallet if needed
                if (skgfile_settings::storeInKdeWallet() && !useKWallet)
                {
                    //Use KWallet
                    KWallet::Wallet* w=KWallet::Wallet::openWallet("kdewallet", SKGMainPanel::getMainPanel()->winId());
                    if (w)
                    {
                        //Change folder
                        w->setFolder(programName);

                        //Write password
                        w->writePassword(fileName, pwd);

                        delete w;
                        w=NULL;
                    }
                }
            }
            else  err.addError ( ERR_FAIL, i18nc("Error message",  "Failed to open '%1'.", fileName ) );

            //Display error
            if ( SKGMainPanel::getMainPanel() ) SKGMainPanel::getMainPanel()->displayErrorMessage ( err );

        }
    }
}

void SKGFilePlugin::actionChangePassword()
{
    SKGError err;
    SKGTRACEINRC ( 10, "SKGFilePlugin::actionChangePassword",err );
    if ( currentDocument && SKGMainPanel::getMainPanel() )
    {
        QPointer<KNewPasswordDialog> dlg=new KNewPasswordDialog( SKGMainPanel::getMainPanel() );
        dlg->setPrompt( i18n( "Take care, if you lose your <b>password</b> then it will be <u><b>impossible</b></u> to open your document." ));
        if ( !dlg->exec() ) err=SKGError(0, i18nc("Successful message after an user action", "Changing password was canceled." ) );
        else
        {
            SKGBEGINTRANSACTION ( *currentDocument, i18nc("Noun, name of the user action", "Change password" ), err );
            QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
            err=currentDocument->changePassword ( dlg->password() );
            QApplication::restoreOverrideCursor();

            //status
            if ( err.isSucceeded() )  err=SKGError(0, i18nc("Successful message after an user action", "Password changed." ) );
            else err.addError ( ERR_FAIL, i18nc("Error message",  "Failed to change password." ) );
        }
        delete dlg;

        //Display error
        SKGMainPanel::getMainPanel()->displayErrorMessage ( err );
    }
}

QList< SKGInterfacePlugin::SKGAdvice > SKGFilePlugin::advices() const
{
    QList< SKGInterfacePlugin::SKGAdvice > output;

    //Backup
    SKGInterfacePlugin::SKGAdvice ad;
    ad.uuid="skgfileplugin_notvalidated";
    ad.priority=2;
    ad.shortMessage=i18nc("Advice on making the best (short)", "Backup your document");
    ad.longMessage=i18nc("Advice on making the best (long)", "Don't forget to backup your document on an other device.");
    output.push_back(ad);

    return output;
}

#include "skgfileplugin.moc"
