#include "sjog.h"
#include "sjog_list.h"
#include "sjog_enlightenment.h"
#include "sjog_config.h"


/* -------------------------------------------------------------------------- */

/*
 * Reads enlightenment files config recursively
 * and creates a tree of sjog_cmd
 */

void
sjog_init_tree_entries()
{
   GNode *tree = NULL;
   gchar rcname[LINE_MAX];
   
   snprintf(rcname, LINE_MAX, "%s/.enlightenment/file.menu", getenv("HOME"));
   tree = g_node_new(NULL);
   
   sjog_init_tree_file(tree, rcname);

   currentTreeNode = tree;
}

void
sjog_init_tree_file(GNode *parent, gchar *filename)
{
   FILE *config_stream;
   GNode *new_node;
   sjog_cmd_t *cmd, *parent_cmd;
   gboolean err = FALSE;
   gchar line[LINE_MAX];
   gchar rcname[LINE_MAX];

   config_stream = fopen(filename, "r");

   if (!config_stream)
      return;

   /* Skip the first line : Menu title */
   fgets(line, sizeof(line), config_stream);

   while (!err && fgets(line, sizeof(line), config_stream) != NULL)
   {
      cmd = sjog_enlightenment_parse(line, &err);
      if (err)
      {
         fclose(config_stream);
         fprintf(stderr, "Error in file : %s\n", filename);
         exit(1);
      }
      else if (cmd == NULL || cmd->path == NULL)
      {
         /* white or comment line, skip it */
               continue;
      }

#ifdef DEBUG
      fprintf(stdout, "sjog_init_config: name=%s path=%s\n",
              cmd->name, cmd->path);
#endif
      new_node = g_node_new(cmd);
      g_node_append(parent, new_node);

      if (cmd->type == SJOG_CMD_TYPE_MENU)
      {
         if (*cmd->path != '/')
            snprintf(rcname, LINE_MAX, "%s/.enlightenment/%s", getenv("HOME"),
                                                               cmd->path);
         else
            strncpy(rcname, cmd->path, LINE_MAX);
         sjog_init_tree_file(new_node, rcname);
      }
   }

   /* add an entry at the end of the list for models without back button */
   parent_cmd = sjog_cmd_new();
   parent_cmd->name = "..";
   parent_cmd->path = "parent";
   parent_cmd->type = SJOG_CMD_TYPE_MENU;
   new_node = g_node_new(parent_cmd);
   g_node_append(parent, new_node);
}


void
sjog_tree_set_list()
{
   GNode *currentNode;
   guint i = 0;
   gchar *item[1];
   sjog_cmd_t *cmd;

   GdkPixbuf *pixbuf;
   GdkPixmap *pixmap;
   GdkBitmap *bitmap;

   /* default is to select 1st item */
   selectedIndex = 0;

   currentNode = g_node_first_child(currentTreeNode);

   gtk_clist_freeze(GTK_CLIST(cmdListUI));
   
   while (currentNode)
   {
      cmd = (sjog_cmd_t *) currentNode->data;
      item[0] = cmd->name;

      if (cmd->icon_path)
      {
         i = gtk_clist_append(GTK_CLIST(cmdListUI), item);
         if (pixbuf = gdk_pixbuf_new_from_file(cmd->icon_path))
         {
            gdk_pixbuf_render_pixmap_and_mask(pixbuf, &pixmap, &bitmap, 1);
            gdk_pixbuf_finalize(pixbuf);
            gtk_clist_set_pixtext(GTK_CLIST(cmdListUI), i, 0, cmd->name, 10, pixmap, bitmap); 
         }
      }
      else
         gtk_clist_append(GTK_CLIST(cmdListUI), item);
      
      currentNode = g_node_next_sibling(currentNode);
   }

   gtk_clist_thaw(GTK_CLIST(cmdListUI));   

   /* select the 1st item */
   gtk_clist_select_row(GTK_CLIST(cmdListUI), selectedIndex, 0);
}


sjog_cmd_t *
sjog_enlightenment_parse(gchar *line, gboolean *err)
{
   gint i;
   gchar *tmp;
   gchar name[LINE_MAX], icon_path[LINE_MAX], type[LINE_MAX], path[LINE_MAX];
   sjog_cmd_t *sjog_cmd;
   gboolean foundQuote;

   *err = FALSE;

   sjog_cmd = (sjog_cmd_t *) malloc (sizeof(sjog_cmd_t));
   if (!sjog_cmd)
   {
      *err = TRUE;
      return NULL;
   }

   /* strip comment or final \n */

   for (tmp = line; *tmp != '\0'; tmp++)
   {
      if (*tmp == '#' || *tmp == '\n')
      {
         *tmp = '\0';
         break;
      }
   }

   /* strip for trailing withespaces */

   while (tmp != line && (*tmp == ' ' || *tmp == '\t'))
      tmp--;
   *(tmp + 1) = '\0';

   
   /* strip the heading whitespaces */

   tmp = line;
   while (*tmp == ' ' || *tmp == '\t') tmp++;

   
   /* read the name */
   
   foundQuote = FALSE;
   i = 0;
   while (((*tmp != ' ' && *tmp != '\t') || foundQuote) && *tmp != '\0')
   {
      if (*tmp == '"')
         foundQuote = !foundQuote;
      else
         name[i++] = *tmp;

      tmp++;
   }
   
   name[i] = '\0';
   sjog_cmd->name = strdup(name);



   /* read the icon path */
   
   foundQuote = FALSE;
   i = 0;
   
   /* strip the spaces */
   while (*tmp == ' ' || *tmp == '\t') tmp++;

   while (((*tmp != ' ' && *tmp != '\t') || foundQuote) && *tmp != '\0')
   {
      if (*tmp == '"')
         foundQuote = !foundQuote;
      else
         icon_path[i++] = *tmp;

      tmp++;
   }
   
   icon_path[i] = '\0';
   if (strcasecmp(icon_path, "NULL"))
      sjog_cmd->icon_path = strdup(icon_path);
   else
      sjog_cmd->icon_path = NULL;


   /* read the type */
   
   foundQuote = FALSE;
   i = 0;

   /* strip the spaces */
   while (*tmp == ' ' || *tmp == '\t') tmp++;

   while (((*tmp != ' ' && *tmp != '\t') || foundQuote) && *tmp != '\0')
   {
      if (*tmp == '"')
         foundQuote = !foundQuote;
      else
         type[i++] = *tmp;

      tmp++;
   }
   
   type[i] = '\0';
   if (!strcasecmp(type, "menu"))
      sjog_cmd->type = SJOG_CMD_TYPE_MENU;
   else if (!strcasecmp(type, "exec"))
      sjog_cmd->type = SJOG_CMD_TYPE_EXEC;
   else
   {
      *err = TRUE;
      g_free(sjog_cmd);
      return NULL;
   }
   /* TODO autres types */



   /* read the path */
   
   foundQuote = FALSE;
   i = 0;

   /* strip the spaces */
   while (*tmp == ' ' || *tmp == '\t') tmp++;
   
   while (((*tmp != ' ' && *tmp != '\t') || foundQuote) && *tmp != '\0')
   {
      if (*tmp == '"')
         foundQuote = !foundQuote;
      else
         path[i++] = *tmp;

      tmp++;
   }
   
   path[i] = '\0';
   sjog_cmd->path = strdup(path);

   if (sjog_cmd->path == NULL)
   {
      *err = TRUE;
      g_free(sjog_cmd);
      return NULL;
   }

   return sjog_cmd;
}

/*
 * modelines
 * ex:set ai ts=3 sw=3:
 * ei:set itab=spaces noat:
 * vim:set et:
 */
