/*
 * __BEGIN_COPYRIGHT
 * SimpleDB API
 * 
 * Copyright (C) 2005 Eminence Technology Pty Ltd
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You can view the GNU Lesser General Public Licence at
 * http://www.gnu.org/licenses/lgpl.html or you can write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Eminence Technology Pty Ltd can be contacted by writing to
 * Eminence Technology, PO Box 118, Moorooka QLD 4105, Australia.
 * Alternatively, you may email opensource [at] eminence [dot] com [dot] au
 * __END_COPYRIGHT
 */

#include "SimpleDB.h"

using namespace SimpleDB ;

Database::Database(std::string dsn) {
  long return_value;
  return_value = SQLAllocHandle(SQL_HANDLE_ENV,SQL_NULL_HANDLE,
				&odbc_environment_handle);
  std::string s;
  if ((return_value != SQL_SUCCESS) &&
      (return_value != SQL_SUCCESS_WITH_INFO))
    throw Exception(std::string("Error AllocHandle"));

  return_value = SQLSetEnvAttr(odbc_environment_handle, SQL_ATTR_ODBC_VERSION, 
                               (void*)SQL_OV_ODBC3, 0); 
  if ((return_value != SQL_SUCCESS) &&
      (return_value != SQL_SUCCESS_WITH_INFO))
    throw Exception(std::string("Error SetEnv"));
	
  return_value = SQLAllocHandle(SQL_HANDLE_DBC, odbc_environment_handle,
				&connectionHandle); 
  if ((return_value != SQL_SUCCESS) &&
      (return_value != SQL_SUCCESS_WITH_INFO)) {
    SQLFreeHandle(SQL_HANDLE_ENV, odbc_environment_handle);
    throw Exception(std::string("Error AllocHDB"));
  }

  SQLSetConnectAttr(connectionHandle, SQL_LOGIN_TIMEOUT, (SQLPOINTER *)5, 0);
  return_value = SQLConnect(connectionHandle,
			    (SQLCHAR*) dsn.c_str(), SQL_NTS,
			    NULL, SQL_NTS,
			    NULL, SQL_NTS);
  if ((return_value != SQL_SUCCESS) &&
      (return_value != SQL_SUCCESS_WITH_INFO)) {
    std::string errorString = "Error SQLConnect: ";
    SQLCHAR sqlState[6];
    SQLINTEGER errorCode;
    SQLSMALLINT messageLength;
    SQLCHAR errorMessage[400];
  long errValue = SQLGetDiagRec(SQL_HANDLE_DBC, connectionHandle,1,
				  (SQLCHAR *)sqlState,&errorCode,
				  (SQLCHAR *)errorMessage,
				  400,&messageLength);
    if(errValue == SQL_SUCCESS) {
      errorString += (char *)errorMessage;
    } else if (errValue == SQL_SUCCESS_WITH_INFO) {
      
      std::cerr << "Error message too long at " << messageLength <<
	" characters" <<  std::endl;
    } else if (errValue == SQL_ERROR) {
      errorString += "RecNumber was negative or 0 or BufferLength was less tha 0";
    } else if (errValue == SQL_NO_DATA) {
      errorString += "SQL no data";
    }

    SQLFreeHandle(SQL_HANDLE_DBC, connectionHandle);
    SQLFreeHandle(SQL_HANDLE_ENV, odbc_environment_handle);
    throw Exception(errorString);
  }
}

Query Database::newQuery() const {
  return Query(connectionHandle);
}

void Database::voidQuery (const std::string& query) const {
  try {
    SimpleDB::Query queryWrapper = this->newQuery();
    queryWrapper.execute(query);
  } catch(SimpleDB::Query::Exception &s)  {
    throw (SimpleDB::Database::Exception (std::string(s.what()))) ;
  }
  catch(SimpleDB::Column::UnboundException)  {
    throw (SimpleDB::Database::Exception 
	   (std::string("A Column was not bound"))) ;
  }
}

long Database::intQuery (const std::string& query) const {
  try {
    SimpleDB::Query queryWrapper = this->newQuery();
    SimpleDB::LongColumn longCol = SimpleDB::LongColumn();
    SimpleDB::Column * columns[1] = {&longCol};
    queryWrapper.bind(columns, 1);
    queryWrapper.execute(query);
    if(!queryWrapper.fetchRow())
      throw SimpleDB::Database::NoDataException();
    return longCol.value() ;
  } catch(SimpleDB::Query::Exception &s)  {
    throw SimpleDB::Database::Exception (std::string(s.what()));
  } catch(SimpleDB::Column::UnboundException)  {
    throw SimpleDB::Database::Exception (std::string("A Column was not bound"));
  }
}

std::string Database::strQuery (const std::string& query, long bufSize) const {
  try {
    SimpleDB::Query queryWrapper = this->newQuery(); 
    SimpleDB::StringColumn col = SimpleDB::StringColumn(bufSize);
    SimpleDB::Column * columns[1] = {&col};
    queryWrapper.bind(columns, sizeof(columns)/sizeof(columns[0]));
    queryWrapper.execute(query);
    queryWrapper.fetchRow();
    return col.value() ;
  }
  catch(SimpleDB::Query::Exception &s)  {
    throw (SimpleDB::Database::Exception (std::string(s.what()))) ;
  }
  catch(SimpleDB::Column::UnboundException)  {
    throw (SimpleDB::Database::Exception (std::string("A Column was not bound"))) ;
  }
  return "" ;
}

const std::string Database::escapeString(const std::string& input) {
  std::string output = "";
  char c;
  for (unsigned int i = 0; i < input.length(); i++) {
    c = input.at(i) ;
    switch (c) {
    case '\'' :
      output += "''";
      break;
    case '\\' :
      output += "\\\\";
      break ;
    default:
      output += c;
      break ;
    }
  }
  return output;
}

const std::string  Database::boolToStr (const bool in) {
  std::string rtStr = "FALSE" ;
  if (in)
    rtStr = "TRUE" ;
  return rtStr ;
}
    
const SQLHDBC Database::getConnectionHandle() const {
  return connectionHandle;
}


Database::~Database() 
{
  SQLDisconnect(connectionHandle);
  SQLFreeHandle(SQL_HANDLE_DBC, connectionHandle);
  SQLFreeHandle(SQL_HANDLE_ENV, odbc_environment_handle);
}
