;;  Filename : test-number-arith.scm
;;  About    : unit tests for arithmetic procedures
;;
;;  Copyright (C) 2006 YAMAMOTO Kengo <yamaken AT bp.iij4u.or.jp>
;;  Copyright (c) 2007 SigScheme Project <uim-en AT googlegroups.com>
;;
;;  All rights reserved.
;;
;;  Redistribution and use in source and binary forms, with or without
;;  modification, are permitted provided that the following conditions
;;  are met:
;;
;;  1. Redistributions of source code must retain the above copyright
;;     notice, this list of conditions and the following disclaimer.
;;  2. Redistributions in binary form must reproduce the above copyright
;;     notice, this list of conditions and the following disclaimer in the
;;     documentation and/or other materials provided with the distribution.
;;  3. Neither the name of authors nor the names of its contributors
;;     may be used to endorse or promote products derived from this software
;;     without specific prior written permission.
;;
;;  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
;;  IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
;;  THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;;  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
;;  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
;;  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
;;  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

(require-extension (unittest))

(if (not (symbol-bound? 'number?))
    (test-skip "R5RS numbers is not enabled"))

(define tn test-name)


(tn "max invalid forms")
(assert-error  (tn) (lambda () (max)))
(assert-error  (tn) (lambda () (max #t)))
(assert-error  (tn) (lambda () (max #f)))
(assert-error  (tn) (lambda () (max '())))
(tn "max 2 args")
(assert-equal? (tn) -2 (max -2 -2))
(assert-equal? (tn) -1 (max -2 -1))
(assert-equal? (tn) 0  (max -2 0))
(assert-equal? (tn) 1  (max -2 1))
(assert-equal? (tn) 2  (max -2 2))
(assert-equal? (tn) -1 (max -1 -2))
(assert-equal? (tn) -1 (max -1 -1))
(assert-equal? (tn) 0  (max -1 0))
(assert-equal? (tn) 1  (max -1 1))
(assert-equal? (tn) 2  (max -1 2))
(assert-equal? (tn) 0  (max 0 -2))
(assert-equal? (tn) 0  (max 0 -1))
(assert-equal? (tn) 0  (max 0 0))
(assert-equal? (tn) 1  (max 0 1))
(assert-equal? (tn) 2  (max 0 2))
(assert-equal? (tn) 1  (max 1 -2))
(assert-equal? (tn) 1  (max 1 -1))
(assert-equal? (tn) 1  (max 1 0))
(assert-equal? (tn) 1  (max 1 1))
(assert-equal? (tn) 2  (max 1 2))
(assert-equal? (tn) 2  (max 2 -2))
(assert-equal? (tn) 2  (max 2 -1))
(assert-equal? (tn) 2  (max 2 0))
(assert-equal? (tn) 2  (max 2 1))
(assert-equal? (tn) 2  (max 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "max 2 args 64-bit")
      (string-eval "(assert-equal? (tn)    956397711204 (max    956397711204    956397711204))")
      (string-eval "(assert-equal? (tn)    956397711204 (max    956397711204   -956397711204))")
      (string-eval "(assert-equal? (tn)    956397711204 (max   -956397711204    956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (max   -956397711204   -956397711204))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max    956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max   -956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)    956397711204 (max    956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)   -956397711204 (max   -956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max  13121090146595    956397711204))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max  13121090146595   -956397711204))")
      (string-eval "(assert-equal? (tn)    956397711204 (max -13121090146595    956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (max -13121090146595   -956397711204))")))
(tn "max 3 args")
(assert-equal? (tn) -2 (max -2 -2 -2))
(assert-equal? (tn) -1 (max -2 -2 -1))
(assert-equal? (tn) 0  (max -2 -2 0))
(assert-equal? (tn) 1  (max -2 -2 1))
(assert-equal? (tn) 2  (max -2 -2 2))
(assert-equal? (tn) -1 (max -2 -1 -2))
(assert-equal? (tn) -1 (max -2 -1 -1))
(assert-equal? (tn) 0  (max -2 -1 0))
(assert-equal? (tn) 1  (max -2 -1 1))
(assert-equal? (tn) 2  (max -2 -1 2))
(assert-equal? (tn) 0  (max -2 0 -2))
(assert-equal? (tn) 0  (max -2 0 -1))
(assert-equal? (tn) 0  (max -2 0 0))
(assert-equal? (tn) 1  (max -2 0 1))
(assert-equal? (tn) 2  (max -2 0 2))
(assert-equal? (tn) 1  (max -2 1 -2))
(assert-equal? (tn) 1  (max -2 1 -1))
(assert-equal? (tn) 1  (max -2 1 0))
(assert-equal? (tn) 1  (max -2 1 1))
(assert-equal? (tn) 2  (max -2 1 2))
(assert-equal? (tn) 2  (max -2 2 -2))
(assert-equal? (tn) 2  (max -2 2 -1))
(assert-equal? (tn) 2  (max -2 2 0))
(assert-equal? (tn) 2  (max -2 2 1))
(assert-equal? (tn) 2  (max -2 2 2))
(assert-equal? (tn) -1 (max -1 -2 -2))
(assert-equal? (tn) -1 (max -1 -2 -1))
(assert-equal? (tn) 0  (max -1 -2 0))
(assert-equal? (tn) 1  (max -1 -2 1))
(assert-equal? (tn) 2  (max -1 -2 2))
(assert-equal? (tn) -1 (max -1 -1 -2))
(assert-equal? (tn) -1 (max -1 -1 -1))
(assert-equal? (tn) 0  (max -1 -1 0))
(assert-equal? (tn) 1  (max -1 -1 1))
(assert-equal? (tn) 2  (max -1 -1 2))
(assert-equal? (tn) 0  (max -1 0 -2))
(assert-equal? (tn) 0  (max -1 0 -1))
(assert-equal? (tn) 0  (max -1 0 0))
(assert-equal? (tn) 1  (max -1 0 1))
(assert-equal? (tn) 2  (max -1 0 2))
(assert-equal? (tn) 1  (max -1 1 -2))
(assert-equal? (tn) 1  (max -1 1 -1))
(assert-equal? (tn) 1  (max -1 1 0))
(assert-equal? (tn) 1  (max -1 1 1))
(assert-equal? (tn) 2  (max -1 1 2))
(assert-equal? (tn) 2  (max -1 2 -2))
(assert-equal? (tn) 2  (max -1 2 -1))
(assert-equal? (tn) 2  (max -1 2 0))
(assert-equal? (tn) 2  (max -1 2 1))
(assert-equal? (tn) 2  (max -1 2 2))
(assert-equal? (tn) 0  (max 0 -2 -2))
(assert-equal? (tn) 0  (max 0 -2 -1))
(assert-equal? (tn) 0  (max 0 -2 0))
(assert-equal? (tn) 1  (max 0 -2 1))
(assert-equal? (tn) 2  (max 0 -2 2))
(assert-equal? (tn) 0  (max 0 -1 -2))
(assert-equal? (tn) 0  (max 0 -1 -1))
(assert-equal? (tn) 0  (max 0 -1 0))
(assert-equal? (tn) 1  (max 0 -1 1))
(assert-equal? (tn) 2  (max 0 -1 2))
(assert-equal? (tn) 0  (max 0 0 -2))
(assert-equal? (tn) 0  (max 0 0 -1))
(assert-equal? (tn) 0  (max 0 0 0))
(assert-equal? (tn) 1  (max 0 0 1))
(assert-equal? (tn) 2  (max 0 0 2))
(assert-equal? (tn) 1  (max 0 1 -2))
(assert-equal? (tn) 1  (max 0 1 -1))
(assert-equal? (tn) 1  (max 0 1 0))
(assert-equal? (tn) 1  (max 0 1 1))
(assert-equal? (tn) 2  (max 0 1 2))
(assert-equal? (tn) 2  (max 0 2 -2))
(assert-equal? (tn) 2  (max 0 2 -1))
(assert-equal? (tn) 2  (max 0 2 0))
(assert-equal? (tn) 2  (max 0 2 1))
(assert-equal? (tn) 2  (max 0 2 2))
(assert-equal? (tn) 1  (max 1 -2 -2))
(assert-equal? (tn) 1  (max 1 -2 -1))
(assert-equal? (tn) 1  (max 1 -2 0))
(assert-equal? (tn) 1  (max 1 -2 1))
(assert-equal? (tn) 2  (max 1 -2 2))
(assert-equal? (tn) 1  (max 1 -1 -2))
(assert-equal? (tn) 1  (max 1 -1 -1))
(assert-equal? (tn) 1  (max 1 -1 0))
(assert-equal? (tn) 1  (max 1 -1 1))
(assert-equal? (tn) 2  (max 1 -1 2))
(assert-equal? (tn) 1  (max 1 0 -2))
(assert-equal? (tn) 1  (max 1 0 -1))
(assert-equal? (tn) 1  (max 1 0 0))
(assert-equal? (tn) 1  (max 1 0 1))
(assert-equal? (tn) 2  (max 1 0 2))
(assert-equal? (tn) 1  (max 1 1 -2))
(assert-equal? (tn) 1  (max 1 1 -1))
(assert-equal? (tn) 1  (max 1 1 0))
(assert-equal? (tn) 1  (max 1 1 1))
(assert-equal? (tn) 2  (max 1 1 2))
(assert-equal? (tn) 2  (max 1 2 -2))
(assert-equal? (tn) 2  (max 1 2 -1))
(assert-equal? (tn) 2  (max 1 2 0))
(assert-equal? (tn) 2  (max 1 2 1))
(assert-equal? (tn) 2  (max 1 2 2))
(assert-equal? (tn) 2  (max 2 -2 -2))
(assert-equal? (tn) 2  (max 2 -2 -1))
(assert-equal? (tn) 2  (max 2 -2 0))
(assert-equal? (tn) 2  (max 2 -2 1))
(assert-equal? (tn) 2  (max 2 -2 2))
(assert-equal? (tn) 2  (max 2 -1 -2))
(assert-equal? (tn) 2  (max 2 -1 -1))
(assert-equal? (tn) 2  (max 2 -1 0))
(assert-equal? (tn) 2  (max 2 -1 1))
(assert-equal? (tn) 2  (max 2 -1 2))
(assert-equal? (tn) 2  (max 2 0 -2))
(assert-equal? (tn) 2  (max 2 0 -1))
(assert-equal? (tn) 2  (max 2 0 0))
(assert-equal? (tn) 2  (max 2 0 1))
(assert-equal? (tn) 2  (max 2 0 2))
(assert-equal? (tn) 2  (max 2 1 -2))
(assert-equal? (tn) 2  (max 2 1 -1))
(assert-equal? (tn) 2  (max 2 1 0))
(assert-equal? (tn) 2  (max 2 1 1))
(assert-equal? (tn) 2  (max 2 1 2))
(assert-equal? (tn) 2  (max 2 2 -2))
(assert-equal? (tn) 2  (max 2 2 -1))
(assert-equal? (tn) 2  (max 2 2 0))
(assert-equal? (tn) 2  (max 2 2 1))
(assert-equal? (tn) 2  (max 2 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "max 3 args 64-bit")
      (string-eval "(assert-equal? (tn)    956397711204 (max  956397711204  956397711204  956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (max -956397711204 -956397711204 -956397711204))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max 0  956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max 0 -956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)    956397711204 (max 0  956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)               0 (max 0 -956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max  13121090146595  956397711204 0))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max  13121090146595 -956397711204 0))")
      (string-eval "(assert-equal? (tn)    956397711204 (max -13121090146595  956397711204 0))")
      (string-eval "(assert-equal? (tn)               0 (max -13121090146595 -956397711204 0))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max  956397711204 0  13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max -956397711204 0  13121090146595))")
      (string-eval "(assert-equal? (tn)    956397711204 (max  956397711204 0 -13121090146595))")
      (string-eval "(assert-equal? (tn)               0 (max -956397711204 0 -13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max  13121090146595 0  956397711204))")
      (string-eval "(assert-equal? (tn)  13121090146595 (max  13121090146595 0 -956397711204))")
      (string-eval "(assert-equal? (tn)    956397711204 (max -13121090146595 0  956397711204))")
      (string-eval "(assert-equal? (tn)               0 (max -13121090146595 0 -956397711204))")))
(tn "max many args")
(assert-equal? (tn) 2  (max 2 2 2 2))
(assert-equal? (tn) 2  (max 0 2 2 2))
(assert-equal? (tn) 2  (max 2 0 2 2))
(assert-equal? (tn) 2  (max 2 2 0 2))
(assert-equal? (tn) 2  (max 2 2 2 0))
(assert-equal? (tn) 2  (max -2 -1 0 1 2))
(assert-equal? (tn) 2  (max 2 1 0 -1 -2))
(assert-equal? (tn) 2  (max -2 -1 0 -1 1 2))
(assert-equal? (tn) 2  (max 2 1 0 -1 1 -2))
(assert-equal? (tn) 2  (max -2 -2 -1 -1 0 0 1 1 2 2))
(assert-equal? (tn) 2  (max 2 2 1 1 0 0 -1 -1 -2 -2))
(assert-equal? (tn) 13 (max 3 1 5 -7 2 13))

(tn "min invalid forms")
(assert-error  (tn) (lambda () (min)))
(assert-error  (tn) (lambda () (min #t)))
(assert-error  (tn) (lambda () (min #f)))
(assert-error  (tn) (lambda () (min '())))
(tn "min 2 args")
(assert-equal? (tn) -2 (min -2 -2))
(assert-equal? (tn) -2 (min -2 -1))
(assert-equal? (tn) -2 (min -2 0))
(assert-equal? (tn) -2 (min -2 1))
(assert-equal? (tn) -2 (min -2 2))
(assert-equal? (tn) -2 (min -1 -2))
(assert-equal? (tn) -1 (min -1 -1))
(assert-equal? (tn) -1 (min -1 0))
(assert-equal? (tn) -1 (min -1 1))
(assert-equal? (tn) -1 (min -1 2))
(assert-equal? (tn) -2 (min 0 -2))
(assert-equal? (tn) -1 (min 0 -1))
(assert-equal? (tn) 0  (min 0 0))
(assert-equal? (tn) 0  (min 0 1))
(assert-equal? (tn) 0  (min 0 2))
(assert-equal? (tn) -2 (min 1 -2))
(assert-equal? (tn) -1 (min 1 -1))
(assert-equal? (tn) 0  (min 1 0))
(assert-equal? (tn) 1  (min 1 1))
(assert-equal? (tn) 1  (min 1 2))
(assert-equal? (tn) -2 (min 2 -2))
(assert-equal? (tn) -1 (min 2 -1))
(assert-equal? (tn) 0  (min 2 0))
(assert-equal? (tn) 1  (min 2 1))
(assert-equal? (tn) 2  (min 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "min 2 args 64-bit")
      (string-eval "(assert-equal? (tn)    956397711204 (min    956397711204    956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min    956397711204   -956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min   -956397711204    956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min   -956397711204   -956397711204))")
      (string-eval "(assert-equal? (tn)    956397711204 (min    956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min   -956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min    956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min   -956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)    956397711204 (min  13121090146595    956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min  13121090146595   -956397711204))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min -13121090146595    956397711204))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min -13121090146595   -956397711204))")))
(tn "min 3 args")
(assert-equal? (tn) -2 (min -2 -2 -2))
(assert-equal? (tn) -2 (min -2 -2 -1))
(assert-equal? (tn) -2 (min -2 -2 0))
(assert-equal? (tn) -2 (min -2 -2 1))
(assert-equal? (tn) -2 (min -2 -2 2))
(assert-equal? (tn) -2 (min -2 -1 -2))
(assert-equal? (tn) -2 (min -2 -1 -1))
(assert-equal? (tn) -2 (min -2 -1 0))
(assert-equal? (tn) -2 (min -2 -1 1))
(assert-equal? (tn) -2 (min -2 -1 2))
(assert-equal? (tn) -2 (min -2 0 -2))
(assert-equal? (tn) -2 (min -2 0 -1))
(assert-equal? (tn) -2 (min -2 0 0))
(assert-equal? (tn) -2 (min -2 0 1))
(assert-equal? (tn) -2 (min -2 0 2))
(assert-equal? (tn) -2 (min -2 1 -2))
(assert-equal? (tn) -2 (min -2 1 -1))
(assert-equal? (tn) -2 (min -2 1 0))
(assert-equal? (tn) -2 (min -2 1 1))
(assert-equal? (tn) -2 (min -2 1 2))
(assert-equal? (tn) -2 (min -2 2 -2))
(assert-equal? (tn) -2 (min -2 2 -1))
(assert-equal? (tn) -2 (min -2 2 0))
(assert-equal? (tn) -2 (min -2 2 1))
(assert-equal? (tn) -2 (min -2 2 2))
(assert-equal? (tn) -2 (min -1 -2 -2))
(assert-equal? (tn) -2 (min -1 -2 -1))
(assert-equal? (tn) -2 (min -1 -2 0))
(assert-equal? (tn) -2 (min -1 -2 1))
(assert-equal? (tn) -2 (min -1 -2 2))
(assert-equal? (tn) -2 (min -1 -1 -2))
(assert-equal? (tn) -1 (min -1 -1 -1))
(assert-equal? (tn) -1 (min -1 -1 0))
(assert-equal? (tn) -1 (min -1 -1 1))
(assert-equal? (tn) -1 (min -1 -1 2))
(assert-equal? (tn) -2 (min -1 0 -2))
(assert-equal? (tn) -1 (min -1 0 -1))
(assert-equal? (tn) -1 (min -1 0 0))
(assert-equal? (tn) -1 (min -1 0 1))
(assert-equal? (tn) -1 (min -1 0 2))
(assert-equal? (tn) -2 (min -1 1 -2))
(assert-equal? (tn) -1 (min -1 1 -1))
(assert-equal? (tn) -1 (min -1 1 0))
(assert-equal? (tn) -1 (min -1 1 1))
(assert-equal? (tn) -1 (min -1 1 2))
(assert-equal? (tn) -2 (min -1 2 -2))
(assert-equal? (tn) -1 (min -1 2 -1))
(assert-equal? (tn) -1 (min -1 2 0))
(assert-equal? (tn) -1 (min -1 2 1))
(assert-equal? (tn) -1 (min -1 2 2))
(assert-equal? (tn) -2 (min 0 -2 -2))
(assert-equal? (tn) -2 (min 0 -2 -1))
(assert-equal? (tn) -2 (min 0 -2 0))
(assert-equal? (tn) -2 (min 0 -2 1))
(assert-equal? (tn) -2 (min 0 -2 2))
(assert-equal? (tn) -2 (min 0 -1 -2))
(assert-equal? (tn) -1 (min 0 -1 -1))
(assert-equal? (tn) -1 (min 0 -1 0))
(assert-equal? (tn) -1 (min 0 -1 1))
(assert-equal? (tn) -1 (min 0 -1 2))
(assert-equal? (tn) -2 (min 0 0 -2))
(assert-equal? (tn) -1 (min 0 0 -1))
(assert-equal? (tn) 0  (min 0 0 0))
(assert-equal? (tn) 0  (min 0 0 1))
(assert-equal? (tn) 0  (min 0 0 2))
(assert-equal? (tn) -2 (min 0 1 -2))
(assert-equal? (tn) -1 (min 0 1 -1))
(assert-equal? (tn) 0  (min 0 1 0))
(assert-equal? (tn) 0  (min 0 1 1))
(assert-equal? (tn) 0  (min 0 1 2))
(assert-equal? (tn) -2 (min 0 2 -2))
(assert-equal? (tn) -1 (min 0 2 -1))
(assert-equal? (tn) 0  (min 0 2 0))
(assert-equal? (tn) 0  (min 0 2 1))
(assert-equal? (tn) 0  (min 0 2 2))
(assert-equal? (tn) -2 (min 1 -2 -2))
(assert-equal? (tn) -2 (min 1 -2 -1))
(assert-equal? (tn) -2 (min 1 -2 0))
(assert-equal? (tn) -2 (min 1 -2 1))
(assert-equal? (tn) -2 (min 1 -2 2))
(assert-equal? (tn) -2 (min 1 -1 -2))
(assert-equal? (tn) -1 (min 1 -1 -1))
(assert-equal? (tn) -1 (min 1 -1 0))
(assert-equal? (tn) -1 (min 1 -1 1))
(assert-equal? (tn) -1 (min 1 -1 2))
(assert-equal? (tn) -2 (min 1 0 -2))
(assert-equal? (tn) -1 (min 1 0 -1))
(assert-equal? (tn) 0  (min 1 0 0))
(assert-equal? (tn) 0  (min 1 0 1))
(assert-equal? (tn) 0  (min 1 0 2))
(assert-equal? (tn) -2 (min 1 1 -2))
(assert-equal? (tn) -1 (min 1 1 -1))
(assert-equal? (tn) 0  (min 1 1 0))
(assert-equal? (tn) 1  (min 1 1 1))
(assert-equal? (tn) 1  (min 1 1 2))
(assert-equal? (tn) -2 (min 1 2 -2))
(assert-equal? (tn) -1 (min 1 2 -1))
(assert-equal? (tn) 0  (min 1 2 0))
(assert-equal? (tn) 1  (min 1 2 1))
(assert-equal? (tn) 1  (min 1 2 2))
(assert-equal? (tn) -2 (min 2 -2 -2))
(assert-equal? (tn) -2 (min 2 -2 -1))
(assert-equal? (tn) -2 (min 2 -2 0))
(assert-equal? (tn) -2 (min 2 -2 1))
(assert-equal? (tn) -2 (min 2 -2 2))
(assert-equal? (tn) -2 (min 2 -1 -2))
(assert-equal? (tn) -1 (min 2 -1 -1))
(assert-equal? (tn) -1 (min 2 -1 0))
(assert-equal? (tn) -1 (min 2 -1 1))
(assert-equal? (tn) -1 (min 2 -1 2))
(assert-equal? (tn) -2 (min 2 0 -2))
(assert-equal? (tn) -1 (min 2 0 -1))
(assert-equal? (tn) 0  (min 2 0 0))
(assert-equal? (tn) 0  (min 2 0 1))
(assert-equal? (tn) 0  (min 2 0 2))
(assert-equal? (tn) -2 (min 2 1 -2))
(assert-equal? (tn) -1 (min 2 1 -1))
(assert-equal? (tn) 0  (min 2 1 0))
(assert-equal? (tn) 1  (min 2 1 1))
(assert-equal? (tn) 1  (min 2 1 2))
(assert-equal? (tn) -2 (min 2 2 -2))
(assert-equal? (tn) -1 (min 2 2 -1))
(assert-equal? (tn) 0  (min 2 2 0))
(assert-equal? (tn) 1  (min 2 2 1))
(assert-equal? (tn) 2  (min 2 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "min 3 args 64-bit")
      (string-eval "(assert-equal? (tn)    956397711204 (min  956397711204  956397711204  956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min -956397711204 -956397711204 -956397711204))")
      (string-eval "(assert-equal? (tn)               0 (min 0  956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min 0 -956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min 0  956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min 0 -956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)               0 (min  13121090146595  956397711204 0))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min  13121090146595 -956397711204 0))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min -13121090146595  956397711204 0))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min -13121090146595 -956397711204 0))")
      (string-eval "(assert-equal? (tn)               0 (min  956397711204 0  13121090146595))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min -956397711204 0  13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min  956397711204 0 -13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min -956397711204 0 -13121090146595))")
      (string-eval "(assert-equal? (tn)               0 (min  13121090146595 0  956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (min  13121090146595 0 -956397711204))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min -13121090146595 0  956397711204))")
      (string-eval "(assert-equal? (tn) -13121090146595 (min -13121090146595 0 -956397711204))")))
(tn "min many args")
(assert-equal? (tn) 2  (min 2 2 2 2))
(assert-equal? (tn) 0  (min 0 2 2 2))
(assert-equal? (tn) 0  (min 2 0 2 2))
(assert-equal? (tn) 0  (min 2 2 0 2))
(assert-equal? (tn) 0  (min 2 2 2 0))
(assert-equal? (tn) -2 (min -2 -1 0 1 2))
(assert-equal? (tn) -2 (min 2 1 0 -1 -2))
(assert-equal? (tn) -2 (min -2 -1 0 -1 1 2))
(assert-equal? (tn) -2 (min 2 1 0 -1 1 -2))
(assert-equal? (tn) -2 (min -2 -2 -1 -1 0 0 1 1 2 2))
(assert-equal? (tn) -2 (min 2 2 1 1 0 0 -1 -1 -2 -2))
(assert-equal? (tn) -7 (min 3 1 5 -7 2 13))

(tn "+ invalid forms")
(assert-error  (tn) (lambda () (+ #t)))
(assert-error  (tn) (lambda () (+ #f)))
(assert-error  (tn) (lambda () (+ '())))
(assert-error  (tn) (lambda () (+ 1 #t)))
(assert-error  (tn) (lambda () (+ 1 #f)))
(assert-error  (tn) (lambda () (+ 1 '())))
(assert-error  (tn) (lambda () (+ 1 2 #t)))
(assert-error  (tn) (lambda () (+ 1 2 #f)))
(assert-error  (tn) (lambda () (+ 1 2 '())))
(tn "+ 0 arg")
(assert-equal? (tn) 0  (+))
(tn "+ 1 arg")
(assert-equal? (tn) -2 (+ -2))
(assert-equal? (tn) -1 (+ -1))
(assert-equal? (tn) 0  (+ 0))
(assert-equal? (tn) 1  (+ 1))
(assert-equal? (tn) 2  (+ 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "+ 1 arg 64-bit")
      (string-eval "(assert-equal? (tn)    956397711204 (+    956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (+   -956397711204))")
      (string-eval "(assert-equal? (tn)  13121090146595 (+  13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090146595 (+ -13121090146595))")))
(tn "+ 2 args")
(assert-equal? (tn) -4 (+ -2 -2))
(assert-equal? (tn) -3 (+ -2 -1))
(assert-equal? (tn) -2 (+ -2 0))
(assert-equal? (tn) -1 (+ -2 1))
(assert-equal? (tn) 0  (+ -2 2))
(assert-equal? (tn) -3 (+ -1 -2))
(assert-equal? (tn) -2 (+ -1 -1))
(assert-equal? (tn) -1 (+ -1 0))
(assert-equal? (tn) 0  (+ -1 1))
(assert-equal? (tn) 1  (+ -1 2))
(assert-equal? (tn) -2 (+ 0 -2))
(assert-equal? (tn) -1 (+ 0 -1))
(assert-equal? (tn) 0  (+ 0 0))
(assert-equal? (tn) 1  (+ 0 1))
(assert-equal? (tn) 2  (+ 0 2))
(assert-equal? (tn) -1 (+ 1 -2))
(assert-equal? (tn) 0  (+ 1 -1))
(assert-equal? (tn) 1  (+ 1 0))
(assert-equal? (tn) 2  (+ 1 1))
(assert-equal? (tn) 3  (+ 1 2))
(assert-equal? (tn) 0  (+ 2 -2))
(assert-equal? (tn) 1  (+ 2 -1))
(assert-equal? (tn) 2  (+ 2 0))
(assert-equal? (tn) 3  (+ 2 1))
(assert-equal? (tn) 4  (+ 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "+ 2 args 64-bit")
      (string-eval "(assert-equal? (tn)   1912795422408 (+    956397711204    956397711204))")
      (string-eval "(assert-equal? (tn)               0 (+    956397711204   -956397711204))")
      (string-eval "(assert-equal? (tn)               0 (+   -956397711204    956397711204))")
      (string-eval "(assert-equal? (tn)  -1912795422408 (+   -956397711204   -956397711204))")
      (string-eval "(assert-equal? (tn)  14077487857799 (+    956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (+   -956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn) -12164692435391 (+    956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn) -14077487857799 (+   -956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)  14077487857799 (+  13121090146595    956397711204))")
      (string-eval "(assert-equal? (tn)  12164692435391 (+  13121090146595   -956397711204))")
      (string-eval "(assert-equal? (tn) -12164692435391 (+ -13121090146595    956397711204))")
      (string-eval "(assert-equal? (tn) -14077487857799 (+ -13121090146595   -956397711204))")))
(tn "+ 3 args")
(assert-equal? (tn) -6 (+ -2 -2 -2))
(assert-equal? (tn) -5 (+ -2 -2 -1))
(assert-equal? (tn) -4 (+ -2 -2 0))
(assert-equal? (tn) -3 (+ -2 -2 1))
(assert-equal? (tn) -2 (+ -2 -2 2))
(assert-equal? (tn) -5 (+ -2 -1 -2))
(assert-equal? (tn) -4 (+ -2 -1 -1))
(assert-equal? (tn) -3 (+ -2 -1 0))
(assert-equal? (tn) -2 (+ -2 -1 1))
(assert-equal? (tn) -1 (+ -2 -1 2))
(assert-equal? (tn) -4 (+ -2 0 -2))
(assert-equal? (tn) -3 (+ -2 0 -1))
(assert-equal? (tn) -2 (+ -2 0 0))
(assert-equal? (tn) -1 (+ -2 0 1))
(assert-equal? (tn) 0  (+ -2 0 2))
(assert-equal? (tn) -3 (+ -2 1 -2))
(assert-equal? (tn) -2 (+ -2 1 -1))
(assert-equal? (tn) -1 (+ -2 1 0))
(assert-equal? (tn) 0  (+ -2 1 1))
(assert-equal? (tn) 1  (+ -2 1 2))
(assert-equal? (tn) -2 (+ -2 2 -2))
(assert-equal? (tn) -1 (+ -2 2 -1))
(assert-equal? (tn) 0  (+ -2 2 0))
(assert-equal? (tn) 1  (+ -2 2 1))
(assert-equal? (tn) 2  (+ -2 2 2))
(assert-equal? (tn) -5 (+ -1 -2 -2))
(assert-equal? (tn) -4 (+ -1 -2 -1))
(assert-equal? (tn) -3 (+ -1 -2 0))
(assert-equal? (tn) -2 (+ -1 -2 1))
(assert-equal? (tn) -1 (+ -1 -2 2))
(assert-equal? (tn) -4 (+ -1 -1 -2))
(assert-equal? (tn) -3 (+ -1 -1 -1))
(assert-equal? (tn) -2 (+ -1 -1 0))
(assert-equal? (tn) -1 (+ -1 -1 1))
(assert-equal? (tn) 0  (+ -1 -1 2))
(assert-equal? (tn) -3 (+ -1 0 -2))
(assert-equal? (tn) -2 (+ -1 0 -1))
(assert-equal? (tn) -1 (+ -1 0 0))
(assert-equal? (tn) 0  (+ -1 0 1))
(assert-equal? (tn) 1  (+ -1 0 2))
(assert-equal? (tn) -2 (+ -1 1 -2))
(assert-equal? (tn) -1 (+ -1 1 -1))
(assert-equal? (tn) 0  (+ -1 1 0))
(assert-equal? (tn) 1  (+ -1 1 1))
(assert-equal? (tn) 2  (+ -1 1 2))
(assert-equal? (tn) -1 (+ -1 2 -2))
(assert-equal? (tn) 0  (+ -1 2 -1))
(assert-equal? (tn) 1  (+ -1 2 0))
(assert-equal? (tn) 2  (+ -1 2 1))
(assert-equal? (tn) 3  (+ -1 2 2))
(assert-equal? (tn) -4 (+ 0 -2 -2))
(assert-equal? (tn) -3 (+ 0 -2 -1))
(assert-equal? (tn) -2 (+ 0 -2 0))
(assert-equal? (tn) -1 (+ 0 -2 1))
(assert-equal? (tn) 0  (+ 0 -2 2))
(assert-equal? (tn) -3 (+ 0 -1 -2))
(assert-equal? (tn) -2 (+ 0 -1 -1))
(assert-equal? (tn) -1 (+ 0 -1 0))
(assert-equal? (tn) 0  (+ 0 -1 1))
(assert-equal? (tn) 1  (+ 0 -1 2))
(assert-equal? (tn) -2 (+ 0 0 -2))
(assert-equal? (tn) -1 (+ 0 0 -1))
(assert-equal? (tn) 0  (+ 0 0 0))
(assert-equal? (tn) 1  (+ 0 0 1))
(assert-equal? (tn) 2  (+ 0 0 2))
(assert-equal? (tn) -1 (+ 0 1 -2))
(assert-equal? (tn) 0  (+ 0 1 -1))
(assert-equal? (tn) 1  (+ 0 1 0))
(assert-equal? (tn) 2  (+ 0 1 1))
(assert-equal? (tn) 3  (+ 0 1 2))
(assert-equal? (tn) 0  (+ 0 2 -2))
(assert-equal? (tn) 1  (+ 0 2 -1))
(assert-equal? (tn) 2  (+ 0 2 0))
(assert-equal? (tn) 3  (+ 0 2 1))
(assert-equal? (tn) 4  (+ 0 2 2))
(assert-equal? (tn) -3 (+ 1 -2 -2))
(assert-equal? (tn) -2 (+ 1 -2 -1))
(assert-equal? (tn) -1 (+ 1 -2 0))
(assert-equal? (tn) 0  (+ 1 -2 1))
(assert-equal? (tn) 1  (+ 1 -2 2))
(assert-equal? (tn) -2 (+ 1 -1 -2))
(assert-equal? (tn) -1 (+ 1 -1 -1))
(assert-equal? (tn) 0  (+ 1 -1 0))
(assert-equal? (tn) 1  (+ 1 -1 1))
(assert-equal? (tn) 2  (+ 1 -1 2))
(assert-equal? (tn) -1 (+ 1 0 -2))
(assert-equal? (tn) 0  (+ 1 0 -1))
(assert-equal? (tn) 1  (+ 1 0 0))
(assert-equal? (tn) 2  (+ 1 0 1))
(assert-equal? (tn) 3  (+ 1 0 2))
(assert-equal? (tn) 0  (+ 1 1 -2))
(assert-equal? (tn) 1  (+ 1 1 -1))
(assert-equal? (tn) 2  (+ 1 1 0))
(assert-equal? (tn) 3  (+ 1 1 1))
(assert-equal? (tn) 4  (+ 1 1 2))
(assert-equal? (tn) 1  (+ 1 2 -2))
(assert-equal? (tn) 2  (+ 1 2 -1))
(assert-equal? (tn) 3  (+ 1 2 0))
(assert-equal? (tn) 4  (+ 1 2 1))
(assert-equal? (tn) 5  (+ 1 2 2))
(assert-equal? (tn) -2 (+ 2 -2 -2))
(assert-equal? (tn) -1 (+ 2 -2 -1))
(assert-equal? (tn) 0  (+ 2 -2 0))
(assert-equal? (tn) 1  (+ 2 -2 1))
(assert-equal? (tn) 2  (+ 2 -2 2))
(assert-equal? (tn) -1 (+ 2 -1 -2))
(assert-equal? (tn) 0  (+ 2 -1 -1))
(assert-equal? (tn) 1  (+ 2 -1 0))
(assert-equal? (tn) 2  (+ 2 -1 1))
(assert-equal? (tn) 3  (+ 2 -1 2))
(assert-equal? (tn) 0  (+ 2 0 -2))
(assert-equal? (tn) 1  (+ 2 0 -1))
(assert-equal? (tn) 2  (+ 2 0 0))
(assert-equal? (tn) 3  (+ 2 0 1))
(assert-equal? (tn) 4  (+ 2 0 2))
(assert-equal? (tn) 1  (+ 2 1 -2))
(assert-equal? (tn) 2  (+ 2 1 -1))
(assert-equal? (tn) 3  (+ 2 1 0))
(assert-equal? (tn) 4  (+ 2 1 1))
(assert-equal? (tn) 5  (+ 2 1 2))
(assert-equal? (tn) 2  (+ 2 2 -2))
(assert-equal? (tn) 3  (+ 2 2 -1))
(assert-equal? (tn) 4  (+ 2 2 0))
(assert-equal? (tn) 5  (+ 2 2 1))
(assert-equal? (tn) 6  (+ 2 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "+ 3 args 64-bit")
      (string-eval "(assert-equal? (tn)   2869193133612 (+  956397711204  956397711204  956397711204))")
      (string-eval "(assert-equal? (tn)  -2869193133612 (+ -956397711204 -956397711204 -956397711204))")
      (string-eval "(assert-equal? (tn)  14077487857799 (+ 0  956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (+ 0 -956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn) -12164692435391 (+ 0  956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn) -14077487857799 (+ 0 -956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)  14077487857799 (+  13121090146595  956397711204 0))")
      (string-eval "(assert-equal? (tn)  12164692435391 (+  13121090146595 -956397711204 0))")
      (string-eval "(assert-equal? (tn) -12164692435391 (+ -13121090146595  956397711204 0))")
      (string-eval "(assert-equal? (tn) -14077487857799 (+ -13121090146595 -956397711204 0))")
      (string-eval "(assert-equal? (tn)  14077487857799 (+  956397711204 0  13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (+ -956397711204 0  13121090146595))")
      (string-eval "(assert-equal? (tn) -12164692435391 (+  956397711204 0 -13121090146595))")
      (string-eval "(assert-equal? (tn) -14077487857799 (+ -956397711204 0 -13121090146595))")
      (string-eval "(assert-equal? (tn)  14077487857799 (+  13121090146595 0  956397711204))")
      (string-eval "(assert-equal? (tn)  12164692435391 (+  13121090146595 0 -956397711204))")
      (string-eval "(assert-equal? (tn) -12164692435391 (+ -13121090146595 0  956397711204))")
      (string-eval "(assert-equal? (tn) -14077487857799 (+ -13121090146595 0 -956397711204))")))
(tn "+ many args")
(assert-equal? (tn) 8  (+ 2 2 2 2))
(assert-equal? (tn) 6  (+ 0 2 2 2))
(assert-equal? (tn) 6  (+ 2 0 2 2))
(assert-equal? (tn) 6  (+ 2 2 0 2))
(assert-equal? (tn) 6  (+ 2 2 2 0))
(assert-equal? (tn) 0  (+ -2 -1 0 1 2))
(assert-equal? (tn) 0  (+ 2 1 0 -1 -2))
(assert-equal? (tn) -1 (+ -2 -1 0 -1 1 2))
(assert-equal? (tn) 1  (+ 2 1 0 -1 1 -2))
(assert-equal? (tn) 0  (+ -2 -2 -1 -1 0 0 1 1 2 2))
(assert-equal? (tn) 0  (+ 2 2 1 1 0 0 -1 -1 -2 -2))
(assert-equal? (tn) 17 (+ 3 1 5 -7 2 13))
(tn "+ boundary numbers")
(case fixnum-bits
  ((28)
   (string-eval "(assert-error  (tn) (lambda () (+  134217727  1)))")
   (string-eval "(assert-error  (tn) (lambda () (+ -134217728 -1)))")
   (string-eval "(assert-error  (tn) (lambda () (+  134217727  134217727)))")
   (string-eval "(assert-equal? (tn) 0          (+  134217727 -134217727))")
   (string-eval "(assert-equal? (tn) -1         (+  134217727 -134217728))")
   (string-eval "(assert-equal? (tn) 0          (+ -134217727  134217727))")
   (string-eval "(assert-equal? (tn) -1         (+ -134217728  134217727))")
   (string-eval "(assert-error  (tn) (lambda () (+ -134217728 -134217728)))"))
  ((32)
   (string-eval "(assert-error  (tn) (lambda () (+  2147483647  1)))")
   (string-eval "(assert-error  (tn) (lambda () (+ -2147483648 -1)))")
   (string-eval "(assert-error  (tn) (lambda () (+  2147483647  2147483647)))")
   (string-eval "(assert-equal? (tn) 0          (+  2147483647 -2147483647))")
   (string-eval "(assert-equal? (tn) -1         (+  2147483647 -2147483648))")
   (string-eval "(assert-equal? (tn) 0          (+ -2147483647  2147483647))")
   (string-eval "(assert-equal? (tn) -1         (+ -2147483648  2147483647))")
   (string-eval "(assert-error  (tn) (lambda () (+ -2147483648 -2147483648)))"))
  ((60)
   (string-eval "(assert-error  (tn) (lambda () (+  576460752303423487  1)))")
   (string-eval "(assert-error  (tn) (lambda () (+ -576460752303423488 -1)))")
   (string-eval "(assert-error  (tn) (lambda () (+  576460752303423487
                                                    576460752303423487)))")
   (string-eval "(assert-equal? (tn) 0          (+  576460752303423487
                                                   -576460752303423487))")
   (string-eval "(assert-equal? (tn) -1         (+  576460752303423487
                                                   -576460752303423488))")
   (string-eval "(assert-equal? (tn) 0          (+ -576460752303423487
                                                    576460752303423487))")
   (string-eval "(assert-equal? (tn) -1         (+ -576460752303423488
                                                    576460752303423487))")
   (string-eval "(assert-error  (tn) (lambda () (+ -576460752303423488
                                                   -576460752303423488)))"))
  ((64)
   (string-eval "(assert-error  (tn) (lambda () (+  9223372036854775807  1)))")
   (string-eval "(assert-error  (tn) (lambda () (+ -9223372036854775808 -1)))")
   (string-eval "(assert-error  (tn) (lambda () (+  9223372036854775807
                                                    9223372036854775807)))")
   (string-eval "(assert-equal? (tn) 0          (+  9223372036854775807
                                                   -9223372036854775807))")
   (string-eval "(assert-equal? (tn) -1         (+  9223372036854775807
                                                   -9223372036854775808))")
   (string-eval "(assert-equal? (tn) 0          (+ -9223372036854775807
                                                    9223372036854775807))")
   (string-eval "(assert-equal? (tn) -1         (+ -9223372036854775808
                                                    9223372036854775807))")
   (string-eval "(assert-error  (tn) (lambda () (+ -9223372036854775808
                                                   -9223372036854775808)))"))
  (else
   (assert-fail (tn) "unknown int bitwidth")))


(tn "* invalid forms")
(assert-error  (tn) (lambda () (* #t)))
(assert-error  (tn) (lambda () (* #f)))
(assert-error  (tn) (lambda () (* '())))
(assert-error  (tn) (lambda () (* 1 #t)))
(assert-error  (tn) (lambda () (* 1 #f)))
(assert-error  (tn) (lambda () (* 1 '())))
(assert-error  (tn) (lambda () (* 1 2 #t)))
(assert-error  (tn) (lambda () (* 1 2 #f)))
(assert-error  (tn) (lambda () (* 1 2 '())))
(tn "* 0 arg")
(assert-equal? (tn) 1 (*))
(tn "* 1 arg")
(assert-equal? (tn) -2 (* -2))
(assert-equal? (tn) -1 (* -1))
(assert-equal? (tn) 0  (* 0))
(assert-equal? (tn) 1  (* 1))
(assert-equal? (tn) 2  (* 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "* 1 arg 64-bit")
      (string-eval "(assert-equal? (tn)    956397711204 (*    956397711204))")
      (string-eval "(assert-equal? (tn)   -956397711204 (*   -956397711204))")
      (string-eval "(assert-equal? (tn)  13121090146595 (*  13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090146595 (* -13121090146595))")))
(tn "* 2 args")
(assert-equal? (tn) 4  (* -2 -2))
(assert-equal? (tn) 2  (* -2 -1))
(assert-equal? (tn) 0  (* -2 0))
(assert-equal? (tn) -2 (* -2 1))
(assert-equal? (tn) -4 (* -2 2))
(assert-equal? (tn) 2  (* -1 -2))
(assert-equal? (tn) 1  (* -1 -1))
(assert-equal? (tn) 0  (* -1 0))
(assert-equal? (tn) -1 (* -1 1))
(assert-equal? (tn) -2 (* -1 2))
(assert-equal? (tn) 0  (* 0 -2))
(assert-equal? (tn) 0  (* 0 -1))
(assert-equal? (tn) 0  (* 0 0))
(assert-equal? (tn) 0  (* 0 1))
(assert-equal? (tn) 0  (* 0 2))
(assert-equal? (tn) -2 (* 1 -2))
(assert-equal? (tn) -1 (* 1 -1))
(assert-equal? (tn) 0  (* 1 0))
(assert-equal? (tn) 1  (* 1 1))
(assert-equal? (tn) 2  (* 1 2))
(assert-equal? (tn) -4 (* 2 -2))
(assert-equal? (tn) -2 (* 2 -1))
(assert-equal? (tn) 0  (* 2 0))
(assert-equal? (tn) 2  (* 2 1))
(assert-equal? (tn) 4  (* 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "* 2 args 64-bit")
      (string-eval "(assert-equal? (tn)  67193102640712995 (*  13121090146595  5121))")
      (string-eval "(assert-equal? (tn) -67193102640712995 (*  13121090146595 -5121))")
      (string-eval "(assert-equal? (tn) -67193102640712995 (* -13121090146595  5121))")
      (string-eval "(assert-equal? (tn)  67193102640712995 (* -13121090146595 -5121))")
      (string-eval "(assert-equal? (tn)                  0 (*  13121090146595 0))")
      (string-eval "(assert-equal? (tn)                  0 (* -13121090146595 0))")
      (string-eval "(assert-equal? (tn)  67193102640712995 (*  5121  13121090146595))")
      (string-eval "(assert-equal? (tn) -67193102640712995 (* -5121  13121090146595))")
      (string-eval "(assert-equal? (tn) -67193102640712995 (*  5121 -13121090146595))")
      (string-eval "(assert-equal? (tn)  67193102640712995 (* -5121 -13121090146595))")
      (string-eval "(assert-equal? (tn)                  0 (*     0  13121090146595))")
      (string-eval "(assert-equal? (tn)                  0 (*     0 -13121090146595))")))
(tn "* 3 args")     
(assert-equal? (tn) -8 (* -2 -2 -2))
(assert-equal? (tn) -4 (* -2 -2 -1))
(assert-equal? (tn) 0  (* -2 -2 0))
(assert-equal? (tn) 4  (* -2 -2 1))
(assert-equal? (tn) 8  (* -2 -2 2))
(assert-equal? (tn) -4 (* -2 -1 -2))
(assert-equal? (tn) -2 (* -2 -1 -1))
(assert-equal? (tn) 0  (* -2 -1 0))
(assert-equal? (tn) 2  (* -2 -1 1))
(assert-equal? (tn) 4  (* -2 -1 2))
(assert-equal? (tn) 0  (* -2 0 -2))
(assert-equal? (tn) 0  (* -2 0 -1))
(assert-equal? (tn) 0  (* -2 0 0))
(assert-equal? (tn) 0  (* -2 0 1))
(assert-equal? (tn) 0  (* -2 0 2))
(assert-equal? (tn) 4  (* -2 1 -2))
(assert-equal? (tn) 2  (* -2 1 -1))
(assert-equal? (tn) 0  (* -2 1 0))
(assert-equal? (tn) -2 (* -2 1 1))
(assert-equal? (tn) -4 (* -2 1 2))
(assert-equal? (tn) 8  (* -2 2 -2))
(assert-equal? (tn) 4  (* -2 2 -1))
(assert-equal? (tn) 0  (* -2 2 0))
(assert-equal? (tn) -4 (* -2 2 1))
(assert-equal? (tn) -8 (* -2 2 2))
(assert-equal? (tn) -4 (* -1 -2 -2))
(assert-equal? (tn) -2 (* -1 -2 -1))
(assert-equal? (tn) 0  (* -1 -2 0))
(assert-equal? (tn) 2  (* -1 -2 1))
(assert-equal? (tn) 4  (* -1 -2 2))
(assert-equal? (tn) -2 (* -1 -1 -2))
(assert-equal? (tn) -1 (* -1 -1 -1))
(assert-equal? (tn) 0  (* -1 -1 0))
(assert-equal? (tn) 1  (* -1 -1 1))
(assert-equal? (tn) 2  (* -1 -1 2))
(assert-equal? (tn) 0  (* -1 0 -2))
(assert-equal? (tn) 0  (* -1 0 -1))
(assert-equal? (tn) 0  (* -1 0 0))
(assert-equal? (tn) 0  (* -1 0 1))
(assert-equal? (tn) 0  (* -1 0 2))
(assert-equal? (tn) 2  (* -1 1 -2))
(assert-equal? (tn) 1  (* -1 1 -1))
(assert-equal? (tn) 0  (* -1 1 0))
(assert-equal? (tn) -1 (* -1 1 1))
(assert-equal? (tn) -2 (* -1 1 2))
(assert-equal? (tn) 4  (* -1 2 -2))
(assert-equal? (tn) 2  (* -1 2 -1))
(assert-equal? (tn) 0  (* -1 2 0))
(assert-equal? (tn) -2 (* -1 2 1))
(assert-equal? (tn) -4 (* -1 2 2))
(assert-equal? (tn) 0  (* 0 -2 -2))
(assert-equal? (tn) 0  (* 0 -2 -1))
(assert-equal? (tn) 0  (* 0 -2 0))
(assert-equal? (tn) 0  (* 0 -2 1))
(assert-equal? (tn) 0  (* 0 -2 2))
(assert-equal? (tn) 0  (* 0 -1 -2))
(assert-equal? (tn) 0  (* 0 -1 -1))
(assert-equal? (tn) 0  (* 0 -1 0))
(assert-equal? (tn) 0  (* 0 -1 1))
(assert-equal? (tn) 0  (* 0 -1 2))
(assert-equal? (tn) 0  (* 0 0 -2))
(assert-equal? (tn) 0  (* 0 0 -1))
(assert-equal? (tn) 0  (* 0 0 0))
(assert-equal? (tn) 0  (* 0 0 1))
(assert-equal? (tn) 0  (* 0 0 2))
(assert-equal? (tn) 0  (* 0 1 -2))
(assert-equal? (tn) 0  (* 0 1 -1))
(assert-equal? (tn) 0  (* 0 1 0))
(assert-equal? (tn) 0  (* 0 1 1))
(assert-equal? (tn) 0  (* 0 1 2))
(assert-equal? (tn) 0  (* 0 2 -2))
(assert-equal? (tn) 0  (* 0 2 -1))
(assert-equal? (tn) 0  (* 0 2 0))
(assert-equal? (tn) 0  (* 0 2 1))
(assert-equal? (tn) 0  (* 0 2 2))
(assert-equal? (tn) 4  (* 1 -2 -2))
(assert-equal? (tn) 2  (* 1 -2 -1))
(assert-equal? (tn) 0  (* 1 -2 0))
(assert-equal? (tn) -2 (* 1 -2 1))
(assert-equal? (tn) -4 (* 1 -2 2))
(assert-equal? (tn) 2  (* 1 -1 -2))
(assert-equal? (tn) 1  (* 1 -1 -1))
(assert-equal? (tn) 0  (* 1 -1 0))
(assert-equal? (tn) -1 (* 1 -1 1))
(assert-equal? (tn) -2 (* 1 -1 2))
(assert-equal? (tn) 0  (* 1 0 -2))
(assert-equal? (tn) 0  (* 1 0 -1))
(assert-equal? (tn) 0  (* 1 0 0))
(assert-equal? (tn) 0  (* 1 0 1))
(assert-equal? (tn) 0  (* 1 0 2))
(assert-equal? (tn) -2 (* 1 1 -2))
(assert-equal? (tn) -1 (* 1 1 -1))
(assert-equal? (tn) 0  (* 1 1 0))
(assert-equal? (tn) 1  (* 1 1 1))
(assert-equal? (tn) 2  (* 1 1 2))
(assert-equal? (tn) -4 (* 1 2 -2))
(assert-equal? (tn) -2 (* 1 2 -1))
(assert-equal? (tn) 0  (* 1 2 0))
(assert-equal? (tn) 2  (* 1 2 1))
(assert-equal? (tn) 4  (* 1 2 2))
(assert-equal? (tn) 8  (* 2 -2 -2))
(assert-equal? (tn) 4  (* 2 -2 -1))
(assert-equal? (tn) 0  (* 2 -2 0))
(assert-equal? (tn) -4 (* 2 -2 1))
(assert-equal? (tn) -8 (* 2 -2 2))
(assert-equal? (tn) 4  (* 2 -1 -2))
(assert-equal? (tn) 2  (* 2 -1 -1))
(assert-equal? (tn) 0  (* 2 -1 0))
(assert-equal? (tn) -2 (* 2 -1 1))
(assert-equal? (tn) -4 (* 2 -1 2))
(assert-equal? (tn) 0  (* 2 0 -2))
(assert-equal? (tn) 0  (* 2 0 -1))
(assert-equal? (tn) 0  (* 2 0 0))
(assert-equal? (tn) 0  (* 2 0 1))
(assert-equal? (tn) 0  (* 2 0 2))
(assert-equal? (tn) -4 (* 2 1 -2))
(assert-equal? (tn) -2 (* 2 1 -1))
(assert-equal? (tn) 0  (* 2 1 0))
(assert-equal? (tn) 2  (* 2 1 1))
(assert-equal? (tn) 4  (* 2 1 2))
(assert-equal? (tn) -8 (* 2 2 -2))
(assert-equal? (tn) -4 (* 2 2 -1))
(assert-equal? (tn) 0  (* 2 2 0))
(assert-equal? (tn) 4  (* 2 2 1))
(assert-equal? (tn) 8  (* 2 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "* 3 args 64-bit")
      (string-eval "(assert-equal? (tn)  470351718484990965 (*  13121090146595  5121  7))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (*  13121090146595 -5121  7))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (* -13121090146595  5121  7))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (* -13121090146595 -5121  7))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (*  13121090146595  5121 -7))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (*  13121090146595 -5121 -7))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (* -13121090146595  5121 -7))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (* -13121090146595 -5121 -7))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (*  5121  13121090146595  7))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (* -5121  13121090146595  7))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (*  5121 -13121090146595  7))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (* -5121 -13121090146595  7))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (*  5121  13121090146595 -7))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (* -5121  13121090146595 -7))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (*  5121 -13121090146595 -7))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (* -5121 -13121090146595 -7))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (*  5121  7  13121090146595))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (* -5121  7  13121090146595))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (*  5121  7 -13121090146595))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (* -5121  7 -13121090146595))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (*  5121 -7  13121090146595))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (* -5121 -7  13121090146595))")
      (string-eval "(assert-equal? (tn)  470351718484990965 (*  5121 -7 -13121090146595))")
      (string-eval "(assert-equal? (tn) -470351718484990965 (* -5121 -7 -13121090146595))")
      (string-eval "(assert-equal? (tn) 0 (* 0 13121090146595 5121))")
      (string-eval "(assert-equal? (tn) 0 (* 13121090146595 0 5121))")
      (string-eval "(assert-equal? (tn) 0 (* 13121090146595 5121 0))")
      (string-eval "(assert-equal? (tn) 0 (* 0 -13121090146595 5121))")
      (string-eval "(assert-equal? (tn) 0 (* -13121090146595 0 5121))")
      (string-eval "(assert-equal? (tn) 0 (* -13121090146595 5121 0))")))
(tn "* many args")  
(assert-equal? (tn) 16 (* 2 2 2 2))
(assert-equal? (tn) 0  (* 0 2 2 2))
(assert-equal? (tn) 0  (* 2 0 2 2))
(assert-equal? (tn) 0  (* 2 2 0 2))
(assert-equal? (tn) 0  (* 2 2 2 0))
(assert-equal? (tn) 0  (* -2 -1 0 1 2))
(assert-equal? (tn) 0  (* 2 1 0 -1 -2))
(assert-equal? (tn) 0  (* -2 -1 0 -1 1 2))
(assert-equal? (tn) 0  (* 2 1 0 -1 1 -2))
(assert-equal? (tn) 0  (* -2 -2 -1 -1 0 0 1 1 2 2))
(assert-equal? (tn) 0  (* 2 2 1 1 0 0 -1 -1 -2 -2))
(assert-equal? (tn) -2730 (* 3 1 5 -7 2 13))
(tn "* boundary numbers")
;; SigScheme does not check overflow on '*'
(if sigscheme?
    (case fixnum-bits
      ((28)
       (string-eval "(assert-equal? (tn) -2 (*  134217727 2))")
       (string-eval "(assert-equal? (tn) 0  (* -134217728 2))")
       (string-eval "(assert-equal? (tn)  134217725 (*  134217727 3))")
       (string-eval "(assert-equal? (tn) -134217728 (* -134217728 3))"))
      ((32)
       (string-eval "(assert-equal? (tn) -2 (*  2147483647 2))")
       (string-eval "(assert-equal? (tn) 0  (* -2147483648 2))")
       (string-eval "(assert-equal? (tn)  2147483645 (*  2147483647 3))")
       (string-eval "(assert-equal? (tn) -2147483648 (* -2147483648 3))"))
      ((60)
       (string-eval "(assert-equal? (tn) -2 (*  576460752303423487 2))")
       (string-eval "(assert-equal? (tn) 0  (* -576460752303423488 2))")
       (string-eval
        "(assert-equal? (tn)  576460752303423485 (*  576460752303423487 3))")
       (string-eval
        "(assert-equal? (tn) -576460752303423488 (* -576460752303423488 3))"))
      ((64)
       (string-eval "(assert-equal? (tn) -2 (*  9223372036854775807 2))")
       (string-eval "(assert-equal? (tn) 0  (* -9223372036854775808 2))")
       (string-eval
        "(assert-equal? (tn)  9223372036854775805 (*  9223372036854775807 3))")
       (string-eval
        "(assert-equal? (tn) -9223372036854775808 (* -9223372036854775808 3))"))
      (else
       (assert-fail (tn) "unknown int bitwidth"))))

(tn "- invalid forms")
(assert-error  (tn) (lambda () (-)))
(assert-error  (tn) (lambda () (- #t)))
(assert-error  (tn) (lambda () (- #f)))
(assert-error  (tn) (lambda () (- '())))
(assert-error  (tn) (lambda () (- #t #t)))
(assert-error  (tn) (lambda () (- #f #f)))
(assert-error  (tn) (lambda () (- '() '())))
(assert-error  (tn) (lambda () (- 1 #t)))
(assert-error  (tn) (lambda () (- 1 #f)))
(assert-error  (tn) (lambda () (- 1 '())))
(assert-error  (tn) (lambda () (- 1 2 #t)))
(assert-error  (tn) (lambda () (- 1 2 #f)))
(assert-error  (tn) (lambda () (- 1 2 '())))
(tn "- 1 arg")
(assert-equal? (tn) 2  (- -2))
(assert-equal? (tn) 1  (- -1))
(assert-equal? (tn) 0  (- 0))
(assert-equal? (tn) -1 (- 1))
(assert-equal? (tn) -2 (- 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "- 1 arg 64-bit")
      (string-eval "(assert-equal? (tn)   -956397711204 (-    956397711204))")
      (string-eval "(assert-equal? (tn)    956397711204 (-   -956397711204))")
      (string-eval "(assert-equal? (tn) -13121090146595 (-  13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090146595 (- -13121090146595))")))
(tn "- 2 args")
(assert-equal? (tn) 0  (- -2 -2))
(assert-equal? (tn) -1 (- -2 -1))
(assert-equal? (tn) -2 (- -2 0))
(assert-equal? (tn) -3 (- -2 1))
(assert-equal? (tn) -4 (- -2 2))
(assert-equal? (tn) 1  (- -1 -2))
(assert-equal? (tn) 0  (- -1 -1))
(assert-equal? (tn) -1 (- -1 0))
(assert-equal? (tn) -2 (- -1 1))
(assert-equal? (tn) -3 (- -1 2))
(assert-equal? (tn) 2  (- 0 -2))
(assert-equal? (tn) 1  (- 0 -1))
(assert-equal? (tn) 0  (- 0 0))
(assert-equal? (tn) -1 (- 0 1))
(assert-equal? (tn) -2 (- 0 2))
(assert-equal? (tn) 3  (- 1 -2))
(assert-equal? (tn) 2  (- 1 -1))
(assert-equal? (tn) 1  (- 1 0))
(assert-equal? (tn) 0  (- 1 1))
(assert-equal? (tn) -1 (- 1 2))
(assert-equal? (tn) 4  (- 2 -2))
(assert-equal? (tn) 3  (- 2 -1))
(assert-equal? (tn) 2  (- 2 0))
(assert-equal? (tn) 1  (- 2 1))
(assert-equal? (tn) 0  (- 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "- 2 args 64-bit")
      (string-eval "(assert-equal? (tn)               0 (-    956397711204    956397711204))")
      (string-eval "(assert-equal? (tn)   1912795422408 (-    956397711204   -956397711204))")
      (string-eval "(assert-equal? (tn)  -1912795422408 (-   -956397711204    956397711204))")
      (string-eval "(assert-equal? (tn)               0 (-   -956397711204   -956397711204))")
      (string-eval "(assert-equal? (tn) -12164692435391 (-    956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn) -14077487857799 (-   -956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)  14077487857799 (-    956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (-   -956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (-  13121090146595    956397711204))")
      (string-eval "(assert-equal? (tn)  14077487857799 (-  13121090146595   -956397711204))")
      (string-eval "(assert-equal? (tn) -14077487857799 (- -13121090146595    956397711204))")
      (string-eval "(assert-equal? (tn) -12164692435391 (- -13121090146595   -956397711204))")))
(tn "- 3 args")
(assert-equal? (tn) 2  (- -2 -2 -2))
(assert-equal? (tn) 1  (- -2 -2 -1))
(assert-equal? (tn) 0  (- -2 -2 0))
(assert-equal? (tn) -1 (- -2 -2 1))
(assert-equal? (tn) -2 (- -2 -2 2))
(assert-equal? (tn) 1  (- -2 -1 -2))
(assert-equal? (tn) 0  (- -2 -1 -1))
(assert-equal? (tn) -1 (- -2 -1 0))
(assert-equal? (tn) -2 (- -2 -1 1))
(assert-equal? (tn) -3 (- -2 -1 2))
(assert-equal? (tn) 0  (- -2 0 -2))
(assert-equal? (tn) -1 (- -2 0 -1))
(assert-equal? (tn) -2 (- -2 0 0))
(assert-equal? (tn) -3 (- -2 0 1))
(assert-equal? (tn) -4 (- -2 0 2))
(assert-equal? (tn) -1 (- -2 1 -2))
(assert-equal? (tn) -2 (- -2 1 -1))
(assert-equal? (tn) -3 (- -2 1 0))
(assert-equal? (tn) -4 (- -2 1 1))
(assert-equal? (tn) -5 (- -2 1 2))
(assert-equal? (tn) -2 (- -2 2 -2))
(assert-equal? (tn) -3 (- -2 2 -1))
(assert-equal? (tn) -4 (- -2 2 0))
(assert-equal? (tn) -5 (- -2 2 1))
(assert-equal? (tn) -6 (- -2 2 2))
(assert-equal? (tn) 3  (- -1 -2 -2))
(assert-equal? (tn) 2  (- -1 -2 -1))
(assert-equal? (tn) 1  (- -1 -2 0))
(assert-equal? (tn) 0  (- -1 -2 1))
(assert-equal? (tn) -1 (- -1 -2 2))
(assert-equal? (tn) 2  (- -1 -1 -2))
(assert-equal? (tn) 1  (- -1 -1 -1))
(assert-equal? (tn) 0  (- -1 -1 0))
(assert-equal? (tn) -1 (- -1 -1 1))
(assert-equal? (tn) -2 (- -1 -1 2))
(assert-equal? (tn) 1  (- -1 0 -2))
(assert-equal? (tn) 0  (- -1 0 -1))
(assert-equal? (tn) -1 (- -1 0 0))
(assert-equal? (tn) -2 (- -1 0 1))
(assert-equal? (tn) -3 (- -1 0 2))
(assert-equal? (tn) 0  (- -1 1 -2))
(assert-equal? (tn) -1 (- -1 1 -1))
(assert-equal? (tn) -2 (- -1 1 0))
(assert-equal? (tn) -3 (- -1 1 1))
(assert-equal? (tn) -4 (- -1 1 2))
(assert-equal? (tn) -1 (- -1 2 -2))
(assert-equal? (tn) -2 (- -1 2 -1))
(assert-equal? (tn) -3 (- -1 2 0))
(assert-equal? (tn) -4 (- -1 2 1))
(assert-equal? (tn) -5 (- -1 2 2))
(assert-equal? (tn) 4  (- 0 -2 -2))
(assert-equal? (tn) 3  (- 0 -2 -1))
(assert-equal? (tn) 2  (- 0 -2 0))
(assert-equal? (tn) 1  (- 0 -2 1))
(assert-equal? (tn) 0  (- 0 -2 2))
(assert-equal? (tn) 3  (- 0 -1 -2))
(assert-equal? (tn) 2  (- 0 -1 -1))
(assert-equal? (tn) 1  (- 0 -1 0))
(assert-equal? (tn) 0  (- 0 -1 1))
(assert-equal? (tn) -1 (- 0 -1 2))
(assert-equal? (tn) 2  (- 0 0 -2))
(assert-equal? (tn) 1  (- 0 0 -1))
(assert-equal? (tn) 0  (- 0 0 0))
(assert-equal? (tn) -1 (- 0 0 1))
(assert-equal? (tn) -2 (- 0 0 2))
(assert-equal? (tn) 1  (- 0 1 -2))
(assert-equal? (tn) 0  (- 0 1 -1))
(assert-equal? (tn) -1 (- 0 1 0))
(assert-equal? (tn) -2 (- 0 1 1))
(assert-equal? (tn) -3 (- 0 1 2))
(assert-equal? (tn) 0  (- 0 2 -2))
(assert-equal? (tn) -1 (- 0 2 -1))
(assert-equal? (tn) -2 (- 0 2 0))
(assert-equal? (tn) -3 (- 0 2 1))
(assert-equal? (tn) -4 (- 0 2 2))
(assert-equal? (tn) 5  (- 1 -2 -2))
(assert-equal? (tn) 4  (- 1 -2 -1))
(assert-equal? (tn) 3  (- 1 -2 0))
(assert-equal? (tn) 2  (- 1 -2 1))
(assert-equal? (tn) 1  (- 1 -2 2))
(assert-equal? (tn) 4  (- 1 -1 -2))
(assert-equal? (tn) 3  (- 1 -1 -1))
(assert-equal? (tn) 2  (- 1 -1 0))
(assert-equal? (tn) 1  (- 1 -1 1))
(assert-equal? (tn) 0  (- 1 -1 2))
(assert-equal? (tn) 3  (- 1 0 -2))
(assert-equal? (tn) 2  (- 1 0 -1))
(assert-equal? (tn) 1  (- 1 0 0))
(assert-equal? (tn) 0  (- 1 0 1))
(assert-equal? (tn) -1 (- 1 0 2))
(assert-equal? (tn) 2  (- 1 1 -2))
(assert-equal? (tn) 1  (- 1 1 -1))
(assert-equal? (tn) 0  (- 1 1 0))
(assert-equal? (tn) -1 (- 1 1 1))
(assert-equal? (tn) -2 (- 1 1 2))
(assert-equal? (tn) 1  (- 1 2 -2))
(assert-equal? (tn) 0  (- 1 2 -1))
(assert-equal? (tn) -1 (- 1 2 0))
(assert-equal? (tn) -2 (- 1 2 1))
(assert-equal? (tn) -3 (- 1 2 2))
(assert-equal? (tn) 6  (- 2 -2 -2))
(assert-equal? (tn) 5  (- 2 -2 -1))
(assert-equal? (tn) 4  (- 2 -2 0))
(assert-equal? (tn) 3  (- 2 -2 1))
(assert-equal? (tn) 2  (- 2 -2 2))
(assert-equal? (tn) 5  (- 2 -1 -2))
(assert-equal? (tn) 4  (- 2 -1 -1))
(assert-equal? (tn) 3  (- 2 -1 0))
(assert-equal? (tn) 2  (- 2 -1 1))
(assert-equal? (tn) 1  (- 2 -1 2))
(assert-equal? (tn) 4  (- 2 0 -2))
(assert-equal? (tn) 3  (- 2 0 -1))
(assert-equal? (tn) 2  (- 2 0 0))
(assert-equal? (tn) 1  (- 2 0 1))
(assert-equal? (tn) 0  (- 2 0 2))
(assert-equal? (tn) 3  (- 2 1 -2))
(assert-equal? (tn) 2  (- 2 1 -1))
(assert-equal? (tn) 1  (- 2 1 0))
(assert-equal? (tn) 0  (- 2 1 1))
(assert-equal? (tn) -1 (- 2 1 2))
(assert-equal? (tn) 2  (- 2 2 -2))
(assert-equal? (tn) 1  (- 2 2 -1))
(assert-equal? (tn) 0  (- 2 2 0))
(assert-equal? (tn) -1 (- 2 2 1))
(assert-equal? (tn) -2 (- 2 2 2))
(if (>= fixnum-bits 60)
    (begin
      (tn "- 3 args 64-bit")
      (string-eval "(assert-equal? (tn)   -956397711204 (-  956397711204  956397711204  956397711204))")
      (string-eval "(assert-equal? (tn)    956397711204 (- -956397711204 -956397711204 -956397711204))")
      (string-eval "(assert-equal? (tn) -14077487857799 (- 0  956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn) -12164692435391 (- 0 -956397711204  13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (- 0  956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)  14077487857799 (- 0 -956397711204 -13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (-  13121090146595  956397711204 0))")
      (string-eval "(assert-equal? (tn)  14077487857799 (-  13121090146595 -956397711204 0))")
      (string-eval "(assert-equal? (tn) -14077487857799 (- -13121090146595  956397711204 0))")
      (string-eval "(assert-equal? (tn) -12164692435391 (- -13121090146595 -956397711204 0))")
      (string-eval "(assert-equal? (tn) -12164692435391 (-  956397711204 0  13121090146595))")
      (string-eval "(assert-equal? (tn) -14077487857799 (- -956397711204 0  13121090146595))")
      (string-eval "(assert-equal? (tn)  14077487857799 (-  956397711204 0 -13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (- -956397711204 0 -13121090146595))")
      (string-eval "(assert-equal? (tn)  12164692435391 (-  13121090146595 0  956397711204))")
      (string-eval "(assert-equal? (tn)  14077487857799 (-  13121090146595 0 -956397711204))")
      (string-eval "(assert-equal? (tn) -14077487857799 (- -13121090146595 0  956397711204))")
      (string-eval "(assert-equal? (tn) -12164692435391 (- -13121090146595 0 -956397711204))")))
(tn "- many args")
(assert-equal? (tn) -4 (- 2 2 2 2))
(assert-equal? (tn) -6 (- 0 2 2 2))
(assert-equal? (tn) -2 (- 2 0 2 2))
(assert-equal? (tn) -2 (- 2 2 0 2))
(assert-equal? (tn) -2 (- 2 2 2 0))
(assert-equal? (tn) -4 (- -2 -1 0 1 2))
(assert-equal? (tn) 4  (- 2 1 0 -1 -2))
(assert-equal? (tn) -3 (- -2 -1 0 -1 1 2))
(assert-equal? (tn) 3  (- 2 1 0 -1 1 -2))
(assert-equal? (tn) -4 (- -2 -2 -1 -1 0 0 1 1 2 2))
(assert-equal? (tn) 4  (- 2 2 1 1 0 0 -1 -1 -2 -2))
(assert-equal? (tn) -11 (- 3 1 5 -7 2 13))
(tn "- boundary numbers")
(case fixnum-bits
  ((28)
   (string-eval "(assert-error  (tn) (lambda () (- -134217728)))")
   (string-eval "(assert-error  (tn) (lambda () (- -134217728 1)))")
   (string-eval "(assert-equal? (tn) -134217727  (-  134217727))")
   (string-eval "(assert-equal? (tn)  134217727  (- -134217727))")
   (string-eval "(assert-equal? (tn) 0 (-  134217727  134217727))")
   (string-eval "(assert-equal? (tn) 0 (- -134217728 -134217728))"))
  ((32)
   (string-eval "(assert-error  (tn) (lambda () (- -2147483648)))")
   (string-eval "(assert-error  (tn) (lambda () (- -2147483648 1)))")
   (string-eval "(assert-equal? (tn) -2147483647 (-  2147483647))")
   (string-eval "(assert-equal? (tn)  2147483647 (- -2147483647))")
   (string-eval "(assert-equal? (tn) 0 (-  2147483647  2147483647))")
   (string-eval "(assert-equal? (tn) 0 (- -2147483648 -2147483648))"))
  ((60)
   (string-eval "(assert-error  (tn) (lambda () (- -576460752303423488)))")
   (string-eval "(assert-error  (tn) (lambda () (- -576460752303423488 1)))")
   (string-eval
    "(assert-equal? (tn) -576460752303423487  (-  576460752303423487))")
   (string-eval
    "(assert-equal? (tn)  576460752303423487  (- -576460752303423487))")
   (string-eval
    "(assert-equal? (tn) 0 (-  576460752303423487  576460752303423487))")
   (string-eval
    "(assert-equal? (tn) 0 (- -576460752303423488 -576460752303423488))"))
  ((64)
   (string-eval "(assert-error  (tn) (lambda () (- -9223372036854775808)))")
   (string-eval "(assert-error  (tn) (lambda () (- -9223372036854775808 1)))")
   (string-eval
    "(assert-equal? (tn) -9223372036854775807 (-  9223372036854775807))")
   (string-eval
    "(assert-equal? (tn)  9223372036854775807 (- -9223372036854775807))")
   (string-eval
    "(assert-equal? (tn) 0 (-  9223372036854775807  9223372036854775807))")
   (string-eval
    "(assert-equal? (tn) 0 (- -9223372036854775808 -9223372036854775808))"))
  (else
   (assert-fail (tn) "unknown int bitwidth")))

(tn "/ invalid forms")
(assert-error  (tn) (lambda () (/)))
(assert-error  (tn) (lambda () (/ #t)))
(assert-error  (tn) (lambda () (/ #f)))
(assert-error  (tn) (lambda () (/ '())))
(assert-error  (tn) (lambda () (/ #t #t)))
(assert-error  (tn) (lambda () (/ #f #f)))
(assert-error  (tn) (lambda () (/ '() '())))
(assert-error  (tn) (lambda () (/ 1 #t)))
(assert-error  (tn) (lambda () (/ 1 #f)))
(assert-error  (tn) (lambda () (/ 1 '())))
(assert-error  (tn) (lambda () (/ 1 2 #t)))
(assert-error  (tn) (lambda () (/ 1 2 #f)))
(assert-error  (tn) (lambda () (/ 1 2 '())))
(if (symbol-bound? 'inexact?)
    (assert-fail "/" "no tests for inexact results")
    (begin
      (tn "/ 1 arg")
      (assert-equal? (tn) 0          (/ -2))
      (assert-equal? (tn) -1         (/ -1))
      (assert-error  (tn) (lambda () (/ 0)))
      (assert-equal? (tn) 1          (/ 1))
      (assert-equal? (tn) 0          (/ 2))
      (if (>= fixnum-bits 60)
          (begin
            (tn "/ 1 arg 64-bit")
            (string-eval "(assert-equal? (tn) 0 (/    956397711204))")
            (string-eval "(assert-equal? (tn) 0 (/   -956397711204))")
            (string-eval "(assert-equal? (tn) 0 (/  13121090146595))")
            (string-eval "(assert-equal? (tn) 0 (/ -13121090146595))")))
      (tn "/ 2 args")
      (assert-equal? (tn) 1          (/ -2 -2))
      (assert-equal? (tn) 2          (/ -2 -1))
      (assert-error  (tn) (lambda () (/ -2 0)))
      (assert-equal? (tn) -2         (/ -2 1))
      (assert-equal? (tn) -1         (/ -2 2))
      (assert-equal? (tn) 0          (/ -1 -2))
      (assert-equal? (tn) 1          (/ -1 -1))
      (assert-error  (tn) (lambda () (/ -1 0)))
      (assert-equal? (tn) -1         (/ -1 1))
      (assert-equal? (tn) 0          (/ -1 2))
      (assert-equal? (tn) 0          (/ 0 -2))
      (assert-equal? (tn) 0          (/ 0 -1))
      (assert-error  (tn) (lambda () (/ 0 0)))
      (assert-equal? (tn) 0          (/ 0 1))
      (assert-equal? (tn) 0          (/ 0 2))
      (assert-equal? (tn) 0          (/ 1 -2))
      (assert-equal? (tn) -1         (/ 1 -1))
      (assert-error  (tn) (lambda () (/ 1 0)))
      (assert-equal? (tn) 1          (/ 1 1))
      (assert-equal? (tn) 0          (/ 1 2))
      (assert-equal? (tn) -1         (/ 2 -2))
      (assert-equal? (tn) -2         (/ 2 -1))
      (assert-error  (tn) (lambda () (/ 2 0)))
      (assert-equal? (tn) 2          (/ 2 1))
      (assert-equal? (tn) 1          (/ 2 2))
      (if (>= fixnum-bits 60)
          (begin
            (tn "/ 2 args 64-bit")
            (string-eval "(assert-equal? (tn)  2         (/  13121090146595  5121297354163))")
            (string-eval "(assert-equal? (tn) -2         (/  13121090146595 -5121297354163))")
            (string-eval "(assert-equal? (tn) -2         (/ -13121090146595  5121297354163))")
            (string-eval "(assert-equal? (tn)  2         (/ -13121090146595 -5121297354163))")
            (string-eval "(assert-error  (tn) (lambda () (/  13121090146595 0)))")
            (string-eval "(assert-error  (tn) (lambda () (/ -13121090146595 0)))")
            (string-eval "(assert-equal? (tn) 0          (/  5121297354163  13121090146595))")
            (string-eval "(assert-equal? (tn) 0          (/ -5121297354163  13121090146595))")
            (string-eval "(assert-equal? (tn) 0          (/  5121297354163 -13121090146595))")
            (string-eval "(assert-equal? (tn) 0          (/ -5121297354163 -13121090146595))")
            (string-eval "(assert-equal? (tn) 0          (/     0  13121090146595))")
            (string-eval "(assert-equal? (tn) 0          (/     0 -13121090146595))")))
      (tn "/ 3 args")
      (assert-equal? (tn) 0          (/ -2 -2 -2))
      (assert-equal? (tn) -1         (/ -2 -2 -1))
      (assert-error  (tn) (lambda () (/ -2 -2 0)))
      (assert-equal? (tn) 1          (/ -2 -2 1))
      (assert-equal? (tn) 0          (/ -2 -2 2))
      (assert-equal? (tn) -1         (/ -2 -1 -2))
      (assert-equal? (tn) -2         (/ -2 -1 -1))
      (assert-error  (tn) (lambda () (/ -2 -1 0)))
      (assert-equal? (tn) 2          (/ -2 -1 1))
      (assert-equal? (tn) 1          (/ -2 -1 2))
      (assert-error  (tn) (lambda () (/ -2 0 -2)))
      (assert-error  (tn) (lambda () (/ -2 0 -1)))
      (assert-error  (tn) (lambda () (/ -2 0 0)))
      (assert-error  (tn) (lambda () (/ -2 0 1)))
      (assert-error  (tn) (lambda () (/ -2 0 2)))
      (assert-equal? (tn) 1          (/ -2 1 -2))
      (assert-equal? (tn) 2          (/ -2 1 -1))
      (assert-error  (tn) (lambda () (/ -2 1 0)))
      (assert-equal? (tn) -2         (/ -2 1 1))
      (assert-equal? (tn) -1         (/ -2 1 2))
      (assert-equal? (tn) 0          (/ -2 2 -2))
      (assert-equal? (tn) 1          (/ -2 2 -1))
      (assert-error  (tn) (lambda () (/ -2 2 0)))
      (assert-equal? (tn) -1         (/ -2 2 1))
      (assert-equal? (tn) 0          (/ -2 2 2))
      (assert-equal? (tn) 0          (/ -1 -2 -2))
      (assert-equal? (tn) 0          (/ -1 -2 -1))
      (assert-error  (tn) (lambda () (/ -1 -2 0)))
      (assert-equal? (tn) 0          (/ -1 -2 1))
      (assert-equal? (tn) 0          (/ -1 -2 2))
      (assert-equal? (tn) 0          (/ -1 -1 -2))
      (assert-equal? (tn) -1         (/ -1 -1 -1))
      (assert-error  (tn) (lambda () (/ -1 -1 0)))
      (assert-equal? (tn) 1          (/ -1 -1 1))
      (assert-equal? (tn) 0          (/ -1 -1 2))
      (assert-error  (tn) (lambda () (/ -1 0 -2)))
      (assert-error  (tn) (lambda () (/ -1 0 -1)))
      (assert-error  (tn) (lambda () (/ -1 0 0)))
      (assert-error  (tn) (lambda () (/ -1 0 1)))
      (assert-error  (tn) (lambda () (/ -1 0 2)))
      (assert-equal? (tn) 0          (/ -1 1 -2))
      (assert-equal? (tn) 1          (/ -1 1 -1))
      (assert-error  (tn) (lambda () (/ -1 1 0)))
      (assert-equal? (tn) -1         (/ -1 1 1))
      (assert-equal? (tn) 0          (/ -1 1 2))
      (assert-equal? (tn) 0          (/ -1 2 -2))
      (assert-equal? (tn) 0          (/ -1 2 -1))
      (assert-error  (tn) (lambda () (/ -1 2 0)))
      (assert-equal? (tn) 0          (/ -1 2 1))
      (assert-equal? (tn) 0          (/ -1 2 2))
      (assert-equal? (tn) 0          (/ 0 -2 -2))
      (assert-equal? (tn) 0          (/ 0 -2 -1))
      (assert-error  (tn) (lambda () (/ 0 -2 0)))
      (assert-equal? (tn) 0          (/ 0 -2 1))
      (assert-equal? (tn) 0          (/ 0 -2 2))
      (assert-equal? (tn) 0          (/ 0 -1 -2))
      (assert-equal? (tn) 0          (/ 0 -1 -1))
      (assert-error  (tn) (lambda () (/ 0 -1 0)))
      (assert-equal? (tn) 0          (/ 0 -1 1))
      (assert-equal? (tn) 0          (/ 0 -1 2))
      (assert-error  (tn) (lambda () (/ 0 0 -2)))
      (assert-error  (tn) (lambda () (/ 0 0 -1)))
      (assert-error  (tn) (lambda () (/ 0 0 0)))
      (assert-error  (tn) (lambda () (/ 0 0 1)))
      (assert-error  (tn) (lambda () (/ 0 0 2)))
      (assert-equal? (tn) 0          (/ 0 1 -2))
      (assert-equal? (tn) 0          (/ 0 1 -1))
      (assert-error  (tn) (lambda () (/ 0 1 0)))
      (assert-equal? (tn) 0          (/ 0 1 1))
      (assert-equal? (tn) 0          (/ 0 1 2))
      (assert-equal? (tn) 0          (/ 0 2 -2))
      (assert-equal? (tn) 0          (/ 0 2 -1))
      (assert-error  (tn) (lambda () (/ 0 2 0)))
      (assert-equal? (tn) 0          (/ 0 2 1))
      (assert-equal? (tn) 0          (/ 0 2 2))
      (assert-equal? (tn) 0          (/ 1 -2 -2))
      (assert-equal? (tn) 0          (/ 1 -2 -1))
      (assert-error  (tn) (lambda () (/ 1 -2 0)))
      (assert-equal? (tn) 0          (/ 1 -2 1))
      (assert-equal? (tn) 0          (/ 1 -2 2))
      (assert-equal? (tn) 0          (/ 1 -1 -2))
      (assert-equal? (tn) 1          (/ 1 -1 -1))
      (assert-error  (tn) (lambda () (/ 1 -1 0)))
      (assert-equal? (tn) -1         (/ 1 -1 1))
      (assert-equal? (tn) 0          (/ 1 -1 2))
      (assert-error  (tn) (lambda () (/ 1 0 -2)))
      (assert-error  (tn) (lambda () (/ 1 0 -1)))
      (assert-error  (tn) (lambda () (/ 1 0 0)))
      (assert-error  (tn) (lambda () (/ 1 0 1)))
      (assert-error  (tn) (lambda () (/ 1 0 2)))
      (assert-equal? (tn) 0          (/ 1 1 -2))
      (assert-equal? (tn) -1         (/ 1 1 -1))
      (assert-error  (tn) (lambda () (/ 1 1 0)))
      (assert-equal? (tn) 1          (/ 1 1 1))
      (assert-equal? (tn) 0          (/ 1 1 2))
      (assert-equal? (tn) 0          (/ 1 2 -2))
      (assert-equal? (tn) 0          (/ 1 2 -1))
      (assert-error  (tn) (lambda () (/ 1 2 0)))
      (assert-equal? (tn) 0          (/ 1 2 1))
      (assert-equal? (tn) 0          (/ 1 2 2))
      (assert-equal? (tn) 0          (/ 2 -2 -2))
      (assert-equal? (tn) 1          (/ 2 -2 -1))
      (assert-error  (tn) (lambda () (/ 2 -2 0)))
      (assert-equal? (tn) -1         (/ 2 -2 1))
      (assert-equal? (tn) 0          (/ 2 -2 2))
      (assert-equal? (tn) 1          (/ 2 -1 -2))
      (assert-equal? (tn) 2          (/ 2 -1 -1))
      (assert-error  (tn) (lambda () (/ 2 -1 0)))
      (assert-equal? (tn) -2         (/ 2 -1 1))
      (assert-equal? (tn) -1         (/ 2 -1 2))
      (assert-error  (tn) (lambda () (/ 2 0 -2)))
      (assert-error  (tn) (lambda () (/ 2 0 -1)))
      (assert-error  (tn) (lambda () (/ 2 0 0)))
      (assert-error  (tn) (lambda () (/ 2 0 1)))
      (assert-error  (tn) (lambda () (/ 2 0 2)))
      (assert-equal? (tn) -1         (/ 2 1 -2))
      (assert-equal? (tn) -2         (/ 2 1 -1))
      (assert-error  (tn) (lambda () (/ 2 1 0)))
      (assert-equal? (tn) 2          (/ 2 1 1))
      (assert-equal? (tn) 1          (/ 2 1 2))
      (assert-equal? (tn) 0          (/ 2 2 -2))
      (assert-equal? (tn) -1         (/ 2 2 -1))
      (assert-error  (tn) (lambda () (/ 2 2 0)))
      (assert-equal? (tn) 1          (/ 2 2 1))
      (assert-equal? (tn) 0          (/ 2 2 2))
      (if (>= fixnum-bits 60)
          (begin
            (tn "/ 3 args 64-bit")
            (string-eval "(assert-equal? (tn)  366030355 (/  13121090146595  5121  7))")
            (string-eval "(assert-equal? (tn) -366030355 (/  13121090146595 -5121  7))")
            (string-eval "(assert-equal? (tn) -366030355 (/ -13121090146595  5121  7))")
            (string-eval "(assert-equal? (tn)  366030355 (/ -13121090146595 -5121  7))")
            (string-eval "(assert-equal? (tn) -366030355 (/  13121090146595  5121 -7))")
            (string-eval "(assert-equal? (tn)  366030355 (/  13121090146595 -5121 -7))")
            (string-eval "(assert-equal? (tn)  366030355 (/ -13121090146595  5121 -7))")
            (string-eval "(assert-equal? (tn) -366030355 (/ -13121090146595 -5121 -7))")
            (string-eval "(assert-equal? (tn)          0 (/  5121  13121090146595  7))")
            (string-eval "(assert-equal? (tn)          0 (/ -5121  13121090146595  7))")
            (string-eval "(assert-equal? (tn)          0 (/  5121 -13121090146595  7))")
            (string-eval "(assert-equal? (tn)          0 (/ -5121 -13121090146595  7))")
            (string-eval "(assert-equal? (tn)          0 (/  5121  13121090146595 -7))")
            (string-eval "(assert-equal? (tn)          0 (/ -5121  13121090146595 -7))")
            (string-eval "(assert-equal? (tn)          0 (/  5121 -13121090146595 -7))")
            (string-eval "(assert-equal? (tn)          0 (/ -5121 -13121090146595 -7))")
            (string-eval "(assert-equal? (tn)          0 (/  5121  7  13121090146595))")
            (string-eval "(assert-equal? (tn)          0 (/ -5121  7  13121090146595))")
            (string-eval "(assert-equal? (tn)          0 (/  5121  7 -13121090146595))")
            (string-eval "(assert-equal? (tn)          0 (/ -5121  7 -13121090146595))")
            (string-eval "(assert-equal? (tn)          0 (/  5121 -7  13121090146595))")
            (string-eval "(assert-equal? (tn)          0 (/ -5121 -7  13121090146595))")
            (string-eval "(assert-equal? (tn)          0 (/  5121 -7 -13121090146595))")
            (string-eval "(assert-equal? (tn)          0 (/ -5121 -7 -13121090146595))")
            (string-eval "(assert-equal? (tn)          0 (/ 0 13121090146595 5121))")
            (string-eval "(assert-error  (tn) (lambda () (/ 13121090146595 0 5121)))")
            (string-eval "(assert-error  (tn) (lambda () (/ 13121090146595 5121 0)))")
            (string-eval "(assert-equal? (tn)          0 (/ 0 -13121090146595 5121))")
            (string-eval "(assert-error  (tn) (lambda () (/ -13121090146595 0 5121)))")
            (string-eval "(assert-error  (tn) (lambda () (/ -13121090146595 5121 0)))")))
      (tn "/ many args")
      (assert-equal? (tn) 0 (/ 2 2 2 2))
      (assert-equal? (tn) 0 (/ 0 2 2 2))
      (assert-error  (tn) (lambda () (/ 2 0 2 2)))
      (assert-error  (tn) (lambda () (/ 2 2 0 2)))
      (assert-error  (tn) (lambda () (/ 2 2 2 0)))
      (assert-error  (tn) (lambda () (/ -2 -1 0 1 2)))
      (assert-error  (tn) (lambda () (/ 2 1 0 -1 -2)))
      (assert-error  (tn) (lambda () (/ -2 -1 0 -1 1 2)))
      (assert-error  (tn) (lambda () (/ 2 1 0 -1 1 -2)))
      (assert-error  (tn) (lambda () (/ -2 -2 -1 -1 0 0 1 1 2 2)))
      (assert-error  (tn) (lambda () (/ 2 2 1 1 0 0 -1 -1 -2 -2)))
      (assert-equal? (tn) 121 (/ 2349873 8 23 21 5))))
(tn "/ boundary numbers")
(case fixnum-bits
  ((28)
   (string-eval "(assert-equal? (tn) 1 (/  134217727  134217727))")
   (string-eval "(assert-equal? (tn) 1 (/ -134217728 -134217728))"))
  ((32)
   (string-eval "(assert-equal? (tn) 1 (/  2147483647  2147483647))")
   (string-eval "(assert-equal? (tn) 1 (/ -2147483648 -2147483648))"))
  ((60)
   (string-eval
    "(assert-equal? (tn) 1 (/  576460752303423487  576460752303423487))")
   (string-eval
    "(assert-equal? (tn) 1 (/ -576460752303423488 -576460752303423488))"))
  ((64)
   (string-eval
    "(assert-equal? (tn) 1 (/  9223372036854775807  9223372036854775807))")
   (string-eval
    "(assert-equal? (tn) 1 (/ -9223372036854775808 -9223372036854775808))"))
  (else
   (assert-fail (tn) "unknown int bitwidth")))

(tn "abs invalid forms")
(assert-error  (tn) (lambda () (abs)))
(assert-error  (tn) (lambda () (abs #t)))
(assert-error  (tn) (lambda () (abs #f)))
(assert-error  (tn) (lambda () (abs '())))
(assert-error  (tn) (lambda () (abs #t #t)))
(assert-error  (tn) (lambda () (abs #f #f)))
(assert-error  (tn) (lambda () (abs '() '())))
(assert-error  (tn) (lambda () (abs 1 2)))
(assert-error  (tn) (lambda () (abs 1 #t)))
(assert-error  (tn) (lambda () (abs 1 #f)))
(assert-error  (tn) (lambda () (abs 1 '())))
(tn "abs")
(assert-equal? (tn) 3 (abs -3))
(assert-equal? (tn) 2 (abs -2))
(assert-equal? (tn) 1 (abs -1))
(assert-equal? (tn) 0 (abs 0))
(assert-equal? (tn) 1 (abs 1))
(assert-equal? (tn) 2 (abs 2))
(assert-equal? (tn) 3 (abs 3))
(assert-equal? (tn) 5921370 (abs #x-5a5a5a))
(assert-equal? (tn) 5921370 (abs #x5a5a5a))
(if (>= fixnum-bits 60)
    (begin
      (tn "abs 64-bit")
      (string-eval "(assert-equal? (tn)    956397711204 (abs    956397711204))")
      (string-eval "(assert-equal? (tn)    956397711204 (abs   -956397711204))")
      (string-eval "(assert-equal? (tn)  13121090146595 (abs  13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090146595 (abs -13121090146595))")))
(tn "abs boundary numbers")
(case fixnum-bits
  ((28)
   (string-eval "(assert-equal? (tn) 134217727  (abs  134217727))")
   (string-eval "(assert-equal? (tn) 134217727  (abs -134217727))")
   (string-eval "(assert-error  (tn) (lambda () (abs -134217728)))"))
  ((32)
   (string-eval "(assert-equal? (tn) 2147483647 (abs  2147483647))")
   (string-eval "(assert-equal? (tn) 2147483647 (abs -2147483647))")
   (string-eval "(assert-error  (tn) (lambda () (abs -2147483648)))"))
  ((60)
   (string-eval
    "(assert-equal? (tn) 576460752303423487 (abs  576460752303423487))")
   (string-eval
    "(assert-equal? (tn) 576460752303423487 (abs -576460752303423487))")
   (string-eval
    "(assert-error  (tn) (lambda ()         (abs -576460752303423488)))"))
  ((64)
   (string-eval
    "(assert-equal? (tn) 9223372036854775807 (abs  9223372036854775807))")
   (string-eval
    "(assert-equal? (tn) 9223372036854775807 (abs -9223372036854775807))")
   (string-eval
    "(assert-error  (tn) (lambda ()          (abs -9223372036854775808))))"))
  (else
   (assert-fail (tn) "unknown int bitwidth")))

(tn "quotient invalid forms")
(assert-error  (tn) (lambda () (quotient)))
(assert-error  (tn) (lambda () (quotient #t)))
(assert-error  (tn) (lambda () (quotient #f)))
(assert-error  (tn) (lambda () (quotient '())))
(assert-error  (tn) (lambda () (quotient #t #t)))
(assert-error  (tn) (lambda () (quotient #f #f)))
(assert-error  (tn) (lambda () (quotient '() '())))
(assert-error  (tn) (lambda () (quotient 1)))
(assert-error  (tn) (lambda () (quotient 1 #t)))
(assert-error  (tn) (lambda () (quotient 1 #f)))
(assert-error  (tn) (lambda () (quotient 1 '())))
(assert-error  (tn) (lambda () (quotient 1 2 #t)))
(assert-error  (tn) (lambda () (quotient 1 2 #f)))
(assert-error  (tn) (lambda () (quotient 1 2 '())))
(tn "quotient")
(assert-equal? (tn) 1          (quotient -2 -2))
(assert-equal? (tn) 2          (quotient -2 -1))
(assert-error  (tn) (lambda () (quotient -2 0)))
(assert-equal? (tn) -2         (quotient -2 1))
(assert-equal? (tn) -1         (quotient -2 2))
(assert-equal? (tn) 0          (quotient -1 -2))
(assert-equal? (tn) 1          (quotient -1 -1))
(assert-error  (tn) (lambda () (quotient -1 0)))
(assert-equal? (tn) -1         (quotient -1 1))
(assert-equal? (tn) 0          (quotient -1 2))
(assert-equal? (tn) 0          (quotient 0 -2))
(assert-equal? (tn) 0          (quotient 0 -1))
(assert-error  (tn) (lambda () (quotient 0 0)))
(assert-equal? (tn) 0          (quotient 0 1))
(assert-equal? (tn) 0          (quotient 0 2))
(assert-equal? (tn) 0          (quotient 1 -2))
(assert-equal? (tn) -1         (quotient 1 -1))
(assert-error  (tn) (lambda () (quotient 1 0)))
(assert-equal? (tn) 1          (quotient 1 1))
(assert-equal? (tn) 0          (quotient 1 2))
(assert-equal? (tn) -1         (quotient 2 -2))
(assert-equal? (tn) -2         (quotient 2 -1))
(assert-error  (tn) (lambda () (quotient 2 0)))
(assert-equal? (tn) 2          (quotient 2 1))
(assert-equal? (tn) 1          (quotient 2 2))
(assert-equal? (tn) 17  (quotient 121 7))
(assert-equal? (tn) -17 (quotient 121 -7))
(assert-equal? (tn) -17 (quotient -121 7))
(assert-equal? (tn) 17  (quotient -121 -7))
(if (>= fixnum-bits 60)
    (begin
      (tn "quotient 64-bit")
      (string-eval "(assert-equal? (tn)  2         (quotient  13121090146595  5121297354163))")
      (string-eval "(assert-equal? (tn) -2         (quotient  13121090146595 -5121297354163))")
      (string-eval "(assert-equal? (tn) -2         (quotient -13121090146595  5121297354163))")
      (string-eval "(assert-equal? (tn)  2         (quotient -13121090146595 -5121297354163))")
      (string-eval "(assert-error  (tn) (lambda () (quotient  13121090146595 0)))")
      (string-eval "(assert-error  (tn) (lambda () (quotient -13121090146595 0)))")
      (string-eval "(assert-equal? (tn) 0          (quotient  5121  13121090146595))")
      (string-eval "(assert-equal? (tn) 0          (quotient -5121  13121090146595))")
      (string-eval "(assert-equal? (tn) 0          (quotient  5121 -13121090146595))")
      (string-eval "(assert-equal? (tn) 0          (quotient -5121 -13121090146595))")
      (string-eval "(assert-equal? (tn) 0          (quotient     0  13121090146595))")
      (string-eval "(assert-equal? (tn) 0          (quotient     0 -13121090146595))")))
(tn "quotient boundary numbers")
(case fixnum-bits
  ((28)
   (string-eval "(assert-equal? (tn) 1 (quotient  134217727  134217727))")
   (string-eval "(assert-equal? (tn) 1 (quotient -134217728 -134217728))"))
  ((32)
   (string-eval "(assert-equal? (tn) 1 (quotient  2147483647  2147483647))")
   (string-eval "(assert-equal? (tn) 1 (quotient -2147483648 -2147483648))"))
  ((60)
   (string-eval "(assert-equal? (tn) 1 (quotient  576460752303423487
                                                  576460752303423487))")
   (string-eval "(assert-equal? (tn) 1 (quotient -576460752303423488
                                                 -576460752303423488))"))
  ((64)
   (string-eval "(assert-equal? (tn) 1 (quotient  9223372036854775807
                                                  9223372036854775807))")
   (string-eval "(assert-equal? (tn) 1 (quotient -9223372036854775808
                                                 -9223372036854775808))"))
  (else
   (assert-fail (tn) "unknown int bitwidth")))

(tn "remainder invalid forms")
(assert-error  (tn) (lambda () (remainder)))
(assert-error  (tn) (lambda () (remainder #t)))
(assert-error  (tn) (lambda () (remainder #f)))
(assert-error  (tn) (lambda () (remainder '())))
(assert-error  (tn) (lambda () (remainder #t #t)))
(assert-error  (tn) (lambda () (remainder #f #f)))
(assert-error  (tn) (lambda () (remainder '() '())))
(assert-error  (tn) (lambda () (remainder 1)))
(assert-error  (tn) (lambda () (remainder 1 #t)))
(assert-error  (tn) (lambda () (remainder 1 #f)))
(assert-error  (tn) (lambda () (remainder 1 '())))
(assert-error  (tn) (lambda () (remainder 1 2 #t)))
(assert-error  (tn) (lambda () (remainder 1 2 #f)))
(assert-error  (tn) (lambda () (remainder 1 2 '())))
(tn "remainder")
(assert-equal? (tn) 0          (remainder -2 -2))
(assert-equal? (tn) 0          (remainder -2 -1))
(assert-error  (tn) (lambda () (remainder -2 0)))
(assert-equal? (tn) 0          (remainder -2 1))
(assert-equal? (tn) 0          (remainder -2 2))
(assert-equal? (tn) -1         (remainder -1 -2))
(assert-equal? (tn) 0          (remainder -1 -1))
(assert-error  (tn) (lambda () (remainder -1 0)))
(assert-equal? (tn) 0          (remainder -1 1))
(assert-equal? (tn) -1         (remainder -1 2))
(assert-equal? (tn) 0          (remainder 0 -2))
(assert-equal? (tn) 0          (remainder 0 -1))
(assert-error  (tn) (lambda () (remainder 0 0)))
(assert-equal? (tn) 0          (remainder 0 1))
(assert-equal? (tn) 0          (remainder 0 2))
(assert-equal? (tn) 1          (remainder 1 -2))
(assert-equal? (tn) 0          (remainder 1 -1))
(assert-error  (tn) (lambda () (remainder 1 0)))
(assert-equal? (tn) 0          (remainder 1 1))
(assert-equal? (tn) 1          (remainder 1 2))
(assert-equal? (tn) 0          (remainder 2 -2))
(assert-equal? (tn) 0          (remainder 2 -1))
(assert-error  (tn) (lambda () (remainder 2 0)))
(assert-equal? (tn) 0          (remainder 2 1))
(assert-equal? (tn) 0          (remainder 2 2))
(assert-equal? (tn) 2  (remainder 121 7))
(assert-equal? (tn) 2  (remainder 121 -7))
(assert-equal? (tn) -2 (remainder -121 7))
(assert-equal? (tn) -2 (remainder -121 -7))
(tn "remainder R5RS examples")
(assert-equal? (tn) 1  (remainder 13 4))
(assert-equal? (tn) -1 (remainder -13 4))
(assert-equal? (tn) 1  (remainder 13 -4))
(assert-equal? (tn) -1 (remainder -13 -4))
(if (>= fixnum-bits 60)
    (begin
      (tn "remainder 64-bit")
      (string-eval "(assert-equal? (tn)  668       (remainder  13121090146595  5121))")
      (string-eval "(assert-equal? (tn)  668       (remainder  13121090146595 -5121))")
      (string-eval "(assert-equal? (tn) -668       (remainder -13121090146595  5121))")
      (string-eval "(assert-equal? (tn) -668       (remainder -13121090146595 -5121))")
      (string-eval "(assert-error  (tn) (lambda () (remainder  13121090146595 0)))")
      (string-eval "(assert-error  (tn) (lambda () (remainder -13121090146595 0)))")
      (string-eval "(assert-equal? (tn)  5121      (remainder  5121  13121090146595))")
      (string-eval "(assert-equal? (tn) -5121      (remainder -5121  13121090146595))")
      (string-eval "(assert-equal? (tn)  5121      (remainder  5121 -13121090146595))")
      (string-eval "(assert-equal? (tn) -5121      (remainder -5121 -13121090146595))")
      (string-eval "(assert-equal? (tn) 0          (remainder     0  13121090146595))")
      (string-eval "(assert-equal? (tn) 0          (remainder     0 -13121090146595))")))
(tn "remainder boundary numbers")
(case fixnum-bits
  ((28)
   (string-eval "(assert-equal? (tn) 0 (remainder  134217727  134217727))")
   (string-eval "(assert-equal? (tn) 0 (remainder -134217728 -134217728))"))
  ((32)
   (string-eval "(assert-equal? (tn) 0 (remainder  2147483647  2147483647))")
   (string-eval "(assert-equal? (tn) 0 (remainder -2147483648 -2147483648))"))
  ((60)
   (string-eval "(assert-equal? (tn) 0 (remainder  576460752303423487
                                                   576460752303423487))")
   (string-eval "(assert-equal? (tn) 0 (remainder -576460752303423488
                                                  -576460752303423488))"))
  ((64)
   (string-eval "(assert-equal? (tn) 0 (remainder  9223372036854775807
                                                   9223372036854775807))")
   (string-eval "(assert-equal? (tn) 0 (remainder -9223372036854775808
                                                  -9223372036854775808))"))
  (else
   (assert-fail (tn) "unknown int bitwidth")))

(tn "modulo invalid forms")
(assert-error  (tn) (lambda () (modulo)))
(assert-error  (tn) (lambda () (modulo #t)))
(assert-error  (tn) (lambda () (modulo #f)))
(assert-error  (tn) (lambda () (modulo '())))
(assert-error  (tn) (lambda () (modulo #t #t)))
(assert-error  (tn) (lambda () (modulo #f #f)))
(assert-error  (tn) (lambda () (modulo '() '())))
(assert-error  (tn) (lambda () (modulo 1)))
(assert-error  (tn) (lambda () (modulo 1 #t)))
(assert-error  (tn) (lambda () (modulo 1 #f)))
(assert-error  (tn) (lambda () (modulo 1 '())))
(assert-error  (tn) (lambda () (modulo 1 2 #t)))
(assert-error  (tn) (lambda () (modulo 1 2 #f)))
(assert-error  (tn) (lambda () (modulo 1 2 '())))
(tn "modulo")
(assert-equal? (tn) 0          (modulo -2 -2))
(assert-equal? (tn) 0          (modulo -2 -1))
(assert-error  (tn) (lambda () (modulo -2 0)))
(assert-equal? (tn) 0          (modulo -2 1))
(assert-equal? (tn) 0          (modulo -2 2))
(assert-equal? (tn) -1         (modulo -1 -2))
(assert-equal? (tn) 0          (modulo -1 -1))
(assert-error  (tn) (lambda () (modulo -1 0)))
(assert-equal? (tn) 0          (modulo -1 1))
(assert-equal? (tn) 1          (modulo -1 2))
(assert-equal? (tn) 0          (modulo 0 -2))
(assert-equal? (tn) 0          (modulo 0 -1))
(assert-error  (tn) (lambda () (modulo 0 0)))
(assert-equal? (tn) 0          (modulo 0 1))
(assert-equal? (tn) 0          (modulo 0 2))
(assert-equal? (tn) -1         (modulo 1 -2))
(assert-equal? (tn) 0          (modulo 1 -1))
(assert-error  (tn) (lambda () (modulo 1 0)))
(assert-equal? (tn) 0          (modulo 1 1))
(assert-equal? (tn) 1          (modulo 1 2))
(assert-equal? (tn) 0          (modulo 2 -2))
(assert-equal? (tn) 0          (modulo 2 -1))
(assert-error  (tn) (lambda () (modulo 2 0)))
(assert-equal? (tn) 0          (modulo 2 1))
(assert-equal? (tn) 0          (modulo 2 2))
(assert-equal? (tn) 2  (modulo 121 7))
(assert-equal? (tn) -5 (modulo 121 -7))
(assert-equal? (tn) 5  (modulo -121 7))
(assert-equal? (tn) -2 (modulo -121 -7))
(tn "modulo R5RS examples")
(assert-equal? (tn) 1  (modulo 13 4))
(assert-equal? (tn) 3  (modulo -13 4))
(assert-equal? (tn) -3 (modulo 13 -4))
(assert-equal? (tn) -1 (modulo -13 -4))
(if (>= fixnum-bits 60)
    (begin
      (tn "modulo 64-bit")
      (string-eval "(assert-equal? (tn)             668 (modulo  13121090146595  5121))")
      (string-eval "(assert-equal? (tn)           -4453 (modulo  13121090146595 -5121))")
      (string-eval "(assert-equal? (tn)            4453 (modulo -13121090146595  5121))")
      (string-eval "(assert-equal? (tn)            -668 (modulo -13121090146595 -5121))")
      (string-eval "(assert-error  (tn) (lambda ()      (modulo  13121090146595 0)))")
      (string-eval "(assert-error  (tn) (lambda ()      (modulo -13121090146595 0)))")
      (string-eval "(assert-equal? (tn)            5121 (modulo  5121  13121090146595))")
      (string-eval "(assert-equal? (tn)  13121090141474 (modulo -5121  13121090146595))")
      (string-eval "(assert-equal? (tn) -13121090141474 (modulo  5121 -13121090146595))")
      (string-eval "(assert-equal? (tn)           -5121 (modulo -5121 -13121090146595))")
      (string-eval "(assert-equal? (tn)               0 (modulo     0  13121090146595))")
      (string-eval "(assert-equal? (tn)               0 (modulo     0 -13121090146595))")))
(tn "modulo boundary numbers")
(case fixnum-bits
  ((28)
   (string-eval "(assert-equal? (tn) 0 (modulo  134217727  134217727))")
   (string-eval "(assert-equal? (tn) 0 (modulo -134217728 -134217728))"))
  ((32)
   (string-eval "(assert-equal? (tn) 0 (modulo  2147483647  2147483647))")
   (string-eval "(assert-equal? (tn) 0 (modulo -2147483648 -2147483648))"))
  ((60)
   (string-eval "(assert-equal? (tn) 0 (modulo  576460752303423487
                                                576460752303423487))")
   (string-eval "(assert-equal? (tn) 0 (modulo -576460752303423488
                                               -576460752303423488))"))
  ((64)
   (string-eval "(assert-equal? (tn) 0 (modulo  9223372036854775807
                                                9223372036854775807))")
   (string-eval "(assert-equal? (tn) 0 (modulo -9223372036854775808
                                               -9223372036854775808))"))
  (else
   (assert-fail (tn) "unknown int bitwidth")))

(total-report)
