/*
 *  Part of the shrinkta program, a dvd copy tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include <bonobo.h>
#include <gnome.h>

#include <dvd.h>

#include "support.h"
#include "preferences.h"
#include "preferences-dialog.h"
#include "app.h"
#include "dialogs.h"
#include "pipe.h"
#include "shrink.h"
#include "multiplex.h"
#include "author.h"
#include "image.h"
#include "nautilus-burn.h"

const gchar *DVD_AUDIO_FORMAT[] = {
	"AC3",	/* 5.1ch 448bps */
	"UNKNOWN1",/* ?? */
	"MPEG1",/* 2ch 384bps */
	"MPEG2",/* 7.1ch 912bps */
	"LPCM",	/* 8ch 6144bps*/
	"SDDS",	/* 7.1ch 1280bps [not in use] */
	"DTS"	/* 5.1ch 1536bps */
};

const gchar *DVD_AUDIO_QUANT[] = {
	"16BIT",
	"20BIT",
	"24BIT",
	"DRC",
};

GtkWidget *label_source_name;
GtkWidget *combobox_source_title;
GtkWidget *combobox_source_first_chapter;
GtkWidget *combobox_source_last_chapter;
GtkWidget *combobox_source_audio_track1;
GtkWidget *combobox_source_audio_track2;
GtkWidget *toolbutton_eject;
GtkWidget *toolbutton_read;
GtkWidget *toolbutton_copy;
GtkWidget *toolbutton_burn;
GtkWidget *toolbutton_stop;
GtkWidget *menu_open;
GtkWidget *menu_open_folder;
GtkWidget *menu_eject;
GtkWidget *menu_read;
GtkWidget *menu_copy;
GtkWidget *menu_burn;
GtkWidget *menu_stop;
GtkWidget *menu_prefs;

const gchar *DVD_VIDEO_FILENAMES[] = {
	"video_mpeg1.m2v",
	"video_mpeg2.m2v"
};

const gchar *DVD_AUDIO_FILENAMES[] = {
	"_audio.ac3",
	"_unknown.unknown",
	"_audio_mpeg1.mp2",
	"_audio_mpeg2.mp2",
	"_audio.lpcm",
	"_audio.sdds",
	"_audio.dts"
};

/* ToDo - Fix this! */
typedef struct _RawDvdTracks {
	DvdDisk	 *dvd;
	DvdTitleDisk *title;
	guint8	  first_chapter;
	guint8	  last_chapter;
	
	/* working_directory/"dvd name" */
	gchar	 *path;
	
	/* movie */
	DvdVideo *dvd_video;
	gchar	 *movie_path;
	gint	  movie_file;
	gdouble	  movie_size;
	guint32	  movie_time_stamp;
	
	/* audio1 */
	DvdAudio *dvd_audio1;
	gchar	 *audio1_path;
	gint	  audio1_file;
	gdouble	  audio1_size;
	guint32	  audio1_time_stamp;
	guint64	  audio1_clocks;
	
	/* audio2 */
	DvdAudio *dvd_audio2;
	gchar	 *audio2_path;
	gint	  audio2_file;
	gdouble	  audio2_size;
	guint32	  audio2_time_stamp;
	guint64	  audio2_clocks;
	
	/* audio sync */
	gboolean  audio_sync;
	gboolean  first_video;
	
	/* dvd structure path */
	gchar	 *dvd_dir;
	guint	  dvd_size;
	
	/* image_file */
	gchar	 *dvd_image;
} RawDvdTracks;

RawDvdTracks tracks = {
	NULL, NULL,  0, 0,
	NULL,
	NULL, NULL, -1, 0, 0,
	NULL, NULL, -1, 0, 0, 0,
	NULL, NULL, -1, 0, 0, 0,
	FALSE, TRUE,
	NULL, 0,
	NULL
};

/* read stage */
enum {
	STAGE_BACKUP, /* next stage is STAGE_MAKE_UDF */
	STAGE_READ,
	STAGE_SHRINK,
	STAGE_MULTIPLEX,
	STAGE_AUTHOR,
	STAGE_MAKE_UDF,
	STAGE_COMPLETE
};

GStaticMutex read_stage_mutex = G_STATIC_MUTEX_INIT;
gboolean canned = FALSE;
gboolean full_backup = FALSE;
gint read_stage;

GStaticMutex busy_mutex = G_STATIC_MUTEX_INIT;
guint8 flags;

typedef enum _AppWindowFlag {
	/* sensitivity flags */
	APP_SENSITIVE_READ	= 1 << 0,
	APP_SENSITIVE_BACKUP	= 1 << 1,
	APP_SENSITIVE_EJECT	= 1 << 2,
	APP_SENSITIVE_BURN	= 1 << 3,
	APP_SENSITIVE_STOP	= 1 << 4,
	APP_SENSITIVE_FIT	= 1 << 5,
	APP_SENSITIVE_PREFS	= 1 << 6
} AppWindowFlag;

void		app_window_set_flags	(AppWindowFlag	 app_flags);
void		app_window_unset_flags	(AppWindowFlag	 app_flags);
void		app_window_set_flag	(AppWindowFlag	 app_flag);
void		app_window_unset_flag	(AppWindowFlag	 app_flag);
gboolean	app_window_flag_is_set	(AppWindowFlag	 app_flag);
static gboolean	on_window_delete_event	(GtkWidget	*widget,
					 GdkEvent	*event,
					 gpointer	 user_data);
static void	on_quit_activate	(GtkMenuItem	*menuitem,
					 gpointer	 user_data);
static void	on_preferences_activate	(GtkMenuItem	*menuitem,
					 gpointer	 user_data);
static void	on_about_activate	(GtkMenuItem	*menuitem,
					 gpointer	 user_data);
static void	on_source_first_chapter_changed	(GtkComboBox	*combo,
						 gpointer	 user_data);
static void	on_source_last_chapter_changed	(GtkComboBox	*combo,
						 gpointer	 user_data);
static void	on_source_audio1_changed(GtkComboBox	*combo,
					 gpointer	 user_data);
static void	on_source_audio2_changed(GtkComboBox	*combo,
					 gpointer	 user_data);
static void	on_source_title_changed	(GtkComboBox	*combo,
					 gpointer	 user_data);
static void	on_open_clicked		(GtkWidget	*widget,
					 gpointer	 user_data);
static void	on_open_folder_clicked	(GtkWidget	*widget,
					 gpointer	 user_data);
static void	on_read_clicked		(GtkWidget	*widget,
					 gpointer	 path);
static void	on_eject_clicked	(GtkWidget	*widget,
					 gpointer	 user_data);
static void	output_vob_data		(DvdTitle	*title,
					 DvdStream	 output,
					 gint		 track,
					 gint		 bytes,
					 guint8		*buffer,
					 guint32	 pts,
					 guint64	 frame_clocks,
					 gpointer	 user_data);
static void	output_time		(DvdTitle	*title,
					 const DvdTime	*cell_t,
					 const DvdTime	*chapter_t,
					 const DvdTime	*title_t,
					 gpointer	 user_data);
static void	process_progress_func	(gchar		*msg,
					 gdouble	 progress);
static void	clear_app		(void);
static gboolean	delete_dir		(const gchar	*dir);
static gboolean	delete_file		(const gchar	*file);
static gpointer	read_title		(gpointer	data);
static gpointer	write_disk		(gpointer	data);
static void	on_copy_clicked		(GtkWidget	*widget,
					 gpointer	 user_data);
static void	on_burn_clicked		(GtkWidget	*widget,
					 gpointer	 user_data);
static void	on_stop_clicked		(GtkWidget	*widget,
					 gpointer	 user_data);
static void	on_help_clicked		(GtkWidget	*widget,
					 gpointer	 user_data);

static gboolean
on_window_delete_event	(GtkWidget	*widget,
			 GdkEvent	*event,
			 gpointer	 user_data)
{
	on_quit_activate (NULL, NULL);
	return TRUE;
}


static void
on_quit_activate	(GtkMenuItem	*menuitem,
			 gpointer	 user_data)
{
	if (g_static_mutex_trylock (&busy_mutex) == TRUE) {
		if (tracks.dvd_image != NULL) {
			if (run_shrinkta_dialog (GTK_WINDOW (app), SHRINKTA_DIALOG_TYPE_QUESTION_DELETE_IMAGE) == TRUE) {
				gdk_threads_leave ();
				if (delete_file (tracks.dvd_image) == -1) {
					gdk_threads_enter ();
					run_dialog_warning (GTK_WINDOW (app), _("Unable to delete DVD image"));
				} else {
					gdk_threads_enter ();
				}
			}
		}
		gtk_main_quit ();
	} else {
		run_dialog_warning (GTK_WINDOW (app), _("<b>The program is busy</b>\n\n"
						        "Cancelling operations will be supported soon"));
	}
}

static void
on_preferences_activate	(GtkMenuItem	*menuitem,
			 gpointer	 user_data)
{
	GtkWidget *prefs;
	DvdMedia media;
	gint response;
	
	media = preferences_get_media (preferences);
	prefs = preferences_dialog_new (preferences, GTK_WINDOW (app));
	response = gtk_dialog_run (GTK_DIALOG (prefs));
	while (response == GTK_RESPONSE_HELP) {
		GError *error = NULL;
		PreferencesDialogTab tab;
		const gchar *PREFERENCE_HELP[] = {"shrinkta-prefs-general",
						  "shrinkta-prefs-media",
						  "shrinkta-prefs-audio",
						  "shrinkta-prefs-video"};
		
		g_message ("preferences help clicked");
		
		tab = preferences_dialog_get_tab (PREFERENCES_DIALOG (prefs));
		gnome_help_display ("shrinkta", PREFERENCE_HELP[tab], &error);
		if (error != NULL) {
			g_message ("%s", error->message);
			g_error_free (error);
		}
		response = gtk_dialog_run (GTK_DIALOG (prefs));
	}
	gtk_widget_destroy (prefs);
	if (preferences_get_media (preferences) != media) {
		if (tracks.dvd != NULL) {
			on_read_clicked (NULL, NULL);
		}
	}
}

static void
on_about_activate	(GtkMenuItem	*menuitem,
			 gpointer	 user_data)
{
	show_about (GTK_WINDOW (app));
}

static void
on_source_last_chapter_changed	(GtkComboBox	*combo,
				 gpointer	 user_data)
{
	if (gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) < 0) {
		return;
	}
	tracks.last_chapter = gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) + tracks.first_chapter;
}

static void
on_source_first_chapter_changed	(GtkComboBox	*combo,
				 gpointer	 user_data)
{
	guint8 id;
	guint8 chapters;
	guint32 elapsed_ts = 0;
	guint32 total_ts;
	const DvdTime *tmp_time;
	
	if (gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) < 0) {
		return;
	}
	
	chapters = dvd_title_get_chapters (DVD_TITLE (tracks.title));
	tmp_time = dvd_title_get_time (DVD_TITLE (tracks.title));
	total_ts = tmp_time->ts;
	for (id = tracks.first_chapter - 1;
	     id < chapters;
	     id++) {
		gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_last_chapter), 0);
	}
	tracks.first_chapter = gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) + 1;
	if (tracks.first_chapter > tracks.last_chapter) {
		tracks.last_chapter = tracks.first_chapter;
	}

	for (id = tracks.first_chapter;
	     id <= chapters;
	     id++) {
		gchar *tmp;
		DvdChapter *chapter;
		gint percent;
		
		chapter =  dvd_title_get_chapter (DVD_TITLE (tracks.title), id);
		g_assert (chapter != NULL);
		tmp_time = dvd_chapter_get_time (chapter);
		elapsed_ts += tmp_time->ts;
		g_object_unref (G_OBJECT (chapter));
		percent = (((gdouble) elapsed_ts  / (gdouble) total_ts) * 100.0);
		tmp = g_strdup_printf ("%2.d, Selected %d%%", id, percent);
		gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_source_last_chapter), tmp);
		g_free (tmp);
	}
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_last_chapter), tracks.last_chapter - tracks.first_chapter);
	gtk_widget_set_size_request (combobox_source_last_chapter, -1, -1);
	
}

static void
on_source_audio1_changed(GtkComboBox	*combo,
			 gpointer	 user_data)
{
	gint track_id;
	guint8 audio_tracks;
	
	track_id = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
	if (track_id < 0) {
		return;
	}
	if (tracks.dvd_audio1 != NULL) {
		g_object_unref (G_OBJECT (tracks.dvd_audio1));
	}
	tracks.dvd_audio1 = NULL;
	g_free (tracks.audio1_path);
	tracks.audio1_path = NULL;
	audio_tracks = dvd_title_get_audio_tracks (DVD_TITLE (tracks.title));
	g_message ("audio_tracks = %d", audio_tracks);
	if (audio_tracks == 0) {
		return;
	}
	/* may have "none" */
	if (track_id < audio_tracks) {
		tracks.dvd_audio1 = dvd_title_get_audio (DVD_TITLE (tracks.title), track_id);
		tracks.audio1_path = g_strdup_printf ("%s/track1%s",
						     tracks.path,
					 	    DVD_AUDIO_FILENAMES[dvd_audio_get_format (tracks.dvd_audio1)]);
	}
	gtk_widget_set_size_request (combobox_source_audio_track1, -1, -1);
}

static void
on_source_audio2_changed(GtkComboBox	*combo,
			 gpointer	 user_data)
{
	gint track_id;
	guint8 audio_tracks;
	
	track_id = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
	if (track_id < 0) {
		return;
	}
	if (tracks.dvd_audio2 != NULL) {
		g_object_unref (G_OBJECT (tracks.dvd_audio2));
	}
	tracks.dvd_audio2 = NULL;
	g_free (tracks.audio2_path);
	tracks.audio2_path = NULL;
	
	audio_tracks = dvd_title_get_audio_tracks (DVD_TITLE (tracks.title));
	/* may have "none" */
	if (audio_tracks == 0) {
		return;
	}
	if (track_id < (gint) audio_tracks) {
		tracks.dvd_audio2 = dvd_title_get_audio (DVD_TITLE (tracks.title), track_id);
		tracks.audio2_path = g_strdup_printf ("%s/track2%s",
						     tracks.path,
						     DVD_AUDIO_FILENAMES[dvd_audio_get_format (tracks.dvd_audio2)]);
	}
	gtk_widget_set_size_request (combobox_source_audio_track2, -1, -1);
}

static void
on_source_title_changed	(GtkComboBox	*combo,
			 gpointer	 user_data)
{
	gint title_id;
	guint8 best_id = 0;
	guint8 best_id2 = 0;
	gint id = 0;
	gboolean first_audio_set = FALSE;
	gboolean second_audio_set = FALSE;
	gint audio_tracks;
	guint8 chapters;
	
	title_id = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
	if (title_id < 0) {
		return;
	}
	
	if (tracks.title != NULL) {
		
		chapters = dvd_title_get_chapters (DVD_TITLE (tracks.title));
		g_message ("chapters = %d", chapters);
		for (id = 0;
		     id < chapters;
		     id++) {
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_first_chapter), 0);
		}
		for (id = tracks.first_chapter;
		     id <= chapters;
		     id++) {
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_last_chapter), 0);
		}
		audio_tracks = (gint) dvd_title_get_audio_tracks (DVD_TITLE (tracks.title));
		for (id = 0;
		     id < audio_tracks + 1;
		     id++) {
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_audio_track1), 0);
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_audio_track2), 0);
		}
		g_object_unref (G_OBJECT (tracks.title));
		tracks.title = NULL;
	}

	tracks.title = DVD_TITLE_DISK (dvd_get_title (DVD (tracks.dvd), title_id));
	if (tracks.dvd_video != NULL) {
		g_object_unref (G_OBJECT (tracks.dvd_video));
	}
	tracks.dvd_video = dvd_title_get_video (DVD_TITLE (tracks.title));
	g_free (tracks.movie_path);
	tracks.movie_path = g_strdup_printf ("%s/%s",
					     tracks.path,
					     DVD_VIDEO_FILENAMES[dvd_video_get_format (tracks.dvd_video)]);
	audio_tracks = (gint) dvd_title_get_audio_tracks (DVD_TITLE (tracks.title));
	for (id = 0;
	     id < audio_tracks;
	     id++) {
		DvdAudio *audio;
		DvdLanguage audio_lang;
		DvdAudioPurpose audio_purpose;
		DvdAudioFormat audio_format;
		DvdAudioQuant audio_quant;
		guint8 channels;
		guint8 audio_number;
		gchar *tmp = NULL;
		
		
		audio = dvd_title_get_audio (DVD_TITLE (tracks.title), id);
		audio_purpose = dvd_audio_get_purpose (audio);
		audio_lang = dvd_audio_get_lang (audio);
		audio_format = dvd_audio_get_format (audio);
		audio_quant = dvd_audio_get_quant (audio);
		channels = dvd_audio_get_channels (audio);
		audio_number = dvd_audio_get_number (audio);
		
		switch (audio_purpose) {
		case DVD_AUDIO_PURPOSE_UNDEFINED:
			/* fall through */
		case DVD_AUDIO_PURPOSE_NORMAL:
			tmp = g_strdup_printf ("%s %d channel %s %s",
						DVD_AUDIO_FORMAT[audio_format],
						channels,
						DVD_AUDIO_QUANT[audio_quant],
						dvd_language_get_name (audio_lang));
			if ((audio_lang == preferences_get_language (preferences)) ||
			    ((audio_lang == DVD_LANG_UNDEFINED) &&
			     (preferences_get_use_undefined (preferences) == TRUE))) {
				if ((audio_format == preferences_get_audio_format (preferences)) &&
				    (first_audio_set == FALSE)) {
					best_id = audio_number;
					first_audio_set = TRUE;
				} else if ((audio_format == DVD_AUDIO_FORMAT_DTS) &&
					   (preferences_get_include_dts (preferences) == TRUE)) {
					best_id2 = audio_number;
					second_audio_set = TRUE;
				}
			}
			break;
		case DVD_AUDIO_PURPOSE_IMPAIRED:
			tmp = g_strdup_printf ("%s %d channel %s %s [IMPAIRED]",
						DVD_AUDIO_FORMAT[audio_format],
						channels,
						DVD_AUDIO_QUANT[audio_quant],
						dvd_language_get_name (audio_lang));
			break;
		case DVD_AUDIO_PURPOSE_COMMENTS1:
			/* fall through */
		case DVD_AUDIO_PURPOSE_COMMENTS2:
			tmp = g_strdup_printf ("%s %d channel %s %s [COMMENTS]",
						DVD_AUDIO_FORMAT[audio_format],
						channels,
						DVD_AUDIO_QUANT[audio_quant],
						dvd_language_get_name (audio_lang));
			break;
		case DVD_AUDIO_PURPOSE_UNKNOWN:
			tmp = g_strdup_printf ("%s %d channel %s %s [UNKNOWN]",
						DVD_AUDIO_FORMAT[audio_format],
						channels,
						DVD_AUDIO_QUANT[audio_quant],
						dvd_language_get_name (audio_lang));
			break;
		default:
			g_warning ("unknown audio purpose %d id %d", audio_purpose, id);
			g_assert_not_reached ();
			break;
		}
		g_assert (tmp != NULL);
		gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_source_audio_track1), tmp);
		gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_source_audio_track2), tmp);
		g_free (tmp);
		g_object_unref (G_OBJECT (audio));
	}
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_source_audio_track2), "none");
	gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_source_audio_track1), "none");
	if (first_audio_set == TRUE) {
		gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_audio_track1), best_id);
		if (second_audio_set == TRUE) {
			gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_audio_track2), best_id2);
		} else {
			/* sets to "none" */
			gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_audio_track2), audio_tracks);
		}
	} else {
		gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_audio_track1), 0);
		gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_audio_track2), audio_tracks);
	}
	gtk_widget_set_size_request (combobox_source_audio_track1, -1, -1);
	gtk_widget_set_size_request (combobox_source_audio_track2, -1, -1);
	
	/* chapters */
	chapters = dvd_title_get_chapters (DVD_TITLE (tracks.title));
	tracks.first_chapter = 1;
	tracks.last_chapter = chapters;
	for (id = 0;
	     id < chapters;
	     id++) {
		gchar *tmp;
		
		tmp = g_strdup_printf ("%2.d", id + 1);
		gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_source_first_chapter), tmp);
		g_free (tmp);
	}
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_first_chapter), tracks.first_chapter - 1);
	gtk_widget_set_size_request (combobox_source_first_chapter, -1, -1);
	gtk_widget_set_size_request (combobox_source_audio_track1, -1, -1);
	gtk_widget_set_size_request (combobox_source_audio_track2, -1, -1);
	gtk_widget_set_size_request (app, -1, -1);
}

static void
on_open_clicked		(GtkWidget	*widget,
			 gpointer	 user_data)
{	
	GtkWidget *dialog;

	dialog = gtk_file_chooser_dialog_new ("Dvd Movie Backup - Open ISO file",
					      GTK_WINDOW (app),
					      GTK_FILE_CHOOSER_ACTION_OPEN,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					      NULL);
	
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		gchar *filename;
	
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
		on_read_clicked	(widget, filename);
		g_free (filename);
	}

	gtk_widget_destroy (dialog);
}

static void
on_open_folder_clicked	(GtkWidget	*widget,
			 gpointer	 user_data)
{	
	GtkWidget *dialog;

	dialog = gtk_file_chooser_dialog_new ("Dvd Movie Backup - Open DVD folder",
					      GTK_WINDOW (app),
					      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					      NULL);
	
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		gchar *filename;
	
		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
		on_read_clicked	(widget, filename);
		g_free (filename);
	}

	gtk_widget_destroy (dialog);
}

void
set_drive_path		(const gchar	*cldrive)
{
	gchar *filename;
	
	filename = g_strdup (cldrive);
	on_read_clicked	(app, filename);
	g_free (filename);
}

static void
clear_app		(void)
{
	if (tracks.title != NULL) {
		guint8 title_count;
		guint8 i;
		gint id;
		guint8 chapters;
		gint audio_tracks;
		
		title_count = dvd_get_title_count (DVD (tracks.dvd));
		for (i = 0;
		     i < title_count;
		     i++) {
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_title), 0);
		}
		chapters = dvd_title_get_chapters (DVD_TITLE (tracks.title));
		g_message ("chapters = %d", chapters);
		for (id = 0;
		     id < chapters;
		     id++) {
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_first_chapter), 0);
		}
		for (id = tracks.first_chapter;
		     id <= chapters;
		     id++) {
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_last_chapter), 0);
		}
		g_object_unref (G_OBJECT (tracks.dvd));
		tracks.dvd = NULL;
		gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_title), -1);
		gtk_widget_set_size_request (combobox_source_title, 180, -1);
		/* audio */
		audio_tracks = (gint) dvd_title_get_audio_tracks (DVD_TITLE (tracks.title));
		for (id = 0;
		     id < audio_tracks + 1;
		     id++) {
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_audio_track1), 0);
			gtk_combo_box_remove_text (GTK_COMBO_BOX (combobox_source_audio_track2), 0);
		}
		g_object_unref (G_OBJECT (tracks.title));
		tracks.title = NULL;
		gtk_widget_set_size_request (combobox_source_audio_track1, 180, -1);
		gtk_widget_set_size_request (combobox_source_audio_track2, 180, -1);
	}
	gtk_label_set_text (GTK_LABEL (label_source_name), "");
}
static void
on_read_clicked		(GtkWidget	*widget,
			 gpointer	 path)
{
	guint8 title_count;
	guint8 i;
	guint8 feature_id;
	gboolean read_ok;
	GError *error = NULL;
	gchar *drive_path;
	gchar *tmp_folder;
	
	app_window_unset_flags (APP_SENSITIVE_READ |
				APP_SENSITIVE_BACKUP |
				APP_SENSITIVE_EJECT |
				APP_SENSITIVE_BURN |
				APP_SENSITIVE_STOP |
				APP_SENSITIVE_FIT |
				APP_SENSITIVE_PREFS);
	
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_title), -1);
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_first_chapter), -1);
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_last_chapter), -1);
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_audio_track1), -1);
	gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_audio_track2), -1);
	
	if (tracks.dvd_image != NULL) {
		if (run_shrinkta_dialog (GTK_WINDOW (app), SHRINKTA_DIALOG_TYPE_QUESTION_DELETE_IMAGE) == TRUE) {
			gdk_threads_leave ();
			if (delete_file (tracks.dvd_image) == -1) {
				gdk_threads_enter ();
				run_dialog_warning (GTK_WINDOW (app), _("Unable to delete DVD image"));
			} else {
				gdk_threads_enter ();
			}
		}
		g_free (tracks.dvd_image);
		tracks.dvd_image = NULL;
	}
	clear_app ();
	tracks.dvd = dvd_disk_new ();
	/*no need to leave gdk threads when no signal handlers connected to libdvd objects */
	if (path == NULL) {
		drive_path = preferences_get_drive_path (preferences);
		if (drive_path == NULL) {
			gchar *msg;
		
			g_object_unref (G_OBJECT (tracks.dvd));
			tracks.dvd = NULL;
			g_warning ("No optical drive configured");
			app_window_set_flags (APP_SENSITIVE_READ |
					      APP_SENSITIVE_EJECT |
					      APP_SENSITIVE_PREFS);
			run_dialog_warning (GTK_WINDOW (app), _("<b>Unable to open DVD video disk</b>\n\n"
								"Please configure your optical drive in preferences"));
			return;
		}
		read_ok = dvd_disk_read_properties (tracks.dvd, drive_path, &error);
		g_free (drive_path);
	} else {
		read_ok = dvd_disk_read_properties (tracks.dvd, path, &error);
	}
	if (read_ok == FALSE) {
		gchar *msg;
		
		g_object_unref (G_OBJECT (tracks.dvd));
		tracks.dvd = NULL;
		g_warning ("Unable to read DVD");
		app_window_set_flags (APP_SENSITIVE_READ |
				      APP_SENSITIVE_EJECT |
				      APP_SENSITIVE_PREFS);
		msg = g_strdup_printf (_("<b>Unable to open DVD video disk</b>\n\n%s"), error->message);
		run_dialog_warning (GTK_WINDOW (app), msg);
		g_free (msg);
		g_error_free (error);
		error = NULL;
		return;
	}
	gtk_label_set_text (GTK_LABEL (label_source_name), dvd_get_name (DVD (tracks.dvd)));
	g_free (tracks.path);
	tmp_folder = preferences_get_temp_folder (preferences);
	tracks.path = g_build_filename (tmp_folder, dvd_get_name (DVD (tracks.dvd)), NULL);
	g_free (tmp_folder);
	if ((tracks.dvd->media == DVD_MEDIA_SL) ||
	    (preferences_get_media (preferences) == DVD_MEDIA_DL)) {
		full_backup = run_shrinkta_dialog (GTK_WINDOW (app), SHRINKTA_DIALOG_TYPE_QUESTION_WILL_FIT);
	}
	if (full_backup == TRUE) {
		DvdRegion region;
		gboolean set_region_to_all;
		
		region = dvd_get_region (DVD (tracks.dvd));
		if (region != DVD_REGION_ALL) {
			set_region_to_all = preferences_get_region_free (preferences);
			if (set_region_to_all == TRUE) {
				dvd_set_region (DVD (tracks.dvd), DVD_REGION_ALL);
			}
		}
		
	} else {
		title_count = dvd_get_title_count (DVD (tracks.dvd));
		for (i = 0;
		     i < title_count;
		     i++) {
			const DvdTime *time;
			gchar *item_str;
			DvdTitle *tmptitle;
			gint chapter_count;
			guint8 audios;
			
			tmptitle  = dvd_get_title (DVD (tracks.dvd), i);
			time = dvd_title_get_time (tmptitle);
			chapter_count = dvd_title_get_chapters (tmptitle);
			audios = dvd_title_get_audio_tracks (tmptitle);
			item_str = g_strdup_printf ("Title %2d, %d Chapters, Length %2dh:%02dm:%02ds",
						    i + 1,
						    chapter_count,
						    time->bk_hours,
						    time->bk_minutes,
						    time->bk_seconds);
			gtk_combo_box_append_text (GTK_COMBO_BOX (combobox_source_title), item_str);
			g_object_unref (G_OBJECT (tmptitle));
			g_free (item_str);
		}
		feature_id = dvd_get_feature_title_id (DVD (tracks.dvd));
		/* this calls "on_source_title_changed" and */
		/* sets our title object + adds audio tracks to combos */
		gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_title), feature_id);
		gtk_widget_set_size_request (combobox_source_title, -1, -1);
		app_window_set_flags (APP_SENSITIVE_FIT);
	}
	app_window_set_flags (APP_SENSITIVE_READ |
			      APP_SENSITIVE_BACKUP |
			      APP_SENSITIVE_EJECT |
			      APP_SENSITIVE_PREFS);
	
		
}

static void
on_eject_clicked	(GtkWidget	*widget,
			 gpointer	 user_data)
{
	NautilusBurnDriveMonitor *monitor;
	NautilusBurnDrive *burn_drive;
	gchar *drive_path;
	
	drive_path = preferences_get_drive_path (preferences);
	
	monitor = nautilus_burn_get_drive_monitor ();
	burn_drive = nautilus_burn_drive_monitor_get_drive_for_device (monitor, drive_path);
	g_free (drive_path);
	if (burn_drive != NULL) {
		nautilus_burn_drive_eject (burn_drive);
		g_object_unref (G_OBJECT (burn_drive));
		clear_app ();
		app_window_unset_flags (APP_SENSITIVE_BACKUP);
	}
	g_object_unref (G_OBJECT (monitor));
}

static void
output_vob_data	(DvdTitle	*title,
		 DvdStream	 output,
		 gint		 track,
		 gint		 bytes,
		 guint8		*buffer,
		 guint32	 pts,
		 guint64	 frame_clocks,
		 gpointer	 user_data)
{
	guint64 audio_sync_scr = 0;
	guint32 audio1_start_pts = 0;
	guint32 audio2_start_pts = 0;
	gint written, len;
	if (bytes < 0) {
		return;
	}

	switch (output) {
	case DVD_STREAM_VOB:
		break;
	case DVD_STREAM_MPEG1_VIDEO:
		/* fall_through - only one video track in dvds */
	case DVD_STREAM_MPEG2_VIDEO:
		if (tracks.audio_sync == FALSE) {
			break;
		}
		if (tracks.first_video == TRUE) {
			if (tracks.dvd_audio1 != NULL) {
				/* make sure audio starts first */
				if (pts <= tracks.audio1_time_stamp) {
					break;
				}
			}
			tracks.movie_time_stamp = pts;
			tracks.first_video = FALSE;
			g_message ("video will start at %d pts", tracks.movie_time_stamp);
		}
		written = 0;
		while (written < bytes) {
			len = write (tracks.movie_file, buffer, bytes - written);
			if (len < 0) {
				perror ("write failed");
				break;
			} else {
				written += len;
			}
		}
		break;
	case DVD_STREAM_AC3_AUDIO:
	case DVD_STREAM_MPEG1_AUDIO:
	case DVD_STREAM_MPEG2_AUDIO:
	case DVD_STREAM_LPCM_AUDIO:
	case DVD_STREAM_SDDS_AUDIO:
	case DVD_STREAM_DTS_AUDIO:
		/*g_message ("Audio %d frame length = %u PTS", track, frame_clocks / 300);*/
		if (tracks.audio_sync == FALSE) {
			if ((tracks.dvd_audio1 != NULL) &&
			    (track == dvd_audio_get_number (tracks.dvd_audio1))) {
				tracks.audio1_clocks = frame_clocks;
				tracks.audio1_time_stamp = pts;
				if (tracks.dvd_audio2 == NULL) {
					tracks.audio1_time_stamp += (tracks.audio1_clocks / 300);
					g_message ("audio will start at %d pts", tracks.audio1_time_stamp);
					tracks.audio_sync = TRUE;
					break;
				}
				g_message ("audio 1 %d pts", tracks.audio1_time_stamp);
			} else if ((tracks.dvd_audio2 != NULL) &&
				   (track == dvd_audio_get_number (tracks.dvd_audio2))) {
				tracks.audio2_clocks = frame_clocks;
				tracks.audio2_time_stamp = pts;
				g_message ("audio 2 %d pts", tracks.audio2_time_stamp);
			}
			if ((tracks.audio1_clocks > 0) && (tracks.audio2_clocks > 0)) {
				
				if (tracks.audio2_clocks > tracks.audio1_clocks) {
					switch (tracks.audio2_clocks) {
					case 2304000:
						audio_sync_scr = 15 * 2304000;
						break;
					case 1152000:
						audio_sync_scr = 15 * 1152000;
						break;
					case 576000:
						audio_sync_scr = 15 * 576000;
						break;
					case 288000:
						audio_sync_scr = 15 * 288000;
						break;
					case 144000:
						audio_sync_scr = 30 * 144000;
						break;
					case 864000:
						/* AC3 */
						if (tracks.audio1_clocks == 576000) {
							audio_sync_scr = 10 * 864000;
						} else {
							audio_sync_scr = 5 * 864000;
						}
						break;
					default:
						g_assert_not_reached ();
					}
				} else if (tracks.audio1_clocks > tracks.audio2_clocks) {
					switch (tracks.audio1_clocks) {
					case 2304000:
						audio_sync_scr = 15 * 2304000;
						break;
					case 1152000:
						audio_sync_scr = 15 * 1152000;
						break;
					case 576000:
						audio_sync_scr = 15 * 576000;
						break;
					case 288000:
						audio_sync_scr = 15 * 288000;
						break;
					case 144000:
						audio_sync_scr = 30 * 144000;
						break;
					case 864000:
						/* AC3 */
						if (tracks.audio2_clocks == 576000) {
							audio_sync_scr = 10 * 864000;
						} else {
							audio_sync_scr = 5 * 864000;
						}
						break;
					default:
						g_assert_not_reached ();
					}
				} else {
					/* same PCM samples in both audio streams */
					audio_sync_scr = tracks.audio2_clocks;
				}
				g_message ("audio sync occurs every %d pts", audio_sync_scr / 300);
				g_message ("audio1 frames per sync %d", audio_sync_scr / tracks.audio1_clocks);
				g_message ("audio2 frames per sync %d", audio_sync_scr / tracks.audio2_clocks);
				audio1_start_pts = tracks.audio1_time_stamp - ((tracks.audio1_clocks / 300) * (tracks.audio1_time_stamp / (tracks.audio1_clocks / 300)));
				audio2_start_pts = tracks.audio2_time_stamp - ((tracks.audio2_clocks / 300) * (tracks.audio2_time_stamp / (tracks.audio2_clocks / 300)));
				g_message ("audio1 %d pts %d clocks", tracks.audio1_time_stamp, tracks.audio1_clocks / 300);
				g_message ("audio2 %d pts %d clocks", tracks.audio2_time_stamp, tracks.audio2_clocks / 300);
				if (audio1_start_pts != audio2_start_pts) {
					while ((audio1_start_pts < tracks.audio1_time_stamp) &&
					       (audio2_start_pts < tracks.audio2_time_stamp)) {
						if (audio1_start_pts < audio2_start_pts) {
							audio1_start_pts += (tracks.audio1_clocks / 300);
						} else {
							audio2_start_pts += (tracks.audio2_clocks / 300);
						}
						if (audio1_start_pts == audio2_start_pts) {
							break;
						}
					}
					if (audio1_start_pts != audio2_start_pts) {
						gdk_threads_enter ();
						run_dialog_warning (GTK_WINDOW (app), _("<b>Unable to syncronize two audio streams</b>\n\n"
											"Dropping audio 2 and continuing"));
						/* setting combo box to "none" will invoke callback and null audio2 */
						gtk_combo_box_set_active (GTK_COMBO_BOX (combobox_source_audio_track2),
									  dvd_title_get_audio_tracks (DVD_TITLE (tracks.title)));
						g_assert (tracks.dvd_audio2 == NULL);
						gdk_threads_leave ();
						tracks.audio1_time_stamp += (tracks.audio1_clocks / 300);
						g_message ("audio will start at %d pts", tracks.audio1_time_stamp);
						tracks.audio_sync = TRUE;
						break;
					} else {
						g_message ("audio tracks started at %d pts + a multiple of %d", audio1_start_pts, audio_sync_scr / 300);
					}
				} else {
					g_message ("audio tracks started at %d pts + a multiple of %d", audio1_start_pts, audio_sync_scr / 300);
				}
				/* next audio packet pts */
				tracks.audio1_time_stamp += (tracks.audio1_clocks / 300);
				tracks.audio2_time_stamp += (tracks.audio2_clocks / 300);
				if (tracks.audio1_time_stamp > tracks.audio2_time_stamp) {
					tracks.audio1_time_stamp -= audio1_start_pts;
					tracks.audio1_time_stamp = ((tracks.audio1_time_stamp / (audio_sync_scr / 300)) * (audio_sync_scr / 300)) + audio1_start_pts + (audio_sync_scr / 300);
					tracks.audio2_time_stamp = tracks.audio1_time_stamp;
					g_message ("audio will start at %d pts", tracks.audio1_time_stamp);
				} else {
					tracks.audio2_time_stamp -= audio2_start_pts;
					tracks.audio2_time_stamp = ((tracks.audio2_time_stamp / (audio_sync_scr / 300)) * (audio_sync_scr / 300)) + audio2_start_pts + (audio_sync_scr / 300);
					tracks.audio1_time_stamp = tracks.audio2_time_stamp;
					g_message ("audio will start at %d pts", tracks.audio1_time_stamp);
				}
				tracks.audio_sync = TRUE;
			}
		} else {
			if ((tracks.dvd_audio1 != NULL) &&
			    (track == dvd_audio_get_number (tracks.dvd_audio1))) {
				if (pts < tracks.audio1_time_stamp) {
					break;
				}
				written = 0;
				while (written < bytes) {
					len = write (tracks.audio1_file, &buffer[written], bytes - written);
					if (len < 0) {
						perror ("write failed");
						break;
					} else {
						written += len;
					}
				}
			} else if ((tracks.dvd_audio2 != NULL) &&
				   (track == dvd_audio_get_number (tracks.dvd_audio2))) {
				if (pts < tracks.audio2_time_stamp) {
					break;
				}
				written = 0;
				while (written < bytes) {
					len = write (tracks.audio2_file, &buffer[written], bytes - written);
					if (len < 0) {
						perror ("write failed");
						break;
					} else {
						written += len;
					}
				}
			}
		}
		break;
	default:
		g_assert_not_reached ();
		break;
	}
	
}

static void
backup_progress		(DvdDisk	*disk,
			 guint8		 percent,
			 gpointer	 user_data)
{
	gdouble done;
	
	done = (gdouble) percent / 100.0;
	
	gdk_threads_enter ();
	gnome_appbar_set_progress_percentage (GNOME_APPBAR (appbar), done);
	gdk_threads_leave ();
}

static void
output_time		(DvdTitle	*title,
			 const DvdTime	*cell_t,
			 const DvdTime	*chapter_t,
			 const DvdTime	*title_t,
			 gpointer	 user_data)
{
	gdouble done;
	const DvdTime *title_time;
	
	title_time = dvd_title_get_time (title);
	done = (gdouble) (title_t->ts / 9000) / (gdouble) (title_time->ts / 9000);
	
	gdk_threads_enter ();
	gnome_appbar_set_progress_percentage (GNOME_APPBAR (appbar), done);
	gdk_threads_leave ();
}

static void
process_progress_func		(gchar		*msg,
				 gdouble	 progress)
{
	gdk_threads_enter ();
	if (msg != NULL) {
		gnome_appbar_set_default (GNOME_APPBAR (appbar), msg);
	}
	if (progress > 1.0) {
		progress = 1.0;
	}
	gnome_appbar_set_progress_percentage (GNOME_APPBAR (appbar), progress);
	gdk_threads_leave ();
}

/* do not run in gdk threads */
static gboolean
delete_dir			(const gchar	*path)
{
	if (g_file_test (path, G_FILE_TEST_IS_DIR) == TRUE) {
		GDir *dir;
		const gchar *name;
		
		g_message ("following %s", path);
		dir = g_dir_open (path, 0, NULL);
		if (dir == NULL) {
			g_warning ("Can't open %s", path);
			return FALSE;
		}
		name = g_dir_read_name (dir);
		while (name != NULL) {
			gchar *npath;
			
			npath = g_build_filename (path, name, NULL);
			if (delete_dir (npath) == FALSE) {
				g_free (npath);
				g_dir_close (dir);
				return FALSE;
			}
			g_free (npath);
			name = g_dir_read_name (dir);
		}
		g_message ("deleting %s", path);
		if (rmdir (path) < 0) {
			g_warning ("Unable to delete directory %s", path);
			perror ("delete_dir");
			return FALSE;
		}
		g_dir_close (dir);
	} else if (g_file_test (path, G_FILE_TEST_IS_REGULAR) == TRUE) {
		if (delete_file (path) != TRUE) {
			g_warning ("Unable to delete regular file %s", path);
			perror ("delete_dir");
			return FALSE;
		}
		return TRUE;
	}
	return TRUE;
}

/* do not run in gdk threads */
static gboolean
delete_file			(const gchar	*file)
{
	struct stat filestat;
	guint32 bytes;
	gdouble progress = 0.0;
	gchar *msg;
	gchar *file_name;
		
	file_name = g_path_get_basename (file);
	msg = g_strdup_printf (_("Deleting file %s"), file_name);
	g_free (file_name);
	gdk_threads_enter ();
	gnome_appbar_set_default (GNOME_APPBAR (appbar), msg);
	gnome_appbar_set_progress_percentage (GNOME_APPBAR (appbar), 0.0);
	gdk_threads_leave ();
	if (stat (file, &filestat) == -1) {
		g_warning ("Delete failed - can't stat file %s", file);
		return FALSE;
	}
	bytes = filestat.st_size;
	/* only do progress for files > 10k */
	if (bytes > 0x2800) {
		guint32 fsize;
		guint32 amount;
		gint fd;
		
		fd = open (file, O_WRONLY, 0);
		if (fd < 0) {
			g_warning ("Delete failed - can't open file %s", file);
			return FALSE;
		}
		amount = bytes / 100;
		for (fsize = bytes - amount;
		     fsize > amount;
		     fsize -= amount) {
			if (ftruncate (fd, fsize) == -1) {
				close (fd);
				g_warning ("Delete failed - can't truncate file %s", file);
				return FALSE;
			}
			progress += 0.01;
			gdk_threads_enter ();
			gnome_appbar_set_progress_percentage (GNOME_APPBAR (appbar), progress);
			gdk_threads_leave ();
		}
		fd = close (fd);
		if (fd == -1) {
			g_warning ("Delete failed - can't close truncated file %s", file);
			return FALSE;
		}
	}
	if (unlink (file) < 0) {
		g_warning ("Unable to delete regular file %s", file);
		perror ("delete_file");
		return FALSE;
	}
	gdk_threads_enter ();
	gnome_appbar_set_progress_percentage (GNOME_APPBAR (appbar), 1.0);
	gnome_appbar_set_default (GNOME_APPBAR (appbar), _("File deleted"));
	gdk_threads_leave ();
	return TRUE;
}

static gpointer
read_title			(gpointer	data)
{
	gchar *msg = NULL;
	gchar *movie_file = NULL;
	GError *error = NULL;
	gchar *folder;
	
	const gchar *STAGE_STATUS_MSG[] = {
		N_("Read title failed!"),
		N_("Backup title failed!"),
		N_("Shrink movie failed!"),
		N_("Combine tracks failed!"),
		N_("Author DVD failed!"),
		N_("Create DVD file system failed!"),
		N_("Title copy complete")
	};
	
	if (tracks.dvd_image != NULL) {
		gdk_threads_enter ();
		if (run_shrinkta_dialog (GTK_WINDOW (app), SHRINKTA_DIALOG_TYPE_QUESTION_DELETE_IMAGE) == TRUE) {
			gdk_threads_leave ();
			if (delete_file (tracks.dvd_image) == -1) {
				gdk_threads_enter ();
				run_dialog_warning (GTK_WINDOW (app), _("Unable to delete DVD image"));
				gdk_threads_leave ();
			}
		} else {
			gdk_threads_leave ();
		}
		g_free (tracks.dvd_image);
		tracks.dvd_image = NULL;
	}
	if (full_backup == TRUE) {
		read_stage = STAGE_BACKUP;
	} else {
		read_stage = STAGE_READ;
	}
	/*read_stage = STAGE_READ;*/
	/* this loop is the only place that read_stage is changed */
	g_static_mutex_lock (&read_stage_mutex);
	for (;
	     read_stage < STAGE_COMPLETE;
	     read_stage++) {
		gboolean ok = TRUE;
		gulong data_signal;
		gulong time_signal;
		struct stat filestat;
		gfloat ratio;
		gchar *tmp_path;
		gchar *temp_folder;
		
		
		switch (read_stage) {
		case STAGE_BACKUP:
			g_static_mutex_unlock (&read_stage_mutex);
			temp_folder = preferences_get_temp_folder (preferences);
			rmdir (tracks.dvd_dir);
			time_signal = g_signal_connect (tracks.dvd,
							"backup_progress",
							G_CALLBACK (backup_progress),
							NULL);
			gnome_appbar_set_default (GNOME_APPBAR (appbar), "Backing up entire disk");
			tmp_path = dvd_disk_backup (tracks.dvd, temp_folder, &error);
			g_signal_handler_disconnect (tracks.dvd, time_signal);
			g_free (temp_folder);
			if (error != NULL) {
				msg = g_strdup_printf (_("<b>Unable to read DVD video disk</b>\n\n%s"), error->message);
				g_error_free (error);
				error = NULL;
				ok = FALSE;
				if (tmp_path != NULL) {
					delete_dir (tmp_path);
					g_free (tmp_path);
					tmp_path = NULL;
				}
				break;
			}
			
			g_free (tracks.dvd_dir);
			tracks.dvd_dir = tmp_path;
			tracks.dvd_size = tracks.dvd->blocks / 512;
			read_stage = STAGE_AUTHOR;
			/* loop then adds 1 to stage */
			break;
		case STAGE_READ:
			g_static_mutex_unlock (&read_stage_mutex);
			tracks.audio_sync = FALSE;
			tracks.movie_time_stamp = 0;
			tracks.audio1_time_stamp = 0;
			tracks.audio2_time_stamp = 0;
			tracks.first_video = TRUE;
			tracks.audio1_clocks = 0;
			tracks.audio2_clocks = 0;

			if (mkdir (tracks.path, 0774) == -1) {
				msg = g_strdup_printf (_("<b>Unable to create DVD temporary folder</b>\n\n"
							 "Please check you have the needed permissions to read and write\n"
							 "files in your \"Temp Folder\" selected in preferences"));
		
				ok = FALSE;
				break;
			}
			g_assert (tracks.movie_path != NULL);
			tracks.movie_file = open (tracks.movie_path, O_WRONLY | O_CREAT | O_TRUNC, 0664);
			if (tracks.movie_file < 0) {
				msg = g_strdup_printf (_("<b>Unable to open movie file for writing</b>\n\n"
						       "Please check you have the needed permissions to create\n"
						       "files in your \"Temp Folder\" selected in preferences"));
				ok = FALSE;
				break;
			}
			if (tracks.dvd_audio1 != NULL) {
				tracks.audio1_file = open (tracks.audio1_path, O_WRONLY | O_CREAT | O_TRUNC, 0664);
				if (tracks.audio1_file < 0) {
					msg = g_strdup_printf (_("<b>Unable to open first audio file for writing</b>\n\n"
							       "Please check you have the needed permissions to create\n"
							       "files in your \"Temp Folder\" selected in preferences"));
					ok = FALSE;
					break;
				}
			}
			if (tracks.dvd_audio2 != NULL) {
				tracks.audio2_file = open (tracks.audio2_path, O_WRONLY | O_CREAT | O_TRUNC, 0664);
				if (tracks.audio2_file < 0) {
					msg = g_strdup_printf (_("<b>Unable to open second audio file for writing</b>\n\n"
							       "Please check you have the needed permissions to create\n"
							       "files in your \"Temp Folder\" selected in preferences"));
					ok = FALSE;
					break;
				}
			}
			data_signal = g_signal_connect (tracks.title,
							"output_chapter",
							G_CALLBACK (output_vob_data),
							NULL);
			time_signal = g_signal_connect (tracks.title,
							"title_time",
							G_CALLBACK (output_time),
							NULL);
			
			msg = g_strdup_printf (_("Reading title %d"), dvd_title_get_id (DVD_TITLE (tracks.title)) + 1);
			gdk_threads_enter ();
			gnome_appbar_set_default (GNOME_APPBAR (appbar), msg);
			gdk_threads_leave ();
			g_free (msg);
			msg = NULL;
			ok = dvd_title_disk_read_chapters (tracks.title,
							   tracks.first_chapter,
							   tracks.last_chapter,
							   &error);
			gdk_threads_enter ();
			app_window_unset_flags (APP_SENSITIVE_STOP);
			gdk_threads_leave ();
			close (tracks.movie_file);
			tracks.movie_file = -1;
			if (tracks.audio1_file > 0) {
				close (tracks.audio1_file);
				tracks.audio1_file = -1;
			}
			if (tracks.audio2_file > 0) {
				close (tracks.audio2_file);
				tracks.audio2_file = -1;
			}
			g_signal_handler_disconnect (tracks.title, data_signal);
			g_signal_handler_disconnect (tracks.title, time_signal);
			if (ok == FALSE) {
				msg = g_strdup_printf (_("<b>Unable to read DVD video disk</b>\n\n%s"), error->message);
				g_error_free (error);
				error = NULL;
				break;
			}
			if (stat (tracks.movie_path, &filestat) == -1) {
				msg = g_strdup_printf (_("<b>Unable to determine movie file size</b>\n\n"
						       "Please check you have the needed permissions to read\n"
						       "files in your \"Temp Folder\" selected in preferences"));
				ok = FALSE;
				break;
			}
			tracks.movie_size = filestat.st_size / (gdouble) 0x100000;
			if (tracks.audio1_path != NULL) {
				if (stat (tracks.audio1_path, &filestat) == -1) {
					msg = g_strdup_printf (_("<b>Unable to determine first audio file size</b>\n\n"
							       "Please check you have the needed permissions to read\n"
							       "files in your \"Temp Folder\" selected in preferences"));
					ok = FALSE;
					break;
				}
				tracks.audio1_size = filestat.st_size / (gdouble) 0x100000;
			}
			if (tracks.audio2_path != NULL) {
				if (stat (tracks.audio2_path, &filestat) == -1) {
					msg = g_strdup_printf (_("<b>Unable to determine second audio file size</b>\n\n"
							       "Please check you have the needed permissions to read\n"
							       "files in your \"Temp Folder\" selected in preferences"));
					ok = FALSE;
					break;
				}
				tracks.audio2_size = filestat.st_size / (gdouble) 0x100000;
			}
			break;
		case STAGE_SHRINK:
			g_static_mutex_unlock (&read_stage_mutex);
			ratio = (gfloat) (tracks.movie_size / (4450.0 - tracks.audio1_size - tracks.audio2_size)) * 1.04;
			if (ratio > 1) {
				if (ratio > preferences_get_maximum_shrink (preferences)) {
					if (tracks.audio2_path != NULL) {
						g_warning ("deleting raw audio 2 file - shrink factor too much");
						if (delete_file (tracks.audio2_path) == FALSE) {
							msg = g_strdup_printf (_("<b>Unable to delete second audio file</b>\n\n"
									       "Please check you have the needed permissions to read and write\n"
									       "files in your \"Temp Folder\" selected in preferences"));
							ok = FALSE;
							break;
						}
						g_free (tracks.audio2_path);
						tracks.audio2_path = NULL;
						tracks.audio2_size = 0;
						ratio = (gfloat) (tracks.movie_size / (4450.0 - tracks.audio1_size)) * 1.04;
					}
					if (ratio > preferences_get_maximum_shrink (preferences)) {
						gboolean canit;
						
						msg = g_strdup_printf (_("<b>Shrink factor of %f exeeds the limit\n"
								       "set in preferences.</b>\n\n"
								       "Do you wish to continue?"), ratio);
						gdk_threads_enter ();
						canit = run_dialog_question (GTK_WINDOW (app), msg);
						gdk_threads_leave ();
						g_free (msg);
						msg = NULL;
						if (canit == FALSE) {
							ok = FALSE;
							break;
						}
					}
				}
				if (shrink_video (tracks.movie_path, (guint) tracks.movie_size, ratio, &process_progress_func) == FALSE) {
					msg = g_strdup_printf (_("<b>Unable to shrink movie file</b>\n\n"
							       "Please check you have the transcode package correctly\n"
							       "installed on your system"));
					ok = FALSE;
					break;
					g_static_mutex_unlock (&busy_mutex);
					g_thread_exit (NULL);
				}
			}
			/* get new video file size */
			if (stat (tracks.movie_path, &filestat) == -1) {
				msg = g_strdup_printf (_("<b>Unable to determine movie file size</b>\n\n"
						       "Please check you have the needed permissions to read and write\n"
						       "files in your \"Temp Folder\" selected in preferences"));
				ok = FALSE;
				break;
			}
			tracks.movie_size = filestat.st_size / (gdouble) 0x100000;
			break;
		case STAGE_MULTIPLEX:
			g_static_mutex_unlock (&read_stage_mutex);
			movie_file = g_strdup_printf ("%s/movie.mpg", tracks.path);
			if (tracks.audio2_path != NULL) {
				gint32 video_sync;
				
				if (tracks.audio1_time_stamp != tracks.audio2_time_stamp) {
					g_warning ("Audio track sync error %d vs %d", tracks.audio1_time_stamp, tracks.audio2_time_stamp);
				}
				
				video_sync = (gint32) tracks.movie_time_stamp - (gint32) tracks.audio1_time_stamp;
				g_message ("video sync %d", video_sync);
				if (multiplx_streams (movie_file,
						      tracks.movie_path,
						      video_sync,
						      tracks.audio1_path,
						      tracks.audio2_path,
						      tracks.movie_size + tracks.audio1_size + tracks.audio2_size,
						      &process_progress_func) == FALSE) {
					g_free (movie_file);
					msg = g_strdup_printf (_("<b>Unable to combine video and audio tracks</b>\n\n"
							       "Please check you have the mjpegtools package correctly\n"
							       "installed on your system"));
					ok = FALSE;
					break;
				}
			} else {
				gint32 video_sync;
				
				video_sync = (gint32) tracks.movie_time_stamp - (gint32) tracks.audio1_time_stamp;
				g_message ("video sync %d", video_sync);
				if (multiplx_streams (movie_file,
						      tracks.movie_path,
						      video_sync,
						      tracks.audio1_path,
						      tracks.audio2_path,
						      tracks.movie_size + tracks.audio1_size + tracks.audio2_size,
						      &process_progress_func) == FALSE) {
					g_free (movie_file);
					msg = g_strdup_printf (_("<b>Unable to combine video and audio tracks</b>\n\n"
							       "Please check you have the mjpegtools package correctly\n"
							       "installed on your system"));
					ok = FALSE;
					break;
				}
			}
			gdk_threads_enter ();
			gnome_appbar_set_default (GNOME_APPBAR (appbar), _("Deleting track files..."));
			gdk_threads_leave ();
			g_message ("deleting raw video file");
			if (delete_file (tracks.movie_path) != TRUE) {
				g_warning ("Unable to remove raw video file");
				ok = FALSE;
			}
			g_message ("deleting raw audio 1 file");
			if (delete_file (tracks.audio1_path) != TRUE) {
				g_warning ("Unable to remove raw audio 1 file");
			}
			if (tracks.audio2_path != NULL) {
				g_message ("deleting raw audio 2 file");
				if (delete_file (tracks.audio2_path) != TRUE) {
					g_warning ("Unable to remove raw audio 2 file");
				}
			}
			tracks.movie_size = tracks.audio1_size = tracks.audio2_size = 0;
			if (ok == FALSE) {
				g_free (movie_file);
				msg = g_strdup_printf (_("<b>Unable to delete video or audio tracks</b>\n\n"
						       "Please check you have the needed permissions to read and write\n"
						       "files in your \"Temp Folder\" selected in preferences"));
			}
			break;
		case STAGE_AUTHOR:
			g_static_mutex_unlock (&read_stage_mutex);
			if (stat (movie_file, &filestat) == -1) {
				g_free (movie_file);
				msg = g_strdup_printf (_("<b>Unable to determine movie file size</b>\n\n"
						       "Please check you have the needed permissions to read and write\n"
						       "files in your \"Temp Folder\" selected in preferences"));
				ok = FALSE;
				break;
			}
			tracks.dvd_size = filestat.st_size / 0x100000;
			tracks.dvd_dir = g_strdup_printf ("%s/dvd/", tracks.path);
			
			if (author_dvd (tracks.dvd_dir,
					movie_file,
					DVD_TITLE (tracks.title),
					tracks.dvd_audio1,
					tracks.dvd_audio2,
					tracks.first_chapter,
					tracks.last_chapter,
					tracks.dvd_size,
					&process_progress_func) == FALSE) {
				g_free (movie_file);
				msg = g_strdup_printf (_("<b>Unable to author DVD</b>\n\n"
						       "Please check you have the dvdauthor package correctly\n"
						       "installed on your system"));
				ok = FALSE;
				break;
			}
			/* need to update tracks.dvd_size here */
			/* it is used as progress when making disk image */
			/* for now just add 4Mb */
			tracks.dvd_size += 4;
			
			/* remove old mpg file */
			gdk_threads_enter ();
			gnome_appbar_set_default (GNOME_APPBAR (appbar), _("Removing files..."));
			gdk_threads_leave ();
			if (delete_file (movie_file) != TRUE) {
				msg = g_strdup_printf (_("<b>Unable to delete movie file</b>\n\n"
						       "Please check you have the needed permissions to read and write\n"
						       "files in your \"Temp Folder\" selected in preferences"));
				ok = FALSE;
			}
			g_free (movie_file);
			break;
		case STAGE_MAKE_UDF:
			g_static_mutex_unlock (&read_stage_mutex);
			gdk_threads_enter ();
			gnome_appbar_set_default (GNOME_APPBAR (appbar), _("Writing disk image"));
			gdk_threads_leave ();
			
			folder = preferences_get_image_folder (preferences);	
			tracks.dvd_image = g_strdup_printf ("%s/%s_%d-%d.iso",
							    folder,
							    dvd_get_name (DVD (tracks.dvd)),
							    tracks.first_chapter,
							    tracks.last_chapter);
			g_free (folder);
			if (g_file_test (tracks.dvd_image, G_FILE_TEST_EXISTS) == TRUE) {
				gdk_threads_enter ();
				if (run_dialog_question (GTK_WINDOW (app),
							 _("<b>The DVD disk image already exists. Do you wish to overwrite this file?</b>\n\n"
							   "Select <b><i>Yes</i></b> to delete the image, or,\n"
							   "<b><i>No</i></b>to cancel writing the image to disk.")) == TRUE) {
					gdk_threads_leave ();
					if (delete_file (tracks.dvd_image) == -1) {
						gdk_threads_enter ();
						run_dialog_warning (GTK_WINDOW (app), _("Unable to delete DVD image"));
						gdk_threads_leave ();
						msg = g_strdup_printf (_("<b>Unable to delete existing DVD image file</b>\n\n"
						     			 "Please remove the file manually or change the image\n"
						 			 "file folder in preferences"));
						ok = FALSE;
					}
				} else {
					gdk_threads_leave ();
					msg = g_strdup_printf (_("<b>Unable to master DVD image file</b>\n\n"
					    			 "The file already exists"));
					ok = FALSE;
				}
				
			}
			if (ok == FALSE) {
				g_free (tracks.dvd_image);
				tracks.dvd_image = NULL;
				break;
			}
			ok = make_udf_filesystem (tracks.dvd_dir,
					 	  tracks.dvd_size,
					 	  dvd_get_name (DVD (tracks.dvd)),
					 	  (const gchar *) dvd_get_volume_set (DVD (tracks.dvd)),
					 	  dvd_get_provider (DVD (tracks.dvd)),
					 	  tracks.dvd_image,
						  &process_progress_func);
			if (ok == FALSE) {
				g_free (tracks.dvd_image);
				tracks.dvd_image = NULL;
				msg = g_strdup_printf (_("<b>Unable to create DVD file system</b>\n\n"
						       "Please check you have the genisoimage package correctly\n"
						       "installed on your system"));
			}
			break;
		case STAGE_COMPLETE:
		default:
			g_assert_not_reached ();
			break;
		}
		g_static_mutex_lock (&read_stage_mutex);
		if (ok == FALSE) {
			break;
		}
		
	} /* end for [stage] */
	g_static_mutex_unlock (&read_stage_mutex);
	
	if (tracks.path != NULL) {
		gdk_threads_enter ();
		gnome_appbar_set_default (GNOME_APPBAR (appbar), _("Removing files..."));
		gdk_threads_leave ();
		delete_dir (tracks.path);
	}
	g_static_mutex_lock (&read_stage_mutex);
	gdk_threads_enter ();
	if (canned == TRUE) {
		run_dialog_warning (GTK_WINDOW (app), _("The operation has been cancelled"));
		gnome_appbar_set_default (GNOME_APPBAR (appbar), _("Cancelled by user"));
		g_free (msg);
		app_window_set_flags (APP_SENSITIVE_BACKUP);
		canned = FALSE;
	} else {
		gnome_appbar_set_default (GNOME_APPBAR (appbar), gettext (STAGE_STATUS_MSG[read_stage]));
		if (read_stage == STAGE_COMPLETE) {
			app_window_set_flags (APP_SENSITIVE_BURN);
		} else {
			if (msg != NULL) {
				run_dialog_warning (GTK_WINDOW (app), msg);
				g_free (msg);
			}
		}
		app_window_set_flags (APP_SENSITIVE_BACKUP);
	}
	gnome_appbar_set_progress_percentage (GNOME_APPBAR (appbar), 0.0);
	if (tracks.title != NULL) {
		app_window_set_flags (APP_SENSITIVE_FIT);
	}
	app_window_unset_flags (APP_SENSITIVE_STOP);
	app_window_set_flags (APP_SENSITIVE_READ |
			      APP_SENSITIVE_PREFS |
			      APP_SENSITIVE_EJECT);
	
	gdk_threads_leave ();
	g_static_mutex_unlock (&read_stage_mutex);
	g_static_mutex_unlock (&busy_mutex);
	g_thread_exit (NULL);
	return NULL;
}

static gpointer
write_disk			(gpointer	data)
{
	gchar *tmpargs;
	gchar *burn_command;
	
	burn_command = preferences_get_burn_command (preferences);
	tmpargs = g_strdup_printf ("%s\"%s\"", burn_command, tracks.dvd_image);
	g_free (burn_command);
	g_message ("burn command = %s", tmpargs);
	if (g_spawn_command_line_sync (tmpargs, NULL, NULL, NULL, NULL) == FALSE) {
		gdk_threads_enter ();
		run_dialog_warning (GTK_WINDOW (app), _("<b>Unable to start nautilus cd burner</b>\n\n"
						      "Please make sure the \"Nautilus CD Burner\" package\n"
						      "is correctly installed on your system"));
		gdk_threads_leave ();
	}
	g_free (tmpargs);
	gdk_threads_enter ();
	app_window_set_flags (APP_SENSITIVE_READ |
			      APP_SENSITIVE_BACKUP |
			      APP_SENSITIVE_EJECT |
			      APP_SENSITIVE_BURN |
			      APP_SENSITIVE_PREFS);
	if (tracks.title != NULL) {
		app_window_set_flags (APP_SENSITIVE_FIT);
	}
	gdk_threads_leave ();
	g_static_mutex_unlock (&busy_mutex);
	g_thread_exit (NULL);
	return NULL;
}

static void
on_copy_clicked		(GtkWidget	*widget,
			 gpointer	 user_data)
{
	g_static_mutex_lock (&busy_mutex);
	switch (preferences_get_media (preferences)) {
	case DVD_MEDIA_DL:
		break;
	case DVD_MEDIA_SL:
		if (tracks.dvd->media != DVD_MEDIA_SL) {
			if (tracks.dvd_audio1 == NULL) {
				run_dialog_warning (GTK_WINDOW (app), _("<b>The selected title has no audio!</b>\n\n"
								        "Please select a title with audio"));
				g_static_mutex_unlock (&busy_mutex);
				return;
			}
		}
		break;
	default:
		break;
	}
	
	
	if ((tracks.dvd_audio1 != NULL) &&
	    (tracks.dvd_audio2 != NULL) &&
	    (dvd_audio_get_number (tracks.dvd_audio1) == dvd_audio_get_number (tracks.dvd_audio2))) {
		run_dialog_warning (GTK_WINDOW (app), _("<b>Audio tracks are the same!</b>\n\n"
						      "Please select two different audio tracks or\n"
						      "select \"none\" for the second audio track"));
		g_static_mutex_unlock (&busy_mutex);
		return;
	}
	if ((tracks.path != NULL) &&
	    (g_file_test (tracks.path, G_FILE_TEST_EXISTS)) == TRUE) {
		if (run_dialog_question (GTK_WINDOW (app),
			 _("<b>Delete existing temporary folder?</b>\n\n"
			 "This might be the result of the application previously crashing.\n\n"
			 "Select <b><i>Yes</i></b> to delete the folder and all it's contents, or,\n"
			 "<b><i>No</i></b> to cancel the disk copy.")) == TRUE) {
			gdk_threads_leave ();
			if (delete_dir (tracks.path) == FALSE) {
				gdk_threads_enter ();
				run_dialog_warning (GTK_WINDOW (app), _("<b>Unable to delete old DVD temporary folder</b>\n\n"
								      "Please check you have the needed permissions to read and write\n"
								      "files in your \"Temp Folder\" selected in preferences"));
				g_static_mutex_unlock (&busy_mutex);
				return;
			}
			gdk_threads_enter ();
		} else {
			g_static_mutex_unlock (&busy_mutex);
			return;
		}
	}
	app_window_unset_flags (APP_SENSITIVE_READ |
				APP_SENSITIVE_BACKUP |
				APP_SENSITIVE_EJECT |
				APP_SENSITIVE_BURN |
				APP_SENSITIVE_FIT);
	app_window_set_flags (APP_SENSITIVE_STOP |
			      APP_SENSITIVE_PREFS);
	
	g_thread_create (read_title, NULL, FALSE, NULL);
}

static void
on_burn_clicked		(GtkWidget	*widget,
			 gpointer	 user_data)
{
	g_static_mutex_lock (&busy_mutex);
	if (tracks.dvd_video == NULL) {
		g_static_mutex_unlock (&busy_mutex);
		return;
	}
	app_window_unset_flags (APP_SENSITIVE_READ |
				APP_SENSITIVE_BACKUP |
				APP_SENSITIVE_EJECT |
				APP_SENSITIVE_BURN |
				APP_SENSITIVE_STOP |
				APP_SENSITIVE_FIT |
				APP_SENSITIVE_PREFS);
	g_thread_create (write_disk, NULL, FALSE, NULL);
}

static void
on_stop_clicked		(GtkWidget	*widget,
			 gpointer	 user_data)
{
	
	g_static_mutex_lock (&read_stage_mutex);
	switch (read_stage) {
	case STAGE_BACKUP:
		app_window_unset_flags (APP_SENSITIVE_STOP);
		/* never call libdvd functions in gdk threads when */
		/* signals are attatched to any libdvd object */
		canned = TRUE;
		gdk_threads_leave ();
		dvd_disk_stop_backup (tracks.dvd);
		gdk_threads_enter ();
		g_static_mutex_unlock (&read_stage_mutex);
		break;
	case STAGE_READ:
		app_window_unset_flags (APP_SENSITIVE_STOP);
		/* never call libdvd functions in gdk threads when */
		/* signals are attatched to any libdvd object */
		canned = TRUE;
		gdk_threads_leave ();
		dvd_title_disk_stop_read (tracks.title);
		gdk_threads_enter ();
		g_static_mutex_unlock (&read_stage_mutex);
		break;
	case STAGE_SHRINK:
	case STAGE_MULTIPLEX:
	case STAGE_AUTHOR:
	case STAGE_MAKE_UDF:
	default:
		g_static_mutex_unlock (&read_stage_mutex);
		break;
	}
}

static void
on_help_clicked		(GtkWidget	*widget,
			 gpointer	 user_data)
{
	GError *error = NULL;
	g_message ("help clicked");
	gnome_help_display ("shrinkta", "shrinkta-usage", &error);
	if (error != NULL) {
		g_message ("%s", error->message);
		g_error_free (error);
	}
}

void
create_app		(void)
{
	GdkPixbuf *app_icon_pixbuf;
	GtkWidget *bonobodock;
	GtkWidget *toolbar;
	GtkIconSize tmp_toolbar_icon_size;
	GtkWidget *tmp_image;
	GtkWidget *separatortoolitem;
	GtkWidget *table_read;
	GtkWidget *label_fc;
	GtkWidget *label_lc;
	GtkWidget *label_dn;
	GtkWidget *label_dt;
	GtkWidget *label_at1;
	GtkWidget *label_at2;
	GtkWidget *hbox;
	GtkTooltips *tooltips;
	flags = 0xff;
	GnomeUIInfo file_menu_uiinfo[] = {
		{
			GNOME_APP_UI_ITEM, N_("_Open Disk Image"),
			N_("Open a DVD video ISO file"),
			(gpointer) on_open_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GTK_STOCK_OPEN,
			0, (GdkModifierType) 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("_Open DVD Folder"),
			N_("Open a folder with a DVD disk structure"),
			(gpointer) on_open_folder_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GTK_STOCK_DIRECTORY,
			0, (GdkModifierType) 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("R_ead DVD Disk"),
			N_("Read DVD disk information"),
			(gpointer) on_read_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "shrinkta/read-cd.png",
			0, (GdkModifierType) 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Ej_ect DVD Disk"),
			N_("Eject DVD disk from your drive"),
			(gpointer) on_eject_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "shrinkta/eject-cd.png",
			0, (GdkModifierType) 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Back_up DVD"),
			N_("Backup DVD to ISO image"),
			(gpointer) on_copy_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "shrinkta/copy-cd.png",
			0, (GdkModifierType) 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("_Stop"),
			N_("Stop current process"),
			(gpointer) on_stop_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GTK_STOCK_STOP,
			0, (GdkModifierType) 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("_Burn DVD"),
			N_("Burn to blank DVD disk"),
			(gpointer) on_burn_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "shrinkta/burn-cd.png",
			0, (GdkModifierType) 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_EXIT_ITEM (on_quit_activate, NULL),
		GNOMEUIINFO_END
	};

	GnomeUIInfo edit_menu_uiinfo[] = {
		GNOMEUIINFO_MENU_PREFERENCES_ITEM (on_preferences_activate, NULL),
		GNOMEUIINFO_END
	};

	GnomeUIInfo help_menu_uiinfo[] = {
		{
			GNOME_APP_UI_ITEM, N_("_Help"),
			N_("Get help"),
			(gpointer) on_help_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, "gtk-help",
			GDK_F1, (GdkModifierType) 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_ABOUT_ITEM (on_about_activate, NULL),
		GNOMEUIINFO_END
	};

	GnomeUIInfo menubar_uiinfo[] = {
		GNOMEUIINFO_MENU_FILE_TREE (file_menu_uiinfo),
		GNOMEUIINFO_MENU_EDIT_TREE (edit_menu_uiinfo),
		GNOMEUIINFO_MENU_HELP_TREE (help_menu_uiinfo),
		GNOMEUIINFO_END
	};
	
	tooltips = gtk_tooltips_new ();

	app = gnome_app_new ("shrinkta", _("Dvd Movie Backup"));
	gtk_widget_set_size_request (app, 315, -1);
	gtk_window_set_resizable (GTK_WINDOW (app), FALSE);
	app_icon_pixbuf = gdk_pixbuf_new_from_file ("/usr/share/pixmaps/shrinkta/shrinkta-icon.png", NULL);
	if (app_icon_pixbuf) {
		gtk_window_set_icon (GTK_WINDOW (app), app_icon_pixbuf);
		gdk_pixbuf_unref (app_icon_pixbuf);
	}

	bonobodock = GNOME_APP (app)->dock;
	gtk_widget_show (bonobodock);

	gnome_app_create_menus (GNOME_APP (app), menubar_uiinfo);
	menu_prefs = edit_menu_uiinfo[0].widget;
	menu_open = file_menu_uiinfo[0].widget;
	menu_open_folder = file_menu_uiinfo[1].widget;
	
	menu_read = file_menu_uiinfo[3].widget;
	menu_eject = file_menu_uiinfo[4].widget;
	menu_copy = file_menu_uiinfo[6].widget;
	menu_stop = file_menu_uiinfo[7].widget;
	menu_burn = file_menu_uiinfo[9].widget;

	toolbar = gtk_toolbar_new ();
	gtk_toolbar_set_show_arrow (GTK_TOOLBAR (toolbar), FALSE);
	gtk_widget_show (toolbar);
	gnome_app_add_toolbar (GNOME_APP (app), GTK_TOOLBAR (toolbar), "toolbar",
				BONOBO_DOCK_ITEM_BEH_EXCLUSIVE
				| BONOBO_DOCK_ITEM_BEH_LOCKED,
				BONOBO_DOCK_TOP, 1, 0, 0);
	tmp_toolbar_icon_size = gtk_toolbar_get_icon_size (GTK_TOOLBAR (toolbar));
	
	tmp_image = gtk_image_new_from_file ("/usr/share/pixmaps/shrinkta/read-cd-tb.png");
	gtk_widget_show (tmp_image);
	toolbutton_read = (GtkWidget*) gtk_tool_button_new (tmp_image, _("Read"));
	gtk_widget_show (toolbutton_read);
	gtk_container_add (GTK_CONTAINER (toolbar), toolbutton_read);
	gtk_tool_item_set_tooltip (GTK_TOOL_ITEM (toolbutton_read), tooltips, _("Read DVD disk"), NULL);
	gtk_tool_item_set_is_important (GTK_TOOL_ITEM (toolbutton_read), TRUE);

	tmp_image = gtk_image_new_from_file ("/usr/share/pixmaps/shrinkta/eject-cd-tb.png");
	gtk_widget_show (tmp_image);
	toolbutton_eject = (GtkWidget*) gtk_tool_button_new (tmp_image, _("Eject"));
	gtk_widget_show (toolbutton_eject);
	gtk_container_add (GTK_CONTAINER (toolbar), toolbutton_eject);
	gtk_tool_item_set_tooltip (GTK_TOOL_ITEM (toolbutton_eject), tooltips, _("Eject disk from your drive"), NULL);
	gtk_tool_item_set_is_important (GTK_TOOL_ITEM (toolbutton_eject), FALSE);
	
	separatortoolitem = (GtkWidget*) gtk_separator_tool_item_new ();
	gtk_widget_show (separatortoolitem);
	gtk_container_add (GTK_CONTAINER (toolbar), separatortoolitem);
	
	tmp_image = gtk_image_new_from_file ("/usr/share/pixmaps/shrinkta/copy-cd-tb.png");
	gtk_widget_show (tmp_image);
	toolbutton_copy = (GtkWidget*) gtk_tool_button_new (tmp_image, _("Backup"));
	gtk_widget_show (toolbutton_copy);
	gtk_container_add (GTK_CONTAINER (toolbar), toolbutton_copy);
	gtk_tool_item_set_tooltip (GTK_TOOL_ITEM (toolbutton_copy), tooltips, _("Backup DVD to ISO image"), NULL);
	gtk_tool_item_set_is_important (GTK_TOOL_ITEM (toolbutton_copy), TRUE);

	toolbutton_stop = (GtkWidget*) gtk_tool_button_new_from_stock (GTK_STOCK_STOP);
	gtk_widget_show (toolbutton_stop);
	gtk_container_add (GTK_CONTAINER (toolbar), toolbutton_stop);
	gtk_tool_item_set_tooltip (GTK_TOOL_ITEM (toolbutton_stop), tooltips, _("Stop current process"), NULL);
	gtk_tool_item_set_is_important (GTK_TOOL_ITEM (toolbutton_stop), FALSE);
	
	separatortoolitem = (GtkWidget*) gtk_separator_tool_item_new ();
	gtk_widget_show (separatortoolitem);
	gtk_container_add (GTK_CONTAINER (toolbar), separatortoolitem);

	tmp_image = gtk_image_new_from_file ("/usr/share/pixmaps/shrinkta/burn-cd-tb.png");
	gtk_widget_show (tmp_image);
	toolbutton_burn = (GtkWidget*) gtk_tool_button_new (tmp_image, _("Burn"));
	gtk_widget_show (toolbutton_burn);
	gtk_container_add (GTK_CONTAINER (toolbar), toolbutton_burn);
	gtk_tool_item_set_tooltip (GTK_TOOL_ITEM (toolbutton_burn), tooltips, _("Burn DVD to disk"), NULL);
	gtk_tool_item_set_is_important (GTK_TOOL_ITEM (toolbutton_burn), TRUE);

	table_read = gtk_table_new (6, 2, FALSE);
	gtk_widget_show (table_read);
	gnome_app_set_contents (GNOME_APP (app), table_read);
	gtk_container_set_border_width (GTK_CONTAINER (table_read), 12);
	gtk_table_set_row_spacings (GTK_TABLE (table_read), 6);
	gtk_table_set_col_spacings (GTK_TABLE (table_read), 12);

	label_dn = gtk_label_new (_("Disk Name"));
	gtk_widget_show (label_dn);
	gtk_table_attach (GTK_TABLE (table_read), label_dn, 0, 1, 0, 1,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label_dn), 0, 0.5);
	
	label_source_name = gtk_label_new ("");
	gtk_widget_show (label_source_name);
	gtk_table_attach (GTK_TABLE (table_read), label_source_name, 1, 2, 0, 1,
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label_source_name), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (label_source_name), 0, 5);
	
	label_dt = gtk_label_new (_("Disk Title"));
	gtk_widget_show (label_dt);
	gtk_table_attach (GTK_TABLE (table_read), label_dt, 0, 1, 1, 2,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label_dt), 0, 0.5);

	label_fc = gtk_label_new (_("First Chapter"));
	gtk_widget_show (label_fc);
	gtk_table_attach (GTK_TABLE (table_read), label_fc, 0, 1, 2, 3,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label_fc), 0, 0.5);

	label_lc = gtk_label_new (_("Last Chapter"));
	gtk_widget_show (label_lc);
	gtk_table_attach (GTK_TABLE (table_read), label_lc, 0, 1, 3, 4,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label_lc), 0, 0.5);

	label_at1 = gtk_label_new (_("Audio Track 1"));
	gtk_widget_show (label_at1);
	gtk_table_attach (GTK_TABLE (table_read), label_at1, 0, 1, 4, 5,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label_at1), 0, 0.5);

	label_at2 = gtk_label_new (_("Audio Track 2"));
	gtk_widget_show (label_at2);
	gtk_table_attach (GTK_TABLE (table_read), label_at2, 0, 1, 5, 6,
			  (GtkAttachOptions) (GTK_FILL),
			  (GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (label_at2), 0, 0.5);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox);
	gtk_table_attach (GTK_TABLE (table_read), hbox, 1, 2, 1, 2,
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);

	combobox_source_title = gtk_combo_box_new_text ();
	gtk_widget_show (combobox_source_title);
	gtk_box_pack_start (GTK_BOX (hbox), combobox_source_title, FALSE, TRUE, 0);
	gtk_widget_set_size_request (combobox_source_title, 120, -1);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox);
	gtk_table_attach (GTK_TABLE (table_read), hbox, 1, 2, 2, 3,
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);

	combobox_source_first_chapter = gtk_combo_box_new_text ();
	gtk_widget_show (combobox_source_first_chapter);
	gtk_box_pack_start (GTK_BOX (hbox), combobox_source_first_chapter, FALSE, TRUE, 0);
	gtk_widget_set_size_request (combobox_source_first_chapter, 80, -1);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox);
	gtk_table_attach (GTK_TABLE (table_read), hbox, 1, 2, 3, 4,
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);

	combobox_source_last_chapter = gtk_combo_box_new_text ();
	gtk_widget_show (combobox_source_last_chapter);
	gtk_box_pack_start (GTK_BOX (hbox), combobox_source_last_chapter, FALSE, TRUE, 0);
	gtk_widget_set_size_request (combobox_source_last_chapter, 100, -1);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox);
	gtk_table_attach (GTK_TABLE (table_read), hbox, 1, 2, 4, 5,
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);

	combobox_source_audio_track1 = gtk_combo_box_new_text ();
	gtk_widget_show (combobox_source_audio_track1);
	gtk_box_pack_start (GTK_BOX (hbox), combobox_source_audio_track1, FALSE, FALSE, 0);
	gtk_widget_set_size_request (combobox_source_audio_track1, 100, -1);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox);
	gtk_table_attach (GTK_TABLE (table_read), hbox, 1, 2, 5, 6,
			  (GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			  (GtkAttachOptions) (GTK_FILL), 0, 0);

	combobox_source_audio_track2 = gtk_combo_box_new_text ();
	gtk_widget_show (combobox_source_audio_track2);
	gtk_box_pack_start (GTK_BOX (hbox), combobox_source_audio_track2, FALSE, FALSE, 0);
	gtk_widget_set_size_request (combobox_source_audio_track2, 100, -1);

	appbar = gnome_appbar_new (TRUE, TRUE, GNOME_PREFERENCES_NEVER);
	gtk_widget_show (appbar);
	gnome_app_set_statusbar (GNOME_APP (app), appbar);
 	gnome_app_install_menu_hints (GNOME_APP (app), menubar_uiinfo);
	g_signal_connect ((gpointer) app, "delete_event",
			  G_CALLBACK (on_window_delete_event),
			  NULL);
	g_signal_connect ((gpointer) toolbutton_read, "clicked",
			  G_CALLBACK (on_read_clicked),
			  NULL);
	g_signal_connect ((gpointer) toolbutton_eject, "clicked",
			  G_CALLBACK (on_eject_clicked),
			  NULL);
	g_signal_connect ((gpointer) toolbutton_copy, "clicked",
			  G_CALLBACK (on_copy_clicked),
			  NULL);
        g_signal_connect ((gpointer) toolbutton_stop, "clicked",
			  G_CALLBACK (on_stop_clicked),
			  NULL);
        g_signal_connect ((gpointer) toolbutton_burn, "clicked",
			  G_CALLBACK (on_burn_clicked),
			  NULL);
	g_signal_connect ((gpointer) combobox_source_title, "changed",
			  G_CALLBACK (on_source_title_changed),
			  NULL);
	g_signal_connect ((gpointer) combobox_source_first_chapter, "changed",
			  G_CALLBACK (on_source_first_chapter_changed),
			  NULL);
        g_signal_connect ((gpointer) combobox_source_last_chapter, "changed",
			  G_CALLBACK (on_source_last_chapter_changed),
			  NULL);
        g_signal_connect ((gpointer) combobox_source_audio_track1, "changed",
			  G_CALLBACK (on_source_audio1_changed),
			  NULL);
        g_signal_connect ((gpointer) combobox_source_audio_track2, "changed",
			  G_CALLBACK (on_source_audio2_changed),
			  NULL);
	app_window_unset_flags (APP_SENSITIVE_BACKUP |
				APP_SENSITIVE_BURN |
				APP_SENSITIVE_STOP |
				APP_SENSITIVE_FIT);
	app_window_set_flags (APP_SENSITIVE_READ |
			      APP_SENSITIVE_EJECT |
			      APP_SENSITIVE_PREFS);
	gtk_widget_show (app);
}

#define APP_SET_FLAG(flag)	(flags |= (flag))
#define APP_UNSET_FLAG(flag)	(flags &= ~(flag))
#define APP_FLAG_IS_SET(flag)	((flags & (flag)) != 0)

void
app_window_set_flags		(AppWindowFlag	 app_flags)
{
	guint8 f;
	
	for (f = 0;
	     f < 16;
	     f++) {
		AppWindowFlag flag;
		
		flag = (1 << f);
		if (APP_FLAG_IS_SET (flag) == FALSE) {
			if ((app_flags & flag) != 0) {
				app_window_set_flag (flag);
			}
		}
	}
}

void
app_window_unset_flags		(AppWindowFlag	 app_flags)
{
	guint8 f;
	
	for (f = 0;
	     f < 8;
	     f++) {
		AppWindowFlag flag;
		
		flag = (1 << f);
		if (APP_FLAG_IS_SET (flag) == TRUE) {
			if ((app_flags & flag) != 0) {
				app_window_unset_flag (flag);
			}
		}
	}
}

void
app_window_set_flag		(AppWindowFlag	 app_flag)
{
	guint8 f;
	
	
	for (f = 0;
	     f < 8;
	     f++) {
		AppWindowFlag flag;
		
		flag = (1 << f);
		if ((APP_FLAG_IS_SET (flag) == FALSE) &&
		    ((app_flag & flag) != 0)) {
			APP_SET_FLAG (flag);
			switch (flag) {
			case APP_SENSITIVE_READ:
				gtk_widget_set_sensitive (toolbutton_read, TRUE);
				gtk_widget_set_sensitive (menu_read, TRUE);
				gtk_widget_set_sensitive (menu_open, TRUE);
				gtk_widget_set_sensitive (menu_open_folder, TRUE);
				break;
			case APP_SENSITIVE_BACKUP:
				gtk_widget_set_sensitive (toolbutton_copy, TRUE);
				gtk_widget_set_sensitive (menu_copy, TRUE);
				break;
			case APP_SENSITIVE_EJECT:
				gtk_widget_set_sensitive (toolbutton_eject, TRUE);
				gtk_widget_set_sensitive (menu_eject, TRUE);
				break;
			case APP_SENSITIVE_BURN:
				gtk_widget_set_sensitive (toolbutton_burn, TRUE);
				gtk_widget_set_sensitive (menu_burn, TRUE);
				break;
			case APP_SENSITIVE_STOP:
				gtk_widget_set_sensitive (toolbutton_stop, TRUE);
				gtk_widget_set_sensitive (menu_stop, TRUE);
				break;
			case APP_SENSITIVE_FIT:
				gtk_widget_set_sensitive (combobox_source_title, TRUE);
				gtk_widget_set_sensitive (combobox_source_first_chapter, TRUE);
				gtk_widget_set_sensitive (combobox_source_last_chapter, TRUE);
				gtk_widget_set_sensitive (combobox_source_audio_track1, TRUE);
				gtk_widget_set_sensitive (combobox_source_audio_track2, TRUE);
				break;
			case APP_SENSITIVE_PREFS:
				gtk_widget_set_sensitive (menu_prefs, TRUE);
				break;
			default:
				g_assert_not_reached ();
				break;
			} /* end switch */
		} /* end if */
	} /* end for */
}

void
app_window_unset_flag		(AppWindowFlag	 app_flag)
{
	guint8 f;
	
	for (f = 0;
	     f < 8;
	     f++) {
		AppWindowFlag flag;
		
		flag = 1 << f;
		if ((APP_FLAG_IS_SET (flag) == TRUE) &&
		    ((app_flag & flag) != 0)) {
			APP_UNSET_FLAG (flag);
			switch (flag) {
			case APP_SENSITIVE_READ:
				gtk_widget_set_sensitive (toolbutton_read, FALSE);
				gtk_widget_set_sensitive (menu_read, FALSE);
				gtk_widget_set_sensitive (menu_open, FALSE);
				gtk_widget_set_sensitive (menu_open_folder, FALSE);
				break;
			case APP_SENSITIVE_BACKUP:
				gtk_widget_set_sensitive (toolbutton_copy, FALSE);
				gtk_widget_set_sensitive (menu_copy, FALSE);
				break;
			case APP_SENSITIVE_EJECT:
				gtk_widget_set_sensitive (toolbutton_eject, FALSE);
				gtk_widget_set_sensitive (menu_eject, FALSE);
				break;
			case APP_SENSITIVE_BURN:
				gtk_widget_set_sensitive (toolbutton_burn, FALSE);
				gtk_widget_set_sensitive (menu_burn, FALSE);
				break;
			case APP_SENSITIVE_STOP:
				gtk_widget_set_sensitive (toolbutton_stop, FALSE);
				gtk_widget_set_sensitive (menu_stop, FALSE);
				break;
			case APP_SENSITIVE_FIT:
				gtk_widget_set_sensitive (combobox_source_title, FALSE);
				gtk_widget_set_sensitive (combobox_source_first_chapter, FALSE);
				gtk_widget_set_sensitive (combobox_source_last_chapter, FALSE);
				gtk_widget_set_sensitive (combobox_source_audio_track1, FALSE);
				gtk_widget_set_sensitive (combobox_source_audio_track2, FALSE);
				break;
			case APP_SENSITIVE_PREFS:
				gtk_widget_set_sensitive (menu_prefs, FALSE);
				break;
			default:
				g_assert_not_reached ();
				break;
			} /* end switch */
		} /* end if */
	} /* end for */
}

gboolean
app_window_flag_is_set		(AppWindowFlag	 app_flag)
{
	return (APP_FLAG_IS_SET (app_flag));
}

