/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <string.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "dvd.h"


static DvdFileIOClass *dvd_file_track_parent_class = NULL;

static void   dvd_file_track_class_init		(DvdFileTrackClass *class);
static void   dvd_file_track_instance_init	(GTypeInstance	*instance,
						 gpointer	 g_class);
static void   dvd_file_track_dispose		(GObject	*object);

static void
dvd_file_track_class_init	(DvdFileTrackClass	*class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_file_track_parent_class = g_type_class_ref (DVD_FILE_IO_TYPE);
	object_class->dispose = dvd_file_track_dispose;
}

static void
dvd_file_track_instance_init	(GTypeInstance	*instance,
				 gpointer	 g_class)
{
	DvdFileTrack *file_track;
	
	file_track = DVD_FILE_TRACK (instance);
	file_track->track_type = DVD_FILE_TRACK_UNKNOWN;
}

static void
dvd_file_track_dispose		(GObject	*object)
{
	DvdFileTrack *file_track = DVD_FILE_TRACK (object);
	
	switch (file_track->track_type) {
	case DVD_FILE_TRACK_VIDEO:
		g_object_unref (G_OBJECT (file_track->dvd_video));
		break;
	case DVD_FILE_TRACK_AUDIO:
		g_object_unref (G_OBJECT (file_track->dvd_audio));
		break;
	case DVD_FILE_TRACK_SUB_PICTURE:
		/* do nothing yet */
		break;
	case DVD_FILE_TRACK_UNKNOWN:
		/* do nothing */
		break;
	default:
		g_assert_not_reached ();
		break;
	}
	G_OBJECT_CLASS (dvd_file_track_parent_class)->dispose (G_OBJECT (object));
}

/**
 * dvd_file_track_get_type:
 *
 * Returns the GType for the DvdFileTrack class.
 */

GType
dvd_file_track_get_type		(void)
{
	static GType dvd_file_track_type = 0;

	if (dvd_file_track_type == 0) {
		GTypeInfo dvd_file_track_info = {
			sizeof (DvdFileTrackClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_file_track_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdFileTrack),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_file_track_instance_init
	    	};
		dvd_file_track_type = g_type_register_static (DVD_FILE_IO_TYPE,
							      "DvdFileTrack",
							      &dvd_file_track_info, 0);
	}
	return dvd_file_track_type;
}

/**
 * Create a new DvdFileTrack object.
 * You must call g_object_unref() when you are done with the returned object.
 * @returns A #DvdFileTrack object.
 */
DvdFileTrack
*dvd_file_track_new			(void)
{
	DvdFileTrack *file_track;
	
	file_track = g_object_new (dvd_file_track_get_type (), NULL);
	return file_track;
}

void
dvd_file_track_set_audio_properties	(DvdFileTrack	*file_track,
					 const DvdAudio	*dvd_audio)
{
	g_assert (file_track != NULL);
	g_assert (file_track->track_type == DVD_FILE_TRACK_UNKNOWN);
	
	file_track->dvd_audio = g_object_ref (G_OBJECT (dvd_audio));
	file_track->track_type = DVD_FILE_TRACK_AUDIO;
}


DvdAudio
*dvd_file_track_get_audio_properties	(DvdFileTrack	*file_track)
{
	g_assert (file_track != NULL);
	g_assert (file_track->track_type == DVD_FILE_TRACK_AUDIO);
	
	return g_object_ref (G_OBJECT (file_track->dvd_audio));
}


void
dvd_file_track_set_video_properties	(DvdFileTrack	*file_track,
					 const DvdVideo	*dvd_video)
{
	g_assert (file_track != NULL);
	g_assert (file_track->track_type == DVD_FILE_TRACK_UNKNOWN);
	
	file_track->dvd_video = g_object_ref (G_OBJECT (dvd_video));
	file_track->track_type = DVD_FILE_TRACK_VIDEO;
}


DvdVideo
*dvd_file_track_get_video_properties	(DvdFileTrack	*file_track)
{
	g_assert (file_track != NULL);
	g_assert (file_track->track_type == DVD_FILE_TRACK_VIDEO);
	
	return g_object_ref (G_OBJECT (file_track->dvd_video));
}


void
dvd_file_track_set_time_stamp		(DvdFileTrack	*file_track,
					 guint32	 time_stamp)
{
	g_assert (file_track != NULL);
	
	file_track->time_stamp = time_stamp;
	file_track->time_stamp_set = TRUE;
}


guint32
dvd_file_track_get_time_stamp		(DvdFileTrack	*file_track)
{
	g_assert (file_track != NULL);
	g_assert (file_track->time_stamp_set == TRUE);
	
	return file_track->time_stamp;
}


gboolean
dvd_file_track_get_time_stamp_set	(DvdFileTrack	*file_track)
{
	g_assert (file_track != NULL);
	
	return file_track->time_stamp_set;
}


DvdFileTrackType
dvd_file_track_get_track_type		(DvdFileTrack	*file_track)
{
	g_assert (file_track != NULL);
	
	return file_track->track_type;
}

