/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * $URL: https://scummvm.svn.sourceforge.net/svnroot/scummvm/scummvm/tags/release-1-2-1/tools/create_kyradat/tables.cpp $
 * $Id: tables.cpp 51834 2010-08-07 14:38:07Z athrxx $
 *
 */

#include "tables.h"
#include "create_kyradat.h"

#define EXTRACT_END_ENTRY { UNK_LANG, kPlatformUnknown, { 0, 0, { { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 } } } }

namespace {

// Id provider tables

const ExtractEntrySearchData k1ForestSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000195, 0x00001455, { { 0x86, 0xC2, 0x45, 0xEB, 0x16, 0x8A, 0x44, 0x73, 0xB4, 0x59, 0x13, 0x9B, 0x85, 0x40, 0x82, 0x81 } } } }, // floppy
	{ UNK_LANG, kPlatformUnknown, { 0x000002DE, 0x00003946, { { 0x85, 0x79, 0xDC, 0xB8, 0x62, 0xAD, 0xD6, 0x01, 0xA5, 0x66, 0x6B, 0xC4, 0x61, 0xCA, 0x82, 0xF5 } } } }, // CD + Amiga

	{ UNK_LANG, kPlatformUnknown, { 0x00000273, 0x00001D80, { { 0x18, 0x32, 0xB7, 0xFB, 0xD8, 0x1A, 0x6D, 0x83, 0x75, 0xF5, 0x2B, 0xF7, 0xC0, 0xFD, 0x85, 0xEF } } } }, // Amiga + CD demo

	{ UNK_LANG, kPlatformFMTowns, { 0x00000196, 0x00001468, { { 0x06, 0x8D, 0xC1, 0x9B, 0xBC, 0x28, 0xE8, 0x36, 0x43, 0x67, 0x34, 0xAA, 0x78, 0xB9, 0xCB, 0x72 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x0000013B, 0x0000128A, { { 0xD8, 0xB1, 0xFF, 0x57, 0xC9, 0x5D, 0xE1, 0x23, 0xA5, 0xA0, 0x4B, 0x0F, 0x28, 0xAC, 0x56, 0x83 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1KallakWritingSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000007F5, 0x00006DCD, { { 0x0E, 0x0A, 0x75, 0xA5, 0x71, 0x5A, 0xC6, 0x32, 0x52, 0x7F, 0xEC, 0x2B, 0x0B, 0xF2, 0x22, 0xE6 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000805, 0x0000719E, { { 0xA2, 0xFE, 0x0A, 0xAA, 0xD9, 0x43, 0x5A, 0xBE, 0x56, 0x38, 0x73, 0x42, 0xAC, 0xA9, 0x1B, 0x01 } } } }, // CD

	{ UNK_LANG, kPlatformAmiga, { 0x000007F4, 0x00006DAD, { { 0x84, 0x9D, 0x82, 0xD2, 0x4E, 0x8B, 0xE2, 0x86, 0x82, 0x5E, 0xE7, 0x2C, 0x7A, 0xD6, 0xFE, 0x19 } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x000007F7, 0x00006E23, { { 0x97, 0x8C, 0x63, 0x8D, 0x91, 0x68, 0xA5, 0x63, 0x55, 0x43, 0xE7, 0xE0, 0x12, 0xF9, 0x80, 0xA4 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x000007F3, 0x00006D9E, { { 0x3E, 0x83, 0xA4, 0xBF, 0x43, 0xAC, 0xDD, 0x41, 0x33, 0xC3, 0x33, 0x4E, 0xAE, 0x08, 0xB6, 0xDA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1KyrandiaLogoSeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000005C, 0x000003D3, { { 0x75, 0xC5, 0x55, 0x54, 0x00, 0xD7, 0xE9, 0x26, 0x0F, 0x01, 0xB1, 0xB7, 0xA9, 0x29, 0xF7, 0x69 } } } },
	{ UNK_LANG, kPlatformPC, { 0x000000A6, 0x00000A09, { { 0x70, 0x6E, 0x74, 0xBB, 0xD7, 0x09, 0x05, 0xDD, 0xA3, 0x09, 0x96, 0x40, 0xBD, 0x01, 0x19, 0x05 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000054, 0x00000300, { { 0x4E, 0xAD, 0xD0, 0x73, 0xD2, 0x3C, 0xF6, 0x6B, 0x65, 0x23, 0xA2, 0x30, 0xD6, 0xEE, 0x1C, 0x13 } } } }, // demo

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1KallakMalcolmSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000026B, 0x00002132, { { 0x51, 0x07, 0x32, 0xA2, 0x09, 0x47, 0x97, 0x02, 0x85, 0x31, 0x39, 0x93, 0x3A, 0x53, 0x47, 0xA5 } } } }, // floppy
	{ UNK_LANG, kPlatformUnknown, { 0x0000027B, 0x0000220A, { { 0xB7, 0xC1, 0x57, 0x04, 0x9B, 0x67, 0x82, 0x7B, 0x6E, 0xFD, 0x59, 0xF2, 0x10, 0x93, 0x89, 0x12 } } } }, // CD + Amiga

	{ UNK_LANG, kPlatformUnknown, { 0x000002B8, 0x0000280B, { { 0x98, 0xC8, 0x36, 0x8C, 0xF8, 0x92, 0xC2, 0xB9, 0x1B, 0x71, 0x6B, 0x4C, 0xA4, 0x6C, 0xF6, 0x30 } } } }, // Amiga + CD demo

	{ UNK_LANG, kPlatformFMTowns, { 0x0000026D, 0x00002188, { { 0x7E, 0x3B, 0x06, 0x69, 0x11, 0x59, 0xBE, 0x42, 0xCB, 0x33, 0x9C, 0x15, 0xCF, 0x61, 0xF6, 0xB0 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x0000026C, 0x00002145, { { 0x95, 0xB7, 0x9E, 0x63, 0x99, 0xFA, 0x22, 0xBF, 0x62, 0x37, 0xC2, 0x2B, 0x40, 0x73, 0x53, 0x83 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1MalcolmTreeSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000216, 0x000014FB, { { 0xBA, 0x6F, 0x63, 0xD5, 0x44, 0xCE, 0xE8, 0x20, 0xD4, 0x4C, 0x06, 0x58, 0x21, 0x8E, 0x07, 0x8B } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000230, 0x0000192A, { { 0x28, 0xCC, 0x89, 0x40, 0x8A, 0xF7, 0xCB, 0xC0, 0x11, 0x8F, 0x0F, 0xE5, 0x5F, 0x24, 0x2D, 0x32 } } } }, // CD

	{ UNK_LANG, kPlatformAmiga, { 0x00000214, 0x000014C4, { { 0xA2, 0x60, 0x16, 0x14, 0x20, 0xE4, 0x8D, 0x89, 0xC4, 0x30, 0x07, 0x21, 0xC9, 0x26, 0xA6, 0x84 } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x00000218, 0x00001551, { { 0x30, 0x42, 0xD3, 0x7D, 0x6B, 0x43, 0x3B, 0xC7, 0x63, 0x56, 0x7D, 0x12, 0xE0, 0x53, 0x41, 0x48 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x00000217, 0x0000150E, { { 0x41, 0xFC, 0xD3, 0xAC, 0x8D, 0x1C, 0x11, 0x4E, 0xE0, 0x0C, 0xF6, 0x93, 0xA9, 0x22, 0x53, 0x7F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1WestwoodLogoSeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000004B, 0x000002FE, { { 0x6E, 0xDA, 0x5D, 0x1E, 0xF4, 0x38, 0xAE, 0x7A, 0x9F, 0xBC, 0x00, 0x7C, 0x77, 0x9D, 0x03, 0xAF } } } },
	{ UNK_LANG, kPlatformPC, { 0x0000004B, 0x0000032A, { { 0x9A, 0x54, 0x6F, 0x6C, 0x70, 0xD1, 0x02, 0x94, 0xD1, 0xA1, 0xA7, 0xBE, 0x1B, 0x10, 0xBD, 0x2F } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000044, 0x00000243, { { 0x00, 0x11, 0x10, 0x64, 0x60, 0xE8, 0xB5, 0x59, 0x00, 0x60, 0xF5, 0x10, 0xB5, 0x63, 0x9D, 0x55 } } } }, // demo

	{ UNK_LANG, kPlatformAmiga, { 0x0000004D, 0x00000319, { { 0xE9, 0x5D, 0xD4, 0x60, 0x4E, 0xA6, 0x92, 0x30, 0x9C, 0x77, 0x99, 0xBC, 0xB1, 0x97, 0xFB, 0xBD } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Demo1SeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000026, 0x0000016E, { { 0x61, 0xF2, 0x84, 0x42, 0x89, 0x6E, 0xB8, 0x0D, 0x01, 0xB7, 0xE4, 0x49, 0xA1, 0x72, 0xDC, 0x66 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Demo2SeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000023, 0x00000151, { { 0xE7, 0x9F, 0xA7, 0x80, 0xE3, 0xC1, 0x63, 0xF2, 0x3F, 0x7B, 0x64, 0xD9, 0x70, 0x3A, 0xF1, 0x55 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Demo3SeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000059, 0x000001A8, { { 0x9E, 0x90, 0x4F, 0x9E, 0xBC, 0x39, 0x37, 0xD1, 0x10, 0x67, 0x8C, 0xE5, 0x84, 0xFD, 0xD9, 0x7C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Demo4SeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x0000013E, { { 0xB7, 0x67, 0x1D, 0x50, 0x11, 0xD0, 0x16, 0x91, 0x40, 0x40, 0xD1, 0xC0, 0xA0, 0x79, 0x3D, 0xCE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AmuleteAnimSeqProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002A, 0x0000025D, { { 0xB1, 0x59, 0x3C, 0x9A, 0x61, 0xC0, 0x1D, 0x80, 0xB5, 0x0C, 0x2E, 0xF9, 0xF6, 0x36, 0xED, 0x7D } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1OutroReunionSeqProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000547, 0x0000781C, { { 0xCF, 0xD6, 0x1D, 0x3D, 0x14, 0x40, 0x88, 0x35, 0x36, 0x4F, 0x0B, 0x1F, 0x9A, 0x1C, 0x3D, 0xAC } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x000005E5, 0x00008918, { { 0x6A, 0x33, 0x8C, 0xB0, 0x16, 0x57, 0x2D, 0xEB, 0xB2, 0xE1, 0x64, 0x80, 0x98, 0x99, 0x98, 0x19 } } } }, // CD

	{ UNK_LANG, kPlatformAmiga, { 0x0000054A, 0x0000785F, { { 0x55, 0xEA, 0xB8, 0x7F, 0x3A, 0x86, 0xCD, 0xA6, 0xBC, 0xA7, 0x9A, 0x39, 0xED, 0xF5, 0x30, 0x0A } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x00000547, 0x00007876, { { 0x7A, 0xC7, 0x80, 0x34, 0x7A, 0x1B, 0xAB, 0xF8, 0xA7, 0x2F, 0x63, 0x3C, 0xDA, 0x89, 0x3F, 0x82 } } } }, // some floppy DOS + FM-TOWNS

	{ UNK_LANG, kPlatformPC98, { 0x00000548, 0x00007889, { { 0x17, 0xE9, 0x40, 0xBA, 0x4A, 0xBD, 0xC6, 0x62, 0x56, 0x96, 0xD6, 0x1B, 0x11, 0xF3, 0xD8, 0x14 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1IntroCPSStringsProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000015, 0x0000071D, { { 0x59, 0x65, 0x08, 0xF9, 0x4C, 0x81, 0xA1, 0xE3, 0x68, 0xF7, 0xE3, 0xF6, 0x33, 0x5F, 0xF5, 0x36 } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x00000014, 0x0000071D, { { 0xBA, 0xB6, 0x58, 0xB3, 0x28, 0x5E, 0x9F, 0x77, 0x69, 0x9D, 0x77, 0x53, 0x9D, 0x0D, 0xB0, 0x29 } } } }, // floppy + PC98
	{ UNK_LANG, kPlatformPC, { 0x00000015, 0x00000786, { { 0xCF, 0x09, 0xE1, 0xD9, 0x8E, 0x34, 0x5D, 0xEA, 0xBC, 0xAC, 0xC4, 0xF4, 0x4A, 0xEC, 0xFF, 0xC1 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x0000000C, 0x00000413, { { 0xA1, 0xE3, 0x06, 0x53, 0x23, 0x9A, 0xE0, 0xF1, 0xE4, 0xFD, 0xD9, 0x05, 0x22, 0xA6, 0x28, 0x46 } } } }, // demo

	{ UNK_LANG, kPlatformAmiga, { 0x00000016, 0x0000070A, { { 0xD9, 0xDB, 0x91, 0xCD, 0x93, 0x81, 0xC4, 0x3F, 0x14, 0xF1, 0xC5, 0x02, 0xE7, 0x3F, 0x3A, 0x6C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1IntroCOLStringsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000F09, { { 0x05, 0x7B, 0x69, 0xB7, 0x6A, 0xC3, 0x7F, 0xD9, 0x7E, 0x51, 0x87, 0xA5, 0x31, 0xD8, 0x80, 0xB3 } } } },
	{ UNK_LANG, kPlatformUnknown, { 0x0000002B, 0x00000F09, { { 0x01, 0x40, 0x65, 0xC1, 0x33, 0xAA, 0x64, 0xA7, 0x44, 0xD5, 0x89, 0x68, 0x17, 0x47, 0xC7, 0x1D } } } }, // floppy(?) + PC98
	{ UNK_LANG, kPlatformPC, { 0x00000046, 0x0000174F, { { 0xAA, 0x59, 0x01, 0x20, 0x06, 0xB4, 0x80, 0xC0, 0x84, 0x38, 0xFD, 0x97, 0xBF, 0x78, 0xDD, 0x78 } } } }, // demo

	{ UNK_LANG, kPlatformAmiga, { 0x0000002E, 0x00000F09, { { 0x2A, 0x42, 0xB8, 0x4E, 0xC6, 0xF0, 0x5F, 0x4A, 0x08, 0xB4, 0xE3, 0xE5, 0xBD, 0x32, 0x11, 0x06 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1IntroWSAStringsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000093, 0x00002E5D, { { 0xAB, 0xD1, 0x32, 0x89, 0x2A, 0x8D, 0xC0, 0x2C, 0x87, 0x87, 0xA3, 0x14, 0x2A, 0x2A, 0x22, 0x7F } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x00000086, 0x00002E5D, { { 0xD2, 0x8D, 0x62, 0xEF, 0xEB, 0x75, 0x04, 0x13, 0x64, 0x85, 0x6B, 0x6C, 0xE8, 0x6C, 0xCF, 0x52 } } } },
	{ UNK_LANG, kPlatformPC, { 0x0000004D, 0x00001AEC, { { 0x15, 0xBC, 0x8C, 0xD9, 0x33, 0x0C, 0xC5, 0x66, 0x87, 0x3B, 0x76, 0xEE, 0x0C, 0x41, 0x4D, 0x1F } } } }, // demo

	{ UNK_LANG, kPlatformUnknown, { 0x00000080, 0x00002E5D, { { 0x67, 0x03, 0x20, 0xDF, 0xAA, 0x3C, 0x6A, 0xB5, 0x45, 0xD0, 0x00, 0x2B, 0x30, 0xCC, 0x36, 0xB6 } } } }, // Amiga + CD Demo

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1IntroStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000004F5, 0x0001A7B1, { { 0xC7, 0x65, 0x5B, 0x5A, 0x56, 0x43, 0x94, 0x55, 0x5B, 0x00, 0xFD, 0x5D, 0xF4, 0xB5, 0x04, 0x15 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x000004F6, 0x0001A7B1, { { 0xDE, 0x41, 0xB1, 0x98, 0xD5, 0xAD, 0x6B, 0xBA, 0x33, 0x04, 0x19, 0x5D, 0xCC, 0x07, 0xB5, 0x56 } } } }, // CD
	{ EN_ANY, kPlatformPC, { 0x000002FA, 0x00010181, { { 0x48, 0xD3, 0x1B, 0x8E, 0x8D, 0xBC, 0x1F, 0x51, 0x35, 0x71, 0xE2, 0xAB, 0xC8, 0xBE, 0xFC, 0x88 } } } }, // demo

	{ FR_FRA, kPlatformPC, { 0x000005EF, 0x000207E6, { { 0xDC, 0x6C, 0x9A, 0x7C, 0x19, 0x45, 0x2B, 0x25, 0x84, 0xCC, 0xE1, 0x40, 0xFE, 0x8E, 0xF4, 0x51 } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x000005F0, 0x000207E6, { { 0xB8, 0x51, 0xEA, 0x44, 0x64, 0x04, 0xE4, 0x6A, 0x7B, 0xBD, 0x31, 0x62, 0xC0, 0xDD, 0xEA, 0xEF } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x00000605, 0x000212AB, { { 0x8F, 0x3A, 0x9D, 0x9A, 0x81, 0xF2, 0x4A, 0x90, 0xBC, 0xBD, 0x62, 0x8B, 0xD8, 0xD6, 0xF4, 0x5D } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000606, 0x000212AB, { { 0xB0, 0x78, 0x86, 0xFB, 0xCB, 0x3D, 0x1E, 0x1E, 0x86, 0xE7, 0x2C, 0x2B, 0x3A, 0x3C, 0x6F, 0x1C } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x0000057C, 0x0001DF02, { { 0x4B, 0x09, 0xBC, 0xBC, 0xB3, 0x3C, 0x69, 0xEB, 0xA6, 0xB5, 0x76, 0xED, 0xC7, 0x57, 0x71, 0xB0 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x000005F0, 0x0001CF13, { { 0x4F, 0xA0, 0x47, 0x57, 0x6D, 0x2D, 0xA6, 0x62, 0x06, 0xBE, 0x86, 0xD2, 0xD7, 0xEA, 0x2E, 0xD6 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x000005CF, 0x00020415, { { 0xCC, 0xE5, 0x9F, 0xB8, 0xCA, 0xFA, 0x2D, 0x05, 0xB8, 0xAF, 0x9F, 0x1F, 0x8A, 0xA8, 0x56, 0xDE } } } },

	{ EN_ANY, kPlatformAmiga, { 0x0000050A, 0x0001A7B1, { { 0x1B, 0x74, 0x71, 0x4C, 0xAB, 0x81, 0x10, 0x59, 0x8A, 0x21, 0x50, 0xBB, 0xFE, 0x6F, 0xD0, 0xE8 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000626, 0x00021319, { { 0x80, 0x55, 0x54, 0x14, 0x5D, 0x6F, 0x49, 0x04, 0x4A, 0xEF, 0x92, 0xB8, 0x5B, 0x01, 0x0F, 0x97 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x00000545, 0x0001A84C, { { 0xEF, 0xE2, 0xE4, 0x92, 0xE1, 0x18, 0xF8, 0x3C, 0x91, 0x29, 0x4D, 0x82, 0xA4, 0x3E, 0xE6, 0x7D } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x000005FB, 0x0003163E, { { 0x16, 0x28, 0x83, 0xCE, 0x29, 0x94, 0x9F, 0x82, 0xDC, 0xD4, 0x06, 0xE8, 0x67, 0xC9, 0x56, 0x29 } } } },

	{ JA_JPN, kPlatformPC98, { 0x000005D0, 0x00032AD7, { { 0xEE, 0x36, 0x93, 0xDD, 0x71, 0x02, 0xE9, 0x0D, 0x54, 0xDE, 0xB6, 0xE1, 0x09, 0x59, 0x60, 0x64 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1OutroHomeStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000005, 0x00000189, { { 0xA5, 0xF3, 0x03, 0x3F, 0x43, 0x2B, 0x77, 0xCC, 0x5C, 0x96, 0x64, 0xB1, 0x75, 0xB5, 0x3E, 0xF6 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x00000007, 0x000003E8, { { 0x68, 0x35, 0x87, 0x53, 0xD9, 0x53, 0x1F, 0x13, 0x24, 0x61, 0x0D, 0x8D, 0x33, 0x91, 0xF3, 0x47 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000007, 0x00000267, { { 0xD0, 0xC1, 0x11, 0x49, 0x3D, 0x3F, 0xCE, 0x84, 0x8F, 0xEF, 0xE3, 0xFD, 0x43, 0xCA, 0x92, 0xB0 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000000E, 0x00000473, { { 0xB6, 0xEB, 0xE8, 0x22, 0x67, 0x24, 0xA9, 0xA3, 0x94, 0x55, 0xC6, 0x57, 0x17, 0x15, 0x5B, 0x04 } } } },

	{ ES_ESP, kPlatformPC, { 0x00000005, 0x00000178, { { 0x2E, 0x9C, 0x94, 0x0F, 0x29, 0x77, 0x27, 0x1D, 0x77, 0x1E, 0x5A, 0xF8, 0x0E, 0x8D, 0x09, 0x6B } } } },
	{ IT_ITA, kPlatformPC, { 0x00000005, 0x00000178, { { 0x2E, 0x9C, 0x94, 0x0F, 0x29, 0x77, 0x27, 0x1D, 0x77, 0x1E, 0x5A, 0xF8, 0x0E, 0x8D, 0x09, 0x6B } } } },

	{ IT_ITA, kPlatformPC, { 0x00000007, 0x000001B8, { { 0x17, 0x95, 0x5B, 0x4F, 0xE2, 0x07, 0x5A, 0x49, 0xFA, 0xCE, 0x53, 0x8B, 0xE7, 0x46, 0x69, 0xC7 } } } }, // (fan) CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1RoomFilenamesProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x0000034C, 0x0000B7AA, { { 0x7C, 0x1B, 0x89, 0x09, 0xF0, 0x87, 0x3C, 0x64, 0x3D, 0x0E, 0x61, 0xCC, 0x09, 0x8D, 0xC9, 0x95 } } } },
	{ UNK_LANG, kPlatformUnknown, { 0x000002DC, 0x0000B7AA, { { 0x31, 0x56, 0x41, 0x69, 0x1E, 0x84, 0x54, 0x2C, 0xF7, 0x7B, 0x98, 0x25, 0xF3, 0x91, 0x56, 0xFD } } } }, // DOS + PC98

	{ UNK_LANG, kPlatformAmiga, { 0x00000308, 0x0000B7AA, { { 0x6D, 0xA4, 0xE8, 0x27, 0x68, 0x24, 0xCF, 0xAA, 0x30, 0x32, 0x71, 0x74, 0x36, 0x68, 0x82, 0x20 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1RoomListProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x000064E8, 0x0010312B, { { 0x94, 0x5C, 0x87, 0x35, 0x35, 0x6B, 0x3E, 0xBF, 0x55, 0x3D, 0xDB, 0xD9, 0xFB, 0x97, 0x27, 0x5D } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x00004DD6, 0x0010312B, { { 0xC6, 0xF0, 0xC4, 0x2C, 0x5A, 0xD7, 0x48, 0xE4, 0x41, 0x23, 0x65, 0x6D, 0xC8, 0xC7, 0xCE, 0xF5 } } } }, // DOS + PC98

	{ UNK_LANG, kPlatformAmiga, { 0x00004ECC, 0x0010312B, { { 0x9A, 0x91, 0xF1, 0x9C, 0x8A, 0x96, 0x1C, 0x7B, 0xB7, 0xE4, 0xF1, 0xE9, 0x7D, 0xEF, 0x40, 0xBF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1CharacterImageFilenamesProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x000000F7, 0x00003A8F, { { 0x1F, 0xAB, 0x8E, 0x5F, 0xAD, 0x76, 0x57, 0x37, 0xD7, 0x9B, 0x3F, 0x35, 0x4B, 0xA0, 0x98, 0x37 } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x000000E4, 0x00003A8F, { { 0x7A, 0x77, 0x6D, 0x26, 0xC8, 0x43, 0xCB, 0x74, 0x46, 0x2E, 0x04, 0x22, 0x89, 0x27, 0x33, 0x06 } } } }, // DOS + PC98

	{ UNK_LANG, kPlatformAmiga, { 0x000000F0, 0x00003A8F, { { 0xC0, 0xD9, 0x8C, 0x2B, 0x65, 0x51, 0x7C, 0x13, 0x52, 0x7E, 0xF7, 0x63, 0xA6, 0xA9, 0xCE, 0x40 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AudioTracksProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000041, 0x00000FBF, { { 0xB5, 0xA2, 0x90, 0xE9, 0x73, 0x83, 0x47, 0x5A, 0xB3, 0x3E, 0x04, 0xBB, 0xAA, 0xC8, 0x84, 0x53 } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x0000005D, 0x0000154E, { { 0xA7, 0x7E, 0x03, 0x0A, 0x81, 0x54, 0xD2, 0x5D, 0x7B, 0x33, 0x07, 0xBF, 0x70, 0x01, 0x4B, 0x79 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AudioTracksIntroProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000006, 0x0000022C, { { 0x5F, 0xC9, 0xE1, 0x4B, 0x34, 0x52, 0xB9, 0xF8, 0xFF, 0x37, 0x8B, 0xF4, 0xEF, 0x5E, 0xC5, 0xDA } } } }, // floppy + demo
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x00000458, { { 0xEB, 0xB3, 0x96, 0xA5, 0x07, 0xE6, 0x11, 0x58, 0xDB, 0x3F, 0x34, 0x30, 0xFB, 0x7B, 0x92, 0xC8 } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1ItemNamesProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000468, 0x00018B13, { { 0xCE, 0xFE, 0x19, 0x5D, 0x3F, 0xD7, 0x2A, 0xF6, 0xD0, 0xC5, 0xBC, 0x17, 0x6A, 0x52, 0xDB, 0xFE } } } },

	{ FR_FRA, kPlatformPC, { 0x00000506, 0x0001C3CC, { { 0x46, 0xD4, 0x19, 0x04, 0x2A, 0x46, 0xCD, 0x63, 0x41, 0x5A, 0x8E, 0xA8, 0x00, 0xDD, 0x02, 0x77 } } } },

	{ DE_DEU, kPlatformPC, { 0x00000512, 0x0001DD2C, { { 0x5D, 0xB0, 0x6B, 0x8E, 0x9C, 0x70, 0x09, 0x90, 0x3C, 0xC6, 0x58, 0x1F, 0x71, 0x24, 0x76, 0x56 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000591, 0x0001F869, { { 0x4D, 0xD7, 0x66, 0xBB, 0x9A, 0xD8, 0x79, 0x1E, 0x92, 0x2B, 0xE5, 0xC2, 0x0F, 0x02, 0x0B, 0x35 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000506, 0x0001C1D9, { { 0xF2, 0x2D, 0x18, 0x13, 0x61, 0xFB, 0x57, 0x93, 0x5C, 0xDC, 0x79, 0x44, 0x96, 0x01, 0xA6, 0x90 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000530, 0x0001D90A, { { 0x52, 0xFB, 0xA8, 0x3F, 0xA3, 0x6F, 0xC2, 0x67, 0x55, 0x9F, 0x66, 0x9F, 0xFD, 0x79, 0x44, 0xDF } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000380, 0x00012960, { { 0x2D, 0x81, 0xCF, 0x7A, 0x9D, 0x71, 0x83, 0xB7, 0xE5, 0x00, 0xB0, 0x6E, 0x25, 0x94, 0xCB, 0xA4 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x000003E5, 0x0001607D, { { 0x6D, 0xBE, 0xAD, 0xE5, 0xD1, 0x41, 0x6C, 0x42, 0x71, 0x79, 0x9C, 0x78, 0x93, 0x84, 0xC8, 0x11 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x00000505, 0x00018B13, { { 0x62, 0x76, 0x84, 0x5D, 0x8D, 0xBA, 0x23, 0xC5, 0x67, 0x26, 0x22, 0xBD, 0xBE, 0x81, 0x2D, 0xFF } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000587, 0x000247A1, { { 0xE8, 0xCA, 0x1E, 0x34, 0x05, 0xE5, 0x08, 0xCA, 0xA7, 0xFE, 0x55, 0xCB, 0x75, 0xD0, 0xA7, 0x19 } } } },

	{ JA_JPN, kPlatformPC98, { 0x000004CF, 0x000247A1, { { 0xBE, 0x80, 0x3F, 0x10, 0xF7, 0x3A, 0x85, 0xD1, 0xE8, 0x24, 0xE6, 0x4E, 0x8F, 0x17, 0xC9, 0x2E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TakenStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000010, 0x000004C2, { { 0x08, 0xC0, 0x66, 0xB1, 0x45, 0x04, 0x78, 0xF9, 0xAF, 0x0F, 0x52, 0xC8, 0x0E, 0x0C, 0x69, 0x86 } } } },

	{ FR_FRA, kPlatformPC, { 0x0000001E, 0x000008FA, { { 0xC9, 0xCC, 0x2E, 0x0B, 0xE4, 0xE0, 0x44, 0xBD, 0xC2, 0x10, 0x09, 0xA3, 0x36, 0xB1, 0x5D, 0x1C } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x0000000F, 0x0000047D, { { 0xE8, 0xB5, 0x15, 0xD5, 0x96, 0x81, 0xD2, 0x84, 0xAB, 0x33, 0xC1, 0x1C, 0x47, 0x1B, 0xFE, 0xDB } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x0000001C, 0x000009C0, { { 0xA8, 0x64, 0xDD, 0x9D, 0x91, 0x42, 0x04, 0x26, 0xD0, 0x80, 0x40, 0xED, 0x08, 0x51, 0xC9, 0xBB } } } },

	{ IT_ITA, kPlatformPC, { 0x00000014, 0x00000588, { { 0x5B, 0x75, 0xAE, 0xD7, 0x50, 0x6C, 0x3E, 0x65, 0x5C, 0xF8, 0x84, 0x34, 0x48, 0x58, 0x0E, 0x6E } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000000F, 0x000002F0, { { 0x37, 0x22, 0x62, 0xE4, 0x1B, 0x1F, 0xD6, 0x9E, 0x94, 0xA3, 0x41, 0x00, 0xD5, 0x29, 0x28, 0x09 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000014, 0x000005D8, { { 0xD6, 0x00, 0x90, 0x6A, 0x75, 0x3B, 0xF1, 0xFE, 0xF4, 0x3E, 0x0E, 0x1D, 0x39, 0xEB, 0x2D, 0xC8 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000008, 0x00000261, { { 0x93, 0x5B, 0x79, 0xE8, 0x89, 0x8E, 0xB5, 0x37, 0x39, 0x2A, 0xB0, 0x04, 0x98, 0x80, 0x5A, 0x4E } } } },
	{ DE_DEU, kPlatformAmiga, { 0x0000000E, 0x000004E0, { { 0x52, 0x4D, 0x74, 0x91, 0x70, 0x0D, 0x4C, 0x40, 0x5C, 0x7E, 0xBA, 0xDA, 0x24, 0x49, 0xF3, 0x1A } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000001C, 0x00000AED, { { 0x32, 0xD2, 0x87, 0x67, 0x3D, 0x01, 0x50, 0xF3, 0xFA, 0x19, 0x7A, 0xB0, 0xF8, 0x6F, 0x8A, 0x07 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000001F, 0x00000EB7, { { 0x7F, 0x8C, 0xFF, 0x35, 0x30, 0x71, 0x9A, 0x78, 0xCC, 0x07, 0x11, 0x86, 0xE3, 0x0D, 0x16, 0xDF } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000016, 0x00000C56, { { 0x97, 0xF0, 0x57, 0xCC, 0x03, 0x06, 0x41, 0xDC, 0xFC, 0xD5, 0xA2, 0x35, 0x0C, 0x71, 0x37, 0xCC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PlacedStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000009, 0x000002B7, { { 0xE7, 0xCF, 0x8A, 0x19, 0xEF, 0x98, 0xE5, 0x74, 0x46, 0x46, 0x47, 0x65, 0xB6, 0x81, 0x70, 0x0A } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000000D, 0x0000074C, { { 0x0E, 0x1C, 0x1F, 0xD2, 0xCF, 0xBF, 0x40, 0xE1, 0x59, 0x0F, 0x1B, 0x46, 0xED, 0x8B, 0x96, 0x8E } } } },

	{ FR_FRA, kPlatformPC, { 0x00000011, 0x00000545, { { 0x05, 0xEF, 0x74, 0x54, 0x06, 0xB7, 0x64, 0x47, 0x21, 0x56, 0x55, 0x74, 0x70, 0x2D, 0xA8, 0x23 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000000B, 0x00000389, { { 0xB9, 0x74, 0xBE, 0x63, 0xB4, 0xA4, 0x8A, 0x04, 0xD9, 0x50, 0x73, 0xB7, 0x01, 0x9E, 0x77, 0xD3 } } } },

	{ IT_ITA, kPlatformPC, { 0x0000000D, 0x0000040D, { { 0x9C, 0x71, 0x53, 0x35, 0xC3, 0xE8, 0x46, 0xB9, 0xD2, 0xFA, 0x1C, 0x8C, 0xC3, 0xFF, 0xBC, 0x1F } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000011, 0x000003B8, { { 0xC8, 0xA6, 0xE4, 0x8A, 0xF7, 0x4C, 0x3F, 0xA6, 0x24, 0x7F, 0xEF, 0xE4, 0x63, 0x8B, 0x72, 0xF3 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000000D, 0x00000439, { { 0x57, 0xAE, 0x1C, 0xC1, 0xF5, 0xE8, 0x5B, 0x9E, 0x90, 0x02, 0xB9, 0x8D, 0x86, 0x38, 0xFB, 0xA8 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1DroppedStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000000A, 0x0000033C, { { 0x96, 0x63, 0xFD, 0x73, 0x71, 0xCC, 0x62, 0x59, 0x86, 0x02, 0x47, 0x55, 0x82, 0xF8, 0x5E, 0x1F } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000000B, 0x0000059F, { { 0xDD, 0x5E, 0x51, 0x7E, 0xD9, 0xFC, 0xCD, 0xAD, 0x6B, 0x93, 0x71, 0xBE, 0x83, 0x63, 0x3F, 0x88 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000011, 0x00000579, { { 0x3D, 0x0C, 0x3C, 0xAD, 0x2E, 0xE3, 0x7A, 0x5A, 0x4A, 0x21, 0x1C, 0x96, 0xC5, 0x5A, 0xA7, 0x9E } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000011, 0x00000612, { { 0xEC, 0xCD, 0x99, 0x58, 0xF7, 0x08, 0x92, 0x43, 0x95, 0x10, 0x6B, 0xDD, 0x61, 0xF7, 0x5C, 0xA4 } } } },

	{ IT_ITA, kPlatformPC, { 0x0000000B, 0x0000031B, { { 0x32, 0x0C, 0xCD, 0x83, 0xF6, 0x97, 0xC8, 0x76, 0x57, 0x6B, 0x7C, 0xAD, 0x51, 0xEB, 0x91, 0x1F } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000011, 0x000003B8, { { 0xC8, 0xA6, 0xE4, 0x8A, 0xF7, 0x4C, 0x3F, 0xA6, 0x24, 0x7F, 0xEF, 0xE4, 0x63, 0x8B, 0x72, 0xF3 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000008, 0x00000261, { { 0x1D, 0xB5, 0xFB, 0x23, 0x94, 0xA7, 0x86, 0x7A, 0xAC, 0x53, 0xDA, 0x6F, 0xCC, 0x41, 0x0F, 0xD7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1NoDropStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000047, 0x00001881, { { 0xEF, 0x81, 0x2B, 0x8F, 0x06, 0x0C, 0xA6, 0xBA, 0x50, 0x9C, 0x99, 0x40, 0x53, 0xFD, 0xC1, 0x55 } } } },

	{ FR_FRA, kPlatformPC, { 0x0000004C, 0x00001B39, { { 0xFF, 0xD5, 0x5D, 0x1C, 0xF1, 0x8A, 0xA4, 0x21, 0xB1, 0x56, 0x2B, 0xAC, 0x9F, 0x40, 0x30, 0x53 } } } },

	{ DE_DEU, kPlatformPC, { 0x0000004C, 0x00001AFF, { { 0xE5, 0x05, 0x79, 0x10, 0x91, 0x95, 0x64, 0x40, 0xB2, 0x73, 0x39, 0xFA, 0x50, 0x8D, 0xCE, 0x8A } } } },

	{ IT_ITA, kPlatformPC, { 0x00000045, 0x00001850, { { 0x50, 0x34, 0x82, 0xA9, 0x28, 0xDE, 0x44, 0xB3, 0x2D, 0x59, 0x0B, 0x7F, 0xCE, 0x42, 0x1F, 0x73 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000004C, 0x00001650, { { 0x42, 0x7A, 0x04, 0xCD, 0x69, 0xB1, 0x06, 0x9A, 0xD8, 0x7E, 0x7A, 0x14, 0x5B, 0x1C, 0x70, 0xB8 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000003D, 0x000015FA, { { 0x87, 0x2E, 0xE6, 0x8A, 0xF1, 0xC9, 0xC9, 0xEC, 0xD3, 0x2C, 0x25, 0x7E, 0x36, 0x02, 0xF7, 0xAC } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000048, 0x00001881, { { 0x69, 0xA8, 0x0B, 0x47, 0xFD, 0xA0, 0x94, 0x12, 0x82, 0x1D, 0xE0, 0x9C, 0xB1, 0x10, 0x6D, 0x11 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x0000004D, 0x00001AFF, { { 0xF8, 0x83, 0x2E, 0x08, 0x88, 0x7B, 0x72, 0x7F, 0x71, 0xD4, 0x05, 0xF3, 0x1A, 0x78, 0xF1, 0x9D } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000007A, 0x0000323C, { { 0x00, 0x7F, 0x8E, 0x2E, 0x7B, 0xFC, 0x06, 0x0B, 0x7A, 0x7F, 0x17, 0xAF, 0x47, 0x82, 0xC1, 0x16 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000061, 0x00002E9C, { { 0x8A, 0xFA, 0x43, 0x26, 0x68, 0x6D, 0x5A, 0x59, 0x0C, 0x9E, 0xCB, 0x28, 0x43, 0x33, 0x54, 0x49 } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000042, 0x000025C2, { { 0x6F, 0x34, 0x59, 0x17, 0x22, 0xCE, 0x1A, 0x82, 0x18, 0xE3, 0x9F, 0x86, 0x84, 0xF1, 0x8E, 0x0C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PutDownStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000026, 0x00000D27, { { 0x83, 0x09, 0x51, 0xD3, 0x94, 0x2B, 0xB1, 0x8B, 0x8B, 0x16, 0xA9, 0xEB, 0xE8, 0xA2, 0x4F, 0x0C } } } },

	{ JA_JPN, kPlatformUnknown, { 0x00000029, 0x0000187D, { { 0xDE, 0xEE, 0x66, 0x88, 0x57, 0xF5, 0xF9, 0x2E, 0xD2, 0x14, 0xF5, 0x83, 0xA0, 0x0D, 0x96, 0x86 } } } },

	{ FR_FRA, kPlatformPC, { 0x0000002E, 0x0000101D, { { 0xBC, 0xEA, 0x6C, 0x1F, 0x31, 0xCF, 0x30, 0x26, 0x2D, 0x24, 0xC5, 0xEA, 0x97, 0x49, 0xDC, 0x4C } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000028, 0x00000E7E, { { 0x26, 0x93, 0x76, 0x37, 0x41, 0x2E, 0xF3, 0xED, 0xF5, 0x7B, 0xA7, 0xEB, 0x80, 0x61, 0x3B, 0x84 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000034, 0x0000131D, { { 0x36, 0x87, 0x4D, 0x73, 0x2A, 0x67, 0xBA, 0xCE, 0xCB, 0x33, 0x3C, 0x10, 0xCC, 0x3E, 0x7E, 0xAD } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000002E, 0x00000EB2, { { 0x14, 0x08, 0xE6, 0xD1, 0x54, 0x76, 0x2A, 0x9E, 0xBE, 0x5A, 0x15, 0xEB, 0x52, 0x01, 0x52, 0x97 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000002D, 0x00001052, { { 0x12, 0x0A, 0x23, 0x11, 0xDF, 0x8A, 0x59, 0xD4, 0xF2, 0xCA, 0xA5, 0xA7, 0x76, 0x1B, 0x54, 0xB6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1WaitAmuletStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000003D, 0x0000150D, { { 0x11, 0xC8, 0x7E, 0x83, 0x7A, 0xC3, 0x5D, 0xD9, 0x72, 0xFC, 0x75, 0xCB, 0x57, 0xE9, 0xF1, 0x36 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x00000029, 0x000017A7, { { 0xFD, 0xA3, 0xAD, 0x24, 0x16, 0x49, 0x22, 0x6F, 0x52, 0xE9, 0x50, 0x11, 0x51, 0x38, 0xCA, 0xE0 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000049, 0x00001ADD, { { 0xFD, 0xCF, 0x9C, 0xAD, 0xA2, 0xAA, 0x74, 0x11, 0x8E, 0x89, 0x02, 0xCB, 0x2B, 0x5B, 0x2B, 0x1C } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000048, 0x000019CF, { { 0x82, 0xDD, 0xBD, 0xA3, 0xD0, 0xE0, 0x19, 0xC0, 0x6D, 0xC5, 0xEE, 0x7A, 0x0B, 0x49, 0x06, 0xCE } } } },

	{ IT_ITA, kPlatformPC, { 0x0000004A, 0x00001B4C, { { 0x38, 0x26, 0x54, 0x10, 0x3D, 0x32, 0xCD, 0xAE, 0x3F, 0x96, 0xFE, 0x6F, 0x4A, 0x94, 0x72, 0xB0 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000049, 0x0000184F, { { 0x11, 0x42, 0xB2, 0xD6, 0x76, 0xEA, 0xF9, 0xCA, 0x46, 0xE4, 0xA6, 0x35, 0xE8, 0xB9, 0x8B, 0xE9 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000042, 0x000017FD, { { 0x0A, 0x8A, 0x7E, 0x9A, 0x5F, 0x4A, 0x35, 0x06, 0x4D, 0x6B, 0xBF, 0x29, 0x1B, 0xAD, 0xD8, 0x37 } } } },

	{ EN_ANY, kPlatformUnknown, { 0x0000003E, 0x0000150D, { { 0xA8, 0xBF, 0x99, 0x9B, 0xC1, 0x36, 0x21, 0x47, 0x6D, 0x99, 0x4F, 0x34, 0xE6, 0x61, 0x47, 0xFD } } } }, // Amiga + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1BlackJewelStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000023, 0x00000B73, { { 0xBE, 0xE9, 0xBD, 0x65, 0xB9, 0xB1, 0xCD, 0xF0, 0x45, 0x03, 0x01, 0x70, 0xBE, 0xD0, 0x0F, 0x80 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x00000035, 0x0000205F, { { 0x33, 0xD6, 0x19, 0xEE, 0xEA, 0xDD, 0x9F, 0xB5, 0x77, 0x0C, 0x8E, 0x84, 0x5E, 0x61, 0x22, 0x3E } } } },

	{ FR_FRA, kPlatformPC, { 0x0000002B, 0x00000F11, { { 0x68, 0xA0, 0xA3, 0xA0, 0xB6, 0xD1, 0x40, 0x22, 0xB4, 0x10, 0x3D, 0x5A, 0x7F, 0x1F, 0x4B, 0xE7 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000031, 0x0000113B, { { 0xA5, 0x00, 0xD1, 0xDC, 0x25, 0x99, 0x3E, 0x6B, 0xFF, 0xC8, 0x6C, 0xDA, 0xA4, 0xCF, 0x51, 0xD7 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000022, 0x00000B6E, { { 0xAE, 0xA0, 0xDF, 0x07, 0xDD, 0x88, 0x98, 0x6B, 0xDD, 0x9E, 0x2E, 0xC0, 0xBD, 0x45, 0x2E, 0x2B } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000002B, 0x00000D6D, { { 0x08, 0x5E, 0x1F, 0xA8, 0x8F, 0x1F, 0x6A, 0xE0, 0xF3, 0x49, 0xEF, 0x44, 0xA8, 0xCD, 0xDE, 0x45 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000025, 0x00000CF6, { { 0x4B, 0x13, 0x39, 0xCB, 0x3F, 0x44, 0x18, 0x46, 0x43, 0xDB, 0x94, 0xC5, 0x3E, 0x6B, 0xC4, 0x74 } } } },

	{ EN_ANY, kPlatformUnknown, { 0x00000024, 0x00000B73, { { 0x8D, 0x57, 0x5F, 0x93, 0x85, 0x75, 0xF2, 0xD8, 0x36, 0xC2, 0x7C, 0x0E, 0x3B, 0xEA, 0xE0, 0x0A } } } }, // Amiga + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PoisonGoneStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000002D, 0x00000F59, { { 0x9A, 0x88, 0x07, 0x5A, 0x9C, 0xC3, 0x7B, 0xB0, 0x79, 0x69, 0xE0, 0x84, 0x11, 0x9B, 0x82, 0xCD } } } },

	{ FR_FRA, kPlatformPC, { 0x00000059, 0x00001E91, { { 0xDA, 0x9C, 0xC4, 0x7A, 0x79, 0x73, 0x60, 0xBF, 0x31, 0xF8, 0x8F, 0xAF, 0xC1, 0x4D, 0x25, 0xEC } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x0000002A, 0x00000EAA, { { 0x0B, 0x4E, 0xDF, 0x03, 0x9E, 0x64, 0x3B, 0x13, 0x10, 0x6A, 0xF9, 0x4C, 0x66, 0x0D, 0x09, 0xC5 } } } }, // (fan) CD

	{ DE_DEU, kPlatformPC, { 0x00000036, 0x00001324, { { 0x30, 0x67, 0xEE, 0x7C, 0x4D, 0x36, 0xFE, 0x46, 0x05, 0x44, 0x23, 0xA1, 0xB4, 0x07, 0x16, 0xB5 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000034, 0x0000128D, { { 0x4E, 0x54, 0xFE, 0x58, 0x7F, 0x15, 0xFA, 0xC1, 0x50, 0xF8, 0x47, 0x29, 0x15, 0xE8, 0x83, 0xC7 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000002A, 0x00000DFD, { { 0x57, 0xDC, 0xE0, 0x45, 0x2E, 0x89, 0x77, 0x7F, 0x4B, 0x41, 0xCF, 0x4A, 0xFE, 0x0B, 0xAF, 0xC1 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000033, 0x0000127E, { { 0x67, 0xEB, 0xD3, 0x00, 0xF8, 0x4F, 0xF1, 0x79, 0x48, 0xE6, 0x9C, 0xB2, 0xA7, 0xCF, 0x76, 0x07 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x0000002E, 0x00000F59, { { 0xAD, 0x95, 0xF3, 0xA7, 0xBB, 0x04, 0x08, 0x77, 0xD0, 0x71, 0xFC, 0x8B, 0x33, 0x2A, 0x6D, 0xD3 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000037, 0x00001324, { { 0xB3, 0xE6, 0x0A, 0x49, 0x37, 0x73, 0x3C, 0xAF, 0x78, 0x9E, 0x7D, 0x13, 0x75, 0xAE, 0xA8, 0x89 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000003C, 0x000013B4, { { 0xE9, 0x54, 0xF1, 0xEC, 0x64, 0x5A, 0xEB, 0xC7, 0xAA, 0x7E, 0xA2, 0x9F, 0xE8, 0xFB, 0x3A, 0x82 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000003D, 0x000017D1, { { 0x7C, 0x93, 0x62, 0x32, 0x47, 0xB9, 0x78, 0x03, 0x86, 0x8D, 0x99, 0xB8, 0x86, 0x68, 0xEF, 0x3A } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000016, 0x00000B60, { { 0xFF, 0x8C, 0xC0, 0x1B, 0x2C, 0x0C, 0x23, 0x89, 0x93, 0x79, 0xC5, 0xCD, 0xAC, 0xF9, 0xEA, 0xA7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1HealingTipStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000002D, 0x00000F04, { { 0xC0, 0xC7, 0x0C, 0x3D, 0xA7, 0x62, 0x14, 0xFB, 0xE8, 0x52, 0x05, 0x0D, 0xFE, 0xF6, 0xC7, 0x28 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000002B, 0x00001949, { { 0xC7, 0xE3, 0x0A, 0x6B, 0x8F, 0xCA, 0xBC, 0x3A, 0xDC, 0x76, 0x48, 0xD3, 0x8B, 0xD9, 0x44, 0x2E } } } },

	{ FR_FRA, kPlatformPC, { 0x00000036, 0x0000132C, { { 0x86, 0x65, 0x80, 0x10, 0x40, 0x32, 0x6C, 0x2F, 0x5E, 0x76, 0xFB, 0xDC, 0x19, 0x33, 0x02, 0x3C } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000003C, 0x00001537, { { 0x78, 0xC2, 0x76, 0x24, 0x0D, 0x1F, 0xC0, 0x13, 0x4E, 0x34, 0x0A, 0x50, 0x4C, 0x35, 0x7E, 0xB1 } } } },

	{ IT_ITA, kPlatformPC, { 0x0000003F, 0x0000170E, { { 0x0B, 0x37, 0xA4, 0x5E, 0x05, 0x4A, 0x96, 0x1F, 0x2E, 0x02, 0x43, 0xBE, 0xCC, 0xF9, 0x21, 0x5E } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000036, 0x00001364, { { 0x59, 0x70, 0x13, 0x8D, 0x93, 0x9C, 0x1B, 0x69, 0x2F, 0x13, 0x9A, 0xB2, 0x4C, 0x97, 0x7E, 0x95 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000028, 0x00000E0F, { { 0x3E, 0x40, 0xCA, 0x2A, 0x5F, 0xFE, 0x74, 0x30, 0x8C, 0x31, 0x41, 0x09, 0xBD, 0xFD, 0xA3, 0x7E } } } },

	{ EN_ANY, kPlatformUnknown, { 0x0000002E, 0x00000F04, { { 0x95, 0x39, 0x36, 0x89, 0xC4, 0x60, 0x7C, 0x0C, 0xDC, 0x06, 0xF7, 0x86, 0x1A, 0xF7, 0x93, 0x2B } } } }, // Amiga + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1WispJewelStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000053, 0x00001C62, { { 0x9A, 0xF1, 0xCC, 0xB3, 0x09, 0xF0, 0x9C, 0x33, 0x0E, 0xF0, 0xE3, 0xF8, 0x24, 0x63, 0x40, 0x95 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000052, 0x00001CE8, { { 0x6D, 0x69, 0x6D, 0x15, 0x4F, 0xF1, 0xD8, 0xA6, 0x53, 0x9F, 0xE3, 0x77, 0x75, 0x25, 0xE7, 0x93 } } } }, // CD

	{ FR_FRA, kPlatformPC, { 0x0000005B, 0x00001E9A, { { 0xC1, 0x26, 0xF3, 0x64, 0x18, 0x7D, 0xF7, 0x5A, 0xB6, 0x2B, 0x44, 0x57, 0xDC, 0x60, 0x20, 0x3C } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x00000050, 0x00001B8D, { { 0x9A, 0xCE, 0x1D, 0x62, 0xFA, 0xC6, 0x73, 0x80, 0x02, 0x0A, 0x61, 0x24, 0xDC, 0x2A, 0xEF, 0xE6 } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x00000061, 0x00002184, { { 0x4B, 0x20, 0xD7, 0xE6, 0x00, 0xBF, 0x32, 0x40, 0xB7, 0xD0, 0xD6, 0xEC, 0x9C, 0x0F, 0x0B, 0xF0 } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000058, 0x00001E88, { { 0x0E, 0x14, 0xD1, 0xF3, 0x03, 0x29, 0x48, 0x1D, 0x32, 0x2A, 0xBB, 0x0B, 0x48, 0xB3, 0x2E, 0x47 } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x00000074, 0x00002945, { { 0x17, 0x8B, 0x7E, 0xD9, 0x04, 0x7C, 0xD4, 0x9D, 0xCE, 0xF0, 0x79, 0x00, 0xAC, 0x82, 0x7C, 0x82 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000050, 0x00001A10, { { 0xFB, 0x79, 0x74, 0x14, 0xB9, 0x75, 0x4F, 0x44, 0x83, 0xB1, 0xD4, 0xBE, 0x17, 0x0F, 0xC4, 0xD6 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000005F, 0x0000211E, { { 0xE7, 0x0A, 0x85, 0x25, 0x44, 0x41, 0x47, 0x3B, 0x7A, 0xA6, 0x62, 0xAE, 0xAE, 0xD5, 0x92, 0x45 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000056, 0x00001C62, { { 0x43, 0x28, 0x3C, 0x0F, 0x78, 0x52, 0xE7, 0x2A, 0x77, 0xF3, 0x21, 0x5A, 0xF0, 0xFC, 0x9E, 0xF8 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000063, 0x00002184, { { 0x6B, 0xDC, 0x6B, 0xCF, 0xD4, 0xC7, 0x2A, 0x9A, 0x2E, 0x34, 0x71, 0x4E, 0xB7, 0xF6, 0xAF, 0xDA } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x00000075, 0x00002B94, { { 0xA5, 0x4C, 0xBF, 0x5F, 0xD3, 0xF8, 0x6D, 0xE4, 0xB4, 0x8E, 0x64, 0x84, 0xD9, 0xBA, 0x2B, 0x33 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000067, 0x00002C6B, { { 0x55, 0x56, 0x6B, 0xEA, 0xEF, 0x34, 0xDA, 0x23, 0x23, 0xF4, 0x9D, 0x0F, 0xBF, 0xC3, 0x1F, 0xDA } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000037, 0x00001EDF, { { 0xA8, 0x42, 0x79, 0xEB, 0x82, 0xAF, 0x03, 0xD7, 0x03, 0xF9, 0xE0, 0x60, 0xD4, 0x87, 0xAD, 0x2E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1MagicJewelStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000013, 0x0000069E, { { 0x7C, 0xF2, 0xEE, 0x7C, 0x8A, 0xCE, 0x5B, 0x09, 0x32, 0xDC, 0x31, 0x1F, 0x45, 0x21, 0x84, 0xFF } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000000F, 0x0000087E, { { 0xD8, 0xF1, 0x40, 0x9D, 0x9C, 0x15, 0x9E, 0xBD, 0x69, 0xE5, 0xE1, 0x51, 0x34, 0x22, 0xF3, 0x75 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000011, 0x000005DB, { { 0x44, 0x99, 0x35, 0x77, 0x9B, 0x3D, 0xF8, 0xAF, 0x7A, 0xA6, 0x14, 0xD0, 0x99, 0xC6, 0x80, 0x7B } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000014, 0x000006EF, { { 0xDE, 0x77, 0xB6, 0x8C, 0x39, 0x5C, 0x6F, 0xDF, 0x7D, 0x04, 0x0F, 0x65, 0x24, 0x75, 0xED, 0x95 } } } },

	{ IT_ITA, kPlatformPC, { 0x0000001D, 0x00000A83, { { 0x60, 0x69, 0x18, 0x6C, 0x34, 0xB0, 0x56, 0xE3, 0x06, 0x7C, 0xB1, 0xDD, 0x26, 0x80, 0xAE, 0x30 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000011, 0x000005E4, { { 0xD5, 0x11, 0xDA, 0x40, 0x09, 0x6A, 0x5D, 0x76, 0x5C, 0x20, 0xC4, 0x43, 0x4D, 0xF3, 0x67, 0xF0 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000011, 0x000005CD, { { 0x32, 0x2A, 0xFF, 0x9F, 0x10, 0x75, 0x6B, 0xD6, 0x46, 0xAE, 0x55, 0xD3, 0x68, 0x4F, 0xBB, 0x5A } } } },

	{ EN_ANY, kPlatformUnknown, { 0x00000014, 0x0000069E, { { 0x6A, 0x1C, 0x9B, 0x85, 0x61, 0xC7, 0x28, 0xA9, 0xA3, 0xF4, 0xFA, 0x47, 0x90, 0x8F, 0x06, 0xB4 } } } }, // Amiga + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1ThePoisonStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000057, 0x00001C24, { { 0xBC, 0x31, 0x5C, 0x25, 0x50, 0x36, 0x58, 0x20, 0x55, 0xFC, 0x75, 0x47, 0x2D, 0x43, 0x73, 0x78 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000068, 0x000022D8, { { 0x39, 0xDA, 0xB3, 0xD2, 0xDA, 0x3F, 0xAB, 0x40, 0x1A, 0x4F, 0x9D, 0x02, 0xBA, 0x37, 0xEC, 0x4D } } } },

	{ DE_DEU, kPlatformPC, { 0x00000072, 0x00002690, { { 0xEF, 0x62, 0x85, 0xA3, 0x8B, 0x20, 0x4B, 0x65, 0xF8, 0xE7, 0x2C, 0x02, 0x3F, 0x3F, 0x25, 0x65 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000063, 0x000021F8, { { 0x3C, 0x2E, 0xF8, 0xAD, 0xCF, 0xB5, 0xE7, 0x2C, 0x98, 0x7A, 0xB9, 0x6F, 0x7A, 0x42, 0xE1, 0x92 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000068, 0x00002077, { { 0x5D, 0xF4, 0xF9, 0xD9, 0x87, 0x34, 0x14, 0xAB, 0xE4, 0x9A, 0x67, 0xAF, 0x18, 0x8B, 0x40, 0x98 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000059, 0x00001DF7, { { 0x16, 0x7B, 0x5F, 0x91, 0x06, 0x5B, 0xFC, 0x9C, 0x88, 0x61, 0xCC, 0x1B, 0x52, 0x4F, 0x91, 0xC5 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x00000058, 0x00001C24, { { 0xBA, 0x1F, 0xBD, 0x5C, 0x85, 0x3D, 0x3C, 0x92, 0xD1, 0x13, 0xF3, 0x40, 0x2E, 0xBB, 0x1C, 0xE2 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000073, 0x00002690, { { 0x44, 0xAE, 0xC9, 0xFD, 0x9F, 0x8E, 0x1B, 0xDD, 0x3F, 0xE4, 0x4D, 0x4B, 0x5A, 0x13, 0xE5, 0x99 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x00000092, 0x0000334E, { { 0xC9, 0x02, 0xB5, 0xEF, 0x90, 0xA8, 0x9C, 0x6E, 0xAD, 0xD6, 0x1F, 0xCC, 0x71, 0x9F, 0x19, 0x2C } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x000000A0, 0x00003EEB, { { 0x4B, 0x0C, 0xD5, 0xF0, 0xFE, 0x05, 0xF5, 0x09, 0x96, 0xA9, 0xE4, 0xDD, 0x86, 0x8D, 0x79, 0x3C } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000045, 0x00002350, { { 0xAD, 0x17, 0xA0, 0x40, 0x86, 0x0E, 0x1D, 0x2F, 0xA7, 0x14, 0x5B, 0x81, 0x38, 0x09, 0x80, 0x21 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1FluteStringsProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000003C, 0x00001599, { { 0x96, 0x72, 0x5A, 0x8A, 0xA0, 0xEE, 0xA2, 0xCE, 0x4D, 0x21, 0x01, 0x6C, 0xC5, 0x1A, 0xEB, 0x21 } } } }, // floppy + AMIGA
	{ EN_ANY, kPlatformPC, { 0x0000009C, 0x00002334, { { 0xA8, 0xA3, 0x1F, 0x0D, 0x36, 0x25, 0x19, 0x98, 0x50, 0xA0, 0x13, 0x65, 0xF4, 0xE8, 0x27, 0x19 } } } },      // CD

	{ FR_FRA, kPlatformPC, { 0x00000045, 0x000018EE, { { 0x3E, 0x3B, 0x20, 0x46, 0xEB, 0xB0, 0x88, 0x26, 0xB2, 0x0D, 0xD6, 0x8F, 0xF7, 0x53, 0x74, 0xAE } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000040, 0x000016F2, { { 0x70, 0xFF, 0x86, 0x73, 0x9E, 0x75, 0xDD, 0x17, 0x27, 0x86, 0x44, 0xE9, 0x78, 0x3D, 0x48, 0xDB } } } },

	{ IT_ITA, kPlatformPC, { 0x0000004C, 0x00001BFB, { { 0x06, 0x4F, 0x6A, 0x37, 0x8B, 0x44, 0x27, 0x26, 0xCE, 0x28, 0xB2, 0x47, 0x71, 0xE2, 0x27, 0x73 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000045, 0x00001726, { { 0x5A, 0xCF, 0x64, 0x4B, 0x61, 0x57, 0xEC, 0xB3, 0xBF, 0x32, 0x45, 0x51, 0x5D, 0x02, 0xC5, 0xB6 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000052, 0x00001D8E, { { 0x9D, 0xA5, 0xF1, 0x42, 0xD1, 0x48, 0xEB, 0x8F, 0x4B, 0xDC, 0xD9, 0x10, 0x55, 0xBD, 0x12, 0xBB } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000005A, 0x000024F9, { { 0xCA, 0x1F, 0x62, 0x23, 0x22, 0x25, 0x4A, 0x94, 0x8A, 0x50, 0x59, 0xD5, 0xB4, 0x4E, 0xF1, 0xA6 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000053, 0x00002745, { { 0x7A, 0xBB, 0xFC, 0x30, 0xB6, 0xCE, 0x61, 0xD4, 0xDB, 0xB0, 0xE6, 0xB2, 0xF4, 0x4D, 0x81, 0x35 } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000032, 0x00001CDF, { { 0xB7, 0x08, 0x8D, 0x67, 0xB2, 0x0F, 0x59, 0x2D, 0xA6, 0x36, 0x21, 0x7B, 0x8B, 0x89, 0xE3, 0xED } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1FlaskFullStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x0000001C, 0x000009C8, { { 0xA5, 0xC7, 0x5E, 0x6E, 0x02, 0xEE, 0x8C, 0xF8, 0xF0, 0xBF, 0x69, 0xB8, 0x05, 0x1E, 0xF5, 0x28 } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000001F, 0x00001135, { { 0x90, 0x52, 0x4A, 0x95, 0xE4, 0x89, 0xD6, 0x0A, 0xE1, 0x2E, 0x98, 0x11, 0x02, 0xF5, 0x79, 0x37 } } } },

	{ FR_FRA, kPlatformPC, { 0x0000001E, 0x00000AC7, { { 0x4E, 0x7A, 0xC8, 0xC5, 0xD4, 0xB8, 0xD0, 0x73, 0x66, 0x2B, 0x2A, 0x13, 0xDE, 0xE1, 0x32, 0xAA } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000001E, 0x00000A71, { { 0x11, 0x88, 0x37, 0x8E, 0x17, 0xF2, 0x82, 0x1E, 0x72, 0xF0, 0xA3, 0x2B, 0x4F, 0x76, 0x5F, 0xBA } } } },

	{ IT_ITA, kPlatformPC, { 0x00000021, 0x00000C19, { { 0xC0, 0xFA, 0xD2, 0xB8, 0xCA, 0x94, 0x67, 0x10, 0x65, 0x01, 0x7F, 0xB0, 0x95, 0xC6, 0x0E, 0xFF } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000001E, 0x00000973, { { 0xD3, 0x93, 0x3C, 0xA6, 0xF7, 0x79, 0xE6, 0x05, 0x49, 0x88, 0x89, 0xDE, 0xCA, 0x77, 0x2F, 0x1B } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000001B, 0x0000099D, { { 0x13, 0x23, 0x5D, 0x38, 0x9B, 0xFB, 0x00, 0x5C, 0xA1, 0x3A, 0x22, 0xD6, 0xCD, 0x5C, 0x09, 0xAE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1FullFlaskStringProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x00000097, 0x00003521, { { 0x9C, 0x30, 0xC7, 0xC9, 0x97, 0xF3, 0x0E, 0xC4, 0x6E, 0xCA, 0x63, 0xC6, 0xD8, 0x06, 0xFF, 0x30 } } } },

	{ FR_FRA, kPlatformPC, { 0x00000086, 0x00002E16, { { 0x94, 0x6C, 0x6D, 0xB6, 0x4C, 0x84, 0xBB, 0xF6, 0xC8, 0x50, 0x4C, 0x00, 0x38, 0xE3, 0x09, 0x5B } } } },

	{ DE_DEU, kPlatformPC, { 0x000000AF, 0x00003E38, { { 0x66, 0x86, 0xAF, 0x1A, 0x2D, 0x33, 0x40, 0xE7, 0x96, 0x9B, 0x32, 0x26, 0xFF, 0x9F, 0x79, 0x8B } } } },

	{ IT_ITA, kPlatformPC, { 0x00000086, 0x00002CDB, { { 0xB1, 0xAE, 0x2A, 0x03, 0x28, 0xEC, 0x6D, 0x2F, 0xED, 0x71, 0xBE, 0x60, 0xCF, 0xAB, 0x8C, 0x00 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x000000B5, 0x00004111, { { 0xA3, 0x4D, 0xA0, 0xF5, 0xE5, 0xD7, 0xFE, 0x11, 0xBB, 0x2C, 0xB5, 0xF2, 0xE0, 0x52, 0xF1, 0xF0 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000009A, 0x0000363B, { { 0x38, 0x25, 0xE6, 0xB5, 0xCB, 0x78, 0x5E, 0xAD, 0x2D, 0xD4, 0x2E, 0x8B, 0x89, 0x20, 0xB1, 0x95 } } } },

	{ EN_ANY, kPlatformAmiga, { 0x0000009A, 0x00003521, { { 0x26, 0xE5, 0xC8, 0x6D, 0x14, 0x81, 0x9F, 0x90, 0x38, 0x3C, 0x00, 0x9D, 0x8E, 0x72, 0xB1, 0x83 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x000000B0, 0x00003E38, { { 0x8A, 0x6D, 0x42, 0x36, 0x29, 0x06, 0xB2, 0xCE, 0xA3, 0x41, 0x14, 0xE8, 0xB1, 0xEF, 0x6E, 0x3B } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000009D, 0x00003521, { { 0xAB, 0x74, 0x71, 0x5C, 0x6E, 0x65, 0xF5, 0x5E, 0xB0, 0x6E, 0x0E, 0xC7, 0x56, 0xCB, 0x55, 0x98 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000007D, 0x000039CE, { { 0x9E, 0xBE, 0x13, 0x7F, 0x4D, 0x4B, 0xCB, 0x08, 0xC3, 0xE5, 0xF1, 0xC2, 0xD3, 0x69, 0x9E, 0xDF } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000079, 0x000039CE, { { 0xEF, 0x3F, 0x36, 0xD4, 0x6C, 0x64, 0x74, 0x9A, 0xFE, 0x13, 0xB5, 0x77, 0x9A, 0xCF, 0x15, 0x5C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1VeryCleverStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000032, 0x0000118D, { { 0x4B, 0x6D, 0xD4, 0xDC, 0x3E, 0xA2, 0x2D, 0x6D, 0x2C, 0x5A, 0xF7, 0x67, 0x4B, 0x6D, 0x40, 0xEF } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000002F, 0x00001BDF, { { 0x32, 0x2C, 0x60, 0x1E, 0xB7, 0xE0, 0xFE, 0x5F, 0xEA, 0xEF, 0xA4, 0x73, 0xAC, 0xCB, 0xBA, 0xFE } } } },

	{ FR_FRA, kPlatformPC, { 0x00000027, 0x00000DE1, { { 0x43, 0xC2, 0xC7, 0xD2, 0xA3, 0x70, 0x36, 0x38, 0xB5, 0x5B, 0x22, 0xEF, 0xBA, 0x0D, 0x37, 0x27 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x00000043, 0x00001871, { { 0x33, 0x3F, 0xEA, 0x7F, 0x5F, 0x56, 0x98, 0xF3, 0x03, 0x8D, 0x3E, 0xAC, 0xA1, 0x62, 0x2A, 0xD7 } } } },

	{ IT_ITA, kPlatformPC, { 0x00000038, 0x000013CF, { { 0x58, 0x7B, 0x59, 0x50, 0x84, 0xBB, 0x5B, 0x70, 0xDF, 0x76, 0x2F, 0xC3, 0x7E, 0xEC, 0x6A, 0x8B } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x00000027, 0x00000D01, { { 0x72, 0x7E, 0x88, 0xFB, 0xDA, 0xC0, 0x9E, 0x31, 0xAC, 0x32, 0xFD, 0x90, 0x7D, 0x01, 0x86, 0xD0 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x00000036, 0x000013F8, { { 0x2D, 0x9B, 0x7D, 0x58, 0xD1, 0x94, 0x04, 0x45, 0x6E, 0x81, 0xCC, 0x1E, 0x2F, 0xC5, 0xC9, 0xEA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1NewGameStringProvider[] = {
	{ EN_ANY, kPlatformUnknown, { 0x00000015, 0x0000052B, { { 0xAB, 0xD2, 0x16, 0x26, 0xC2, 0x86, 0xFA, 0xC8, 0x42, 0xCD, 0x16, 0xCD, 0x25, 0xB7, 0x44, 0xDC } } } },

	{ JA_JPN, kPlatformUnknown, { 0x0000001B, 0x00000EC8, { { 0x13, 0x9A, 0xBC, 0x8F, 0xE2, 0x4B, 0xD7, 0x0B, 0xC0, 0x81, 0x60, 0x10, 0xC2, 0xA6, 0x9C, 0xFA } } } },

	{ FR_FRA, kPlatformPC, { 0x0000001D, 0x0000079B, { { 0x2B, 0x5B, 0x54, 0x29, 0x76, 0x31, 0x5E, 0x89, 0xC2, 0xD1, 0xDB, 0x3B, 0xF0, 0xBC, 0xCD, 0x66 } } } },

	{ DE_DEU, kPlatformUnknown, { 0x0000001D, 0x0000076F, { { 0x5C, 0x84, 0x2A, 0xB1, 0x61, 0xDD, 0x7B, 0xB3, 0xD4, 0x6A, 0xD2, 0xEF, 0xA4, 0x5F, 0x81, 0xFE } } } },

	{ IT_ITA, kPlatformPC, { 0x0000001C, 0x0000075E, { { 0xC8, 0xCB, 0x35, 0x5E, 0x73, 0x09, 0xDB, 0xA1, 0xF1, 0x8D, 0x14, 0x54, 0x8D, 0xF7, 0xB1, 0xD3 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000001D, 0x000007BD, { { 0xF4, 0x5C, 0x50, 0x82, 0x91, 0x7E, 0x09, 0x52, 0x44, 0x95, 0xC2, 0x2E, 0xC9, 0x52, 0x12, 0x2B } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000001B, 0x00000701, { { 0x2B, 0x87, 0xC3, 0x82, 0x68, 0xA5, 0xFC, 0xC5, 0x64, 0x9E, 0xAB, 0xD2, 0x8A, 0x07, 0x9C, 0x1E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1DefaultShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000048A, 0x00012C0A, { { 0x90, 0x9E, 0x21, 0x9F, 0x90, 0x16, 0x1D, 0xA5, 0x57, 0xF8, 0x71, 0xE1, 0x15, 0x44, 0x29, 0x73 } } } },

	// TODO: Where did this come from?
	/*{ UNK_LANG, kPlatformUnknown, { 0x000004F3, 0x00014AE4, { { 0x6E, 0x37, 0x81, 0x15, 0x4D, 0xDD, 0xBA, 0xA6, 0xE3, 0x5F, 0xB6, 0xC8, 0x20, 0xE9, 0xE0, 0x63 } } } },*/

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Healing1ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000009A, 0x00002F17, { { 0x92, 0x2C, 0xF1, 0x5C, 0x8B, 0xCE, 0xEB, 0x0B, 0xE4, 0x07, 0x16, 0xAA, 0x25, 0x4F, 0x83, 0xEB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Healing2ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000D2, 0x00004257, { { 0xAC, 0x35, 0xAA, 0x72, 0x50, 0x67, 0x58, 0xE4, 0x70, 0xB0, 0xBB, 0xCA, 0xA7, 0x4C, 0x91, 0x82 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PoisonDeathShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000008C, 0x00002E90, { { 0xBC, 0x44, 0xFB, 0x98, 0xE7, 0x42, 0xF6, 0xC8, 0x87, 0xDD, 0x00, 0x42, 0x85, 0xD8, 0x1E, 0x81 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1FluteShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000FC, 0x00004F0F, { { 0xB1, 0xF0, 0x56, 0x6E, 0xAB, 0x3C, 0x88, 0x13, 0x97, 0x7C, 0xCC, 0x61, 0x85, 0x8A, 0x28, 0x12 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Winter1ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000031, 0x00000E70, { { 0x3A, 0x15, 0x36, 0x51, 0x2F, 0xF8, 0xAF, 0xE6, 0x41, 0x71, 0x8D, 0x44, 0x99, 0x48, 0xB3, 0x4F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Winter2ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000F5, 0x00005192, { { 0xD3, 0x64, 0xAC, 0x64, 0x42, 0x9F, 0xB9, 0xCA, 0x0C, 0xC8, 0x97, 0x67, 0x8C, 0x92, 0xB4, 0x79 } } } },

	{ UNK_LANG, kPlatformUnknown, { 0x000000F5, 0x000029E4, { { 0xF0, 0x7B, 0x05, 0xD8, 0xC1, 0x97, 0x27, 0x17, 0xFA, 0x32, 0xE9, 0x1A, 0x46, 0xA5, 0xEB, 0x84 } } } }, // CD + Amiga

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1Winter3ShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001C, 0x00000A5E, { { 0x84, 0xDD, 0xC0, 0xEA, 0xE3, 0xBE, 0x98, 0x66, 0xE0, 0xA3, 0x4D, 0x00, 0x88, 0x31, 0x61, 0x1F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1DrinkShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000069, 0x00001EDA, { { 0x21, 0xFA, 0x2F, 0x41, 0x8F, 0x70, 0x3B, 0xBB, 0x93, 0x35, 0x49, 0xEC, 0x1C, 0xEE, 0xDD, 0xAC } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x0000006A, 0x00001EDA, { { 0xE2, 0xDA, 0xEC, 0x87, 0x4A, 0x27, 0xD8, 0xC1, 0x57, 0xE7, 0xD8, 0xD7, 0x87, 0x5F, 0x3A, 0xEB } } } },

	{ UNK_LANG, kPlatformPC98, { 0x0000006A, 0x00001EDA, { { 0xE2, 0xDA, 0xEC, 0x87, 0x4A, 0x27, 0xD8, 0xC1, 0x57, 0xE7, 0xD8, 0xD7, 0x87, 0x5F, 0x3A, 0xEB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1WispShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B6, 0x00003A53, { { 0x34, 0x52, 0x9D, 0x06, 0x4F, 0x1B, 0xCD, 0xB7, 0x30, 0xF7, 0xB4, 0x9F, 0x0F, 0x34, 0xDD, 0xF7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1MagicAnimShapesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000023, 0x00000A2D, { { 0x48, 0x8E, 0x2A, 0x61, 0x93, 0xCD, 0x58, 0x3B, 0x45, 0x62, 0xD9, 0x01, 0x8F, 0xBA, 0x2A, 0x1A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1BranStoneShapesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000006E, 0x00001E41, { { 0x77, 0x99, 0x54, 0xED, 0x4C, 0x31, 0x08, 0x50, 0x41, 0x22, 0x6B, 0xED, 0x34, 0xF4, 0x17, 0xA1 } } } },

	{ UNK_LANG, kPlatformAmiga, { 0x0000006E, 0x00002026, { { 0x5A, 0xAF, 0x28, 0xF3, 0x0F, 0x82, 0x05, 0xB4, 0x33, 0x83, 0xCE, 0x56, 0xDB, 0xE7, 0x5C, 0xB4 } } } },

	{ UNK_LANG, kPlatformFMTowns, { 0x00000063, 0x00001E41, { { 0xDA, 0xA1, 0x93, 0x68, 0xBF, 0xB3, 0xBB, 0xD1, 0x2E, 0xF9, 0xB2, 0x25, 0x72, 0xD9, 0xA6, 0x00 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x00000063, 0x00001E41, { { 0xDA, 0xA1, 0x93, 0x68, 0xBF, 0xB3, 0xBB, 0xD1, 0x2E, 0xF9, 0xB2, 0x25, 0x72, 0xD9, 0xA6, 0x00 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x0000024B, { { 0x6C, 0x6C, 0xE9, 0xB6, 0xC4, 0xC2, 0x29, 0xA5, 0xB6, 0xB3, 0x82, 0xB6, 0x03, 0x5B, 0x66, 0x7A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x00000470, { { 0x63, 0x02, 0x97, 0x7D, 0xBE, 0x97, 0x5E, 0x88, 0x2E, 0x50, 0x18, 0x6A, 0x4A, 0x61, 0x09, 0xA6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette3Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x0000042A, { { 0xF8, 0x32, 0x65, 0x55, 0x1C, 0xC5, 0x6E, 0x71, 0xC6, 0x6A, 0x8E, 0xC5, 0x3C, 0xBC, 0x1F, 0xA8 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette4Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x000003A4, { { 0xA1, 0xA6, 0x6F, 0x53, 0xA2, 0x1E, 0x27, 0x6F, 0x0D, 0xD0, 0x4F, 0xDC, 0x4C, 0xDC, 0xAE, 0x17 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette5Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x0000033D, { { 0x22, 0x02, 0xB3, 0xCD, 0x4E, 0xD5, 0x6E, 0x65, 0x2B, 0x38, 0x7D, 0x3E, 0x27, 0x03, 0x1A, 0x59 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette6Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000033, 0x000001EA, { { 0xC7, 0xA7, 0x0E, 0xBA, 0x3F, 0x5E, 0xD3, 0x02, 0x99, 0x3B, 0x32, 0x00, 0x39, 0x3C, 0xD2, 0xAC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette7Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000036, 0x0000040C, { { 0x1A, 0x8E, 0x3E, 0x5F, 0x09, 0xFE, 0x4A, 0xF1, 0x5D, 0x2E, 0x2C, 0xD3, 0xFF, 0x75, 0x7D, 0x1C } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette8Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000036, 0x000008E4, { { 0x55, 0xB4, 0xD3, 0xD1, 0x89, 0x49, 0xA5, 0x7E, 0x40, 0xF9, 0xFE, 0x6E, 0x16, 0xE8, 0xBD, 0xBB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette9Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x000001DB, { { 0x7A, 0x04, 0x16, 0xDD, 0x6A, 0xBA, 0x51, 0x9E, 0x24, 0x9E, 0xEA, 0x76, 0xD0, 0xFA, 0x92, 0xF4 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette10Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x0000017A, { { 0xD2, 0x20, 0x1F, 0x12, 0xE1, 0x65, 0x68, 0xA7, 0x9C, 0x69, 0x8E, 0x99, 0xDF, 0xC5, 0xD2, 0x8E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette11Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x00000136, { { 0xBF, 0x38, 0xB1, 0xFE, 0xBF, 0x25, 0xBC, 0xD3, 0xCC, 0x60, 0x40, 0x97, 0xB6, 0x4A, 0xC7, 0xB2 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette12Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x00000102, { { 0x1C, 0x1E, 0x6A, 0x30, 0xE1, 0xD1, 0xF9, 0x3D, 0x9A, 0xF1, 0x84, 0xBD, 0x44, 0x07, 0x51, 0xEA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette13Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x00000000, { { 0x76, 0x22, 0x21, 0x4B, 0x85, 0x36, 0xAF, 0xE7, 0xB8, 0x9B, 0x1C, 0x66, 0x06, 0x06, 0x9B, 0x0D } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette14Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x000003BF, { { 0x78, 0xE8, 0xA8, 0xF2, 0x4D, 0x54, 0x48, 0x47, 0xC9, 0xC1, 0x5B, 0x6C, 0x69, 0xCB, 0x65, 0xB6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette15Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002D, 0x000002DD, { { 0x9F, 0x66, 0x50, 0xAC, 0xED, 0x0C, 0x63, 0x20, 0xE9, 0x4E, 0x4D, 0x3D, 0xC4, 0xBC, 0x94, 0x8A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette16Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x0000036E, { { 0xA9, 0xDB, 0x1C, 0x35, 0xED, 0xED, 0xE4, 0x40, 0x68, 0x29, 0xB1, 0x6E, 0x60, 0xE4, 0xB0, 0xDB } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette17Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000021, 0x000004D7, { { 0xC0, 0xB5, 0x7B, 0x94, 0xED, 0xD7, 0x48, 0x66, 0x37, 0x54, 0xE8, 0xFC, 0x0F, 0xA5, 0x54, 0x8A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette18Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x000002FD, { { 0x0A, 0xDD, 0x81, 0xA8, 0x89, 0xD2, 0x53, 0x05, 0x25, 0xCC, 0x2B, 0x8B, 0x48, 0x1D, 0xAB, 0x0A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette19Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x00000134, { { 0xCB, 0x0F, 0xA9, 0xD2, 0xD8, 0xEA, 0x2A, 0x1F, 0x16, 0x7B, 0x90, 0xE5, 0x72, 0xB3, 0x52, 0x50 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette20Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x00000396, { { 0x3B, 0xD8, 0xA5, 0xC5, 0x66, 0x3E, 0xC7, 0x88, 0xD0, 0x3B, 0x42, 0x29, 0x84, 0x58, 0xA7, 0x00 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette21Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x00000134, { { 0x96, 0x70, 0x7D, 0x5B, 0xC4, 0xBA, 0x89, 0x89, 0xC9, 0x42, 0xF4, 0x70, 0x33, 0xF5, 0xE8, 0x02 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette22Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x00000376, { { 0xA2, 0x92, 0x28, 0x65, 0xAB, 0xCC, 0x16, 0xE8, 0xA5, 0x72, 0x2F, 0x00, 0x66, 0x8F, 0x8D, 0x96 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette23Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000015, 0x000002C2, { { 0x5C, 0x78, 0x74, 0x81, 0x76, 0xB5, 0x26, 0x8E, 0x20, 0xAE, 0x50, 0xCD, 0x2B, 0x9E, 0x81, 0x8B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette24Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000472, { { 0xC6, 0xF0, 0xA6, 0x33, 0x7E, 0xAE, 0xBC, 0xF9, 0xAE, 0xE2, 0xF4, 0x7B, 0x71, 0x44, 0x50, 0xEF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette25Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x000004CD, { { 0x57, 0x03, 0xB5, 0x27, 0x64, 0x48, 0xBF, 0x6A, 0x0F, 0xC2, 0xC0, 0x4A, 0x73, 0x9A, 0xFE, 0xB3 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette26Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x000004CA, { { 0xF8, 0x39, 0x8D, 0x6F, 0x8D, 0x9C, 0x6E, 0x0A, 0x33, 0xE8, 0xF2, 0xA6, 0xF6, 0x40, 0x70, 0xE5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette27Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x000004D9, { { 0xF6, 0xD4, 0x88, 0x57, 0x6F, 0x1A, 0xA3, 0x1F, 0x6B, 0x17, 0x4D, 0xA0, 0x9E, 0xD8, 0x09, 0x66 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette28Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x0000051D, { { 0x59, 0x02, 0xE3, 0xC6, 0x28, 0xF6, 0xBC, 0xD4, 0xC8, 0x7E, 0x58, 0x6F, 0xE4, 0xA2, 0x28, 0x4F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette29Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000534, { { 0xC2, 0x26, 0x94, 0x47, 0x28, 0x11, 0xAA, 0xE6, 0x10, 0xA2, 0xD4, 0xD9, 0x61, 0x57, 0x62, 0x22 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette30Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000453, { { 0xE1, 0x30, 0xC2, 0xCD, 0x79, 0xD8, 0x36, 0x8D, 0xAC, 0xF7, 0x87, 0x05, 0x49, 0xD7, 0x7A, 0x5F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette31Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000027, 0x000002FB, { { 0xA6, 0x65, 0xE1, 0xE1, 0xCC, 0x71, 0x83, 0x63, 0x9A, 0x04, 0xE2, 0x00, 0xBF, 0x90, 0xDF, 0x95 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette32Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000027, 0x0000016C, { { 0xAA, 0x88, 0x97, 0xE3, 0xCA, 0x06, 0x11, 0x07, 0x8F, 0xFA, 0x75, 0x26, 0xA8, 0x7A, 0x49, 0x28 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1SpecialPalette33Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x0000094C, { { 0x2E, 0x5E, 0x34, 0x3E, 0x99, 0x61, 0x6F, 0x88, 0x91, 0xA7, 0x75, 0xAA, 0x42, 0x4B, 0x1F, 0x5D } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1GUIStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000001B0, 0x00009562, { { 0x78, 0xB0, 0x81, 0x15, 0x96, 0x09, 0x1E, 0x6C, 0x20, 0x7B, 0xF7, 0xB1, 0x29, 0x2A, 0x3D, 0xE4 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000209, 0x00009788, { { 0xDD, 0xA0, 0x5C, 0x30, 0x60, 0x68, 0xDC, 0x24, 0xBE, 0x3F, 0xA2, 0x31, 0xE0, 0x81, 0x6C, 0xCA } } } }, // CD

	{ FR_FRA, kPlatformPC, { 0x00000234, 0x0000C262, { { 0x08, 0x83, 0x32, 0x4E, 0x48, 0xEC, 0x83, 0x4B, 0x1B, 0x82, 0x58, 0x48, 0xA3, 0x30, 0x05, 0x33 } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x0000025D, 0x0000CBDC, { { 0x36, 0xF0, 0x92, 0x9C, 0x5F, 0xFF, 0x5D, 0x65, 0x5F, 0x0B, 0xC0, 0x72, 0x9F, 0x30, 0x47, 0xAB } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x00000236, 0x0000BA9C, { { 0xEB, 0xA6, 0x90, 0x9D, 0x99, 0x4B, 0x4F, 0xD9, 0xCE, 0xFF, 0x44, 0x3F, 0x77, 0x6F, 0xE1, 0x71 } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000261, 0x0000C538, { { 0xF3, 0xED, 0x5E, 0x73, 0x6F, 0x6E, 0x06, 0xF8, 0x48, 0xF8, 0x30, 0x69, 0x34, 0x5D, 0x99, 0x0C } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x00000225, 0x0000B0E1, { { 0x57, 0xE2, 0x39, 0xA8, 0xF5, 0x70, 0x48, 0xB4, 0x3D, 0xD4, 0x2A, 0x8C, 0x7C, 0x76, 0xB7, 0x8D } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000025D, 0x0000AFD0, { { 0x8E, 0x97, 0xA9, 0x55, 0x16, 0xF7, 0x42, 0x83, 0xA3, 0x68, 0x2F, 0xD6, 0x37, 0x1C, 0x9A, 0xD1 } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000023A, 0x0000C3BD, { { 0xED, 0x0D, 0xE7, 0x5B, 0xDC, 0x21, 0x41, 0x54, 0x68, 0x7D, 0x8E, 0x97, 0x1A, 0xB1, 0xA1, 0x4A } } } }, // floppy

	{ EN_ANY, kPlatformAmiga, { 0x000001DF, 0x00009042, { { 0x0D, 0xD3, 0x1A, 0x92, 0x8D, 0x9C, 0x72, 0x55, 0xEF, 0xFB, 0x81, 0x21, 0x3B, 0x43, 0xA7, 0xE8 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000237, 0x0000BAF7, { { 0xD7, 0x1A, 0x8E, 0xCC, 0x6D, 0x3E, 0xA9, 0xDD, 0x9A, 0x6B, 0x71, 0xFE, 0xD4, 0x50, 0x30, 0x6E } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x000001FC, 0x00009D9A, { { 0xE8, 0x39, 0x46, 0x0A, 0x41, 0xC3, 0xDC, 0x33, 0x6E, 0xC5, 0x5D, 0x99, 0xCC, 0xF6, 0xDD, 0xCC } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000025F, 0x0000E56A, { { 0x77, 0x9A, 0x02, 0x52, 0xED, 0x62, 0x23, 0x5A, 0x58, 0x78, 0x19, 0x52, 0x03, 0xD7, 0x88, 0x40 } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000213, 0x0000EC5A, { { 0x3B, 0xBA, 0xFB, 0x61, 0x34, 0x60, 0x25, 0xDA, 0x48, 0xBF, 0x95, 0x7F, 0x7F, 0xB6, 0xE4, 0x55 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1ConfigStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000003F, 0x000016E9, { { 0x4F, 0x19, 0x60, 0x67, 0xA8, 0x31, 0x0B, 0xD5, 0x3D, 0x06, 0x39, 0xF1, 0x42, 0xB0, 0xFD, 0x5C } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000061, 0x00002249, { { 0x5D, 0xE2, 0x05, 0xA7, 0xEC, 0x0A, 0x73, 0xF5, 0x01, 0x5B, 0x64, 0x71, 0x83, 0xEC, 0x56, 0x24 } } } }, // CD

	{ FR_FRA, kPlatformPC, { 0x0000004B, 0x00001AE8, { { 0xE3, 0xC9, 0x0F, 0x54, 0x48, 0x31, 0x6E, 0x5F, 0x51, 0x8E, 0xF8, 0xE6, 0xAC, 0x16, 0xAC, 0x1C } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x0000007A, 0x00002C37, { { 0x9E, 0xD8, 0x85, 0xB4, 0x9D, 0x32, 0x9D, 0x80, 0x58, 0xE4, 0x08, 0xC7, 0x75, 0xD2, 0x9C, 0x1E } } } }, // CD

	{ DE_DEU, kPlatformPC, { 0x0000005E, 0x00002280, { { 0xC7, 0xD2, 0x22, 0xB5, 0xD3, 0xBC, 0x7A, 0x97, 0x26, 0x04, 0x07, 0x78, 0xAA, 0xAB, 0x1D, 0x5C } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000082, 0x00002E94, { { 0xC5, 0xDC, 0x44, 0x6C, 0x4B, 0x3E, 0x4E, 0x27, 0x0C, 0xCC, 0x65, 0x6C, 0x20, 0x8D, 0x71, 0x07 } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x00000054, 0x00001E98, { { 0x10, 0x6B, 0x6B, 0x9B, 0x91, 0x05, 0xE9, 0x30, 0xE2, 0x75, 0xE2, 0x45, 0x79, 0x17, 0x73, 0xC9 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x0000007A, 0x00002904, { { 0xD3, 0xD2, 0x96, 0x6E, 0xE7, 0xE8, 0x64, 0x77, 0x73, 0xCD, 0xC2, 0x9E, 0x57, 0xB5, 0xD3, 0xCD } } } }, // (fan) CD

	{ ES_ESP, kPlatformPC, { 0x0000004A, 0x00001B7B, { { 0x6B, 0x69, 0x50, 0x92, 0x9B, 0x35, 0x58, 0xE1, 0xEA, 0xBF, 0x42, 0x0B, 0xEB, 0x88, 0x41, 0x8D } } } }, // floppy

	{ EN_ANY, kPlatformAmiga, { 0x0000002E, 0x00000FA1, { { 0x5E, 0xFF, 0xFF, 0x3D, 0xF8, 0x11, 0x6F, 0x3B, 0xC5, 0x39, 0x8F, 0x25, 0x8F, 0x0F, 0xE9, 0x2B } } } },
	{ DE_DEU, kPlatformAmiga, { 0x00000043, 0x00001783, { { 0xB2, 0x2B, 0xAB, 0x27, 0x06, 0x9A, 0x1E, 0x4B, 0xA7, 0xD3, 0xFF, 0xEB, 0xFD, 0x12, 0xDC, 0x94 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000004A, 0x000016E9, { { 0x37, 0x12, 0xAD, 0x4C, 0xB8, 0x2C, 0xF5, 0x75, 0x20, 0x74, 0xA0, 0xFD, 0x58, 0xB1, 0x71, 0x77 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x00000057, 0x00002262, { { 0xC5, 0x9A, 0x4A, 0xE3, 0xDF, 0xD3, 0x0E, 0x8E, 0xBB, 0xAF, 0x75, 0xD3, 0x67, 0xB1, 0xF6, 0x08 } } } },

	{ JA_JPN, kPlatformPC98, { 0x0000003F, 0x00001E9D, { { 0x8A, 0x3C, 0x78, 0x6B, 0x3F, 0x0A, 0x7D, 0x92, 0x16, 0x39, 0xEB, 0x1C, 0x84, 0x9E, 0x4F, 0x84 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TownsMusicFadeTableProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000B10, 0x000076DE, { { 0x9F, 0x08, 0x5B, 0xD6, 0x25, 0x7F, 0x11, 0x08, 0x87, 0x45, 0x92, 0xD3, 0xE5, 0xA8, 0x7C, 0x2F } } } },

	EXTRACT_END_ENTRY
};
	
const ExtractEntrySearchData k1TownsSFXwdTableProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00012608, 0x006717A1, { { 0x34, 0xDD, 0x2D, 0xA5, 0x14, 0x05, 0xEE, 0x2F, 0x93, 0x7C, 0x78, 0x4D, 0xCA, 0x13, 0xED, 0x93 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TownsSFXbtTableProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000100, 0x00007FFE, { { 0xEF, 0x3D, 0x9F, 0x31, 0xE7, 0x19, 0x1E, 0x4F, 0xF3, 0x66, 0x42, 0xC8, 0x0D, 0x89, 0x96, 0x20 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1TownsCDATableProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x000002C4, 0x0000C5ED, { { 0x76, 0x86, 0x77, 0xF9, 0x85, 0x23, 0x32, 0x56, 0x3C, 0x06, 0x03, 0xCF, 0x0C, 0x49, 0x94, 0xB5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PC98StoryStringsProvider[] = {
	{ JA_JPN, kPlatformPC98, { 0x00000054, 0x00002FAD, { { 0xCB, 0x25, 0xD1, 0xFA, 0x65, 0xCE, 0xD5, 0xE6, 0xAD, 0xE3, 0x48, 0x25, 0xC8, 0x51, 0xC6, 0x12 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1PC98IntroSfxProvider[] = {
	{ UNK_LANG, kPlatformPC98, { 0x00000300, 0x00017445, { { 0xC3, 0x54, 0x4A, 0x58, 0x00, 0x33, 0xEA, 0xF1, 0x20, 0xA8, 0x0C, 0xDE, 0x7A, 0x94, 0xAB, 0x5E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1CreditsStringsProvider[] = {
	{ EN_ANY, kPlatformAmiga, { 0x000004CC, 0x000182F7, { { 0x7D, 0x14, 0xA6, 0xEC, 0x12, 0x67, 0xAC, 0xA7, 0x7B, 0xF8, 0x44, 0x19, 0x52, 0xCC, 0x3A, 0x65 } } } },
	{ DE_DEU, kPlatformAmiga, { 0x000004FB, 0x00019987, { { 0x8B, 0xF1, 0x61, 0xB9, 0x3E, 0xA4, 0x28, 0x9B, 0x45, 0x86, 0xD1, 0xF0, 0xC3, 0x35, 0x44, 0xA0 } } } },

	{ EN_ANY, kPlatformFMTowns, { 0x0000058C, 0x0001AD16, { { 0xFC, 0x1B, 0xE6, 0xEB, 0x03, 0x1C, 0x54, 0x93, 0x64, 0x53, 0xB1, 0x17, 0x9D, 0x95, 0x1B, 0x4A } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x0000058C, 0x0001AD16, { { 0xFC, 0x1B, 0xE6, 0xEB, 0x03, 0x1C, 0x54, 0x93, 0x64, 0x53, 0xB1, 0x17, 0x9D, 0x95, 0x1B, 0x4A } } } },

	{ JA_JPN, kPlatformPC98, { 0x00000530, 0x00019571, { { 0x38, 0x4F, 0xE0, 0x74, 0x27, 0x21, 0xEC, 0x63, 0x41, 0x32, 0xB1, 0xD9, 0xA5, 0x65, 0x62, 0x71 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AmigaIntroSFXTableProvider[] = {
	{ UNK_LANG, kPlatformAmiga, { 0x00000140, 0x00000D82, { { 0x61, 0xF6, 0x8F, 0xE5, 0xCD, 0x15, 0x58, 0x9F, 0xDF, 0x0E, 0x22, 0xAA, 0x9A, 0xCF, 0xC4, 0xA6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k1AmigaGameSFXTableProvider[] = {
	{ UNK_LANG, kPlatformAmiga, { 0x000003C0, 0x00005505, { { 0x8D, 0x6D, 0x2C, 0x58, 0x50, 0xF7, 0xBF, 0x60, 0xC3, 0x15, 0x20, 0x23, 0x1F, 0x90, 0x3E, 0xD6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayPakFilesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000022, 0x000008E0, { { 0xB7, 0x3B, 0x51, 0x46, 0x78, 0x2D, 0x4D, 0x82, 0xD4, 0x30, 0xFC, 0x6E, 0xC4, 0x5B, 0x27, 0x3E } } } }, // DOS floppy + PC98

	{ UNK_LANG, kPlatformPC, { 0x00000046, 0x0000121A, { { 0x42, 0xC2, 0x5A, 0xDC, 0x27, 0x2D, 0xB4, 0x44, 0x85, 0x58, 0x0F, 0xB6, 0x6D, 0x76, 0x04, 0x4F } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000014, 0x000004FA, { { 0xBB, 0x4C, 0x16, 0xEB, 0x56, 0xEF, 0xAC, 0x68, 0x6B, 0x49, 0x96, 0x3A, 0x9F, 0x00, 0x75, 0xF6 } } } }, // demo
	{ UNK_LANG, kPlatformPC, { 0x0000000C, 0x00000308, { { 0xC0, 0xAC, 0x10, 0xF1, 0x12, 0xD1, 0x21, 0x92, 0xA1, 0x62, 0x5B, 0x6E, 0xCF, 0x88, 0x32, 0x7C } } } }, // Lands of Lore demo

	{ UNK_LANG, kPlatformFMTowns, { 0x00000033, 0x00000BBD, { { 0x92, 0x5B, 0xDA, 0xE3, 0x7C, 0x39, 0xC4, 0x75, 0x24, 0xBA, 0x67, 0x27, 0x6E, 0x21, 0xA3, 0xF7 } } } },

	// TODO: Where did this come from?
	//{ UNK_LANG, kPlatformUnknown, { 0x0000002E, 0x00000C09, { { 0x6A, 0x25, 0xF6, 0x03, 0xA6, 0x88, 0xA0, 0x53, 0xE1, 0xF4, 0xB9, 0x0C, 0x71, 0x17, 0x36, 0x6B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000008C8, 0x0002FDE3, { { 0x62, 0xD1, 0x6F, 0xBC, 0xEC, 0xE6, 0xCF, 0xE8, 0xD8, 0xE9, 0xDE, 0xFB, 0x09, 0xAF, 0x34, 0x92 } } } }, // floppy
	{ EN_ANY, kPlatformPC, { 0x00000916, 0x00031417, { { 0x3E, 0x15, 0xDA, 0xF4, 0x77, 0x44, 0x80, 0x47, 0xDB, 0x32, 0x0E, 0x6D, 0xCA, 0x32, 0x65, 0x1B } } } }, // CD
	{ EN_ANY, kPlatformPC, { 0x00000102, 0x00005BD6, { { 0xB3, 0x00, 0xE5, 0x39, 0x02, 0x3D, 0xBF, 0xDD, 0x54, 0x70, 0xEA, 0xC1, 0xCB, 0xAC, 0xAA, 0xF7 } } } }, // Lands of Lore demo

	{ DE_DEU, kPlatformPC, { 0x000009EC, 0x000370F9, { { 0x56, 0x94, 0xCC, 0xE5, 0x38, 0x19, 0xAD, 0xD1, 0xBB, 0x6A, 0x6E, 0xEC, 0xC5, 0x1B, 0x57, 0x7B } } } }, // floppy
	{ DE_DEU, kPlatformPC, { 0x00000A3F, 0x000389F4, { { 0x51, 0xC1, 0x87, 0x16, 0x53, 0x8C, 0x68, 0x8E, 0x9B, 0x81, 0xA1, 0xD2, 0xCE, 0x5F, 0x83, 0x31 } } } }, // CD

	{ FR_FRA, kPlatformPC, { 0x000009C9, 0x00036324, { { 0x2C, 0xC2, 0xD9, 0xCF, 0x90, 0x2A, 0xDF, 0xE9, 0x85, 0x9E, 0x5D, 0xBB, 0x1D, 0x9A, 0x14, 0x69 } } } }, // floppy
	{ FR_FRA, kPlatformPC, { 0x00000995, 0x000352D7, { { 0xED, 0x7C, 0x49, 0x7B, 0x1A, 0x2C, 0x73, 0x61, 0x73, 0xAF, 0x16, 0x89, 0x1E, 0x01, 0xE2, 0xAE } } } }, // CD

	{ IT_ITA, kPlatformPC, { 0x00000916, 0x0003188F, { { 0xDC, 0x46, 0x06, 0xE1, 0xB0, 0x66, 0xBC, 0x18, 0x2E, 0x6E, 0x9E, 0xC9, 0xA4, 0x14, 0x8D, 0x08 } } } }, // floppy
	{ IT_ITA, kPlatformPC, { 0x000008C8, 0x00030947, { { 0x7F, 0x75, 0x5F, 0x99, 0x94, 0xFE, 0xA1, 0xE6, 0xEF, 0xB8, 0x93, 0x71, 0x83, 0x1B, 0xAC, 0x4A } } } }, // (fan) CD

	{ EN_ANY, kPlatformFMTowns, { 0x00000990, 0x00030C61, { { 0x60, 0x51, 0x11, 0x83, 0x3F, 0x06, 0xC3, 0xA3, 0xE0, 0xC0, 0x2F, 0x41, 0x29, 0xDE, 0x65, 0xB1 } } } },
	{ JA_JPN, kPlatformFMTowns, { 0x000008A8, 0x00036831, { { 0x56, 0x5B, 0x23, 0x61, 0xE8, 0x3B, 0xE1, 0x36, 0xD6, 0x62, 0xD0, 0x84, 0x00, 0x04, 0x05, 0xAD } } } },

	{ EN_ANY, kPlatformPC98, { 0x000008F4, 0x00030C61, { { 0xA4, 0x4C, 0xD9, 0x05, 0x65, 0x24, 0xE5, 0x3E, 0x6C, 0x8B, 0xEE, 0x8D, 0x4F, 0x09, 0x6F, 0x00 } } } },
	{ JA_JPN, kPlatformPC98, { 0x000007F5, 0x00036831, { { 0xF7, 0xCF, 0xF1, 0x2B, 0x26, 0x1E, 0xEC, 0xC8, 0x13, 0xDE, 0xC7, 0x71, 0x72, 0x05, 0x0F, 0xA7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplaySfxFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000173, 0x00008198, { { 0x2D, 0xF5, 0x5C, 0xAE, 0xDB, 0x71, 0x60, 0xFE, 0x78, 0x04, 0xFC, 0xAD, 0x4B, 0x43, 0xB1, 0x4B } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000234, 0x0000B0B2, { { 0xE4, 0x19, 0x4F, 0x8C, 0x53, 0xFD, 0x5F, 0x2A, 0xFB, 0x77, 0x1C, 0x32, 0xB3, 0x39, 0x25, 0x85 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x000000DE, 0x00004EA4, { { 0x97, 0xA8, 0x4A, 0xED, 0x12, 0x87, 0xEB, 0x82, 0x42, 0xFA, 0x45, 0x0D, 0x45, 0x8F, 0xFE, 0x76 } } } }, // demo
	{ UNK_LANG, kPlatformPC, { 0x000000C5, 0x00004508, { { 0xBF, 0xA9, 0x1A, 0x37, 0x0B, 0xF8, 0x13, 0x9D, 0x2D, 0x67, 0x3D, 0x9B, 0xE9, 0x16, 0x07, 0x0C } } } }, // Lands of Lore demo

	{ UNK_LANG, kPlatformFMTowns, { 0x000001A3, 0x00008098, { { 0xA9, 0xD2, 0xE2, 0x32, 0x59, 0xC4, 0x97, 0x74, 0x11, 0x49, 0x8F, 0x2B, 0xBF, 0x00, 0xF9, 0xF1 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x00000173, 0x00008098, { { 0x29, 0xA6, 0xF1, 0x2D, 0x35, 0x38, 0xFD, 0x3B, 0xF0, 0xB1, 0xE1, 0xFE, 0x45, 0xB1, 0x43, 0xAE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayTlkFilesProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x0000009D, 0x0000286B, { { 0x58, 0x30, 0x72, 0x62, 0xC8, 0x77, 0x2A, 0x06, 0xD6, 0x24, 0x1A, 0x7A, 0xAF, 0x79, 0xFF, 0xAE } } } },
	{ FR_FRA, kPlatformPC, { 0x0000009D, 0x00002878, { { 0x28, 0x5D, 0x7F, 0x5B, 0x57, 0xC2, 0xFF, 0x73, 0xC1, 0x8E, 0xD6, 0xE0, 0x4D, 0x03, 0x99, 0x2C } } } },
	{ DE_DEU, kPlatformPC, { 0x0000009D, 0x00002885, { { 0x87, 0x24, 0xB6, 0xE9, 0xD6, 0xAA, 0x68, 0x2D, 0x6B, 0x05, 0xDF, 0xE1, 0x2B, 0xA4, 0x79, 0xE5 } } } },
	{ IT_ITA, kPlatformPC, { 0x0000009D, 0x0000286B, { { 0x58, 0x30, 0x72, 0x62, 0xC8, 0x77, 0x2A, 0x06, 0xD6, 0x24, 0x1A, 0x7A, 0xAF, 0x79, 0xFF, 0xAE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplaySeqDataProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000006F4, 0x0000F5AD, { { 0x23, 0x1E, 0x30, 0xF0, 0xF1, 0x5B, 0xFF, 0x4B, 0xDF, 0xB1, 0x78, 0xD8, 0x48, 0x7A, 0xA3, 0xDD } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x000006F4, 0x0000FA52, { { 0xC8, 0x33, 0xCC, 0x0A, 0x40, 0x8A, 0x96, 0x4F, 0x42, 0x7E, 0x8B, 0x62, 0x9C, 0x2F, 0x33, 0x0B } } } }, // floppy (French)
	{ UNK_LANG, kPlatformPC, { 0x000006F4, 0x0000F8A6, { { 0x8B, 0xF3, 0x80, 0xE1, 0xB8, 0xCA, 0xE1, 0x18, 0x72, 0xA3, 0xF0, 0x4C, 0xBF, 0x81, 0x97, 0x71 } } } }, // floppy (German)
	{ UNK_LANG, kPlatformPC, { 0x000006F4, 0x0000FA35, { { 0x4C, 0x26, 0x1B, 0xF6, 0x8E, 0x73, 0xB7, 0xD2, 0x58, 0x59, 0xB8, 0x97, 0x58, 0xFB, 0x0E, 0x2A } } } }, // floppy (Italian)
	{ UNK_LANG, kPlatformPC, { 0x000006BE, 0x0000E9FF, { { 0x9A, 0xCD, 0x40, 0x44, 0xEF, 0x9F, 0x05, 0x86, 0xDA, 0x49, 0x8A, 0x85, 0x68, 0xDF, 0xA7, 0x55 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x0000027C, 0x00007742, { { 0x42, 0x37, 0xF8, 0xDD, 0xA2, 0x8C, 0xA6, 0x26, 0x89, 0x5F, 0xCF, 0x61, 0xDD, 0x8F, 0xC7, 0x1E } } } }, // demo
	{ UNK_LANG, kPlatformPC, { 0x0000024C, 0x00004C10, { { 0x21, 0x3A, 0x86, 0x60, 0xA8, 0xFF, 0x42, 0x19, 0x35, 0x32, 0xA5, 0xB9, 0x3A, 0xDD, 0xA9, 0xC7 } } } }, // Lands of Lore demo

	{ UNK_LANG, kPlatformFMTowns, { 0x000006BC, 0x0000DCE3, { { 0x27, 0x09, 0x22, 0xDC, 0xED, 0xDE, 0x88, 0xAE, 0xDB, 0x97, 0x52, 0x12, 0x96, 0x8D, 0x8D, 0x37 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x000006BE, 0x0000E043, { { 0x35, 0xA0, 0xF1, 0x14, 0xC7, 0x60, 0x80, 0x35, 0xA2, 0xDB, 0xF2, 0x73, 0x96, 0xC0, 0xAE, 0x84 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayCreditsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000AE8, 0x0002E520, { { 0x19, 0x59, 0xEA, 0x70, 0x23, 0xE2, 0x41, 0x78, 0x2D, 0xF5, 0xE6, 0x28, 0xEA, 0x0C, 0xCF, 0x79 } } } },
	{ IT_ITA, kPlatformPC, { 0x00000AE8, 0x0002EE4A, { { 0x3E, 0x5C, 0x6F, 0x9E, 0x72, 0x2D, 0xAB, 0x17, 0x1D, 0x27, 0xFF, 0x32, 0xD3, 0x48, 0x46, 0xBA } } } }, // (fan) CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayCreditsSpecialProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001A, 0x00000829, { { 0xDF, 0x31, 0x0F, 0xB9, 0xC9, 0xC4, 0xF2, 0xB5, 0xF7, 0x98, 0x6F, 0x44, 0x4E, 0x9C, 0xFE, 0xA7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayIntroTracksProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x00000592, { { 0xDE, 0xD9, 0x5E, 0xCA, 0x5A, 0x9F, 0x8B, 0xB1, 0xE2, 0xDB, 0xA4, 0xA1, 0xAF, 0xD4, 0x75, 0x46 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000008, 0x000002C9, { { 0xFA, 0x80, 0x3C, 0x78, 0x66, 0x2C, 0xD5, 0x91, 0x5D, 0xF1, 0xE9, 0xC1, 0x6A, 0x65, 0xEB, 0xD6 } } } }, // CD
	{ UNK_LANG, kPlatformPC, { 0x00000010, 0x00000542, { { 0xE5, 0xAA, 0x6A, 0xB9, 0x19, 0xAB, 0x35, 0x26, 0x43, 0x70, 0x32, 0xEF, 0xB9, 0x7F, 0x0F, 0x19 } } } }, // demo
	{ UNK_LANG, kPlatformPC, { 0x0000001A, 0x0000096C, { { 0xC2, 0xBD, 0x2C, 0x65, 0xD3, 0xFE, 0xF1, 0x6A, 0xE7, 0x34, 0x18, 0x0C, 0x86, 0x95, 0x66, 0x35 } } } }, // Lands of Lore demo

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayFinaleTracksProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000012, 0x00000618, { { 0x22, 0x61, 0x4D, 0x6F, 0xCE, 0x38, 0x64, 0xAB, 0x20, 0xD8, 0x34, 0x33, 0x44, 0x21, 0xE0, 0x23 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000009, 0x0000030C, { { 0x24, 0x3F, 0x57, 0xFA, 0x7E, 0x6D, 0x61, 0x6C, 0x53, 0xBC, 0x44, 0x38, 0x61, 0xE9, 0x3E, 0xE6 } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayIntroCDAProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000018, 0x00000036, { { 0xCE, 0x10, 0x68, 0x7E, 0x84, 0x68, 0x3A, 0x23, 0x36, 0x41, 0x07, 0xE8, 0x59, 0xDB, 0xA3, 0x31 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayFinaleCDAProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x00000008, 0x000000AE, { { 0x38, 0x47, 0x35, 0xAA, 0xEC, 0xBA, 0xD3, 0xD0, 0x3E, 0x17, 0xFD, 0xB8, 0x9F, 0xB9, 0xAD, 0x61 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2SeqplayShapeAnimDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B8, 0x00000270, { { 0xBD, 0x3B, 0xFA, 0xE6, 0xA0, 0x40, 0x4E, 0x6E, 0x3E, 0x43, 0x81, 0xF3, 0x35, 0x57, 0x62, 0x27 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngamePakFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000049F, 0x00012F40, { { 0x4B, 0x2A, 0x84, 0xE3, 0xA5, 0x34, 0xE9, 0xB3, 0xB7, 0x39, 0xF1, 0x1B, 0x4C, 0xE6, 0x43, 0x79 } } } }, // floppy

	{ UNK_LANG, kPlatformFMTowns, { 0x0000011C, 0x00003FB8, { { 0x66, 0x34, 0xE8, 0x1C, 0xF9, 0xFF, 0x84, 0x90, 0x20, 0x71, 0x42, 0xA3, 0x2C, 0x4A, 0xE9, 0x46 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x000000F4, 0x00003F57, { { 0x8A, 0xA7, 0x2C, 0xB9, 0x6E, 0x8D, 0x95, 0xBF, 0xF8, 0xB9, 0x98, 0x63, 0x20, 0xF8, 0xB0, 0x7B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameSfxFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000006F1, 0x0001545E, { { 0xD3, 0x8A, 0xA1, 0xD4, 0x83, 0x77, 0x96, 0x6D, 0x87, 0xB1, 0x71, 0x8F, 0x38, 0x6A, 0x34, 0xDC } } } },
	{ UNK_LANG, kPlatformFMTowns, { 0x00000967, 0x0002101A, { { 0x09, 0xC7, 0xB7, 0x2A, 0x76, 0xF1, 0x4B, 0x87, 0xC5, 0x83, 0xFF, 0xF3, 0xDB, 0x3C, 0x66, 0x60 } } } },
	{ UNK_LANG, kPlatformPC98, 	{ 0x000006F1, 0x0001545E, { { 0xD3, 0x8A, 0xA1, 0xD4, 0x83, 0x77, 0x96, 0x6D, 0x87, 0xB1, 0x71, 0x8F, 0x38, 0x6A, 0x34, 0xDC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameSfxIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000358, 0x000074F4, { { 0xC8, 0x30, 0x1D, 0x9C, 0xFC, 0xF8, 0xD5, 0xBA, 0x7E, 0xAB, 0x13, 0x3F, 0x43, 0x50, 0xFA, 0xC4 } } } }, // CD

	{ UNK_LANG, kPlatformUnknown, { 0x00000358, 0x0000747E, { { 0x20, 0x99, 0x4D, 0xB8, 0xF2, 0x05, 0xF2, 0xA7, 0x07, 0x28, 0x97, 0xFB, 0x70, 0x1F, 0x2A, 0x73 } } } }, // floppy + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameTracksProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000007E, 0x00001E45, { { 0xC6, 0x52, 0xCC, 0x1A, 0x23, 0x3B, 0xDA, 0x03, 0xF3, 0xC2, 0x46, 0xDF, 0x01, 0xA3, 0x9E, 0x50 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameCDAProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000130, 0x00001B9D, { { 0x9E, 0xFB, 0x01, 0xE7, 0xBA, 0xE1, 0x4F, 0x3C, 0xC5, 0xB9, 0x75, 0x80, 0xEA, 0xAC, 0xB0, 0x0F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameTalkObjIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000078, 0x00000A9C, { { 0x27, 0x65, 0xFA, 0xB1, 0xF8, 0x04, 0x7F, 0x93, 0x2B, 0x2D, 0x44, 0x3C, 0x95, 0x13, 0x50, 0xD2 } } } }, // floppy (English)
	{ UNK_LANG, kPlatformPC, { 0x00000078, 0x00000A53, { { 0x18, 0x5B, 0x79, 0x92, 0x54, 0xDB, 0x4B, 0x0C, 0x78, 0x89, 0x7C, 0x59, 0xBF, 0x53, 0x37, 0x14 } } } }, // floppy (German + French)
	{ UNK_LANG, kPlatformPC, { 0x00000078, 0x00000A7C, { { 0x33, 0x89, 0x69, 0xEF, 0x62, 0x94, 0x3A, 0xC0, 0xB2, 0xBA, 0xFB, 0x0C, 0x86, 0x2C, 0xAC, 0x5C } } } }, // floppy (Italian)
	{ UNK_LANG, kPlatformPC, { 0x00000084, 0x00000CD0, { { 0x82, 0x2E, 0xF4, 0xDD, 0x28, 0x75, 0xDD, 0x79, 0x78, 0x62, 0x55, 0x48, 0xDD, 0xF7, 0x16, 0x81 } } } }, // CD

	{ UNK_LANG, kPlatformFMTowns, { 0x0000007A, 0x00000744, { { 0x74, 0x84, 0xB4, 0xA6, 0x9D, 0x91, 0xBF, 0x24, 0x1E, 0xD5, 0xDE, 0xD4, 0x73, 0x93, 0x1F, 0xE6 } } } },

	{ UNK_LANG, kPlatformPC98, { 0x00000078, 0x0000099D, { { 0x2F, 0x06, 0x8D, 0x4E, 0x76, 0x9E, 0x92, 0xFD, 0x43, 0xA4, 0x12, 0xED, 0xBA, 0x30, 0xFE, 0x94 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameTimJpStringsProvider[] = {
	{ UNK_LANG, kPlatformFMTowns, { 0x0000002D, 0x00001302, { { 0x15, 0x29, 0xFD, 0xB1, 0xEB, 0xC0, 0x00, 0xA4, 0x19, 0xDA, 0xC7, 0x2E, 0x17, 0x0F, 0x3F, 0xB7 } } } },
	{ UNK_LANG, kPlatformPC98, { 0x0000002C, 0x00001302, { { 0xAB, 0xBA, 0xC8, 0x2A, 0xD0, 0xD1, 0x1C, 0xEA, 0x6F, 0x35, 0x12, 0x5D, 0x85, 0xFB, 0x92, 0x84 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameShapeAnimDataProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000348, 0x0001AAEB, { { 0x67, 0xD1, 0x03, 0xBF, 0x4B, 0xEC, 0x80, 0x26, 0x5D, 0x1D, 0x0E, 0x5B, 0xF1, 0xE8, 0x4C, 0x64 } } } }, // CD

	{ UNK_LANG, kPlatformUnknown, { 0x00000348, 0x00002473, { { 0xBC, 0x18, 0x51, 0xFA, 0x51, 0x1D, 0x24, 0x7D, 0x44, 0xD0, 0xE2, 0x38, 0x31, 0x33, 0x3B, 0x56 } } } }, // floppy + FM-TOWNS

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k2IngameTlkDemoStringsProvider[] = {
	{ EN_ANY, kPlatformPC, { 0x000000CB, 0x00004741, { { 0xF3, 0x54, 0x37, 0xEE, 0x61, 0x9E, 0xE0, 0x82, 0x64, 0x8B, 0xF3, 0x53, 0xC7, 0x2F, 0x49, 0x1F } } } },
	{ FR_FRA, kPlatformPC, { 0x000000BE, 0x00004327, { { 0x8C, 0xF2, 0x5E, 0x85, 0xDA, 0xA1, 0x91, 0x90, 0xE5, 0xC7, 0x2A, 0xBA, 0x48, 0xC3, 0x9A, 0xCA } } } },
	{ DE_DEU, kPlatformPC, { 0x000000DC, 0x00004C7B, { { 0xC9, 0x75, 0x3A, 0x4A, 0xF4, 0xB0, 0xE8, 0x61, 0x90, 0x74, 0x34, 0x84, 0x53, 0x54, 0xA2, 0x4F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3MainMenuStringsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000B0, 0x00003E75, { { 0xA7, 0x91, 0x9B, 0x61, 0x31, 0x12, 0x74, 0xB5, 0x4B, 0x4F, 0x8A, 0x44, 0xA9, 0xB1, 0x03, 0xC0 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3MusicFilesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000001D5, 0x00007436, { { 0x27, 0xC2, 0x96, 0xFF, 0x22, 0xEC, 0xB5, 0x26, 0xF9, 0x4F, 0x04, 0xDD, 0x58, 0x17, 0xFE, 0xC6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3ScoreTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000000C8, 0x0000038F, { { 0x4E, 0x2E, 0xC3, 0xD9, 0x13, 0xA0, 0x74, 0x7E, 0xAB, 0xA1, 0x23, 0x2F, 0x11, 0x35, 0x4D, 0xA5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3SfxFilesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000008BF, 0x0001A766, { { 0xBD, 0x5D, 0x65, 0x5D, 0xC4, 0x34, 0xDF, 0x36, 0xF5, 0xC0, 0xE8, 0x7B, 0x51, 0x9F, 0xE5, 0xC7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3SfxMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000004C0, 0x0000C4F9, { { 0x06, 0x61, 0x45, 0x38, 0xC2, 0x6A, 0x12, 0x23, 0x55, 0x2C, 0x7F, 0xA5, 0xE2, 0x79, 0x81, 0x9E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3ItemAnimDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000001C8, 0x00003777, { { 0x8F, 0xE8, 0x80, 0x4A, 0x88, 0x83, 0xC0, 0x48, 0x55, 0x47, 0x81, 0xD3, 0xB7, 0xD5, 0x09, 0x8F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3ItemMagicTableProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00000F9E, { { 0x87, 0xA7, 0xC0, 0xA2, 0x0F, 0xA3, 0x5B, 0x08, 0x76, 0x6F, 0x02, 0x68, 0x13, 0x17, 0x3B, 0x76 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData k3ItemStringMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x0000003C, { { 0x4C, 0xA7, 0xAA, 0xB0, 0x3C, 0xCC, 0x78, 0x91, 0x0A, 0xF2, 0x7F, 0x25, 0x80, 0xB3, 0x74, 0x74 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolIngamePakFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000088, 0x0000224F, { { 0xDA, 0x24, 0x18, 0xA3, 0xEF, 0x16, 0x70, 0x8F, 0xA8, 0xC2, 0x2E, 0xC2, 0xED, 0x39, 0x03, 0xD1 } } } },
	{ UNK_LANG, kPlatformPC98, { 0x00000084, 0x00002125, { { 0x7A, 0x89, 0xE2, 0x36, 0xEC, 0x6F, 0x52, 0x2B, 0xEF, 0xBA, 0x3D, 0x28, 0x54, 0xDA, 0xFB, 0x72 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCharacterDefsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000492, 0x000046B0, { { 0x7A, 0x94, 0x8B, 0xC6, 0xF7, 0xF1, 0x2F, 0xF3, 0xBC, 0x1B, 0x0B, 0x4E, 0x00, 0xC9, 0x44, 0x58 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000492, 0x000047FD, { { 0x8C, 0x0B, 0x8B, 0xCE, 0xE0, 0xB0, 0x8F, 0xA9, 0x06, 0xC3, 0x98, 0xE6, 0x2E, 0x09, 0xB6, 0x93 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x00000492, 0x00004ACD, { { 0xDF, 0x87, 0xFE, 0x89, 0x59, 0xCC, 0x01, 0xD7, 0xC7, 0xEB, 0x16, 0xA4, 0x09, 0xAF, 0x5D, 0xC0 } } } }, // CD
	{ UNK_LANG, kPlatformPC98, { 0x00000492, 0x00005893, { { 0x7C, 0x7E, 0xFB, 0x80, 0xD9, 0xB6, 0x16, 0x87, 0x80, 0xB7, 0x46, 0x9B, 0x96, 0x1A, 0x6A, 0xBE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolIngameSfxFilesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000008F2, 0x0001E5B6, { { 0x63, 0x5E, 0x37, 0xAA, 0x27, 0x80, 0x4C, 0x85, 0xB1, 0x9D, 0x7B, 0x1D, 0x64, 0xA3, 0xEB, 0x97 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x000008F2, 0x0001E5B7, { { 0x9E, 0xC8, 0xE8, 0x19, 0x2F, 0x58, 0x0B, 0xC7, 0x2D, 0x41, 0x72, 0xE7, 0xF4, 0x80, 0x03, 0xCB } } } }, // CD
	{ UNK_LANG, kPlatformPC98, { 0x000008EF, 0x0001E585, { { 0x85, 0x81, 0x5C, 0xA4, 0x34, 0x44, 0xF4, 0x58, 0xF9, 0x82, 0xEE, 0x0F, 0x6A, 0x0D, 0xA2, 0x7F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolIngameSfxIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x000003E8, 0x0000E8D2, { { 0x19, 0x39, 0x17, 0xED, 0xAE, 0xDC, 0x7A, 0xAC, 0x45, 0x5F, 0x2D, 0xCD, 0x65, 0x8D, 0xAD, 0xAE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMusicTrackMapProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000000F0, 0x0000210D, { { 0x55, 0x25, 0x3E, 0x35, 0xD2, 0xD8, 0x13, 0xE3, 0x1D, 0xB1, 0xB3, 0x00, 0x2E, 0x17, 0x91, 0x2F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolIngameGMSfxIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000000FA, 0x00006281, { { 0x25, 0x89, 0xB0, 0x3B, 0x12, 0x09, 0x02, 0xF6, 0xFE, 0x76, 0xD5, 0xC9, 0x5B, 0x88, 0xAC, 0xAA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolIngameMT32SfxIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000000FA, 0x00006579, { { 0x16, 0x40, 0x1C, 0x09, 0x69, 0xA9, 0x0D, 0x6D, 0x4B, 0x0C, 0x99, 0xF0, 0x40, 0x5D, 0xBB, 0x6E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolIngamePcSpkSfxIndexProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000000FA, 0x00005EFC, { { 0xA3, 0x5C, 0x69, 0xED, 0x13, 0xEC, 0x08, 0x0E, 0xFA, 0x72, 0x83, 0x0D, 0xD7, 0x8D, 0x9C, 0x70 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolSpellPropertiesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000118, 0x00000B06, { { 0x27, 0x69, 0x53, 0x01, 0xA0, 0xE3, 0x76, 0xAA, 0x33, 0xA4, 0x52, 0x11, 0x52, 0xB1, 0x0E, 0xDA } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolGameShapeMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000114, 0x000038D3, { { 0xB2, 0x8A, 0x5D, 0x9A, 0x51, 0x63, 0x4D, 0x65, 0xE4, 0xF5, 0xBA, 0x88, 0x70, 0x6C, 0xA6, 0xF8 } } } }, // floppy + PC98
	{ UNK_LANG, kPlatformPC, { 0x00000114, 0x00003B97, { { 0x29, 0xE5, 0x0F, 0x51, 0xF0, 0x10, 0x35, 0x3E, 0x70, 0x3A, 0xAA, 0xFE, 0xD7, 0xD5, 0xAA, 0x9F } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolSceneItemOffsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000008, 0x00000300, { { 0x69, 0x80, 0x5A, 0x3E, 0x63, 0xC1, 0x04, 0x60, 0x09, 0x2F, 0x49, 0xD7, 0x26, 0x32, 0xAA, 0xE2 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCharInvIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x00000006, { { 0x19, 0x79, 0x4E, 0xFC, 0x05, 0x14, 0x89, 0x23, 0xEB, 0xCA, 0x94, 0x50, 0xE8, 0xD3, 0x81, 0x24 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCharInvDefsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000058, 0x00001D7A, { { 0x25, 0xE4, 0xEB, 0x6D, 0xBE, 0xEA, 0xBD, 0x9A, 0x9F, 0xA5, 0x9E, 0xEB, 0x3D, 0x03, 0x1D, 0x72 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCharDefsManProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x0000003D, { { 0xEB, 0x82, 0x32, 0x9D, 0x76, 0xC8, 0x3D, 0x5E, 0x8C, 0x26, 0x53, 0xDF, 0xC1, 0xFD, 0x0F, 0xC5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCharDefsWomanProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x0000003D, { { 0xEB, 0x82, 0x32, 0x9D, 0x76, 0xC8, 0x3D, 0x5E, 0x8C, 0x26, 0x53, 0xDF, 0xC1, 0xFD, 0x0F, 0xC5 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCharDefsKieranProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x000002E3, { { 0xBF, 0xB1, 0x0F, 0x40, 0xBF, 0xA1, 0xD0, 0x2B, 0xC9, 0x80, 0x35, 0x40, 0xA9, 0xA3, 0x01, 0xC8 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCharDefsAkshelProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x000002FB, { { 0x47, 0x3C, 0x07, 0x15, 0x20, 0xE6, 0x90, 0x59, 0x55, 0xF2, 0xA7, 0xC3, 0x27, 0x22, 0xAB, 0xDC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolExpRequirementsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000002C, 0x00000C0A, { { 0x3F, 0x36, 0xFA, 0xE3, 0xB0, 0x76, 0x5E, 0xFF, 0xE9, 0xBA, 0xDF, 0xD0, 0x9D, 0xFF, 0xDD, 0x27 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMonsterModifiersProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x000002C6, { { 0x38, 0x9A, 0x8B, 0x50, 0xD2, 0x9B, 0x95, 0x38, 0x91, 0x02, 0xA9, 0xBE, 0x78, 0xE5, 0x89, 0x65 } } } }, // floppy + PC98
	{ UNK_LANG, kPlatformPC, { 0x00000018, 0x000002EE, { { 0x4E, 0x37, 0x56, 0xE3, 0x42, 0xB3, 0x15, 0x2C, 0x7E, 0x9B, 0x7E, 0x50, 0x32, 0x91, 0x55, 0xBE } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMonsterShiftOffsetsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x00000803, { { 0x30, 0x55, 0x74, 0x0D, 0xC7, 0x3B, 0xD9, 0x5C, 0x26, 0xF0, 0x4E, 0x8F, 0xE4, 0x4D, 0xCB, 0x2A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMonsterDirFlagsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x00000080, { { 0xE5, 0xA1, 0xE3, 0xCE, 0xA0, 0x5F, 0x15, 0xE9, 0x5B, 0x28, 0x90, 0xC0, 0xDF, 0x21, 0xEC, 0x24 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMonsterScaleYProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001E, 0x00000982, { { 0xE2, 0x71, 0x5F, 0x57, 0x4A, 0x8F, 0x50, 0xDB, 0x3E, 0xDA, 0xAB, 0x10, 0xEB, 0xDB, 0x0D, 0x14 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMonsterScaleXProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000020, 0x00000918, { { 0xF6, 0x14, 0xE6, 0x48, 0x4E, 0x5B, 0x43, 0xCC, 0xCE, 0x4E, 0x98, 0x71, 0x5A, 0xC2, 0x00, 0x1E } } } },
	{ UNK_LANG, kPlatformPC98, { 0x0000001D, 0x000008D2, { { 0x1C, 0x25, 0x38, 0xE2, 0xBB, 0xB2, 0xDB, 0x93, 0x1B, 0x25, 0xB6, 0x89, 0xA9, 0x9B, 0x0A, 0xFE } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMonsterScaleWHProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000082, 0x00001D2A, { { 0x85, 0x7E, 0x18, 0xDD, 0x74, 0x1C, 0x62, 0x6F, 0xF4, 0xE5, 0xAF, 0x65, 0xEC, 0x6A, 0x90, 0xAD } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolFlyingObjectShpProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000082, 0x00000252, { { 0xDE, 0x9D, 0x89, 0xAF, 0x0F, 0x50, 0x14, 0x60, 0x68, 0xAF, 0x19, 0xD8, 0x54, 0x8A, 0x36, 0x27 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolInventoryDescProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000016, 0x0000082D, { { 0x86, 0xB4, 0xB9, 0x50, 0xB6, 0xDA, 0x29, 0xB2, 0xC0, 0x0D, 0x34, 0x3F, 0x8D, 0x88, 0xAA, 0xE4 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolLevelShpListProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000007F, 0x00002090, { { 0x17, 0x31, 0x8A, 0xB5, 0x9B, 0x3A, 0xDA, 0x16, 0x9E, 0xE3, 0xD1, 0x5F, 0xB4, 0x7B, 0xB2, 0x25 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolLevelDatListProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000007F, 0x00001FB8, { { 0xF6, 0xE9, 0x98, 0x79, 0x51, 0xCA, 0xA0, 0x35, 0xE4, 0xD0, 0xA1, 0xCD, 0x23, 0x89, 0x7D, 0x11 } } } }, // floppy + PC98
	{ UNK_LANG, kPlatformPC, { 0x000000FF, 0x000047EC, { { 0x0D, 0xA5, 0xFD, 0x8A, 0x33, 0xDB, 0x93, 0x43, 0xE2, 0x57, 0x35, 0xEC, 0xA6, 0xCF, 0x7A, 0xA1 } } } }, // CD

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCompassDefsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000100, 0x000030EF, { { 0x6F, 0xF0, 0x46, 0x6E, 0xB3, 0x72, 0xCF, 0xC7, 0xE3, 0xAF, 0xBE, 0x63, 0xA1, 0x1C, 0x33, 0x20 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolItemPricesProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000005C, 0x00001251, { { 0x18, 0x62, 0x5E, 0xE2, 0xE4, 0x2A, 0xB0, 0xA0, 0x8B, 0x8D, 0x9D, 0x07, 0x5F, 0x83, 0x53, 0xF7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolStashSetupProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x0000001E, { { 0x1C, 0x93, 0x66, 0x56, 0xDB, 0xD7, 0xA4, 0xB3, 0xE7, 0x2F, 0xEA, 0x88, 0xE2, 0xC8, 0x79, 0xD0 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscUnk1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000024, 0x00000A2A, { { 0xAC, 0x4E, 0x73, 0x2C, 0xB0, 0xEE, 0x24, 0x0E, 0x66, 0x8D, 0x48, 0xE5, 0xCA, 0x6B, 0x7F, 0x7F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscShapeIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000024, 0x00000728, { { 0x14, 0xBA, 0x6D, 0x5C, 0x7D, 0x20, 0x0D, 0x35, 0xA7, 0xB0, 0x8D, 0x2F, 0x1D, 0x2A, 0x49, 0xA4 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscOvlMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x0000001F, { { 0x9C, 0xF2, 0xCC, 0x48, 0x42, 0xC6, 0x76, 0x83, 0xD3, 0x1A, 0x43, 0x42, 0x7F, 0xEF, 0x19, 0x0F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscScaleWidthDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00000ABE, { { 0x28, 0x9A, 0xAA, 0x16, 0xC4, 0xFD, 0x52, 0xA9, 0x76, 0x98, 0x72, 0x0C, 0x2D, 0xE4, 0xB0, 0x57 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscScaleHeightDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x000009E8, { { 0x25, 0x35, 0x07, 0xBC, 0xF9, 0x82, 0x8B, 0x5B, 0x67, 0x7C, 0x38, 0xD1, 0xF8, 0x35, 0x81, 0xC7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscXProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00001468, { { 0x55, 0xC5, 0x30, 0x76, 0x0A, 0xDC, 0xEC, 0xAB, 0x68, 0x9B, 0x61, 0xF0, 0x58, 0x78, 0x56, 0xA6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscYProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000024, 0x00000282, { { 0x09, 0x98, 0x3A, 0x33, 0x15, 0xA1, 0x4A, 0xFF, 0x76, 0x19, 0x2B, 0xB1, 0x74, 0x89, 0xF4, 0x37 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscTileIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x00000099, { { 0x18, 0xD9, 0x39, 0x27, 0x5B, 0x34, 0xAE, 0x7C, 0xA9, 0xA9, 0xDB, 0x42, 0x49, 0x61, 0x6B, 0x37 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscUnk2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000017, 0x000000D2, { { 0xDE, 0xDA, 0x75, 0x15, 0x2B, 0xDC, 0x90, 0x3F, 0xC9, 0x92, 0x04, 0x01, 0x23, 0x7A, 0xDA, 0x2E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDoorShapeIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000017, 0x0000000A, { { 0x2E, 0xC4, 0xA1, 0x47, 0x7C, 0xAE, 0xAD, 0xD8, 0x8A, 0x72, 0x95, 0x2F, 0x18, 0xC5, 0x08, 0x19 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDimData1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000144, 0x0001007D, { { 0x18, 0x3D, 0xA5, 0xF7, 0x1A, 0x5A, 0x90, 0xA7, 0x4E, 0x66, 0x1A, 0x4E, 0x0C, 0x69, 0x58, 0x31 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDimData2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000289, 0x00001BC2, { { 0x7F, 0x9D, 0xD3, 0x5A, 0x57, 0x73, 0xEA, 0x37, 0x44, 0x5E, 0x1A, 0x88, 0xFB, 0xE8, 0xE8, 0x8F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscBlockMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000C, 0x00000012, { { 0x01, 0xEE, 0x32, 0xA6, 0x71, 0x15, 0x8D, 0xFB, 0x33, 0xF2, 0xD6, 0x8A, 0x30, 0x00, 0x10, 0x4B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDimMapProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000012, 0x00000014, { { 0x4D, 0x53, 0x2E, 0xF2, 0xA3, 0xF9, 0xE2, 0xEC, 0x44, 0xBE, 0x5F, 0x04, 0x91, 0xF8, 0xE1, 0x04 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscOvlIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000028, 0x00000048, { { 0x3E, 0x8E, 0x62, 0xAF, 0xD1, 0x28, 0x39, 0x73, 0x0D, 0xD8, 0x4A, 0xA7, 0xF4, 0xD7, 0x32, 0x25 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscBlockIndexProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000048, 0x00002200, { { 0xF4, 0x8B, 0x32, 0xC3, 0xD3, 0xFB, 0x46, 0xF2, 0xB8, 0x3A, 0x58, 0x39, 0x94, 0x57, 0x97, 0x4B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDoor1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000080, 0x00000348, { { 0xC6, 0x58, 0x8B, 0xFE, 0x18, 0x72, 0x47, 0xF1, 0xB6, 0x3A, 0x0F, 0xFB, 0x3D, 0x99, 0x74, 0xD0 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDoorScaleProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000010, 0x0000024C, { { 0x8D, 0x83, 0x26, 0xEE, 0xDC, 0xF7, 0x13, 0xC0, 0xAA, 0x88, 0xC2, 0xAA, 0x66, 0xA7, 0x59, 0x41 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDoor4Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000008, 0x00000103, { { 0x29, 0xC0, 0x4B, 0x7F, 0x36, 0x23, 0xBB, 0x38, 0x4C, 0x83, 0xC6, 0x9D, 0xB4, 0x8F, 0x29, 0x2E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDoorXProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000480, 0x0001654D, { { 0x2A, 0x1F, 0xBF, 0xE3, 0xC4, 0xEF, 0x7E, 0xD1, 0x61, 0x51, 0xFE, 0x88, 0x8D, 0x1F, 0x59, 0x70 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolDscDoorYProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000480, 0x00026666, { { 0x06, 0xBF, 0xA4, 0xD4, 0x6E, 0x29, 0x42, 0xA2, 0xA0, 0x8E, 0x3C, 0x14, 0xF3, 0xD6, 0x3F, 0x87 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolScrollXTopProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x0000004B, { { 0x18, 0x1E, 0x6E, 0xE9, 0x34, 0xF0, 0x02, 0xC6, 0x57, 0x34, 0xDF, 0x55, 0xD9, 0x39, 0xE8, 0x98 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolScrollYTopProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x00000022, { { 0xF3, 0x20, 0x5A, 0xC1, 0xBB, 0x0C, 0x79, 0x52, 0x23, 0xC1, 0x36, 0x81, 0x70, 0x2F, 0x92, 0xFC } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolScrollXBottomProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x0000004B, { { 0x18, 0x1E, 0x6E, 0xE9, 0x34, 0xF0, 0x02, 0xC6, 0x57, 0x34, 0xDF, 0x55, 0xD9, 0x39, 0xE8, 0x98 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolScrollYBottomProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000000A, 0x0000003C, { { 0x5B, 0x4F, 0xB7, 0xB5, 0x55, 0xA2, 0x9A, 0x21, 0xEF, 0xB4, 0x98, 0x47, 0x05, 0x57, 0x49, 0x55 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonDefsProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x0000082A, 0x0000CAAE, { { 0xC1, 0x83, 0x0D, 0xA0, 0x66, 0x16, 0x3D, 0x31, 0xCE, 0x30, 0x9F, 0x4E, 0x00, 0x65, 0x5A, 0xC8 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x0000082A, 0x0000C34E, { { 0x7F, 0x9A, 0x0F, 0x28, 0x1A, 0x8F, 0x03, 0x46, 0x48, 0xEB, 0xC9, 0xB9, 0x23, 0x29, 0x5E, 0x50 } } } }, // floppy
	{ UNK_LANG, kPlatformPC, { 0x0000082A, 0x0000C47B, { { 0xDF, 0x1A, 0x18, 0x1F, 0x58, 0x05, 0x1F, 0x56, 0xD8, 0x6D, 0xBB, 0x93, 0xEC, 0x35, 0x9D, 0xA5 } } } }, // CD
	{ UNK_LANG, kPlatformPC98, { 0x0000082A, 0x0000AB58, { { 0xDD, 0x2B, 0xA9, 0x54, 0x60, 0x25, 0x2C, 0x74, 0xF8, 0x5D, 0xC6, 0xD2, 0x2C, 0x1A, 0x24, 0x44 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonList1Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000050, 0x00000A37, { { 0x0F, 0x73, 0xEC, 0xDD, 0xAB, 0xFF, 0x49, 0x46, 0x5E, 0x8F, 0x0D, 0xC3, 0xE7, 0x1B, 0x89, 0x51 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonList2Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001E, 0x00000522, { { 0xEA, 0x41, 0x46, 0xE2, 0xFE, 0xAA, 0x7D, 0x5E, 0x89, 0x7F, 0xBF, 0x9B, 0x30, 0x60, 0x74, 0xF3 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonList3Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x0000023E, { { 0x70, 0xAA, 0xCA, 0xAC, 0x5C, 0x21, 0xCF, 0xA5, 0xBF, 0x7F, 0x5F, 0xBC, 0xF1, 0x24, 0x8A, 0xAF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonList4Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001E, 0x0000054D, { { 0x19, 0x2A, 0xBE, 0x7F, 0x94, 0x10, 0xA0, 0x60, 0x2A, 0x33, 0xD6, 0x11, 0x85, 0xF0, 0xA4, 0xA9 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonList5Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000020, 0x0000045D, { { 0xE3, 0x7C, 0xC2, 0x36, 0x21, 0x46, 0xDB, 0xF3, 0xDD, 0x38, 0x4B, 0x40, 0xE0, 0x35, 0x09, 0xC3 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonList6Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000001C, 0x000004C4, { { 0x21, 0x7C, 0x29, 0x3F, 0x95, 0x6F, 0x91, 0x8C, 0xB2, 0x30, 0x09, 0xA6, 0x7B, 0x48, 0x44, 0x8F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonList7Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000006, 0x0000021D, { { 0xDC, 0xCE, 0x1B, 0xEB, 0x11, 0x6D, 0xDE, 0x37, 0x17, 0xC8, 0x06, 0x51, 0xC3, 0x0C, 0xCB, 0xA6 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolButtonList8Provider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000004, 0x00000253, { { 0x0C, 0x7B, 0x10, 0x99, 0x93, 0xD0, 0x33, 0xCA, 0xAB, 0x8D, 0x7E, 0x24, 0xE5, 0x7E, 0x6C, 0x91 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolLegendDataProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000030, 0x00000858, { { 0x63, 0x5E, 0x60, 0xC7, 0x62, 0x2C, 0x5D, 0x8F, 0x74, 0x71, 0x98, 0xB7, 0x09, 0xD2, 0x51, 0xC7 } } } },
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000A52, { { 0x81, 0xC5, 0xA4, 0xE7, 0x60, 0xDA, 0xD6, 0x5E, 0x19, 0xAB, 0xF3, 0xC7, 0xDD, 0xDB, 0x92, 0x5E } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMapCursorOvlProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x00000019, 0x000009CD, { { 0xF6, 0xD2, 0xFA, 0x36, 0x62, 0x95, 0x1D, 0x99, 0x7F, 0x11, 0x5F, 0xA8, 0x4D, 0x47, 0x72, 0x40 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolMapStringIdProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x0000003C, 0x00000CFB, { { 0xE3, 0xC3, 0x41, 0x06, 0xD1, 0x71, 0x77, 0x78, 0xAD, 0x39, 0xAE, 0x2C, 0x16, 0x21, 0x45, 0xB7 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolSpellbookAnimProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000038, 0x000003A1, { { 0x50, 0xA0, 0xF6, 0xA7, 0x53, 0x96, 0x86, 0x49, 0xB0, 0x8D, 0xA8, 0xB2, 0x2D, 0x9A, 0xE2, 0x1F } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolSpellbookCoordsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000018, 0x0000018F, { { 0xA5, 0xF6, 0x8A, 0x58, 0x9A, 0xC7, 0x3C, 0x3A, 0xB5, 0x87, 0x89, 0x87, 0x73, 0x51, 0x9B, 0x1B } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolHealShapeFramesProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000050, 0x000000F4, { { 0xC9, 0x6E, 0x39, 0xE1, 0xD7, 0xAD, 0x10, 0x4F, 0xE2, 0xFE, 0xDC, 0xAD, 0x00, 0x9D, 0x41, 0xEF } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolLightningDefsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000014, 0x00000385, { { 0x68, 0x39, 0x65, 0xCB, 0xA9, 0x80, 0x90, 0xFB, 0xDD, 0x77, 0x0C, 0x76, 0x5A, 0xB5, 0x05, 0x03 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolFireballCoordsProvider[] = {
	{ UNK_LANG, kPlatformUnknown, { 0x00000200, 0x0000FD81, { { 0xB3, 0xE0, 0x6F, 0x89, 0xCD, 0xE5, 0xA9, 0x6A, 0x4B, 0x61, 0x7A, 0x3F, 0x47, 0x26, 0x73, 0x58 } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolCreditsProvider[] = {
	{ JA_JPN , kPlatformPC98, { 0x000005E7, 0x0001A1B0, { { 0x2A, 0xD0, 0x38, 0x84, 0x0C, 0x38, 0xCB, 0x52, 0x5D, 0x82, 0xBE, 0x03, 0x76, 0xFA, 0x0A, 0x4A } } } },

	EXTRACT_END_ENTRY
};

const ExtractEntrySearchData kLolHistoryProvider[] = {
	{ UNK_LANG, kPlatformPC, { 0x000001D1, 0x00007F9B, { { 0x25, 0x10, 0x86, 0x40, 0xAC, 0x53, 0xFE, 0x11, 0x4D, 0xE2, 0xD9, 0x35, 0xD6, 0x89, 0xBB, 0x09 } } } },

	EXTRACT_END_ENTRY
};

// Provider tables

struct ExtractEntry {
	int id;
	const ExtractEntrySearchData *providers;
};

const ExtractEntry extractProviders[] = {
	{ k1ForestSeq, k1ForestSeqProvider },
	{ k1KallakWritingSeq, k1KallakWritingSeqProvider },
	{ k1KyrandiaLogoSeq, k1KyrandiaLogoSeqProvider },
	{ k1KallakMalcolmSeq, k1KallakMalcolmSeqProvider },
	{ k1MalcolmTreeSeq, k1MalcolmTreeSeqProvider },
	{ k1WestwoodLogoSeq, k1WestwoodLogoSeqProvider },
	{ k1Demo1Seq, k1Demo1SeqProvider },
	{ k1Demo2Seq, k1Demo2SeqProvider },
	{ k1Demo3Seq, k1Demo3SeqProvider },
	{ k1Demo4Seq, k1Demo4SeqProvider },
	{ k1AmuleteAnimSeq, k1AmuleteAnimSeqProvider },
	{ k1OutroReunionSeq, k1OutroReunionSeqProvider },
	{ k1IntroCPSStrings, k1IntroCPSStringsProvider },
	{ k1IntroCOLStrings, k1IntroCOLStringsProvider },
	{ k1IntroWSAStrings, k1IntroWSAStringsProvider },
	{ k1IntroStrings, k1IntroStringsProvider },
	{ k1OutroHomeString, k1OutroHomeStringProvider },
	{ k1RoomFilenames, k1RoomFilenamesProvider },
	{ k1RoomList, k1RoomListProvider },
	{ k1CharacterImageFilenames, k1CharacterImageFilenamesProvider },
	{ k1AudioTracks, k1AudioTracksProvider },
	{ k1AudioTracksIntro, k1AudioTracksIntroProvider },
	{ k1ItemNames, k1ItemNamesProvider },
	{ k1TakenStrings, k1TakenStringsProvider },
	{ k1PlacedStrings, k1PlacedStringsProvider },
	{ k1DroppedStrings, k1DroppedStringsProvider },
	{ k1NoDropStrings, k1NoDropStringsProvider },
	{ k1PutDownString, k1PutDownStringProvider },
	{ k1WaitAmuletString, k1WaitAmuletStringProvider },
	{ k1BlackJewelString, k1BlackJewelStringProvider },
	{ k1PoisonGoneString, k1PoisonGoneStringProvider },
	{ k1HealingTipString, k1HealingTipStringProvider },
	{ k1WispJewelStrings, k1WispJewelStringsProvider },
	{ k1MagicJewelStrings, k1MagicJewelStringsProvider },
	{ k1ThePoisonStrings, k1ThePoisonStringsProvider },
	{ k1FluteStrings, k1FluteStringsProvider },
	{ k1FlaskFullString, k1FlaskFullStringProvider },
	{ k1FullFlaskString, k1FullFlaskStringProvider },
	{ k1VeryCleverString, k1VeryCleverStringProvider },
	{ k1NewGameString, k1NewGameStringProvider },
	{ k1DefaultShapes, k1DefaultShapesProvider },
	{ k1Healing1Shapes, k1Healing1ShapesProvider },
	{ k1Healing2Shapes, k1Healing2ShapesProvider },
	{ k1PoisonDeathShapes, k1PoisonDeathShapesProvider },
	{ k1FluteShapes, k1FluteShapesProvider },
	{ k1Winter1Shapes, k1Winter1ShapesProvider },
	{ k1Winter2Shapes, k1Winter2ShapesProvider },
	{ k1Winter3Shapes, k1Winter3ShapesProvider },
	{ k1DrinkShapes, k1DrinkShapesProvider },
	{ k1WispShapes, k1WispShapesProvider },
	{ k1MagicAnimShapes, k1MagicAnimShapesProvider },
	{ k1BranStoneShapes, k1BranStoneShapesProvider },
	{ k1SpecialPalette1, k1SpecialPalette1Provider },
	{ k1SpecialPalette2, k1SpecialPalette2Provider },
	{ k1SpecialPalette3, k1SpecialPalette3Provider },
	{ k1SpecialPalette4, k1SpecialPalette4Provider },
	{ k1SpecialPalette5, k1SpecialPalette5Provider },
	{ k1SpecialPalette6, k1SpecialPalette6Provider },
	{ k1SpecialPalette7, k1SpecialPalette7Provider },
	{ k1SpecialPalette8, k1SpecialPalette8Provider },
	{ k1SpecialPalette9, k1SpecialPalette9Provider },
	{ k1SpecialPalette10, k1SpecialPalette10Provider },
	{ k1SpecialPalette11, k1SpecialPalette11Provider },
	{ k1SpecialPalette12, k1SpecialPalette12Provider },
	{ k1SpecialPalette13, k1SpecialPalette13Provider },
	{ k1SpecialPalette14, k1SpecialPalette14Provider },
	{ k1SpecialPalette15, k1SpecialPalette15Provider },
	{ k1SpecialPalette16, k1SpecialPalette16Provider },
	{ k1SpecialPalette17, k1SpecialPalette17Provider },
	{ k1SpecialPalette18, k1SpecialPalette18Provider },
	{ k1SpecialPalette19, k1SpecialPalette19Provider },
	{ k1SpecialPalette20, k1SpecialPalette20Provider },
	{ k1SpecialPalette21, k1SpecialPalette21Provider },
	{ k1SpecialPalette22, k1SpecialPalette22Provider },
	{ k1SpecialPalette23, k1SpecialPalette23Provider },
	{ k1SpecialPalette24, k1SpecialPalette24Provider },
	{ k1SpecialPalette25, k1SpecialPalette25Provider },
	{ k1SpecialPalette26, k1SpecialPalette26Provider },
	{ k1SpecialPalette27, k1SpecialPalette27Provider },
	{ k1SpecialPalette28, k1SpecialPalette28Provider },
	{ k1SpecialPalette29, k1SpecialPalette29Provider },
	{ k1SpecialPalette30, k1SpecialPalette30Provider },
	{ k1SpecialPalette31, k1SpecialPalette31Provider },
	{ k1SpecialPalette32, k1SpecialPalette32Provider },
	{ k1SpecialPalette33, k1SpecialPalette33Provider },
	{ k1GUIStrings, k1GUIStringsProvider },
	{ k1ConfigStrings, k1ConfigStringsProvider },
	{ k1TownsMusicFadeTable, k1TownsMusicFadeTableProvider },
	{ k1TownsSFXwdTable, k1TownsSFXwdTableProvider },
	{ k1TownsSFXbtTable, k1TownsSFXbtTableProvider },
	{ k1TownsCDATable, k1TownsCDATableProvider },
	{ k1PC98StoryStrings, k1PC98StoryStringsProvider },
	{ k1PC98IntroSfx, k1PC98IntroSfxProvider },
	{ k1CreditsStrings, k1CreditsStringsProvider },
	{ k1AmigaIntroSFXTable, k1AmigaIntroSFXTableProvider },
	{ k1AmigaGameSFXTable, k1AmigaGameSFXTableProvider },
	{ k2SeqplayPakFiles, k2SeqplayPakFilesProvider },
	{ k2SeqplayStrings, k2SeqplayStringsProvider },
	{ k2SeqplaySfxFiles, k2SeqplaySfxFilesProvider },
	{ k2SeqplayTlkFiles, k2SeqplayTlkFilesProvider },
	{ k2SeqplaySeqData, k2SeqplaySeqDataProvider },
	{ k2SeqplayCredits, k2SeqplayCreditsProvider },
	{ k2SeqplayCreditsSpecial, k2SeqplayCreditsSpecialProvider },
	{ k2SeqplayIntroTracks, k2SeqplayIntroTracksProvider },
	{ k2SeqplayFinaleTracks, k2SeqplayFinaleTracksProvider },
	{ k2SeqplayIntroCDA, k2SeqplayIntroCDAProvider },
	{ k2SeqplayFinaleCDA, k2SeqplayFinaleCDAProvider },
	{ k2SeqplayShapeAnimData, k2SeqplayShapeAnimDataProvider },
	{ k2IngamePakFiles, k2IngamePakFilesProvider },
	{ k2IngameSfxFiles, k2IngameSfxFilesProvider },
	{ k2IngameSfxIndex, k2IngameSfxIndexProvider },
	{ k2IngameTracks, k2IngameTracksProvider },
	{ k2IngameCDA, k2IngameCDAProvider },
	{ k2IngameTalkObjIndex, k2IngameTalkObjIndexProvider },
	{ k2IngameTimJpStrings, k2IngameTimJpStringsProvider },
	{ k2IngameShapeAnimData, k2IngameShapeAnimDataProvider },
	{ k2IngameTlkDemoStrings, k2IngameTlkDemoStringsProvider },
	{ k3MainMenuStrings, k3MainMenuStringsProvider },
	{ k3MusicFiles, k3MusicFilesProvider },
	{ k3ScoreTable, k3ScoreTableProvider },
	{ k3SfxFiles, k3SfxFilesProvider },
	{ k3SfxMap, k3SfxMapProvider },
	{ k3ItemAnimData, k3ItemAnimDataProvider },
	{ k3ItemMagicTable, k3ItemMagicTableProvider },
	{ k3ItemStringMap, k3ItemStringMapProvider },
	{ kLolIngamePakFiles, kLolIngamePakFilesProvider },
	{ kLolCharacterDefs, kLolCharacterDefsProvider },
	{ kLolIngameSfxFiles, kLolIngameSfxFilesProvider },
	{ kLolIngameSfxIndex, kLolIngameSfxIndexProvider },
	{ kLolMusicTrackMap, kLolMusicTrackMapProvider },
	{ kLolIngameGMSfxIndex, kLolIngameGMSfxIndexProvider },
	{ kLolIngameMT32SfxIndex, kLolIngameMT32SfxIndexProvider },
	{ kLolIngamePcSpkSfxIndex, kLolIngamePcSpkSfxIndexProvider },
	{ kLolSpellProperties, kLolSpellPropertiesProvider },
	{ kLolGameShapeMap, kLolGameShapeMapProvider },
	{ kLolSceneItemOffs, kLolSceneItemOffsProvider },
	{ kLolCharInvIndex, kLolCharInvIndexProvider },
	{ kLolCharInvDefs, kLolCharInvDefsProvider },
	{ kLolCharDefsMan, kLolCharDefsManProvider },
	{ kLolCharDefsWoman, kLolCharDefsWomanProvider },
	{ kLolCharDefsKieran, kLolCharDefsKieranProvider },
	{ kLolCharDefsAkshel, kLolCharDefsAkshelProvider },
	{ kLolExpRequirements, kLolExpRequirementsProvider },
	{ kLolMonsterModifiers, kLolMonsterModifiersProvider },
	{ kLolMonsterShiftOffsets, kLolMonsterShiftOffsetsProvider },
	{ kLolMonsterDirFlags, kLolMonsterDirFlagsProvider },
	{ kLolMonsterScaleY, kLolMonsterScaleYProvider },
	{ kLolMonsterScaleX, kLolMonsterScaleXProvider },
	{ kLolMonsterScaleWH, kLolMonsterScaleWHProvider },
	{ kLolFlyingObjectShp, kLolFlyingObjectShpProvider },
	{ kLolInventoryDesc, kLolInventoryDescProvider },
	{ kLolLevelShpList, kLolLevelShpListProvider },
	{ kLolLevelDatList, kLolLevelDatListProvider },
	{ kLolCompassDefs, kLolCompassDefsProvider },
	{ kLolItemPrices, kLolItemPricesProvider },
	{ kLolStashSetup, kLolStashSetupProvider },
	{ kLolDscUnk1, kLolDscUnk1Provider },
	{ kLolDscShapeIndex, kLolDscShapeIndexProvider },
	{ kLolDscOvlMap, kLolDscOvlMapProvider },
	{ kLolDscScaleWidthData, kLolDscScaleWidthDataProvider },
	{ kLolDscScaleHeightData, kLolDscScaleHeightDataProvider },
	{ kLolDscX, kLolDscXProvider },
	{ kLolDscY, kLolDscYProvider },
	{ kLolDscTileIndex, kLolDscTileIndexProvider },
	{ kLolDscUnk2, kLolDscUnk2Provider },
	{ kLolDscDoorShapeIndex, kLolDscDoorShapeIndexProvider },
	{ kLolDscDimData1, kLolDscDimData1Provider },
	{ kLolDscDimData2, kLolDscDimData2Provider },
	{ kLolDscBlockMap, kLolDscBlockMapProvider },
	{ kLolDscDimMap, kLolDscDimMapProvider },
	{ kLolDscOvlIndex, kLolDscOvlIndexProvider },
	{ kLolDscBlockIndex, kLolDscBlockIndexProvider },
	{ kLolDscDoor1, kLolDscDoor1Provider },
	{ kLolDscDoorScale, kLolDscDoorScaleProvider },
	{ kLolDscDoor4, kLolDscDoor4Provider },
	{ kLolDscDoorX, kLolDscDoorXProvider },
	{ kLolDscDoorY, kLolDscDoorYProvider },
	{ kLolScrollXTop, kLolScrollXTopProvider },
	{ kLolScrollYTop, kLolScrollYTopProvider },
	{ kLolScrollXBottom, kLolScrollXBottomProvider },
	{ kLolScrollYBottom, kLolScrollYBottomProvider },
	{ kLolButtonDefs, kLolButtonDefsProvider },
	{ kLolButtonList1, kLolButtonList1Provider },
	{ kLolButtonList2, kLolButtonList2Provider },
	{ kLolButtonList3, kLolButtonList3Provider },
	{ kLolButtonList4, kLolButtonList4Provider },
	{ kLolButtonList5, kLolButtonList5Provider },
	{ kLolButtonList6, kLolButtonList6Provider },
	{ kLolButtonList7, kLolButtonList7Provider },
	{ kLolButtonList8, kLolButtonList8Provider },
	{ kLolLegendData, kLolLegendDataProvider },
	{ kLolMapCursorOvl, kLolMapCursorOvlProvider },
	{ kLolMapStringId, kLolMapStringIdProvider },
	{ kLolSpellbookAnim, kLolSpellbookAnimProvider },
	{ kLolSpellbookCoords, kLolSpellbookCoordsProvider },
	{ kLolHealShapeFrames, kLolHealShapeFramesProvider },
	{ kLolLightningDefs, kLolLightningDefsProvider },
	{ kLolFireballCoords, kLolFireballCoordsProvider },
	{ kLolCredits, kLolCreditsProvider },
	{ kLolHistory, kLolHistoryProvider },
	{ -1, NULL }
};

} // end of anonymous namespace

ExtractEntryList getProvidersForId(int id) {
	ExtractEntryList list;

	for (const ExtractEntry *p = extractProviders; p->id != -1; ++p) {
		if (p->id == id) {
			for (const ExtractEntrySearchData *d = p->providers; d->hint.size != 0; ++d)
				list.push_back(*d);
		}
	}

	return list;
}

