//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function q = CL_rot_quatSlerp(q1,q2,t)
// Quaternion interpolation on the unit sphere using Slerp method
//
// Calling Sequence
// q = CL_rot_quatSlerp(q1,q2,t)
//
// Description
// <itemizedlist><listitem>
// <p>Interpolates between two quaternions using Slerp method (shortest path).</p>  
// <p>The interpolated quaternion is computed using 
// <b>q1</b>, <b>q2</b> and the
// interpolation parameter <b>t</b>. </p>
// <p>- If t == 0 => q = q1 </p>
// <p>- If t == 1 => q = q2 </p>
// <p></p></listitem>
// <listitem>
// <p><b>Notes:</b></p>
// <p> - Let N be the size of q1 and q2, and P be the size of t. Then one must have: 
// N == P or N == 1 or P == 1. </p>
// <p> - q1 and q2 may not have the same norm (but the norms should be > 0).  </p>
// <p> - See <link linkend="Conventions">Conventions</link> or <link linkend="CL_rot_defQuat">CL_rot_defQuat</link> for more details on quaternions. </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// q1 : Quaternion with norm > 0 (size N)
// q2 : Quaternion with norm > 0 (size N)
// t : Interpolation parameter (1xP)
// q : Interpolated quaternion (size max(N,P))
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_defQuat
// CL_rot_matrix2quat
//
// Examples
// // Rotation of angle 15deg around Z-axis :
// q1 = CL_rot_axAng2quat(CL_deg2rad(15),[0;0;1]); 
// // Rotation of angle 35deg around random axis :
// q2 = CL_rot_axAng2quat(CL_deg2rad(35),[0;3/5;-4/5]);  
//
// q = CL_rot_quatSlerp(q1,q2,0)  // q = q1
// q = CL_rot_quatSlerp(q1,q2,1)  // q = q2
// q = CL_rot_quatSlerp(q1,q2,[0.5 0.6])  
// q_vect =  CL_rot_quatSlerp([q1 conj(q1)],[conj(q2) q2],-0.2) 
//

// Declarations:


// Code:

N = size(q1);
N2 = size(q2);
P = size(t,2);
Nmax = max(N,P);

if ~( N == N2 & (N == 1 | P == 1 | N == P) ) 
  CL__error('Invalid size for q1, q2 or t')
end

q1 = q1./norm(q1);
q2 = q2./norm(q2);

if (N==1 & P>1) 
  q1=q1*ones(1,P);
  q2=q2*ones(1,P);
end

if (P==1 & N>1) 
  t=t*ones(1,N);
end

q = CL_rot_defQuat(zeros(4,Nmax));

q = conj(q1)*q2;

// Avoids numeric errors : 
ii = find(1 < q.r & q.r < 1 + 20*%eps);
if (ii ~= [])
  q.r(ii) = 1;
end

ii = find(-1 -20*%eps < q.r & q.r < -1);
if (ii ~= [])
  q.r(ii) = -1;
end

rotAmp = 2.0*acos(q.r);
// Shortest way:
i = find(rotAmp > %pi);
if (i~=[])
  rotAmp(i) = abs(rotAmp(i)-2*%pi);
  q(i) = CL_rot_defQuat(-q(i).r,-q(i).i);
end

sts2 = CL_norm(q.i);
j = find(sts2 > 1.d-7); // au dela de 10-5 deg environ.
j2 = find(~(sts2 > 1.d-7));
if (j~=[])
  Vect = CL_unitVector(q(j).i) ;
  teta = rotAmp(j) .* t(j);
  q(j) = q1(j) * CL_rot_defQuat(cos(teta/2),CL_dMult(sin(teta/2),Vect));
end
if (j2~=[])
  q(j2) = q1(j2);
end

endfunction
