//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [v1,v2] = CL_man_lambert(pos1,pos2,delta_t,direction,mu)
// Lambert's problem
//
// Calling Sequence
// [v1,v2] = CL_man_lambert(pos1,pos2,delta_t,direction [,mu])
//
// Description
// <itemizedlist><listitem>
// The function computes the velocity vectors <emphasis role="bold">v1</emphasis> and <emphasis role="bold">v2</emphasis>, 
// given the position vectors <emphasis role="bold">r1</emphasis> and <emphasis role="bold">r2</emphasis>, 
// the time of flight <emphasis role="bold">delta_t</emphasis>, and the direction of motion <emphasis role="bold">direction</emphasis>.
// </listitem>
// </itemizedlist>
//
// Parameters
// pos1 : Initial position vector [m] (3xN)
// pos2 : Final position vector [m] (3xN)
// delta_t : Time of flight from r1 to r2 [s] (1xN or 1x1)
// direction : (optional) 'pro' if the transfer orbit is prograde, 'retro' if the transfer orbit is retrograde (default is 'pro')
// mu : (optional) Gravitational constant (default is %CL_mu) [m^3/s^2]
// v1 : Initial velocity vector [m/s] (3xN)
// v2 : Final velocity vector [m/s] (3xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Orbital Mechanics for Engineering Students, H D Curtis, Section 5.3 and Appendix D.11 (algorithm 5.2)
// 2 Modern astrodynamics Fundamentals and perturbation methods, V Bond and M Allman, Chapter 6.2
//
// See also
// CL_stumpS
// CL_stumpC
//
// Examples
// r1 = [5000.e3 ; 10000.e3 ;2100.e3];
// r2 = [-14600.e3 ; 2500.e3 ;7000.e3];
// dt = 3600; //one hour
// // direction = 'pro' by default
// [v1, v2] = CL_man_lambert([r1 r1],[r2 r2],[dt dt]);

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

// Auxiliar functions : 

  function dum = yy(z,r1,r2,A)
    dum = real(r1 + r2 + A .* (z.*CL_stumpS(z)-1) ./ sqrt(CL_stumpC(z)));
  endfunction

  function dum = FF(z,t,r1,r2,A,mu)
    dum = real(( yy(z,r1,r2,A)./CL_stumpC(z) ).^1.5 .* CL_stumpS(z) + A.*sqrt(yy(z,r1,r2,A)) - sqrt(mu).*t);
  endfunction
  
  function dum = dFdz(z,A,r1,r2)
    ii = find(z==0)
    jj = find(z~=0)
    dum(ii) = real( sqrt(2) ./ 40 .* yy(0,r1(ii),r2(ii),A(ii))^1.5 + A(ii) ./ 8 .* ...
           ( sqrt(yy(0,r1(ii),r2(ii),A(ii))) + A(ii) .* sqrt(1/2 ./ yy(0,r1(ii),r2(ii),A(ii))) ) );
    dum(jj) = real( (yy(z(jj),r1(jj),r2(jj),A(jj))./CL_stumpC(z(jj))).^1.5 .* ...
            ( 1/2 ./ z(jj).*(CL_stumpC(z(jj)) - 3.*CL_stumpS(z(jj))./ 2 ./CL_stumpC(z(jj)))+ 3.*CL_stumpS(z(jj)).^2 ./ 4 ./ CL_stumpC(z(jj))) ...
            + A(jj)./ 8 .* ...
            (3.*CL_stumpS(z(jj))./CL_stumpC(z(jj)).*sqrt(yy(z(jj),r1(jj),r2(jj),A(jj))) + A(jj).*sqrt(CL_stumpC(z(jj))./yy(z(jj),r1(jj),r2(jj),A(jj)))) );
  endfunction


if ~exists('direction','local') then direction='pro'; end;
if ~exists('mu','local') then mu=%CL_mu; end;


N = size(pos1,2);

// Consistency of inputs :
Npos2 = size(pos2,2);
Ndelta_t = size(delta_t,2);
if( Ndelta_t == 1) then delta_t = delta_t * ones(1,N); end;
if( Npos2 ~= N) then CL__error('pos1 and pos2 must be of same size'); end;
if( Ndelta_t ~= N) then CL__error('delta_t must be of size 1 or the same size as pos1'); end;

// Magnitudes of pos1 and pos2
r1 = CL_colNorm(pos1);
r2 = CL_colNorm(pos2);

c12 = CL_cross(pos1,pos2);
theta = acos(CL_dot(pos1,pos2)./r1./r2);

// Determine whether the orbit is prograde or retrograde:
if( direction == 'pro')
  ii = find( c12(3,:) <=0 );
  theta(ii) = 2*%pi - theta(ii);
elseif(  direction == 'retro')
  jj = find( c12(3,:) >= 0 );
  theta(jj) = 2*%pi - theta(jj);
else
  CL__error('direction has to be ''pro'' or ''retro'' ');
end

// Equation 5.35:
A = sin(theta).*sqrt(r1.*r2./(1 - cos(theta)));

// Modification of initial algorithm : 
// Before : systematic search of the initial guess such that FF(z) ~= 0 
// with a step of 0.1 starting from -100
// Now : Dichotomy to find the initial guess 

// Dichotomy to find the initial guess
// The solution to FF(z,delta_t,r1,r2,A,mu) = 0 is looked for in [zmin, zmax] 
z_min = -100 ;
z_max = (2*%pi)^2 - 3e-6 ; // such that CL_stumpC(z) > %eps
a = z_min * ones(1,N); // lower bound of dichotomy
b = z_max * ones(1,N); // upper bound of dichotomy
fa = FF(a,delta_t,r1,r2,A,mu); 
fb = FF(b,delta_t,r1,r2,A,mu); 

// 10 iterations => solution at the end of the dichotomy has an 
// accuracy of (z_max-zmin)/2^10 ~= 0.1
for i = 1:10 
  c = (a + b)/2; 
  fc = FF(c,delta_t,r1,r2,A,mu); 
  K = find (fc .* fa >= 0); 
  a(K) = c(K); fa(K) = fc(K);  
  K = find (fc .* fb >= 0); 
  b(K) = c(K); fb(K) = fc(K);
end

// Newton resolution : Iterate on Equation 5.45 until z is determined to within
// the error tolerance (tol)
// NB: The algorithm is assumed to converge given the accuracy of the initial guess (~0.1) 
tol = 1.e-8;  // error tolerance
nmax = 100;   // limit on the number of iterations
z = (a+b)/2;  // initial guess
ratio = ones(1,N);
n = 0;
I = 1 : N;
while ( I ~= [] & n<=nmax)
  n = n + 1;
  ratio(I) = FF(z(I),delta_t(I),r1(I),r2(I),A(I),mu)./dFdz(z(I),A(I),r1(I),r2(I));
  z(I) = z(I) - ratio(I);
  I = find( (abs(ratio)>tol) );
end

// Error if the maximum number of iterations has been exceeded:
if (I ~= []) then CL__error('Maximum number of iterations exceeded'); end

// Equation 5.46a:
f = ( 1 - yy(z,r1,r2,A)./r1 ).*.ones(3,1);

// Equation 5.46b:
g = ( A.*sqrt(yy(z,r1,r2,A)./mu) ).*.ones(3,1);

// Equation 5.46d:
gdot = ( 1 - yy(z,r1,r2,A)./r2 ).*.ones(3,1);

// Equation 5.28:
v1 = 1 ./g.*(pos2 - f.*pos1);

// Equation 5.29:
v2 = 1 ./g.*(gdot.*pos2 - pos1);

endfunction



