//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [f1,f2]=CL_gm_pixelSize(sat_alt,tilt,target_alt,er)
// Footprint dimensions for a conic sensor at an angle to the vertical
//
// Calling Sequence
// [f1,f2]=CL_gm_pixelSize(sat_alt,tilt,target_alt,[er])
//
// Description
// <itemizedlist><listitem>
// This function computes the size factors of the footprint of a conic sensor at an 
// angle to the descending vertical. 
// <para> If the footprint is small enough, it can be considered as
// the intersection between a cone of revolution and the plane tangent to the planet surface (and containing the target location). </para>
// <para> Then: </para>
// <para> - The footprint outline is a circle with radius D when the sensor is looking to the descending vertical. </para>
// <para> - It is an ellipse with semi-major axis <emphasis role="bold">f1</emphasis>*D and semi-minor axis <emphasis role="bold">f2</emphasis>*D when the sensor is tilted at an angle <emphasis role="bold">tilt</emphasis> to the vertical. </para>
// <para> Two effects are responsible for the change of size: increase of distance 
// (reflected in f1 and f2) and increase of incidence angle (reflected in f1 only). </para>
// <para><inlinemediaobject><imageobject><imagedata fileref="pixelSize.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>
// Note: 
// <para> - A spherical planet is assumed. </para>
// </listitem>
// </itemizedlist>
//
// Parameters
// sat_alt: Satellite altitude [m] (1xN)
// tilt: Angle to the descending vertical [rad] (1xN)
// target_alt: (optional) Target altitude [m]  (default is 0)(1xN)
// er : (optional) Planet radius (default is %CL_eqRad) [m]
// f1: Size factor in tilt plane(1xN)
// f2: Size factor in direction perpendicular to tilt plane(1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_gm_visiParams
//
// Bibliography
// Satellites Orbites et Missions (M. Capderou)
//
// Examples
// // Example 1
// sat_alt = 700.e3;
// tilt = CL_deg2rad(20);
// [K1,K2]=CL_gm_pixelSize(sat_alt,tilt)
// // same computation, with CL_gm_visiParams :
// sat_radius = %CL_eqRad + sat_alt;
// target_radius = %CL_eqRad;
// // distance for the given tilt
// [dist] = CL_gm_visiParams(sat_radius,target_radius,'sat',tilt,'dist') 
// // elevation for the given tilt
// [elev] = CL_gm_visiParams(sat_radius,target_radius,'sat',tilt,'elev') 
// f1 = dist/(sat_alt * sin(elev))
// f2 = dist/sat_alt
//

// Declarations:
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;

// Code:

  if ~exists('er','local') then er=%CL_eqRad; end
  if ~exists('target_alt','local') then target_alt=0; end

  sat_radius = er + sat_alt;
  target_radius = er + target_alt;
  [dist] = CL_gm_visiParams(sat_radius,target_radius,'sat',tilt,'dist') // distance for the given tilt
  [elev] = CL_gm_visiParams(sat_radius,target_radius,'sat',tilt,'elev') // elevation for the given tilt
  f1 = dist./(sat_alt .* sin(elev))
  f2 = dist./sat_alt

endfunction
