//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [om,omp] = CL_rot_angularVelocity(axes,angles,angles_der,angles_der2)
// Rotation angles to angular velocity and acceleration vectors
//
// Calling Sequence
// [om,omp] = CL_rot_angularVelocity(axes,angles,angles_der,angles_der2)
//
// Description
// <itemizedlist><listitem>
// Computes the angular rotation and acceleration vectors resulting from a combination of 3 
// rotations. The rotation axes are defined by numbers: 1=x-axis, 2=y-axis, 3=z-axis.  
// </listitem>
// </itemizedlist>
//
// Parameters
// axes : Axes numbers: 1=x-axis, 2=y-axis or 3=z-axis (1x3 or 3x1)
// angles : Rotation angles around respective axes [rad] (3xN) 
// angles_der : First derivatives of angles with respect to time [rad/s] (3xN) 
// angles_der2 : Second derivatives of angles with respect to time [rad/s^2] (3xN) 
// om : Angular rotation vector (3xN)
// omp : Angular acceleration vector (3xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_defQuat
// CL_rot_matrix2quat
//
// Examples
// axes = [1,2,3];
// angles = [%pi/4 ; %pi/8 ; %pi/6]
// angles_der = [%pi/4 ; %pi/8 ; %pi/6]
// angles_der2 = [%pi/4 ; %pi/8 ; %pi/6]
// [om,omp] = CL_rot_angularVelocity(axes,angles,angles_der,angles_der2)

// Declarations:


// Code:

[lhs,rhs]=argn();
if rhs~=4 CL__error("check number of input arguments"); end,

phi=angles(1,:);
theta=angles(2,:);
psi=angles(3,:);

phip=angles_der(1,:);
thetap=angles_der(2,:);
psip=angles_der(3,:);

phipp=angles_der2(1,:);
thetapp=angles_der2(2,:);
psipp=angles_der2(3,:);

// correspondance
// axis sequence - index value - kinematic equations of motions
straxes=string(axes(1))+string(axes(2))+string(axes(3)); 

// set_sequences = [[1,2,3];[2,3,1]; [3,1,2]; [1,3,2]; [3,2,1]; [2,1,3]; [1,2,1];[2,3,2]; [3,1,3]; [1,3,1]; [3,2,3]; [2,1,2]];
set_sequences = ["123" "231" "312" "132" "321" "213" "121" "232" "313" "131" "323" "212"];

//set_I_J_K =     [1      2     3     1     3     2     1     2     3     1     3     2;   //I
//                 2      3     1     3     2     1     2     3     1     3     2     1;   //J
//                 3      1     2     2     1     3     3     1     2     2     1     3;]; //K

nn = find(set_sequences==straxes);
I=axes(1);
J=axes(2);
K=axes(3);

if (nn>=1 & nn<=3) then
  transformation="A";
elseif (nn>=4 & nn<=6) then
  transformation="B"
elseif (nn>=7 & nn<=9) then
  transformation="C";
elseif (nn>=10 & nn<=12) then
  transformation="D";
else
  CL__error("Unknown rotation order");
end;


om = zeros(3,size(theta,2));
omp = zeros(3,size(theta,2));

select transformation

  case "A"

    om(I)  = phip.*cos(theta).*cos(psi) + thetap.*sin(psi);
    om(J)  = -phip.*cos(theta).*sin(psi) + thetap.*cos(psi);
    om(K)  = psip + phip.*sin(theta);

    omp(I) = phipp.*cos(theta).*cos(psi) - phip.*thetap.*sin(theta).*cos(psi) - phip.*cos(theta).*psip.*sin(psi) + ...
        thetapp.*sin(psi) + thetap.*psip.*cos(psi);
    omp(J) = -phipp.*cos(theta).*sin(psi) + phip.*thetap.*sin(theta).*sin(psi) - phip.*cos(theta).*psip.*cos(psi) + ...
        thetapp.*cos(psi) - thetap.*psip.*sin(psi);
    omp(K) = psipp + phipp.*sin(theta) + phip.*thetap.*cos(theta);

  case "B"

    om(I)  = phip.*cos(theta).*cos(psi) - thetap.*sin(psi);
    om(J)  = phip.*cos(theta).*sin(psi) + thetap.*cos(psi);
    om(K)  = psip - phip.*sin(theta);

    omp(I) = phipp.*cos(theta).*cos(psi) - phip.*thetap.*sin(theta).*cos(psi) - phip.*cos(theta).*psip.*sin(psi) - ...
        thetapp.*sin(psi) - thetap.*psip.*cos(psi);
    omp(J) = phipp.*cos(theta).*sin(psi) - phip.*thetap.*sin(theta).*sin(psi) + phip.*cos(theta).*psip.*cos(psi) + ...
        thetapp.*cos(psi) - thetap.*psip.*sin(psi);
    omp(K) = psipp - phipp.*sin(theta) - phip.*thetap.*cos(theta);

  case "C"

    om(K)  = phip.*sin(theta).*cos(psi) - thetap.*sin(psi);
    om(J)  = phip.*sin(theta).*sin(psi) + thetap.*cos(psi);
    om(I)  = psip - phip.*cos(theta);

    omp(K) = phipp.*sin(theta).*cos(psi) + phip.*thetap.*cos(theta).*cos(psi) - phip.*sin(theta).*psip.*sin(psi) - ...
        thetapp.*sin(psi) - thetap.*psip.*cos(psi);
    omp(J) = phipp.*sin(theta).*sin(psi) + phip.*thetap.*cos(theta).*sin(psi) + phip.*sin(theta).*psip.*cos(psi) + ...
        thetapp.*cos(psi) - thetap.*psip.*sin(psi);
    omp(I) = psipp - phipp.*cos(theta) + phip.*thetap.*sin(theta);

  case "D"

    om(K,:) = -phip.*sin(theta).*cos(psi) + thetap.*sin(psi);
    om(J,:) = phip.*sin(theta).*sin(psi) + thetap.*cos(psi);
    om(I,:) = psip + phip.*cos(theta);

    omp(K,:) = -phipp.*sin(theta).*cos(psi) - phip.*thetap.*cos(theta).*cos(psi) + phip.*sin(theta).*psip.*sin(psi) + ...
        thetapp.*sin(psi) + thetap.*psip.*cos(psi);
    omp(J,:) = phipp.*sin(theta).*sin(psi) + phip.*thetap.*cos(theta).*sin(psi) + phip.*sin(theta).*psip.*cos(psi) + ...
        thetapp.*cos(psi) - thetap.*psip.*sin(psi);
    omp(I,:) = psipp + phipp.*cos(theta) - phip.*thetap.*sin(theta);

end

endfunction
