//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [M] = CL_fr_locOrbMat (pos_car,vel_car,frame_name)
// Inertial frame to custom local orbital frame transformation matrix
//
// Calling Sequence
// [M] = CL_fr_locOrbMat(pos_car,vel_car,frame_name)
//
// Description
// <itemizedlist><listitem>
// Computes the frame transformation matrix <emphasis role="bold">M</emphasis> from 
// the inertial reference frame to a custom 'qsw' or 'tnw' local orbital frame.
// <para> The inertial frame is implicitly the frame relative to which the satellite's
// position and velocity are defined. The local frame is built using these 
// position and velocity vectors. </para>
// <para><emphasis role="bold">frame_name</emphasis> is the name of the local
// frame. The name is a 3-letter word with letters among the following: 'q', 's', 'w', 't', 'n', 
// lower case or upper case. If 'a' (lower case!) is the name of some axis, 'A' (upper case!) 
// is the name of the same axis in the opposite direction. </para>
// <para> Example: "qWS" means: x-axis = q, y-axis = -w, z-axis = -s</para>
// <para> The reference frame that results should be direct. </para>
// </listitem>
// <listitem>
// Notes: 
// <para> By convention, <emphasis role="bold">M</emphasis> multiplied by coordinates
// relative to the inertial frame yields coordinates relative to the local frame. </para>
// </listitem>
// <listitem>
// See <link linkend="Local Frames">Local Frames</link> for  more details on the definition of local frames. 
// </listitem>
// </itemizedlist>
//
// Parameters
// pos_car: satellite's position vector relative to the inertial frame [m] (3xN)
// vel_car: satellite's velocity vector relative to the inertial frame [m/s] (3xN)
// frame_name: (string) name of local frame 
// M : inertial to local orbital frame transformation matrix (3x3xN)
//
// Bibliography
// 1 Mecanique spatiale, CNES - Cepadues 1995, Tome I, section 10.2.2.3 (Definition du Repere orbital local)
// 2 CNES - MSLIB FORTRAN 90, Volume O (Les Reperes orbitaux locaux)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_fr_qswMat
// CL_fr_tnwMat
//
// Examples
// pos_car = [[3500.e3;2500.e3;5800.e3] , [4500.e3;2100.e3;6800.e3]];
// vel_car = [[1.e3;3.e3;7.e3] , [2.e3;3.e3;6.e3]];
//
// // Example 1: Standard 'qsw' local frame: 
// [M] = CL_fr_locOrbMat(pos_car,vel_car,"qsw")
// // is equivalent to: 
// [M] = CL_fr_qswMat(pos_car,vel_car)
//
// // Example 2: 'lvlh': is the same as (s, -w, -q): 
// [M] = CL_fr_locOrbMat(pos_car,vel_car,"sWQ")
// // is equivalent to: 
// [M] = CL_fr_lvlhMat(pos_car,vel_car)

// Declarations:


// Code:

[lhs,rhs] = argn()
if rhs~=3 then CL__error('Invalid number of input arguments'); end

N = size(pos_car,2);

// Axes QSW : 
// Axes commencants par q ou Q
frame_names = [ "qsw","qWs","qSW","qSw", "Qws","QsW","QSw","QWS" ];
// Axes commencants par s ou S
frame_names = [frame_names, "swq","sQw","sWQ","sWq", "Sqw","SwQ","SWq","SQW" ];
// Axes commencants par w ou W
frame_names = [frame_names, "wqs","wSq","wQS","wQs", "Wsq","WqS","WQs","WSQ" ];

// Axes TNW : 
// Axes commencants par t ou T
frame_names = [frame_names, "tnw","tWn","tNW","tNw", "Twn","TnW","TNw","TWN" ];
// Axes commencants par n ou N
frame_names = [frame_names, "nwt","nTw","nWT","nWt", "Ntw","NwT","NWt","NTW" ];
// Axes commencants par w ou W
frame_names = [frame_names, "wtn","wNt","wTN","wTn", "Wnt","WtN","WTn","WNT" ];

// Generer les indices des axes a partir des noms :
//axes = zeros(3, size(frame_names,2));
//for k = 1 : size(frame_names,2)
//  for j = 1 : 3
//	str = part(frame_names(k),j);
//	if(str == "q" | str == "t") then axes(j,k) = 1; end;
//	if(str == "s" | str == "n") then axes(j,k) = 2; end;
//	if(str == "w" ) then axes(j,k) = 3; end;
//	if(str == "Q" | str == "T") then axes(j,k) = -1; end;
//	if(str == "S" | str == "N") then axes(j,k) = -2; end;
//	if(str == "W" ) then axes(j,k) = -3; end;
//  end
//end

// Permutations des axes :
axes = [ 1  1  1  1 -1 -1 -1 -1  2  2  2  2 -2 -2 -2 -2  3  3  3  3 -3 -3 -3 -3  1  1  1  1 -1 -1 -1 -1  2  2  2  2 -2 -2 -2 -2  3  3  3  3 -3 -3 -3 -3  
         2 -3 -2 -2  3  2 -2 -3  3 -1 -3 -3  1  3 -3 -1  1 -2 -1 -1  2  1 -1 -2  2 -3 -2 -2  3  2 -2 -3  3 -1 -3 -3  1  3 -3 -1  1 -2 -1 -1  2  1 -1 -2  
         3  2 -3  3  2 -3  3 -2  1  3 -1  1  3 -1  1 -3  2  1 -2  2  1 -2  2 -1  3  2 -3  3  2 -3  3 -2  1  3 -1  1  3 -1  1 -3  2  1 -2  2  1 -2  2 -1 ];  
 
type_frame = [ ones(1,24) , 2*ones(1,24) ];

k = find(frame_name == frame_names);
if(k == [])
  CL__error("Unrecognized frame name");
end

// Frame type = QSW
if(type_frame(k) == 1)
  [P] = CL_fr_qswMat(pos_car,vel_car);
// Frame type = TNW
elseif(type_frame(k) == 2)
  [P] = CL_fr_tnwMat(pos_car,vel_car);
end

// Signes des axes
sa1 = sign(axes(1,k));
sa2 = sign(axes(2,k));
sa3 = sign(axes(3,k));

// Indices des axes
ia1 = abs(axes(1,k));
ia2 = abs(axes(2,k));
ia3 = abs(axes(3,k));

// hypermat remplit element par element en faisant varier l'indice de gauche d'abord :
M = hypermat([3 3 N] , [sa1*P(ia1,1,:); sa2*P(ia2,1,:); sa3*P(ia3,1,:); ...
                        sa1*P(ia1,2,:); sa2*P(ia2,2,:); sa2*P(ia3,2,:); ...
                        sa1*P(ia1,3,:); sa3*P(ia2,3,:); sa3*P(ia3,3,:)]);
if(N == 1)
    M = M(:,:,1);
end

endfunction
