//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [dv,arg_lat]=CL_man_incRaanCirc(sma,inci,raani,incf,raanf,mu)
// Inclination and RAAN maneuver for circular orbits
//
// Calling Sequence
// [dv,arg_lat]=CL_man_incRaanCirc(sma,inci,raani,incf,raanf [,mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the dv needed for an inclination and RAAN change for a circular orbit. 
// <para><emphasis role="bold">dv</emphasis> is the velocity increment vector (in spherical coordinates) in the QSW frame : [lambda; phi; norm], where lambda: angle between the radius vector and the projection of dv on the orbit plane, phi: angle between dv and its projection on the orbit plane, norm = |dv|. </para>
// <para><emphasis role="bold">arg_lat</emphasis> is the argument of latitude of the maneuver.</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// sma : Semi-major axis [m] (1xN)
// inci: Initial inclination [rad] (1xN)
// raani: Initial right ascension of ascending node [rad] (1xN)
// incf: Final inclination [rad] (1xN)
// raanf: Final right ascension of ascending node [rad] (1xN)
// mu : (optionnal) Gravitational constant. [m^3/s^2] (default value is %CL_mu)
// dv : Velocity increment in spherical coordinates in the QSW frame [lambda;phi;|dv|] [rad;rad;m/s] (3xN)
// arg_lat: Argument of latitude of the maneuver [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// sma = 7200.e3;
// inci = CL_deg2rad(111);
// raani = CL_deg2rad(12);
// incf = CL_deg2rad(108);
// raanf = CL_deg2rad(15);
// [dv,arg_lat]=CL_man_incRaanCirc(sma,inci,raani,incf,raanf)
// kep = [sma*ones(inci);zeros(inci);inci;zeros(inci);raani;arg_lat];
// kep1 = CL_man_applyDv(kep,dv)

// Declarations:
if(~exists('%CL_mu')) then global %CL_mu; end;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end

// No need to check the output variable 'intersect' ! 
[arg_lat_1,arg_lat_2] = CL_gm_intersectPlanes(inci,raani,incf,raanf)

w1 = [sin(raani) .* sin(inci)
    -cos(raani) .* sin(inci)
    cos(inci) ];
na1 = [ cos(raani) ; sin(raani) ; zeros(raani)];

r1 = CL_colMult(cos(arg_lat_1),na1)  + CL_colMult(sin(arg_lat_1),CL_cross(w1,na1));

s1 =  CL_cross(w1,r1);

w2 = [sin(raanf) .* sin(incf)
    -cos(raanf) .* sin(incf)
    cos(incf) ];
s2 = CL_cross(w2,r1);


V = sqrt(mu./sma);
dv_iner = CL_colMult(V, s2 - s1);

dv_qsw = [ CL_dot(dv_iner,r1) ; CL_dot(dv_iner,s1) ; CL_dot(dv_iner,w1) ];

dv = CL_co_car2sph(dv_qsw);
arg_lat = arg_lat_1;


endfunction
