//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [E] = CL_kp_v2E(ecc,v)
// True anomaly to eccentric anomaly
//
// Calling Sequence
// E = CL_kp_v2E(ecc,v)
//
// Description
// <itemizedlist><listitem>
// Computes eccentric anomaly from true anomaly and eccentricity for 
// elliptical and hyperbolic orbits.
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// v: True anomaly [rad] (PxN or Px1)
// ecc: Eccentricity (PxN or Px1)
// E: Eccentric anomaly. Undefined (%nan) if the true anomaly is not in the correct range.  [rad] (PxN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 Orbital Mechanics for Engineering Students, H D Curtis, Chapter 3, equation 3.10a
// 2 Mecanique spatiale, CNES - Cepadues 1995, Tome I, section 4.6.6, page 179
//
// See also
// CL_kp_M2Ecir
// CL_kp_M2E
// CL_kp_E2v
//
// Examples
// // consistency test
// v=[3.4794 5.7281];
// ecc=[0.37255 0.55];
// E = CL_kp_v2E(ecc,v);
// v2 = CL_kp_E2v(ecc,E);

// Declarations:
if(~exists('%CL_epsilon')) then global %CL_epsilon; end;

// Code:

Nv = size(v,2)
Ne = size(ecc,2)
N = max(Nv,Ne)

if ~( Nv==1 | Ne==1 | Ne==Nv ) 
   CL__error('Wrong size of input arguments'); 
end

if (Nv==1); v = v * ones(1,N); end
if (Ne==1); ecc = ecc * ones(1,N); end

if (find(ecc < 0) ~= [])
  CL__error("Negative eccentricity");
end

if (find(ecc >= 1 - %CL_epsilon.parab & ecc <= 1 + %CL_epsilon.parab) ~= [])
  CL__error("Parabolic orbits not handled");
end

E = zeros(v)

// -------------------------------
// Ellipse
// -------------------------------

I = find(ecc < 1);

if (I ~= [])
  betaa = ecc(I) ./ (1 + sqrt(1-ecc(I).^2))
  E(I) = v(I) - 2*atan(betaa .* sin(v(I)), 1 + betaa .* cos(v(I)))
end

// -------------------------------
// Hyperbola
// -------------------------------

I = find(ecc > 1);

if (I ~= [])

  E(I) = %nan; 

  if (find(abs(v(I)) > %pi) ~= [])
    CL__error('Hyperbola: true anomaly > pi'); 
  end

  // check v < vinf 
  // vinf = acos(-1/e) : invalid value (E would be infinity)
  k = find(cos(v(I)) .* ecc(I) > -1+%eps & abs(v(I)) < %pi-%eps); 
  I = I(k); 

  E(I) = 2*atanh( sqrt((ecc(I)-1) ./ (ecc(I)+1)) .* tan(v(I)/2) ); 

end

endfunction
