//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [view_angle]=CL_gm_equatorialSwath(alt,inc,er,rotr_pla)
// Sensor view angle for an overlap ratio of 1 at equator
//
// Calling Sequence
// [view_angle]=CL_gm_equatorialSwath(alt,inc,[er,rotr_pla])
//
// Description
// <itemizedlist><listitem>
// This function computes the view angle as seen from the satellite (angle from the descending vertical) so that the overlap ratio at equator is 1. The overlap ratio is the ratio Q=Leq/Ded where: 
// <para> - Leq: portion of the equator seen by the sensor during the time it crosses the equator. </para>
// <para> - Deq: gap between 2 consecutive ground tracks at equator. </para>
// </listitem>
// <listitem>
// Notes: 
// <para> - The planet is assumed spherical, and the orbit circular. </para> 
// <para> - The sensor is conic and oriented towards the descending vertical. </para> 
// <para> - The gap between 2 consecutive ground tracks is computed taking the effect of J2 into account. </para> 
// <para> - The result is undefined (%nan) if inclination is 0 or %pi.</para> 
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-06-03 )</emphasis></para>
//
// Parameters
// alt: Satellite's altitude [m] (1xN)
// inc: Orbit's inclination [rad] (1xN)
// view_angle: Sensor half view angle [rad] (1xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// rotr_pla_sun : (optional) Mean apparent rotation rate of the Sun around the planet (default is %CL_rotrEarthSun)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// Satellites orbites et missions - Michel Capderou
//
// See also
// CL_gm_equatorialOverlap
//
// Examples
// alt = [822.e3 630.e3] ;
// inc = [CL_deg2rad(98.72) CL_deg2rad(96.72)];
// [view_angle] = CL_gm_equatorialSwath(alt,inc)

// Declarations:
if(~exists('%CL_eqRad')) then global %CL_eqRad; end;
if(~exists('%CL_rotrEarth')) then global %CL_rotrEarth; end;

// Code:
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('rotr_pla','local') then rotr_pla=%CL_rotrEarth; end

N = max(size(alt,2), size(inc,2));
alt = alt .* ones(1,N);
inc = inc .* ones(1,N);

I = find (alt <= 0 | inc < 0-%eps | inc > %pi+%eps); 
if ~isempty(I); CL__error('Arguments out of range'); end

I = find (inc >=%pi-%eps & inc <= %eps); 
if ~isempty(I)
  view_angle = %nan * ones(1,N);
  return;
end


sma = er+alt;
ecc = 0;

// Calcul de l'inclinaison apparrente :
[mmj2] = CL_op_paramsJ2('mm',sma,ecc,inc);
[drift_pom,drift_gom,drift_M] = CL_op_driftJ2(sma,ecc,inc);
incapp = atan((mmj2.*sin(inc)),(mmj2.*cos(inc)-(rotr_pla-drift_gom)));
[dlong] = CL_op_paramsJ2('lgap',sma,ecc,inc);
alpha=dlong./(2.*(sin(inc)+cos(inc).*tan(inc-incapp)));
[view_angle]=CL_gm_visiParams(sma,er,"cen",alpha,"sat");

endfunction

