//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [val2] = CL_unitConvert(val1, unit1, unit2)
// Unit conversion
//
// Calling Sequence
// [val2] = CL_unitConvert(val1, unit1, unit2)
//
// Description
// <itemizedlist>
// <listitem>
// Converts a quantity expressed in unit1 into unit2. The recognized units are:
// <para>m: meter</para>
// <para>km: kilometer (1000 meters)</para>
// <para>s: second</para>
// <para>mn: minute (60 seconds) </para>
// <para>h: hour (60 minutes) </para>
// <para>day: day (24 hours)</para>
// <para>yr: year (365.25 days)</para>
// <para>g: gramme</para>
// <para>kg: kilogramme (1000 grammes) </para>
// <para>au: astronomical unit (%CL_au)</para>
// <para>N: newton</para>
// <para>Hz: herz</para>
// <para>kHz: kiloherz (1000 Hz)</para>
// <para>rad: radian</para>
// <para>deg: degree (%pi/180 radians)</para>
// </listitem>
// <listitem>
// <para>Notes: </para>
// <para>- The consistency between unit1 and unit2 is not checked. Thus CL_unitConvert(1, "kg", "m") generates no error!.</para>
//  <para>- The case where some existing variable is passed in place of a valid unit is not detected either.</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// val1: Value of some quantity 'X' expressed in unit1. 
// unit1: Initial unit (string).
// unit2: Final unit (string).
// val2: Value of 'X' expressed in unit2. val2 is [] if the units are not valid.
//
// Authors
// CNES - DCT/SB (AL)
//
// Examples
// val = CL_unitConvert(1, "m/s", "km/h")
// val = CL_unitConvert(1, "au^2", "km^2")
// val = CL_unitConvert(0:90:360, "deg", "rad")

// Declarations:
global %CL_au;

// Code:

  // reference units
  m   = 1;
  s   = 1;
  kg  = 1;
  rad = 1;
  Hz  = 1;

  // derived units
  km  = 1000      * m;
  au  = %CL_au    * m;  // NB: %CL_au must be defined in meters ! 
  h   = 3600      * s;
  mn  = 60        * s;
  day = 86400     * s;
  yr =  365.25    * day;
  g   = 1.e-3     * kg;
  deg = (%pi/180) * rad;
  N   = 1         * kg*m/s^2;
  kHz = 1000      * Hz;

  try
    f1 = evstr(unit1); 
    f2 = evstr(unit2); 
    val2 = val1 * f1 / f2;
  catch
    val2 = [];
  end

endfunction
