//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function CL_init()
// Initialization of CelestLab data
//
// Description
// <itemizedlist><listitem>
// This function initializes CelestLab data. 
// <para> These data consist of physical values and other constants. The data are defined 
// as globals and returned to the caller as locals. </para> 
// <para> Values may be changed if necessary after a call to CL_init(). </para>
// </listitem>
// <listitem>
// <emphasis role="bold">List of CelestLab global variables</emphasis> (see references below): 
// <para>%CL_g0: Standard value for Earth gravity acceleration at sea level [m/s^2] - {1}</para>
// <para>%CL_eqRad: Equator Earth radius [m] - {4}</para>
// <para>%CL_obla: Earth oblateness - {4}</para>
// <para>%CL_mu: Earth gravitational constant [m^3/s^2] - {4}</para>
// <para>%CL_radiusMoon: Moon equatorial radius [m] - {11}</para>
// <para>%CL_muMoon: Moon gravitational constant [m^3/s^2] - {11}</para>
// <para>%CL_radiusSun: Sun equatorial radius [m] - {2},{3}</para>
// <para>%CL_muSun: Sun gravitational constant [m^3/s^2] - {2}</para>
// <para>%CL_rotrEarthSun: Mean apparent rotation rate of the Sun around the Earth [rad/s] - {9}</para>
// <para>%CL_j2:  2nd zonal harmonic - {4}</para>
// <para>%CL_j3:  3rd zonal harmonic - {4}</para>
// <para>%CL_j4:  4th zonal harmonic - {4}</para>
// <para>%CL_j5:  5th zonal harmonic - {4}</para>
// <para>%CL_j6:  6th zonal harmonic - {4}</para>
// <para>%CL_j1jn : Zonal harmonics j1 to j70 (70x1 vector) - {4}</para>
// <para>%CL_obliquity2000: Earth mean pole obliquity at J2000 epoch [rad] - {7}</para>
// <para>%CL_au: Astronomical unit [m] - {2}</para>
// <para>%CL_lightSpeed: Speed of light [m/s] - {6},{3}</para>
// <para>%CL_rotrEarth: Rotation rate of the Earth wrt to Gamma50 Veis reference frame [rad/s] - {9}</para>
// <para>%CL_gravCst: Universal gravitational constant [m^3.s^-2.kg^-1] - {3},{5}</para>
// <para>%CL_criticalInc1: critical inclination (so that drift of argument of perigee = 0) = sqrt(asin(4/5)) [rad] </para>
// <para>%CL_criticalInc2: critical inclination (so that drift of argument of perigee = 0) = %pi-sqrt(asin(4/5)) [rad] </para>
// <para>%CL_planetC: tlist containing solar system planet constants</para>
// <para>%CL_planetC.mercury.eqRad: Mercury radius [m] - {10}</para>
// <para>%CL_planetC.mercury.obla: Mercury oblateness</para>
// <para>%CL_planetC.mercury.mu: Mercury gravitational constant [m^3/s^2] - {12}</para>
// <para>%CL_planetC.venus.eqRad: Venus radius [m] - {10}</para>
// <para>%CL_planetC.venus.obla: Venus oblateness </para>
// <para>%CL_planetC.venus.mu: Venus gravitational constant [m^3/s^2] - {12}</para>
// <para>%CL_planetC.earth.eqRad: Earth radius [m] - {4}</para>
// <para>%CL_planetC.earth.obla: Earth oblateness - {4}</para>
// <para>%CL_planetC.earth.mu: Earth gravitational constant [m^3/s^2] - {4}</para>
// <para>%CL_planetC.mars.eqRad: Mars radius [m] - {10}</para>
// <para>%CL_planetC.mars.obla: Mars oblateness </para>
// <para>%CL_planetC.mars.mu: Mars gravitational constant [m^3/s^2] - {12}</para>
// <para>%CL_planetC.jupiter.eqRad: Jupiter radius [m] - {10}</para>
// <para>%CL_planetC.jupiter.obla: Jupiter oblateness </para>
// <para>%CL_planetC.jupiter.mu: Jupiter gravitational constant [m^3/s^2] - {12}</para>
// <para>%CL_planetC.saturn.eqRad: Saturn radius [m] - {10}</para>
// <para>%CL_planetC.saturn.obla: Saturn oblateness </para>
// <para>%CL_planetC.saturn.mu: Saturn gravitational constant [m^3/s^2] - {12}</para>
// <para>%CL_planetC.uranus.eqRad: Uranus radius [m] - {10}</para>
// <para>%CL_planetC.uranus.obla: Uranus oblateness </para>
// <para>%CL_planetC.uranus.mu: Uranus gravitational constant [m^3/s^2] - {12}</para>
// <para>%CL_planetC.neptune.eqRad: Neptune radius [m] </para>
// <para>%CL_planetC.neptune.obla: Neptune oblateness </para>
// <para>%CL_planetC.neptune.mu: Neptune gravitational constant [m^3/s^2] - {12}</para>
// <para>%CL_planetC.pluto.eqRad: Pluto radius [m] - {3} (Pluto is not considered as a planet any more)</para>
// <para>%CL_planetC.pluto.obla: Pluto oblateness </para>
// <para>%CL_planetC.pluto.mu: Pluto gravitational constant [m^3/s^2] - {12}</para>
// <para>%CL_TAI_UTC: leap seconds (TAI-UTC). 3-column matrix containing: beginning/end of period (in julian days from 1950.0), value of TAI-UTC (number of leap seconds) </para>
// <para>%CL_deg2rad: %pi/180, conversion deg to rad </para>
// <para>%CL_rad2deg: 180/%pi, conversion rad to deg </para>
// <para>%CL_epsilon: structure containing data for orbit type checking </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Bibliography
//   {1} LES SYSTEMES DE REFERENCE UTILISES EN ASTRONOMIE, M. Chapront-Touze, G. Francou et B. Morando, Service des Calculs et de Mecanique Celeste du Bureau des Longitudes
//   {2} DE 200 (Standish and Willians, 1981), (Kaplan, 1981) and (Fukushima, 1991)
//   {3} UAI 1976 (UAI, 1976)
//   {4} JGM3 Potential (AMLIB database)
//   {5} MERIT 1983 (Melbourne, 1983)
//   {6} Mecanique spatiale, CNES - Cepadues 1995
//   {7} IERS 1992 (McCarthy, 1992)
//   {8} A5-NT-1-J-0225-EADS, First simulation flight ATV Jules Verne on L528 A5/ESV launcher
//   {9} MSLIB90
//   {10} Fukushima 1990
//   {11} Introduction aux ephemerides astronomiques, J.-L. Simon, M. Chapront-Touze, B. Morando et W. Thuillot, Bureau des Longitudes
//   {12} IMCCE  http://www.imcce.fr/fr/ephemerides/donnees/pposs/moteur.php
//
// See also
//
// Authors
// CNES - DCT/SB

// Declarations:

// liste des variables globales exportees
global %CL_TAI_UTC %CL_au %CL_criticalInc1 %CL_criticalInc2 %CL_deg2rad ...
       %CL_epsilon %CL_eqRad %CL_g0 %CL_gravCst ...
       %CL_j1jn %CL_j2 %CL_j3 %CL_j4 %CL_j5 %CL_j6 ...
       %CL_lightSpeed %CL_mu %CL_muMoon %CL_muSun %CL_obla ...
       %CL_obliquity2000 %CL_rotrEarth %CL_rotrEarthSun %CL_planetC ...
       %CL_rad2deg %CL_radiusMoon %CL_radiusSun;

global %CL_earthRadius; // not to be used - kept only for compatibility reasons

// Code:

  ///////////
  // EARTH //
  ///////////

  %CL_eqRad = 6378136.3;  // [m] Earth equatorial radius - JGM3 Potential
  %CL_earthRadius = %CL_eqRad;  // [m] Earth equatorial radius - JGM3 Potential
  %CL_obla = 1/298.25781000000; // Earth oblateness - JGM3 Potential
  %CL_mu = 3.986004415e14; // geocentric gravitational constant (m^3/s^2) - JGM3 Potential

  %CL_g0 = 9.80665;  // mean Earth acceleration due to gravity [m/s^2] - Standard value

  %CL_j2 = 0.00108262669059782; //2nd zonal harmonic - JGM3 Potential
  %CL_j3 = -2.53243547943241e-06; //3rd zonal harmonic - JGM3 Potential
  %CL_j4 = -1.61933127174051e-06; //4th zonal harmonic - JGM3 Potential
  %CL_j5 = -2.27716112895682e-07; //5th zonal harmonic - JGM3 Potential
  %CL_j6 = 5.39648498594285e-07; //6th zonal harmonic - JGM3 Potential

  // JGM3 Potential J1 to J70
  %CL_j1jn = [ 0.
          0.00108262669059782
          -2.53243547943241e-06
          -1.61933127174051e-06
          -2.27716112895682e-07
          5.39648498594285e-07
          -3.51368443669702e-07
          -2.02518718143151e-07
          -1.19368714898869e-07
          -2.48056865754093e-07
          2.40565216215095e-07
          -1.81911694596695e-07
          -2.07567730643349e-07
          1.17417389371202e-07
          -1.76272694689137e-08
          3.11943094249257e-08
          -1.07130592684284e-07
          -4.42167249859155e-08
          2.19733398304135e-08
          -1.20314617695259e-07
          -4.92459305790164e-08
          7.55354073197852e-08
          1.57100136082411e-07
          4.26855017998661e-09
          -3.45975500825326e-08
          -5.79992158190519e-08
          -2.49739104998735e-08
          6.15584743530392e-08
          1.5116059801076e-08
          -1.03392057692495e-07
          -4.4735692916947e-08
          2.07584858547697e-08
          6.6397264033699e-09
          6.88679451544847e-08
          -5.77565952474551e-08
          4.31634924417224e-08
          5.95461741021507e-08
          8.22864852043474e-08
          -9.01023120106021e-09
          3.43879724695743e-08
          8.68557607744411e-09
          -2.24712519709114e-09
          -5.98908001617914e-08
          -3.20822040439335e-08
          4.40317791662939e-08
          -2.17993398739857e-08
          1.7212653356465e-09
          -8.68697322129902e-08
          9.90633841472856e-09
          1.30428365947208e-08
          7.33807737493882e-08
          -3.11627077052876e-08
          -5.17067907327928e-08
          -1.8361593860458e-08
          -2.41368628455192e-08
          3.15082802134974e-08
          2.22744022003168e-08
          4.92129929178496e-08
          -2.39564694291193e-09
          4.61046978372794e-08
          -9.41548851112535e-09
          -7.93909519203522e-09
          2.73656964065564e-08
          3.36767136008911e-08
          1.78321140940191e-10
          1.28436129062281e-08
          2.66227214973719e-10
          -5.38782109805533e-09
          -1.63849001673196e-08
          1.29765521503902e-08];


  // Earth mean pole obliquity at J2000 epoch (23deg, 26min, 21.41119secondes) IERS1992
  %CL_obliquity2000 = 0.40909262920459006; //[rad]

  // astronomical unit DE 200
  %CL_au = 1.4959787066E+11; //[m]

  // the speed of light (m/s) UAI 1976, Mecanique Spatiale, CNES
  %CL_lightSpeed = 299792458; //[m/s]

  // angular rotation speed of the Earth
  %CL_rotrEarth = 0.729211514670520957673456605920873977E-4; //[rd/s]

  // universal gravitational constant
  %CL_gravCst = 6672e-14;  //[m^3.s^-2.kg^-1]

   // Sun geocentric gravitational constant (m^3/s^2) DE 200
  %CL_muSun = 1.32712440e20; //[m^3/s^2]

   // moon geocentric gravitational constant (m^3/s^2)
  %CL_muMoon = 49027989e5; //[m^3/s^2]

  // Equatorial moon radius UAI 1976, DE 200
  %CL_radiusMoon = 1737400; //[m]

  // Equatorial Sun radius UAI 1976, DE 200
  %CL_radiusSun = 696000000; //[m]

  // Sun inertial rotation period coherent avec %CL_rotrEarth : 86400 = 2*%pi/ (%CL_rotrEarth - %CL_rotrEarthSun)
  %CL_rotrEarthSun = 0.1990993006216967288585452383952014E-6; //[ RAD/S ]

  // Critical inclinations (so that argument of perigee drift = 0)
  %CL_criticalInc1 = 1.1071487177940905030170654601785370400;
  %CL_criticalInc2 = 2.0344439357957027354455779231009658441;

  // TAI_UTC
  %CL_TAI_UTC = [-1    8035   00.000
           8035  8217   10.000
           8217  8401   11.000
           8401  8766   12.000
           8766  9131   13.000
           9131  9496   14.000
           9496  9862   15.000
           9862 10227   16.000
          10227 10592   17.000
          10592 10957   18.000
          10957 11504   19.000
          11504 11869   20.000
          11869 12234   21.000
          12234 12965   22.000
          12965 13879   23.000
          13879 14610   24.000  //1 janv 1988  1 janv 1990
          14610 14975   25.000  //1 janv 1990  1 janv 1991
          14975 15522   26.000  //1 janv 1991
          15522 15887   27.000  //1 juil 1992
          15887 16252   28.000  //1 juil 1993
          16252 16801   29.000  //1 juil 1994
          16801 17348   30.000  //1 janv 1996
          17348 17897   31.000  //1 juil 1997
          17897 20454   32.000  //1 janv 1999
          20454 21550   33.000  //1 janv 2006
          21550 99999   34.000];  //1 janv 2009


  //equatorial radius oblateness and mu for solar system planets
  //Source IMCCE  http://www.imcce.fr/fr/ephemerides/donnees/pposs/moteur.php

  //mercury
  eqRad = 2439.7e3;  //[m]
  obla = 0;
  mu=22032080000000; //m3s-2
  mercury = tlist(['mercury','name','eqRad','obla','mu'],'mercury',eqRad,obla,mu);

  //venus
  eqRad = 6051.8e3;  //[m]
  obla = 0;
  mu=324858600000000;
  venus = tlist(['venus','name','eqRad','obla','mu'],'venus',eqRad,obla,mu);

  //earth
  // eqRad = 6378137; //[m] MERIT 1983
  // obla = 1/298.257; //MERIT 1983, IERS 1989, IERS 1992
  // mu = ??
  eqRad = %CL_eqRad;
  obla = %CL_obla;
  mu=%CL_mu;
  earth = tlist(['earth','name','eqRad','obla','mu'],'earth',eqRad,obla,mu);

  //mars
  eqRad = 3396.19e3;  //[m]
  obla = (eqRad-3376.20e3)/eqRad; //mean
  mu=42828314000000;
  mars = tlist(['mars','name','eqRad','obla','mu'],'mars',eqRad,obla,mu);

  //jupiter
  eqRad = 71492e3; //[m]
  obla = (eqRad-66854e3)/eqRad;
  mu=1.26686537e+17;
  jupiter = tlist(['jupiter','name','eqRad','obla','mu'],'jupiter',eqRad,obla,mu);

  //saturn
  eqRad = 60268e3;
  obla = (eqRad-54364e3)/eqRad;
  mu=3.79312e+16
  saturn = tlist(['saturn','name','eqRad','obla','mu'],'saturn',eqRad,obla,mu);

  //uranus
  eqRad = 25559e3;
  obla = (eqRad-24973e3)/eqRad;
  mu=5.7939393e+15;
  uranus = tlist(['uranus','name','eqRad','obla','mu'],'uranus',eqRad,obla,mu);

  //neptune
  eqRad = 24764e3;
  obla = (eqRad-24341e3)/eqRad;
  mu=6.835107e+15;
  neptune = tlist(['neptune','name','eqRad','obla','mu'],'neptune',eqRad,obla,mu);

  //pluto
  eqRad = 1195e3;
  obla = 0;
  mu=826100000000;
  pluto = tlist(['pluto','name','eqRad','obla','mu'],'pluto',eqRad,obla,mu);

  //tlist containing planet constants
  %CL_planetC = tlist(['CLplanetC',..
  mercury.name,venus.name,earth.name,mars.name,jupiter.name,saturn.name,uranus.name,neptune.name,pluto.name],..
  mercury,venus,earth,mars,jupiter,saturn,uranus,neptune,pluto);

  clear eqRad obla mercury venus earth mars jupiter saturn uranus neptune pluto;


//Conversion of units constants:
  %CL_deg2rad = %pi/180;
  %CL_rad2deg = 180/%pi;


//   <para>--- TEST CONSTANTS: ---</para>
//   <para>%CL_epsilon: tlist containing threshold parameters for tests</para>
//   <para>%CL_epsilon.test: %eps; used in regression tests</para>
//   <para>%CL_epsilon.cir: 1e-10, circular eccentricity limit (if e .lt. eps_cir then circular orbit) - {9}</para>
//   <para>%CL_epsilon.parab: 1e-10, parabolic eccentricity limit (if e .le. 1-eps_parab then elliptic orbit; if e .ge. 1-eps_parab then hyperbolic orbit, parabolic otherwise) - {9}</para>
//   <para>%CL_epsilon.equa: 1e-10, equatorial inclination limit (if sin(i) .lt. eps_equa then equatorial orbit) - {9}</para>

  //test epsilons on types of orbit and others
  %CL_epsilon = tlist(['CLepsilon','test','cir','parab','equa']);
  %CL_epsilon.test = %eps; //test used in regression tests
  %CL_epsilon.cir = 1.e-10;    //if e < eps_cir, circular orbit
  %CL_epsilon.parab = 1.e-10;  //if e <= 1-eps_parab, elliptic orbit
                               //if e >= 1+eps_parab, hyperbolic orbit, parabolic otherwise
  %CL_epsilon.equa = 1.e-10;   //if sin(i) < eps_equa, equatorial orbit

  // transfert des variables globales dans l'environnement appelant
  [%CL_mu,%CL_g0,%CL_eqRad,%CL_earthRadius,%CL_obla,%CL_mu,%CL_radiusMoon,%CL_muMoon,%CL_radiusSun,...
  %CL_muSun,%CL_rotrEarthSun,%CL_j2,%CL_j3,%CL_j4,%CL_j5,%CL_j6,%CL_j1jn,%CL_obliquity2000,%CL_au,%CL_lightSpeed,...
  %CL_rotrEarth,%CL_gravCst,%CL_criticalInc1,%CL_criticalInc2,%CL_planetC,%CL_TAI_UTC,%CL_deg2rad,%CL_rad2deg,%CL_epsilon] = ...
  ...
  return(%CL_mu,%CL_g0,%CL_eqRad,%CL_earthRadius,%CL_obla,%CL_mu,%CL_radiusMoon,%CL_muMoon,%CL_radiusSun,...
         %CL_muSun,%CL_rotrEarthSun,%CL_j2,%CL_j3,%CL_j4,%CL_j5,%CL_j6,%CL_j1jn,%CL_obliquity2000,%CL_au,%CL_lightSpeed,...
         %CL_rotrEarth,%CL_gravCst,%CL_criticalInc1,%CL_criticalInc2,%CL_planetC,%CL_TAI_UTC,%CL_deg2rad,%CL_rad2deg,...
         %CL_epsilon);


endfunction
