//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function  [angle] = CL__sc2angle(x,y)
// Sinus and cosinus to angle
//
// Calling Sequence
// [angle] = CL__sc2angle(x,y)
//
// Description
// <itemizedlist><listitem>
// calculates an angle given its cosinus <emphasis role="bold">x</emphasis> and sinus <emphasis role="bold">y</emphasis>.
// </listitem>
// </itemizedlist>
//
// Parameters
// x: cosinus (1xN)
// y: sinus (1xN)
// angle: angle with cos(angle)=x and sin(angle)=y. (1xN) [rad]
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume U (mu_angle2)
//
// Examples
// angle = CL__sc2angle(cos(%pi/4),sin(%pi/4))
//

// Declarations:


// Code:

if x==[] & y==[]
  angle = []
  return
end

//angle = CL_rMod(atan(y, x),0,2*%pi);
//return


// initialisation constante de test
presque_zero = 2.2250738585072013e-308  //plus petit reel positif non nul selon mslib

// Controle des parametres d'entree (x=y=0)
ii_0 = find((abs(x) <= presque_zero) & (abs(y) <= presque_zero))
if ii_0~=[] //arbitrary choice x=1 and y=1 to allow calculation. At the end angle(ii_0)=0
  x(ii_0) = 1
  y(ii_0) = 1
end

arctg_angle = zeros(x);
s = zeros(x);
angle = zeros(x);
// Calcul arctg(s) avec s = min( | x | , | y |) / max( | x | , | y |)
ii = find(abs(x) < abs(y)) // si   | x | < | y | :  s = | x | / | y |  et  angle = pi/2 - arctg(s)
jj = find(~(abs(x) < abs(y))) // si | x | > | y | :   s = | y | / | x |  et angle = arctg(s)
s(ii) = abs(x(ii)) ./ abs(y(ii))
arctg_angle(ii) = %pi./2 - atan(s(ii))
s(jj) = abs(y(jj)) ./ abs(x(jj))
arctg_angle(jj) = atan(s(jj))

// Determination signe et quadrant de l'angle suivant les signes de x et y
i1 = find(y >= 0)
angle(i1) = arctg_angle(i1) // si y >= 0 et x >= 0: angle = angle
i2 = find((y >= 0)&(x < 0))
angle(i2) = %pi - arctg_angle(i2) // si y >= 0 et x < 0: angle = pi - angle
j1 = find((~(y >= 0))&(x > 0))
angle(j1) = 2.*%pi - arctg_angle(j1)  // si y < 0 et x > 0: angle = 2pi - angle
j2 = find((~(y >= 0))&(~(x > 0)))
angle(j2) = %pi + arctg_angle(j2) // si y < 0 et x <= 0: angle = pi + angle

//Special case
angle(ii_0)=0

endfunction
