//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [drift_pom,drift_gom,drift_M,dpomdaei,dgomdaei,dMdaei] = CL_op_driftJ2(sma,ecc,inc,er,mu,j2)
// Drift of orbital elements due to J2
//
// Calling Sequence
// [drift_pom,drift_gom,drift_M[,dpomdaei[,dgomdaei[,dMdaei]]]] = CL_op_driftJ2(sma,ecc,inc[,er[,mu[,j2]]])
//
// Description
// <itemizedlist><listitem>
// Computes: 
// <para>- the secular drifts (drift_pom=d(pom)/dt, drift_gom=d(gom)/dt, drift_M=dM/dt) on the keplerian elements, where pom=argument of periapsis, 
// gom=right ascension of ascending node,M=mean anomaly, considering the effect of the first zonal harmonic J2, </para>
// <para>- the partial derivatives of these drifts with respect to semi major axis, eccentricity and inclination.</para>
// <para>(See the formulas, where: R = equatorial radius, n = keplerian mean motion</para>
// </listitem>
// <listitem><inlinemediaobject><imageobject><imagedata fileref="driftJ2.gif"/></imageobject></inlinemediaobject></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// sma: Semi-major axis [m] (1x1 or 1xN)
// ecc: Eccentricity  (1x1 or 1xN)
// inc: Inclination [rad] (1x1 or 1xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// j2: (optional) Second zonal harmonic) (default is %CL_j2)
// drift_pom: drift of the argument of periapsis [rad/s] (1xN)
// drift_gom: drift of the right ascension of ascending node [rad/s] (1xN)
// drift_M: drift of the mean anomaly [rad/s] (1xN)
// dpomdaei: partial derivative of drift_pom with respect to sma, ecc and inc (3xN)
// dgomdaei: partial derivative of drift_gom with respect to sma, ecc and inc (3xN)
// dMdaei: partial derivative of drift_M with respect to sma, ecc and inc (3xN)
//
// Bibliography
// CNES - MSLIB FORTRAN 90, Volume E (me_deriv_secul_J2)
//
// Authors
// CNES - DCT/SB
//
// Examples
// sma=[%CL_eqRad+350000,%CL_eqRad+700000];
// ecc=[0.001,0.001];
// inc=CL_deg2rad([51.6,51.6]);
// [drift_pom,drift_gom,drift_M]=CL_op_driftJ2(sma,ecc,inc)
//

// Declarations:
global %CL_eqRad %CL_j2 %CL_mu;

// Code:

lhs = argn(1) // number of left arguments
if (lhs > 6); CL__error('bad number of output arguments'); end 

Nsma = size(sma,2);
Necc = size(ecc,2);
Ninc = size(inc,2);

N = max(Nsma, Ninc, Necc);

// error and consistency checks
coherence = (Nsma==N|Nsma==1) & (Necc==N|Necc==1) & (Ninc==N|Ninc==1);
if ~coherence; CL__error('bad size of input arguments'); end

I = find (sma <= 0); 
if ~isempty(I); CL__error('semi major axis out of range'); end

I = find (ecc < 0 | ecc >= 1); 
if ~isempty(I); CL__error('eccentricity out of range'); end

I = find (inc < 0 | inc > %pi+%eps); // margin in case !!!
if ~isempty(I); CL__error('inclination out of range'); end

// sma, ecc, inc forced to be all of the same size
sma = sma .* ones(1,N);
ecc = ecc .* ones(1,N); 
inc = inc .* ones(1,N); 

if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('j2','local') then j2=%CL_j2; end

// -------- 
// formulas
// --------
K = 0.75 * j2 * er^2 * sqrt(mu);  // common coefficient

n = sqrt(mu ./ sma.^3); // mean motion 
cosi = cos(inc); 
f2 = 1 - ecc.^2;
f = sqrt(f2); 
D = sma.^(7./2) .* f2.^2; // denominator in following expressions

// 1) secular drifts of parameters
drift_pom = K * (5 * cosi.^2 - 1) ./ D;        // arg of perigee
drift_gom = -2 * K * cosi ./ D;                // RAAN
drift_anm = K * (3 * cosi.^2 - 1) .* f ./ D;   // mean anomaly: effect of J2
drift_M = drift_anm + n; // mean anomaly: effect of J2 + central body

// 2) derivatives
if lhs > 3

  dpomdaei = zeros(3,N);
  dgomdaei = zeros(3,N);
  dMdaei   = zeros(3,N);
  sini     = sin(inc); 

  // derivatives with respect to semi major axis
  dpomdaei(1,:) = -3.5 * drift_pom ./ sma;             // d(drift_pom)/d(sma)
  dgomdaei(1,:) = -3.5 * drift_gom ./ sma;             // d(drift_gom)/d(sma)
  dMdaei(1,:)   = (-3.5 * drift_anm - 1.5 * n)./ sma;  // d(drift_M)  /d(sma)

  // derivatives with respect to eccentricity
  dpomdaei(2,:) = 4 * drift_pom .* ecc ./ f2;           // d(drift_pom)/d(exc)
  dgomdaei(2,:) = 4 * drift_gom .* ecc ./ f2;           // d(drift_gom)/d(exc)
  dMdaei(2,:)   = 3 * drift_anm .* ecc ./ f2;           // d(drift_M)  /d(exc)

  // derivatives with respect to inclination
  dpomdaei(3,:) = -10 * K * cosi .* sini ./ D;          // d(drift_pom)/d(inc)
  dgomdaei(3,:) = 2 * K * sini ./ D;                    // d(drift_gom)/d(inc)
  dMdaei(3,:)   = -6 * K * cosi .* sini .* f ./ D;      // d(drift_M)  /d(inc)

end

endfunction
