#
# SchoolTool - common information systems platform for school administration
# Copyright (c) 2005 Shuttleworth Foundation
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

from zope.catalog.interfaces import ICatalog
from zope.component import getUtility
from zope.intid.interfaces import IIntIds
from zope.security.proxy import removeSecurityProxy

from schooltool.app.catalog import AttributeCatalog
from schooltool.contact.interfaces import IContact
from schooltool.person.interfaces import IPerson
from schooltool.table.catalog import ConvertingIndex

import interfaces, intervention


class InterventionCatalog(AttributeCatalog):
    version = '1 - replaced catalog utility'
    interface = interfaces.IInterventionMarker
    attributes = ('created',)

    def setIndexes(self, catalog):
        super(InterventionCatalog, self).setIndexes(catalog)
        catalog['first_name'] = ConvertingIndex(converter=getStudentFirstName)
        catalog['last_name'] = ConvertingIndex(converter=getStudentLastName)
        catalog['persons_responsible'] = ConvertingIndex(
            converter=interfaces.IInterventionPersonsResponsible)
        catalog['schoolYearId'] = ConvertingIndex(converter=getSchoolYearId)
        catalog['intervention_type'] = ConvertingIndex(
            converter=interfaces.IInterventionType)


getInterventionCatalog = InterventionCatalog.get


def getStudentFirstName(obj):
    return IContact(IPerson(obj)).first_name


def getStudentLastName(obj):
    return IContact(IPerson(obj)).last_name


def getMessagePersonsResponsible(obj):
    util = getUtility(IIntIds)
    return [util.getId(contact) for contact in obj.recipients]


def getGoalPersonsResponsible(obj):
    # XXX not using removeSecurityProxy caused ZODB to crash
    # Cannot pickle <type 'zope.security._proxy._Proxy'> objects
    util = getUtility(IIntIds)
    return [util.getId(contact)
            for contact in removeSecurityProxy(obj.persons_responsible)]


def getMessageType(obj):
    return 'message'


def getGoalType(obj):
    return 'goal'


def getSchoolYearId(obj):
    sy = interfaces.IInterventionSchoolYear(obj)
    return sy.__name__


def messageGetter(obj):
    value = obj.body
    if len(value) > 50:
        value = value[:47] + '...'
    return value


def goalGetter(obj):
    value = obj.presenting_concerns
    if len(value) > 50:
        value = value[:47] + '...'
    return value


def updatePersonsResponsibleIndex(event):
    if event.rel_type != intervention.URIResponsible:
        return
    if (interfaces.IInterventionMessage.providedBy(event.participant1) or
        interfaces.IInterventionGoal.providedBy(event.participant1)):
        obj = event.participant1
    else:
        obj = event.participant2

    util = getUtility(IIntIds)
    catalog = getInterventionCatalog()
    if catalog is None:
        return
    obj_intid = util.queryId(obj)
    if obj_intid is not None and obj_intid in catalog.extent:
        catalog.index_doc(obj_intid, obj)

