module Schleuder
  # Represents a single list: name, config, members.
  class List
    # Name of this list. Must match the name of the subdirectory in
    # +SchleuderConfig::lists_dir+
    attr_reader :listname
    
    # Prepare some variables, set up the SchleuderLogger and set GNUPGHOME
    def initialize(listname,newlist=false)
      @listname = listname
      @config = _load_config(false) if newlist
      @log = ListLogger.new listname, listdir, config
      
      # setting GNUPGHOME to list's home, to make use of the keys there
      Schleuder.log.debug "setting ENV[GNUPGHOME] to #{listdir}"
      ENV["GNUPGHOME"] = listdir
      @members = nil
    end
    
    # Provides an array of Schleuder::Member's, read from +members.conf+
    def members
      unless @members
        Schleuder.log.debug("reading #{members_file}")
        @members = YAML::load_file(members_file).collect do |h|
          h.kind_of?(Schleuder::Member) ? h : Schleuder::Member.new(h,false)
        end
      end
      @members
    end

    def members_file
      @members_file ||= File.join(listdir, Schleuder.config.lists_memberfile)
    end
    
    # Saves an array of Schleuder::Member's into +members.conf++
    def members=(arr)
      Schleuder.log.debug 'writing members'
      Schleuder.log.info("writing #{members_file}")
      @members = arr.collect { |m| m.kind_of?(Hash) ? Member.new(m,false) : m }
      _write(YAML.dump(@members.collect { |m| m.to_hash }), members_file)
      @members
    end
    
    # Finds a member by email address.
    def find_member_by_email(addresses)
      addresses = Array(addresses)
      members.detect { |m| addresses.include?(m.email) } || false
    end

    def find_admin_by_email(addresses)
      addresses = Array(addresses)
      if admin_email = self.config.admins.detect { |a| addresses.include?(a.email) }
        Member.new(:email => admin_email)
      else
        false
      end
    end

    def find_member_by_key(key)
      Schleuder.log.debug "Looking for member for key #{key}"
      find_by_key(members, key)
    end

    def find_admin_by_key(key)
      Schleuder.log.debug "Looking for admin for key #{key}"
      find_by_key(config.admins, key)
    end

    def find_admin_by_address(address)
      config.admins.detect { |admin| admin.email == address }
    end

    def find_member_by_address(address)
      members.detect { |member| member.email == address }
    end

    # Provides the list config as Schleuder::ListConfig-object 
    def config
      @config ||= _load_config
    end

    # Saves +data+ into the list-config-file (default: list.conf). +data+ must
    # be a Schleuder::ListConfig or valid input to +ListConfig.new+
    def config=(data)
      Schleuder.log.info("writing list-config for: #{listname}")
      if data.is_a?(ListConfig)
        @config = data
      else
        @config = ListConfig.new(data)
      end
      _write(YAML::dump(@config.to_hash), File.join(listdir, Schleuder.config.lists_configfile))
      @config
    end
    
    # Builds the bounce-address for the list
    def bounce_addr
      @bounce_addr ||= self.config.myaddr.gsub(/^(.*)@(.*)$/, '\1-bounce@\2')
    end

    # Builds the owner-address for the list
    def owner_addr
      @owner_addr ||= self.config.myaddr.gsub(/^(.*)@(.*)$/, '\1-owner@\2')
    end

    # Builds the request-address for the list
    def request_addr
      @request_addr ||= self.config.myaddr.gsub(/^(.*)@(.*)$/, '\1-request@\2')
    end

    # builds the send-key-command-address for the list
    def sendkey_addr
      self.config.myaddr.gsub(/^(.*)@(.*)$/, '\1-sendkey@\2')
    end

    def self.listdir(listname)
      name = listname.split('@')
      if name.size < 2
        Schleuder.log.warn 'Listname should be a full email-address, using only the local part is deprecated. Please fix this!'
      end
      File.expand_path(File.join([Schleuder.config.lists_dir, name.reverse].flatten))
    end

    def listdir
      @listdir ||=  List.listdir(@listname)
    end

    def listid
      @listid ||= config.myaddr.gsub(/@/, '.')
    end

    def key
      @key ||= lookup_list_key
    end

    def key_fingerprint
      key.subkeys.first.fingerprint
    end

    def archive(mail)
      @list_archiver ||= Schleuder::Archiver.new
      @list_archiver.archive(mail)
    end

    private

    # Loads the configuration
    # fromfile = Whether to load the config from file.
    def _load_config(fromfile=true)
        Schleuder.log.debug("reading list-config for: #{@listname}") unless Schleuder.log.nil?
        @config = ListConfig.new(File.join(listdir, Schleuder.config.lists_configfile),fromfile)
    end

    def find_by_key(ary, key)
      return false unless key.kind_of?(GPGME::Key)
      res = ary.detect { |elem| elem.kind_of?(Member) && elem.uses_key?(key) }
      Schleuder.log.debug "Found #{res} for #{key}" unless res.nil?
      res || false
    end

    def _write(data,filename)
      File.open(filename, 'w') { |f| f << data }
    end

    def lookup_list_key
      key, msg = crypt.get_key(config.key_fingerprint||config.myaddr)
      unless key
        raise "Could not find a key for this List! Reason: #{msg}"
      end
      key
    end

    def crypt
      @@crypt ||= Crypt.new(nil)
    end

  end
end
