/*
 * scamper_file_text_tbit.c
 *
 * Copyright (C) 2009-2011 The University of Waikato
 * Authors: Ben Stasiewicz, Matthew Luckie
 *
 * $Id: scamper_tbit_text.c,v 1.3 2011/03/04 04:46:11 mjl Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef lint
static const char rcsid[] =
  "$Id";
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "internal.h"

#include "scamper_addr.h"
#include "scamper_list.h"
#include "scamper_file.h"
#include "scamper_tbit.h"
#include "scamper_tbit_text.h"
#include "utils.h"

#define TCP_MAX_SEQNUM 4294967295U

int scamper_file_text_tbit_write(const scamper_file_t *sf,
				 const scamper_tbit_t *tbit)
{
  scamper_tbit_pkt_t *pkt;
  char buf[131072], src[64], dst[64], tmp[32], ipid[12], fstr[32], tfstr[32];
  char *str;
  struct timeval diff;
  uint32_t i;
  uint32_t seq, ack, server_isn, client_isn, off, id, u32;
  uint16_t len, u16, datalen;
  uint8_t proto, flags, iphlen, tcphlen, mf, ecn;
  size_t soff = 0, tfoff;
  int frag;
  int fd = scamper_file_getfd(sf);

  switch(tbit->result)
    {
    case SCAMPER_TBIT_RESULT_NONE: str = "none"; break;
    case SCAMPER_TBIT_RESULT_TCP_NOCONN: str = "tcp-noconn"; break;
    case SCAMPER_TBIT_RESULT_TCP_RST: str = "tcp-rst"; break;
    case SCAMPER_TBIT_RESULT_TCP_ERROR: str = "tcp-error"; break;
    case SCAMPER_TBIT_RESULT_ERROR: str = "error"; break;
    case SCAMPER_TBIT_RESULT_ABORTED: str = "aborted"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NOACK: str = "pmtud-noack"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NODATA: str = "pmtud-nodata"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_TOOSMALL: str = "pmtud-toosmall"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NODF: str = "pmtud-nodf"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_FAIL: str = "pmtud-fail"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_SUCCESS: str = "pmtud-success"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_CLEARDF: str = "pmtud-cleardf"; break;
    case SCAMPER_TBIT_RESULT_ECN_SUCCESS: str = "ecn-success"; break;
    case SCAMPER_TBIT_RESULT_ECN_INCAPABLE: str = "ecn-incapable"; break;
    case SCAMPER_TBIT_RESULT_ECN_BADSYNACK: str = "ecn-badsynack"; break;
    case SCAMPER_TBIT_RESULT_ECN_NOECE: str = "ecn-noece"; break;
    case SCAMPER_TBIT_RESULT_ECN_NOACK: str = "ecn-noack"; break;
    case SCAMPER_TBIT_RESULT_ECN_NODATA: str = "ecn-nodata"; break;
    case SCAMPER_TBIT_RESULT_NULL_SUCCESS: str = "null-success"; break;
    case SCAMPER_TBIT_RESULT_NULL_NODATA: str = "null-nodata"; break;
    default: snprintf(tmp, sizeof(tmp), "%d", tbit->result); str = tmp; break;
    }

  string_concat(buf, sizeof(buf), &soff,
		"tbit from %s to %s\n server-mss %d, result: %s\n",
		scamper_addr_tostr(tbit->src, src, sizeof(src)),
		scamper_addr_tostr(tbit->dst, dst, sizeof(dst)),
		tbit->server_mss, str);

  client_isn = 0;
  server_isn = 0;

  for(i=0; i<tbit->pktc; i++)
    {
      pkt = tbit->pkts[i];
      frag = 0; mf = 0; id = 0; off = 0;

      if((pkt->data[0] >> 4) == 4)
        {
	  iphlen = (pkt->data[0] & 0xf) * 4;
	  len = bytes_ntohs(pkt->data+2);
	  proto = pkt->data[9];
	  ecn = pkt->data[1] & 0x3;
	  if(pkt->data[6] & 0x20)
	    mf = 1;
	  id  = bytes_ntohs(pkt->data+4);
	  off = (bytes_ntohs(pkt->data+6) & 0x1fff) * 8;
	  if(mf != 0 || off != 0)
	    frag = 1;
	  snprintf(ipid, sizeof(ipid), " %04x", bytes_ntohs(pkt->data+4));
        }
      else if((pkt->data[0] >> 4) == 6)
        {
	  iphlen = 40;
	  len = bytes_ntohs(pkt->data+4) + iphlen;
	  proto = pkt->data[6];
	  ecn = (pkt->data[1] & 0x30) >> 4;

	  for(;;)
            {
	      switch(proto)
                {
		case IPPROTO_HOPOPTS:
		case IPPROTO_DSTOPTS:
		case IPPROTO_ROUTING:
		  proto = pkt->data[iphlen+0];
		  iphlen += (pkt->data[iphlen+1] * 8) + 8;
		  continue;

		case IPPROTO_FRAGMENT:
		  if(pkt->data[iphlen+3] & 0x1)
		    mf = 1;
		  off = (bytes_ntohs(pkt->data+iphlen+2) & 0xfff8);
		  id  = bytes_ntohl(pkt->data+iphlen+4);
		  proto = pkt->data[iphlen+0];
		  iphlen += 8;
		  frag = 1;
		  continue;
                }
	      break;
            }
        }
      else
	{
	  continue;
	}	

      timeval_diff_tv(&diff, &tbit->start, &pkt->tv);
      string_concat(buf, sizeof(buf), &soff, " [%3d.%03d] %s ",
		    (int)diff.tv_sec, (int)(diff.tv_usec / 1000),
		    pkt->dir == SCAMPER_TBIT_PKT_DIR_TX ? "TX" : "RX");

      if(frag != 0)
	snprintf(fstr,sizeof(fstr),"%u:%u%s", id, off, mf != 0 ? " MF" : "");
      else
	fstr[0] = '\0';

      if(off != 0)
	{
	  string_concat(buf, sizeof(buf), &soff,
			"%-13s %4dF%21s %s %s", "", len, "", ipid, fstr);
	}
      else if(proto == IPPROTO_TCP)
        {
	  seq     = bytes_ntohl(pkt->data+iphlen+4);
	  ack     = bytes_ntohl(pkt->data+iphlen+8);
	  flags   = pkt->data[iphlen+13];
	  tcphlen = ((pkt->data[iphlen+12] & 0xf0) >> 4) * 4;

	  tfoff = 0;
	  if(flags & 0x2)
            {
	      if(flags & 0x10)
                {
		  server_isn = seq;
		  string_concat(tfstr, sizeof(tfstr), &tfoff, "SYN/ACK");
                }
	      else
                {
		  client_isn = seq;
		  string_concat(tfstr, sizeof(tfstr), &tfoff, "SYN");
                }
            }
	  else if(flags & 0x1)
	    string_concat(tfstr, sizeof(tfstr), &tfoff, "FIN");
	  else if(flags & 0x4)
	    string_concat(tfstr, sizeof(tfstr), &tfoff, "RST");
	  if(flags & 0x40)
	    string_concat(tfstr, sizeof(tfstr), &tfoff, "%sECE",
			  tfoff != 0 ? "/" : "");
	  if(flags & 0x80)
	    string_concat(tfstr, sizeof(tfstr), &tfoff, "%sCWR",
			  tfoff != 0 ? "/" : "");
	  if(tfoff == 0)
	    tfstr[0] = '\0';

	  if(pkt->dir == SCAMPER_TBIT_PKT_DIR_TX)
            {
	      seq -= client_isn + ((seq >= client_isn) ? 0 : TCP_MAX_SEQNUM+1);
	      ack -= server_isn + ((ack >= server_isn) ? 0 : TCP_MAX_SEQNUM+1);
            }
	  else    
            {
	      seq -= server_isn + ((seq >= server_isn) ? 0 : TCP_MAX_SEQNUM+1);
	      ack -= client_isn + ((ack >= client_isn) ? 0 : TCP_MAX_SEQNUM+1);
            }

	  datalen = len - iphlen - tcphlen;

	  string_concat(buf, sizeof(buf), &soff, "%-13s %4d%s",
			tfstr, len, frag != 0 ? "F" : " ");
	  if(datalen != 0)
	    snprintf(tmp, sizeof(tmp), " seq = %u:%u(%d)", seq, ack, datalen);
	  else
	    snprintf(tmp, sizeof(tmp), " seq = %u:%u", seq, ack);
	  string_concat(buf, sizeof(buf), &soff, "%-23s%s", tmp, ipid);
	  if(frag != 0) string_concat(buf, sizeof(buf), &soff, fstr);
	  if(datalen > 0 && (pkt->data[0] >> 4) == 4 && pkt->data[6] & 0x40)
	    string_concat(buf, sizeof(buf), &soff, " DF");
	  if(ecn == 3)      string_concat(buf, sizeof(buf), &soff, " CE");
	  else if(ecn != 0) string_concat(buf, sizeof(buf), &soff, " ECT");
        }
      else if(proto == IPPROTO_ICMP)
        {
	  if(pkt->data[iphlen+0] == 3 && pkt->data[iphlen+1] == 4)
	    {
	      u16 = bytes_ntohs(pkt->data+iphlen+6);
	      string_concat(buf, sizeof(buf), &soff,
			    "%-13s %4d  mtu = %d", "PTB", len, u16);
	    }
        }
      else if(proto == IPPROTO_ICMPV6)
        {
	  if(pkt->data[iphlen+0] == 2)
	    {
	      u32 = bytes_ntohl(pkt->data+iphlen+4);
	      string_concat(buf, sizeof(buf), &soff,
			    "%-13s %4d  mtu = %d", "PTB", len, u32);
	    }
	}

      string_concat(buf, sizeof(buf), &soff, "\n");
    }

  write_wrap(fd, buf, NULL, soff);
  return 0;
}
