/*
 * scamper_dlhdr.c
 *
 * $Id: scamper_dlhdr.c,v 1.8 2010/05/11 21:07:39 mjl Exp $
 *
 * Copyright (C) 2003-2010 The University of Waikato
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <sys/types.h>

#if defined(_MSC_VER)
typedef unsigned __int8 uint8_t;
typedef unsigned __int16 uint16_t;
typedef unsigned __int32 uint32_t;
typedef __int16 int16_t;
#define __func__ __FUNCTION__
#endif

#ifdef _WIN32
#include <winsock2.h>
#include <ws2tcpip.h>
#endif

#ifndef _WIN32
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#endif

#if defined(__APPLE__)
#include <stdint.h>
#endif

#include <stdlib.h>
#include <string.h>
#include <errno.h>

#if defined(DMALLOC)
#include <dmalloc.h>
#endif

#include "scamper_debug.h"
#include "scamper_fds.h"
#include "scamper_addr.h"
#include "scamper_addr2mac.h"
#include "scamper_task.h"
#include "scamper_dl.h"
#include "scamper_dlhdr.h"
#include "scamper_if.h"
#include "scamper_list.h"
#include "scamper_do_neighbourdisc.h"
#include "utils.h"

/*
 * ndcb_t
 *
 * struct to remember details about a datalink header request made which
 * we are unable to immediately answer.
 */
typedef struct ndcb
{
  scamper_task_t *task;
  scamper_fd_t   *fd;
  scamper_addr_t *ip;
  void          (*func)(scamper_task_t *, scamper_dlhdr_t *);
} ndcb_t;

/*
 * dl_hdr_alloc
 *
 * allocate a datalink header struct that will be filled in by the caller.
 */
static scamper_dlhdr_t *dlhdr_alloc(uint16_t len)
{
  scamper_dlhdr_t *hdr;

  if((hdr = malloc(sizeof(scamper_dlhdr_t))) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc hdr");
      return NULL;
    }
  hdr->len = len;
  hdr->buf = NULL;

  if(len > 0 && (hdr->buf = malloc(len)) == NULL)
    {
      printerror(errno, strerror, __func__, "could not malloc hdr->buf");
      free(hdr);
      return NULL;
    }

  return hdr;
}

static scamper_dlhdr_t *dlhdr_ethernet_make(scamper_fd_t *fd,
					    scamper_addr_t *mac,
					    scamper_addr_t *ip)
{
  scamper_dlhdr_t *hdr = NULL;
  int ifindex;

  /* need the interface index corresponding */
  if(scamper_fd_ifindex(fd, &ifindex) != 0)
    goto err;

  /*
   * we can return a datalink header immediately using cached values,
   * so create it now
   */
  if((hdr = dlhdr_alloc(14)) == NULL)
    goto err;

  /* copy the destination mac address to use */
  memcpy(hdr->buf, mac->addr, 6);

  /* the source mac address to use */
  if(scamper_if_getmac(ifindex, hdr->buf+6) != 0)
    {
      scamper_debug(__func__, "could not get source mac");
      goto err;
    }

  /* the ethertype */
  if(ip->type == SCAMPER_ADDR_TYPE_IPV4)
    {
      hdr->buf[12] = 0x08;
      hdr->buf[13] = 0x00;
    }
  else if(ip->type == SCAMPER_ADDR_TYPE_IPV6)
    {
      hdr->buf[12] = 0x86;
      hdr->buf[13] = 0xDD;
    }
  else
    {
      scamper_debug(__func__, "unhandled ip->type %d", ip->type);
      goto err;
    }
  return hdr;

 err:
  if(hdr != NULL) scamper_dlhdr_free(hdr);
  return NULL;
}

static void dlhdr_ethernet_ndcb(void *param, scamper_addr_t *mac)
{
  ndcb_t *ndcb = param;
  int ifindex;

  if(mac != NULL)
    {
      ndcb->func(ndcb->task, dlhdr_ethernet_make(ndcb->fd, mac, ndcb->ip));
      if(scamper_fd_ifindex(ndcb->fd, &ifindex) == 0)
	scamper_addr2mac_add(ifindex, ndcb->ip, mac);
    }
  else ndcb->func(ndcb->task, NULL);

  free(ndcb);
  return;
}

/*
 * dlhdr_ethernet
 *
 * form an ethernet header.  as this requires mac addresses, and scamper
 * may not know the mac address of the relevant IP, this function deals with
 * doing a neighbour discovery.
 */
static int dlhdr_ethernet(scamper_task_t *task, scamper_fd_t *fd,
			  scamper_addr_t *dst, scamper_addr_t *gw,
			  void (*func)(scamper_task_t *, scamper_dlhdr_t *))
{
  scamper_addr_t *ip = NULL;
  scamper_addr_t *mac = NULL;
  ndcb_t *ndcb = NULL;
  int ifindex;

  /* need the interface index corresponding */
  if(scamper_fd_ifindex(fd, &ifindex) != 0)
    return -1;

  /* determine what we should be looking up */
  if(gw == NULL)
    ip = dst;
  else if(gw->type == SCAMPER_ADDR_TYPE_ETHERNET)
    mac = gw;
  else
    ip = gw;

  /* if we need to get a mac address, then look it up */
  if(mac == NULL && (mac = scamper_addr2mac_whohas(ifindex, ip)) == NULL)
    {
      if((ndcb = malloc_zero(sizeof(ndcb_t))) == NULL)
	goto err;
      ndcb->task = task;
      ndcb->fd   = fd;
      ndcb->ip   = ip;
      ndcb->func = func;

      if(scamper_do_neighbourdisc_do(ifindex,ip,ndcb,dlhdr_ethernet_ndcb) != 0)
	goto err;

      return 0;
    }

  /* give the user what they asked for */
  func(task, dlhdr_ethernet_make(fd, mac, dst));
  return 0;

 err:
  return -1;
}

static int dlhdr_ethloop(scamper_task_t *task, scamper_fd_t *fd,
			 scamper_addr_t *dst, scamper_addr_t *gw,
			 void (*func)(scamper_task_t *, scamper_dlhdr_t *))
{
  scamper_dlhdr_t *hdr;

  if((hdr = dlhdr_alloc(14)) == NULL)
    goto err;

  memset(hdr->buf, 0, 12);
  if(dst->type == SCAMPER_ADDR_TYPE_IPV4)
    {
      hdr->buf[12] = 0x08;
      hdr->buf[13] = 0x00;
    }
  else if(dst->type == SCAMPER_ADDR_TYPE_IPV6)
    {
      hdr->buf[12] = 0x86;
      hdr->buf[13] = 0xDD;
    }
  else goto err;

  func(task, hdr);
  return 0;

 err:
  if(hdr != NULL) scamper_dlhdr_free(hdr);
  return -1;
}

static int dlhdr_null(scamper_task_t *task, scamper_fd_t *fd,
		      scamper_addr_t *dst, scamper_addr_t *gw,
		      void (*func)(scamper_task_t *, scamper_dlhdr_t *))
{
  scamper_dlhdr_t *hdr;
  int af;

  if(dst->type == SCAMPER_ADDR_TYPE_IPV4)
    af = AF_INET;
  else if(dst->type == SCAMPER_ADDR_TYPE_IPV6)
    af = AF_INET6;
  else return -1;

  if((hdr = dlhdr_alloc(sizeof(int))) == NULL)
    return -1;

  memcpy(hdr->buf, &af, sizeof(int));
  func(task, hdr);
  return 0;
}

static int dlhdr_raw(scamper_task_t *task, scamper_fd_t *fd,
		      scamper_addr_t *dst, scamper_addr_t *gw,
		      void (*func)(scamper_task_t *, scamper_dlhdr_t *))
{
  scamper_dlhdr_t *hdr;
  if((hdr = dlhdr_alloc(0)) == NULL)
    return -1;
  func(task, hdr);
  return 0;
}

static int dlhdr_unsupp(scamper_task_t *task, scamper_fd_t *fd,
			scamper_addr_t *dst, scamper_addr_t *gw,
			void (*func)(scamper_task_t *, scamper_dlhdr_t *))
{
  return -1;
}

/*
 * scamper_dlhdr_get
 *
 * given a datalink socket to use, the ultimate destination, and the
 * gateway (if any) returned by the route socket, determine the datalink
 * header to use when framing a packet.
 *
 */
int scamper_dlhdr_get(scamper_task_t *task, scamper_fd_t *fd,
		      scamper_addr_t *dst, scamper_addr_t *gw,
		      void (*cb)(scamper_task_t *, scamper_dlhdr_t *))
{
  static int (*const func[])(scamper_task_t *, scamper_fd_t *,
			     scamper_addr_t *, scamper_addr_t *,
			     void (*)(scamper_task_t *, scamper_dlhdr_t *)) = {
    dlhdr_unsupp,
    dlhdr_ethernet,
    dlhdr_null,
    dlhdr_raw,
    dlhdr_ethloop,
  };
  scamper_dl_t *dl;
  int tx;

  if((dl = scamper_fd_read_state(fd)) == NULL)
    return -1;

  tx = scamper_dl_tx_type(dl);
  if(tx < 0 || tx > 4)
    return -1;

  return func[tx](task, fd, dst, gw, cb);
}

void scamper_dlhdr_free(scamper_dlhdr_t *hdr)
{
  if(hdr->buf != NULL) free(hdr->buf);
  free(hdr);
  return;
}
