// Copyright (C) 1999-2012
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "boxannulus.h"
#include "fitsimage.h"

BoxAnnulus::BoxAnnulus(const BoxAnnulus& a) : BaseBox(a) {}

BoxAnnulus::BoxAnnulus(Base* p, const Vector& ctr,
		       const Vector& s, 
		       double ang,
		       const char* clr, int* dsh, 
		       int wth, const char* fnt, const char* txt, 
		       unsigned short prop, const char* cmt,
		       const List<Tag>& tg, const List<CallBack>& cb)
  : BaseBox(p, ctr, ang, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = 1;
  annuli_ = new Vector[1];
  annuli_[0] = s;

  strcpy(type_,"boxannulus");
  numHandle = 4;

  updateBBox();
}

BoxAnnulus::BoxAnnulus(Base* p, const Vector& ctr,
		       const Vector& inner, const Vector& outer, int num,
		       double ang,
		       const char* clr, int* dsh, 
		       int wth, const char* fnt, const char* txt, 
		       unsigned short prop, const char* cmt,
		       const List<Tag>& tg, const List<CallBack>& cb)
  : BaseBox(p, ctr, ang, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = num+1;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = ((outer-inner)/num)*i+inner;

  strcpy(type_,"boxannulus");
  numHandle = 4 + numAnnuli_;

  updateBBox();
}

BoxAnnulus::BoxAnnulus(Base* p, const Vector& ctr, 
		       int an, Vector* s,
		       double ang, 
		       const char* clr, int* dsh, 
		       int wth, const char* fnt, const char* txt,
		       unsigned short prop, const char* cmt,
		       const List<Tag>& tg, const List<CallBack>& cb)
  : BaseBox(p, ctr, ang, clr, dsh, wth, fnt, txt, prop, cmt, tg, cb)
{
  numAnnuli_ = an;
  annuli_ = new Vector[numAnnuli_];

  for (int i=0; i<numAnnuli_; i++)
    annuli_[i] = s[i];
  sortAnnuli();

  strcpy(type_, "boxannulus");
  numHandle = 4 + numAnnuli_;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void BoxAnnulus::editBegin(int h)
{
  if (h<5) {
    switch (h) {
    case 1:
      return;
    case 2:
      annuli_[numAnnuli_-1] = Vector(-annuli_[numAnnuli_-1][0],annuli_[numAnnuli_-1][1]);
      return;
    case 3:
      annuli_[numAnnuli_-1] = -annuli_[numAnnuli_-1];
      return;
    case 4:
      annuli_[numAnnuli_-1] = Vector(annuli_[numAnnuli_-1][0],-annuli_[numAnnuli_-1][1]);
      return;
    }
  }

  doCallBack(CallBack::EDITBEGINCB);
}

void BoxAnnulus::edit(const Vector& v, int h)
{
  Matrix mm = bckMatrix();
  Matrix nn = mm.invert();

  // This sizes about the opposite node
  if (h<5) {
    Vector o = annuli_[numAnnuli_-1];
    Vector n = (annuli_[numAnnuli_-1]/2) - (v*mm);

    // don't go thru opposite node
    if (n[0]!=0 && n[1]!=0) {
      Vector ov = annuli_[numAnnuli_-1]/2 * nn;
      annuli_[numAnnuli_-1] = n;
      Vector nv = annuli_[numAnnuli_-1]/2 * nn;
      center -= nv-ov;

      for (int i=0; i<numAnnuli_-1; i++) {
	annuli_[i][0] *= fabs(n[0]/o[0]);
	annuli_[i][1] *= fabs(n[1]/o[1]);
      }
    }
  }
  else {
    // we must have some length
    double l = (v * mm * 2).length();
    annuli_[h-5] = annuli_[numAnnuli_-1] * l/annuli_[numAnnuli_-1][0];
  }

  updateBBox();
  doCallBack(CallBack::EDITCB);
  doCallBack(CallBack::MOVECB);
}

void BoxAnnulus::editEnd()
{
  for (int i=1; i<numAnnuli_; i++)
    annuli_[i] = annuli_[i].abs();
  sortAnnuli();

  updateBBox();
  doCallBack(CallBack::EDITENDCB);
}

int BoxAnnulus::addAnnuli(const Vector& v)
{
  Matrix mm = bckMatrix();
  double l = (v * mm * 2).length();
  Vector rr = annuli_[numAnnuli_-1] * l/annuli_[numAnnuli_-1][0];

  return insertAnnuli(rr);
}

// list

void BoxAnnulus::list(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
		   Coord::SkyFormat format, int conj, int strip)
{
  FitsImage* ptr = parent->findFits(sys,center);
  listPre(str, sys, sky, ptr, strip, 0);

  switch (sys) {
  case Coord::IMAGE:
  case Coord::PHYSICAL:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    {
      Vector v = ptr->mapFromRef(center,sys);
      str << "box(" << setprecision(8) << v[0] << ',' << v[1] << ',';
      for (int i=0; i<numAnnuli_; i++) {
	Vector r = ptr->mapLenFromRef(annuli_[i],sys);
	str << r[0] << ',' << r[1] << ',' ;
      }
      str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSCel(sys)) {
	switch (format) {
	case Coord::DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << "box(" << setprecision(8) << v[0] << ',' << v[1] << ',';
	    for (int i=0; i<numAnnuli_; i++) {
	      Vector r = ptr->mapLenFromRef(annuli_[i],sys,Coord::ARCSEC);
	      str << r[0] << "\"" << ',' << r[1] << "\"" << ',' ;
	    }
	    str << radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
	  }
	  break;
	case Coord::SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
	    string x(buf);
	    istringstream wcs(x);
	    wcs >> ra >> dec;

	    str << "box(" << ra << ',' << dec << ',' ;
	    for (int i=0; i<numAnnuli_; i++) {
	      Vector r = ptr->mapLenFromRef(annuli_[i],sys,Coord::ARCSEC);
	      str << r[0] << "\""<< ',' << r[1] << "\""<< ',' ;
	    }
	    str << radToDeg(parent->mapAngleFromRef(angle,sys,sky)) << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	str << "box(" << setprecision(8) << v[0] << ',' << v[1] << ',';
	for (int i=0; i<numAnnuli_; i++) {
	  Vector r = ptr->mapLenFromRef(annuli_[i],sys);
	  str << r[0] << ',' << r[1] << ',' ;
	}
	str << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
      }
    }
  }

  listPost(str, conj, strip);
}

void BoxAnnulus::listXML(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky, 
			 Coord::SkyFormat format)
{
  FitsImage* ptr = parent->findFits(sys,center);

  XMLRowInit();
  XMLRow(XMLSHAPE,type_);

  XMLRowCenter(ptr,sys,sky,format);
  XMLRowRadius(ptr,sys,annuli_,numAnnuli_);
  XMLRowAng(sys,sky);

  XMLRowProps(ptr,sys);
  XMLRowEnd(str);
}

void BoxAnnulus::listPros(ostream& str, Coord::CoordSystem sys, Coord::SkyFrame sky,
			  Coord::SkyFormat format, int strip)
{
  FitsImage* ptr = parent->findFits();

  switch (sys) {
  case Coord::IMAGE:
  case Coord::DETECTOR:
  case Coord::AMPLIFIER:
    sys = Coord::IMAGE;
  case Coord::PHYSICAL:
    {
      Vector v = ptr->mapFromRef(center,sys);
      for (int i=0; i<numAnnuli_; i++) {
	coord.listProsCoordSystem(str,sys,sky);
	str << "; ";

	Vector r = ptr->mapLenFromRef(annuli_[i],Coord::IMAGE);
	str << "box " << setprecision(8) << v << r << radToDeg(angle);

	if (i!=0) {
	  Vector r1 = ptr->mapLenFromRef(annuli_[i-1],Coord::IMAGE);
	  str << " & !box " << setprecision(8) << v << r1 << radToDeg(angle);
	}

	listProsPost(str, strip);
      }
    }
    break;
  default:
    if (ptr->hasWCSCel(sys)) {
      switch (format) {
      case Coord::DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  for (int i=0; i<numAnnuli_; i++) {
	    coord.listProsCoordSystem(str,sys,sky);
	    str << "; ";

	    Vector r = ptr->mapLenFromRef(annuli_[i],sys,Coord::ARCSEC);
	    str << "box " << setprecision(8) << v[0] << "d " << v[1] << "d "
		<< r[0] << "\" " << r[1] << "\" " << radToDeg(angle);

	    if (i!=0) {
	      Vector r1 = ptr->mapLenFromRef(annuli_[i-1],sys,Coord::ARCSEC);
	      str << " & !box " << setprecision(8) 
		  << v[0] << "d " << v[1] << "d " 
		  << r1[0] << "\" " << r1[1] << "\" " << radToDeg(angle);
	    }

	    listProsPost(str, strip);
	  }
	}
	break;
      case Coord::SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char decc[16];
	  char *dec = decc;
	  string x(buf);
	  istringstream wcs(x);
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    dec++;

	  for (int i=0; i<numAnnuli_; i++) {
	    coord.listProsCoordSystem(str,sys,sky);
	    str << "; ";

	    Vector r = ptr->mapLenFromRef(annuli_[i],sys,Coord::ARCSEC);
	    str << "box " << ra << ' ' << dec << ' ' 
		<< r[0] << "\" " << r[1] << "\" " << radToDeg(angle);

	    if (i!=0) {
	      Vector r1 = ptr->mapLenFromRef(annuli_[i-1],sys,Coord::ARCSEC);
	      str << " & !box " << ra << ' ' << dec << ' ' 
		  << r1[0] << "\" " << r1[1] << "\" " << radToDeg(angle);
	    }

	    listProsPost(str, strip);
	  }
	}
	break;
      }
    }
  }
}

void BoxAnnulus::listSAOimage(ostream& str, int strip)
{
  FitsImage* ptr = parent->findFits();
  listSAOimagePre(str);

  for (int i=0; i<numAnnuli_; i++) {
    Vector v = ptr->mapFromRef(center,Coord::IMAGE);
    str << "box(" << setprecision(8) << v[0] << ',' << v[1] << ',' 
	<< annuli_[i][0] << ',' << annuli_[i][1] << ',' << radToDeg(angle) << ')';

    if (i!=0)
      str << " & !box(" << setprecision(8) << v[0] << ',' << v[1] 
	  << ',' << annuli_[i-1][0] << ',' << annuli_[i-1][1] << ',' 
	  << radToDeg(angle) << ')';

    listSAOimagePost(str, strip);
  }
}


