# Samizdat dispatcher classes
#
#   Copyright (c) 2002-2009  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 3 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

class Route
  ROUTE_PATTERN = Regexp.new(
    %r{\A
      /
      ([[:alpha:]][[:alnum:]_]*)
      (?:
        /([0-9]+)
      )?
      (?:
        /([[:alpha:]][[:alnum:]_]*)
      )?
      (?:
        /
      )?
    \z}x
  ).freeze

  DEFAULT_CONTROLLER = 'frontpage'
  DEFAULT_ACTION = :index

  RESOURCE_ROUTE_PATTERN = Regexp.new(%r{\A/([0-9]+)\z}).freeze
  RESOURCE_CONTROLLER = 'resource'

  def initialize(route)
    if '/' == route
      @controller = DEFAULT_CONTROLLER
      @action = DEFAULT_ACTION
      return
    end

    match = RESOURCE_ROUTE_PATTERN.match(route)
    if match
      @controller = RESOURCE_CONTROLLER
      @id = match[1].to_i.untaint
      @action = DEFAULT_ACTION
      return
    end

    match = ROUTE_PATTERN.match(route)
    if match.nil?
      raise ResourceNotFoundError, route.to_s
    end

    @controller, @id, @action = match[1..3]
    @controller.untaint

    if @id
      @id = @id.to_i.untaint
    end

    if @action
      if Controller.instance_methods(false).include?(@action)
        # methods defined in Controller class are not valid actions
        raise ResourceNotFoundError, route.to_s
      end
      @action = @action.untaint.to_sym
    end

    @action ||= DEFAULT_ACTION
  end

  def render(request)
    controller = create_controller(request)
    controller.send(@action)
    controller.render
  end

  def Route.render_error(request, error)
    controller = ErrorController.new(request)
    controller._error(error)
    controller.render
  end

  private

  # return controller object if controller and action are recognized, raise 404
  # exception otherwise
  #
  def create_controller(request)
    begin
      require %{samizdat/controllers/#{@controller}_controller}
    rescue LoadError
      raise ResourceNotFoundError, @controller
    end

    controller = Object.const_get(@controller.capitalize + 'Controller').new(request, @id)
    # don't rescue NameError here, it's too generic; if the controller file is
    # there, it should have a class with matching name; if it doesn't, internal
    # error should be raised anyway

    controller.respond_to?(@action) or raise ResourceNotFoundError,
      @controller + '/' + @action.to_s

    controller
  end
end

class Dispatcher
  # invoke controller action matching the CGI request
  #
  # throw +:finish+ to interrupt the request processing and return to the
  # dispatcher
  #
  def dispatch(cgi)
    catch :finish do
      request = Request.new(cgi)
      begin
        Route.new(request.route).render(request)
      rescue => error
        Route.render_error(request, error)
      end
    end
  end
end

class CGIDispatcher < Dispatcher
  def run
    dispatch(CGI.new)
    exit
  end
end

class FCGIDispatcher < Dispatcher
  def run
    FCGI.each_cgi do |cgi|
      dispatch(cgi)
    end
  end
end
