# Samizdat HTML templates
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

begin
  require 'redcloth'

  # get rid of <del>, '-' belongs to text, not markup
  RedCloth::QTAGS.delete_if {|rc, ht, re, rtype| 'del' == ht }
rescue LoadError
  class RedCloth < String
    def to_html   # revert to text/plain
      "<pre>#{CGI.escapeHTML(self)}</pre>"
    end
  end
end

class Template
  def initialize(session)
    @session = session
    @link = {:start => session.base}
  end

  # document navigation links (made, start, next, ...)
  attr_accessor :link

  # HTML header, title, and style settings
  #
  def head(title='', options={})
    style = @session.cookie('style')
    style = config['style'][0] unless config['style'].include?(style)
    css = %{  <link rel="stylesheet" type="text/css" href="css/#{style}.css" media="screen" />\n}
    config['style'].each do |s|
      s != style and css <<
%{  <link rel="alternate stylesheet" type="text/css" title="#{s}" href="css/#{s}.css" media="screen" />\n}
    end

    if options[:rss].kind_of? Hash
      rss = ''
      options[:rss].each do |t, l|
        rss << %{  <link rel="alternate" type="application/rss+xml" title="#{t}" href="#{@session.base + l}" />\n}
      end
    elsif options[:rss].kind_of? String
        rss = %{  <link rel="alternate" type="application/rss+xml" href="#{@session.base + options[:rss]}" />\n}
    end

    @link[:icon] = config['site']['icon']

%{<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN"
  "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en">
<head>
  <title>#{title}</title>
  <meta name="generator" content="Samizdat #{SAMIZDAT_VERSION}" />
#{css}#{rss}} +
    @link.to_a.collect {|rel, href|
      %{  <link rel="#{rel}" href="#{href}" />\n} if href
    }.join + %{</head>\n<body>\n}
  end

  # close HTML
  #
  def foot
    %{</body>\n</html>}
  end

  # site head and subhead
  #
  def site_head
%{<div><a href="#main" class="hidden">Skip navigation</a></div>
<div id="head">
  <div id="head-left"><a href="#{@session.base}" title="#{config['site']['name'] + ': ' + _('Front Page')}">#{config['site']['logo']}</a></div>
  <div id="head-right"><a href="message.rb">}+_('Publish')+%{</a>
    :: <a href="item.rb">}+_('Share')+%{</a>
    :: <a href="query.rb">}+_('Search')+%{</a></div>
</div>
<div id="subhead">
  <div id="subhead-right">#{member_line}</div>
</div>\n}
  end

  # site foot (attributions)
  #
  def main_foot
%{<div id="foot">
Powered by <a href="http://www.fsf.org/">Free Software</a>, including
<a href="http://www.apache.org/">Apache</a> web server,
<a href="http://www.ruby-lang.org/">Ruby</a> programming language,
and <a href="http://www.postgresql.org/">PostgreSQL</a> database.
<a href="http://samizdat.nongnu.org/">Samizdat</a> engine is free
software; you can distribute/modify it under the terms of the GNU
<a href="http://www.gnu.org/licenses/gpl.txt">General Public License</a>
version 2 or later. </div>}
  end

  # member options when logged on, link to login form otherwise
  #
  def member_line
    settings = '<a href="member.rb" title="'+_('Account settings, interface language, theme')+'">'+_('Settings')+'</a>, '
    if @session.id
      resource_href(@session.id, CGI.escapeHTML(@session.full_name)) + ': ' +
        settings + '<a href="logout.rb">'+_('Log out')+'</a>'
    else
      settings + '<a href="login.rb">'+_('Log in')+'</a>'
    end
  end

  # wrap content in <div id="sidebar"> if non-empty
  #
  def sidebar(content)
    if content and content != ''
%{<div id="sidebar">
#{content}
</div>\n}
    else '' end
  end

  # wrap title and content into a CSS-rendered box
  #
  def box(title, content, id=nil)
    box_title = %{<div class="box-title">#{title}</div>} if title
    box_id = %{ id="#{id}"} if id
%{<div class="box"#{box_id}>
  #{box_title}<div class="box-content">
#{content}
  </div>
</div>\n}
  end

  # return _string_ truncated to the _limit_, with ellipsis if necessary
  #
  def Template.limit_string(string, limit=config['limit']['title'])
    $KCODE = 'utf8'
    if string.jsize > limit
      string.each_char[0, limit - 1].join + '…'
    else
      string
    end
  end

  # navigation link to a given page number
  #
  def nav(size, skip, script=File.basename(@session.script_name.to_s)+'?', name='skip')
    # todo: reorganize
    return '' if true == size or (size.kind_of?(Numeric) and size < limit_page)
    %{<div class="nav"><a href="#{script}#{name}=#{skip}">} +
      _('next page') + %{</a></div>\n}
  end

  # add link to RSS rendering of the page
  #
  def nav_rss(link)
    link ? %{<div class="nav"><a href="#{link}">rss 1.0</a></div>} : ''
  end

  # resource list with navigation link
  #
  def list(list, nav, foot='')
    foot = %{<div class="foot">\n#{foot + nav}</div>\n} unless
      '' == foot and '' == nav
    even = 1
    %{<ul>\n} +
    list.collect {|li|
      even = 1 - even
      %{<li#{' class="even"' if even == 1}>#{li}</li>\n}
    }.join + %{</ul>\n} + foot
  end

  # resource table with navigation link
  #
  def table(table, nav, foot='')
    foot = %{<div class="foot">\n#{foot + nav}</div>\n} unless
      '' == foot and '' == nav
    even = 1
    %{<table>\n<thead><tr>\n} +
    table.shift.collect {|th| "<th>#{th}</th>\n" }.join +
    %{</tr></thead>\n<tbody>\n} +
    table.collect {|row|
      even = 1 - even   # todo: a CSS-only way to do this
      %{<tr#{' class="even"' if even == 1}>\n} + row.collect {|td|
        "<td>#{td or '&nbsp;'}</td>\n"
      }.join + "</tr>\n"
    }.join + %{</tbody></table>\n} + foot
  end

  # type can be any of the following:
  #
  # [:label]
  #   wrap label _value_ in a <div> tag and associate it with _name_ field
  #   (caveat: ids are not unique across multiple forms)
  # [:br] line break
  # [:textarea] fixed text area 70x20 with _value_
  # [:select] _value_ is an array of options or pairs of [option, label]
  # [:submit] _value_ is a button label
  # [standard HTML input type] copy _type_ as is into <input> tag
  #
  def form_field(type, name=nil, value=nil, default=nil)
    value = CGI.escapeHTML(value) if value.class == String
    attr = %{ name="#{name}"} if name
    attr += %{ id="f_#{name}"} if name and :label != type
    attr += ' disabled="disabled"' if name and :disabled == default
    case type
    when :br then %{<br />\n}
    when :label
      %{<div class="label"><label for="f_#{name}">#{value}</label></div>\n}
    when :textarea
      %{<textarea#{attr} cols="70" rows="20">#{value}\n</textarea>\n}   # mind rexml
    when :select, :select_submit
      attr += ' onchange="submit()"' if :select_submit == type
      %{<select#{attr}>\n} + value.collect {|option|
        v, l = (option.class == Array)? option : [option, option]
        selected = (v == default)? ' selected="selected"' : ''
        %{    <option#{selected} value="#{v}">#{l}</option>\n}
      }.join + "</select>\n"
    when :submit, :submit_moderator
      value = _('Submit') if value.nil?
      %{<input#{attr} type="submit" value="#{value}" class="#{type}" />\n}
    else
      if :checkbox == type
        attr += ' checked="checked"' if value
        value = 'true'
      end
      %{<input#{attr} type="#{type}" value="#{value}" />\n}
    end
  end

  # wrap a list of form fields into a form (see form_field)
  #
  # automatically detects if multipart/form-data is necessary
  #
  def form(action, *fields)
    if fields.assoc(:file)
      enctype = ' enctype="multipart/form-data"'
    end
    %{<form action="#{action}" method="post"#{enctype}><div>\n} +
    fields.collect {|param| form_field(*param) }.join + "</div></form>\n"
  end

  # wrap focus name and rating in <span> tags
  #
  def focus_info(related, focus)
    return nil if focus.uriref == related.uriref   # never relate resource to itself
    info = focus.name
    id = Resource.validate_id(focus.id)
    info = resource_href(id, info) if id
    info += ': ' + focus.print_rating
    info += ' (<a title="'+_('Click to vote on how this resource is related to this focus')+
      '" href="resource.rb?id='+related.id.to_s+
      '&amp;focus='+CGI.escape((focus.id or focus.uriref).to_s)+
      '">'+_('vote')+
      '</a>)' if @session.access('vote')
    '<p>' + info + "</p>\n"
  end

  # list supplied focuses using focus_info
  #
  def focus_box(related, focuses)
    return '' unless focuses.kind_of? Array and
      (focuses.size > 0 or @session.access('vote'))
    fbox = box( _('Related Focuses'),
      focuses.sort_by {|f| -f.sort_index }.collect {|focus|
        focus_info(related, focus)
      }.join,
      'focuses'
    )
    if @session.access('vote')
      fbox += box(nil,
        '<p><a title="'+_('Click to relate this resource to another focus')+%{" href="resource.rb?id=#{related.id}&amp;focus=select">}+_('Add another focus')+'</a></p>'
      )
    end
    fbox
  end

  # list supplied focuses in a straight line
  #
  def focus_line(id, focuses)
    %{<a href="#{id}#focuses">}+_('related to')+'</a>: ' +
    focuses.sort_by {|f| -f.sort_index }.collect {|focus|
      resource_href(focus.id, focus.name)
    }.join(', ') if focuses.size > 0
  end

  # form fields for vote on focus rating
  #
  def focus_fields(id, focus, advanced=('advanced' == @session.cookie('ui')))
    focuses = Focus.collect_focuses {|f,|
      [f, Resource.new(@session, f).render(:title)]
    }
    if focus and ('focus::Translation' == focus.uriref or not focuses.assoc(focus.id))
      focuses.unshift( [(focus.id or focus.uriref), focus.name] )
    end
    focuses.unshift ['select', _('SELECT FOCUS')]
    fields =
      [ [:hidden, 'id', id],
        [:label, 'focus', _('Select focus that this resource will be related to')],
          [:select, 'focus', focuses, (focus ? (focus.id or focus.uriref) : 'select')] ]
    if advanced
      fields.push(
        [:label, 'focus_id', _("Enter focus id if it's not in the list")],
          [:text, 'focus_id'],
        [:label, 'rating', _('Give a rating of how strongly this resource is related to selected focus')],
          [:select, 'rating', [
            [-2, _('-2 (No)')],
            [-1, _('-1 (Not Likely)')],
            [0, _('0 (Uncertain)')],
            [1, _('1 (Likely)')],
            [2, _('2 (Yes)')] ], 0]
      )
    else
      fields.push([:hidden, 'rating', 1])
    end
  end

  # transform date to a standard string representation
  #
  def format_date(date)
    date = date.to_time if date.methods.include? 'to_time'   # duck
    date = date.strftime '%Y-%m-%d %H:%M' if date.kind_of? Time
    date
  end

  # render link to resource with a tooltip
  #
  # _title_ should be HTML-escaped
  #
  def resource_href(id, title)
    return '' unless title.kind_of? String
    id.nil? ? Template.limit_string(title) :
      '<a title="'+_('Click to view the resource')+%{" href="#{id}">#{Template.limit_string(title)}</a>}
  end

  # render link to full message
  #
  def full_message_href(id)
    %{<p><a href="#{id}">}+_('See the full message')+'</a></p>'
  end

  # render resource description for resource listing
  #
  # _title_ should be HTML-escaped
  #
  def resource(id, title, info)
%{<div class="resource">
<div class="title">#{resource_href(id, title)}</div>
<div class="info">#{info}</div>
</div>\n}
  end

  # inline formats that need a link to view source
  #
  SOURCE_FORMAT = {
    'text/textile' => 'textile',
    'text/html' => 'html'
  }

  # render _message_ info line (except focuses)
  #
  def message_info(message, mode)
    creator = message.creator.nil? ?   # no link if published by guest
      _('guest') : %{<a href="#{message.creator}">#{message.creator_name}</a>}
    date = format_date(message.date)
    if :full == mode
      parent = %{<a href="#{message.parent}">} +
        _('parent message') + '</a>' if message.parent
      current = %{<a href="#{message.current}">} + _('current version') +
        '</a>' if message.current
      history = %{<a href="history.rb?id=#{message.id}">} + _('history') +
        '</a>' if message.nversions.to_i > 0
      if format = SOURCE_FORMAT[message.format]
        source = %{<a href="message.rb?parent=#{message.id}&amp;source" title="} +
          _('view source') + %{">#{format}</a>}
      end
    end
    replies = %{<a href="#{message.id}#replies">} + _('replies') +
      '</a>:&nbsp;' + message.nreplies.to_s if message.nreplies.to_i > 0
    translations = _('translation') + ': ' +
      message.translations.sort_by {|l,m,r| -r.to_f }.collect {|l,m,r|
        %{<a href="#{m}">#{l}</a>}
      }.join(' ') if message.translations.to_a.size > 0
    resource = Resource.new(@session, message.id)
    focuses = focus_line(message.id, message.focuses.collect {|f|
      Focus.new(@session, f, resource)
    })
    hidden = _('hidden') if message.hidden
    [ sprintf(_('by&nbsp;%s on&nbsp;%s'), creator, date.to_s),
      parent, current, history, source, replies, translations, focuses, hidden
    ].compact.join(",\n ")
  end

  AUTOURL_SCHEMES = %w[http https ftp]

  # format inline content according to _format_
  #
  def Template.format_inline_content(content, format)
    html =
      case format
      when 'text/plain'   # inline verbatim text
        "<pre>#{CGI.escapeHTML(content)}</pre>"
      when 'text/html'   # limited HTML text
        '<div>' + content + '</div>'
      when 'text/textile'   # textile formatted text
        begin
          RedCloth.new(content).to_html
        rescue Exception
          raise RuntimeError, $!.message
        end
      else   # default text rendering
        CGI.escapeHTML(content).split(/^\s*$/).collect {|p|
          '<p>' + p.gsub(URI::ABS_URI_REF) {|url|
            scheme, host = $1, $4   # see URI::REGEXP::PATTERN::X_ABS_URI
            if AUTOURL_SCHEMES.include?(scheme) and not host.nil?
              url =~ /\A(.*?)([.,;:?!()]+)?\z/   # don't grab punctuation
              url, tail = $1, $2
              %{<a href="#{url}">#{url}</a>#{tail}}
            else
              url
            end
          } + "</p>\n"
        }.join
      end
    Samizdat::Sanitize.new(config.xhtml).sanitize(html)
  end

  # render _message_ content, _mode_ can be :short or :full
  #
  def message_content(message, mode)
    return nil if :list == mode or (message.desc and :short == mode)
    if message.content.nil? and not message.inline?   # file link
      content = config.content_location +
        Message.content_path(message.id, message.format, message.creator_login)
    else   # inline text
      content = message.content.to_s
    end

    if Message.format_cacheable?(message.format)
      case mode   # first try html_{full,short} cache
      when :full
        return message.html_full if message.html_full
      when :short
        if message.html_short   # short version is different from full version
          return message.html_short + full_message_href(message.id)
        elsif message.html_full   # short version is same as full version
          return message.html_full
        end
        # truncate content in :short mode
        full_size = content.size
        content = Template.limit_string(content, config['limit']['short'])
        truncated = full_message_href(message.id) if content.size < full_size
      end
      Template.format_inline_content(content, message.format) + truncated.to_s

    else
      case message.format
      when 'application/x-squish'   # inline query form
        form('query.rb',
          [:textarea, 'query', content],
          [:br], [:submit, 'run', _('Run')])
      when /^image\//   # <img/> reference
        %{<img alt="#{CGI.escapeHTML(message.title.to_s)}" src="#{content}" />}
      else   # <a/> reference
        %{<p><a href="#{content}">} +
          sprintf(_('%s content'), message.format) + '</a></p>'
      end
    end
  end

  # render buttons for _message_ in :full mode
  #
  def message_buttons(message)
    if not message.current   # no buttons for old versions
      buttons =
        if message.nrelated > 0
          # reverted this change because it breaks focus translations
          # []   # don't offer to post replies to focuses
          [ [:submit, 'reply', _('Reply')] ]
        else
          [ [:submit, 'reply', _('Reply')] ]
        end
      open =
        if @session.id.nil?   # guest
          false
        elsif message.creator == @session.id   # creator
          true
        else   # everyone else
          message.open
        end
      buttons.push([:submit, 'edit', _('Edit'), (open ? nil : :disabled )])
      @session.moderator and buttons.push(message.hidden ?
        [:submit_moderator, 'unhide', _('UNHIDE')] :
        [:submit_moderator, 'hide', _('HIDE')])
      @session.moderator and buttons.push(
        [:submit_moderator, 'takeover', _('TAKE OVER')],
        [:submit_moderator, 'displace', _('DISPLACE')],
        [:submit_moderator, 'reparent', _('REPARENT')])
    elsif @session.moderator   # old versions can still be displaced
      buttons = [
        [:submit_moderator, 'displace', _('DISPLACE')]
      ]
    end
    buttons and buttons = %{<div class="foot">\n} +
      form('message.rb',
        [:hidden, 'title', message.title],
        [:hidden, 'parent', message.id],
      *buttons) + %{</div>\n}
    buttons
  end

  # render full message
  #
  # _message_ is an instance of Message class
  #
  # _mode_ can be :short, :full, or :preview
  #
  def message(message, mode)
    info = message_info(message, mode)
    # use translation in :short mode
    translation = (:short == mode)?
      message.select_translation(@session.accept_language) : message
    if :full != mode   # title is already in the page head in :full mode
      title = translation.title
      title = Focus.focus_title(title) if
        :short == mode and message.nrelated > 0
      title = CGI.escapeHTML(Template.limit_string(title))
      title = %{<div class="title">#{resource_href(message.id, title)}</div>\n}
    end
    if translation.desc and translation.desc != translation.id
      # use description of translation when applicable
      desc = message_content(Message.cached(translation.desc), :short)
    end
    content =
      if desc
        if :short == mode
          desc + full_message_href(message.id)
        else
          box(resource_href(message.desc, _('Description')), desc, 'desc') +
            message_content(message, mode)
        end
      else
        message_content(translation, mode)
      end
%{<div class="message" id="id#{message.id}">
#{title}<div class="info">#{info}</div>
<div class="content">#{content}</div>
</div>\n}
  end

  # language selection
  #
  def language_list
    list = ''
    old_language = @session.language
    config['locale']['languages'].sort.each do |lang|
      lang.untaint
      @session.language = lang
      name = _('English')
      @session.language = old_language
      next if 'English' == name and 'en' != lang   # broken localization
      name += _(' (default language)') if
        lang == config['locale']['languages'][0]
      name = %{<strong>#{name}</strong>} if lang == old_language
      list += %{<a href="member.rb?set_lang=#{lang}">#{name}</a>\n}
    end
    list
  end

  # calculate the table of differences
  #
  # return Array of changesets [[left, right], [same], ... ]
  #
  def diff_calculate(old, new)
    begin
      require 'algorithm/diff'
    rescue LoadError
      return [[old.join, new.join]]
    end
    table = []
    offset = 0
    i = 0
    diff = old.diff(new)   # [[op, pos, [lines]], ... ]
    while i < old.size or diff.size > 0 do
      if 0 == diff.size or
        i < diff[0][1] - ((diff[0][0] == :+)? offset : 0)   # copy

        if table[-1] and 1 == table[-1].size
          table[-1][0] << old[i]
        else
          table.push [old[i]]
        end
        i += 1
        next
      end
      if diff[0][0] == :-
        if diff[1] and diff[1][0] == :+ and diff[1][1] - offset == diff[0][1]
          # replace
          table.push [diff[0][2], diff[1][2]]
          offset += diff[1][2].size - diff[0][2].size
          i += diff[0][2].size
          diff.slice!(0, 2)
        else   # delete
          table.push [diff[0][2], []]
          offset -= diff[0][2].size
          i += diff[0][2].size
          diff.shift
        end
      else   # add
        table.push [[], diff[0][2]]
        offset += diff[0][2].size
        diff.shift
      end
    end
    table
  end

  # render difference table
  #
  # _old_ and _new_ are Hashes with keys :info, :format, :content
  #
  def diff(old, new)
    [old, new].each do |msg|
      msg[:format] = 'text/plain' if 'application/x-squish' == msg[:format]
      msg[:content] =
      case msg[:format]
      when nil
        msg[:content].split(/(?=\n\n)/)
      when 'text/plain', 'text/textile'
        msg[:content].split(/(?=\n)/)
      when 'text/html'
        msg[:content].split(%r{(?=<(?!\s*/))})
      else
        [msg[:content]]
      end
    end
    %{<table class="diff">\n<thead><tr>\n} +
    [old, new].collect {|msg| "<th>#{msg[:info]}</th>\n" }.join +
    %{</tr></thead>\n<tbody>\n} +
    diff_calculate(old[:content], new[:content]).collect {|left, right|
      left = left.join if left.kind_of? Array
      right = right.join if right.kind_of? Array
      width = right ? ' style="width: 50%"' : ' colspan="2"'
      left and left = {:format => old[:format], :content => left,
        :class => (right ? ' class="delete"' : '')}
      right and right = {:format => new[:format], :content => right,
        :class => ' class="add"'}
      %{<tr>\n} + [left, right].compact.collect {|line|
%{<td#{width}#{line[:class] if line[:content].size > 0}><div class="content">#{Template.format_inline_content(line[:content], line[:format])}#{'&nbsp;' unless line[:content].size > 0}</div></td>\n}
      }.join + "</tr>\n"
    }.join + %{</tbody></table>\n}
  end

  # wrap page title and body with heads and foots
  #
  # _body_ can be String or Array of pairs [title, body]; in latter case,
  # _title_ can be defaulted to the title of the first pair
  #
  # if _options_ includes :front_page flag, body isn't wrapped in box
  #
  def page(title, body, options={})
    main =
      if title and options.delete(:front_page)
        body = %{<div id="languages">#{language_list}</div>} + body if
          defined?(GetText) and 'yes' != @session.cookie('nostatic')
        body
      else
        body = [[title, body]] unless body.class == Array
        title = body[0][0] if title.nil?
        body.collect {|t, b, id| box(t, b, id) }.join
      end
    html = head(title.to_s, options) +
      site_head() +
      %{\n<div id="main">\n} + main + main_foot + "\n</div>\n" +
      foot
  end
end
