# Samizdat focus handling
#
#   Copyright (c) 2002-2005  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'

class Focus
  # get focus id from current focus if not supplied; get current focus from
  # config if focus cookie is not set; derive focus resource id from uriref
  # when necessary; keep external uriref if applies
  #
  # derive readable focus name from resource title or focus uriref
  #
  # if _related_ is a Resource, #rating and #rating= can be used
  #
  def initialize(session, id=nil, related=nil)   # todo: unit-test this beast
    @session = session

    if id and Resource.validate_id(id).nil?   # try uriref
      focus_ns = config['ns']['focus']
      if /\A(focus::|#{focus_ns})(Translation)\z/ =~ id
        @name = $2
        @uriref = focus_ns + @name   # keep external uriref
        id = rdf.get_property(@uriref, 's::id')   # derive id from uriref
      end
    end

    if id   # existing resource
      @resource = Resource.new(session, id)
      @id = @resource.id
      @name = @resource.render(:title)   # watch out for recursion!
    elsif @uriref   # virtual uriref resource
      [ _('Translation') ]   # rgettext hack
      @name = _(@name)   # translate focus names
    else
      raise ResourceNotFoundError, 'Focus ' + id.to_s
    end

    @related = related   # if nil, won't operate with rating
  end

  attr_reader :id, :uriref, :resource, :name, :related

  # rating is an integer from -2 to 2
  #
  def Focus.validate_rating(rating)
    rating = rating.to_i if rating
    (rating and rating >= -2 and rating <= 2)? rating: nil
  end

  # render focus title
  #
  def Focus.focus_title(title)
    _(title) + ' (' + _('Focus') + ')'   # translate focus names
  end

  # site focuses, ordered by number of related resources
  #
  def Focus.collect_focuses
    list = cache.fetch_or_add('focuses/0') do
      rdf.select_all( %{
SELECT ?focus, count(?resource)
WHERE (rdf::predicate ?stmt dc::relation)
      (rdf::subject ?stmt ?resource)
      (rdf::object ?stmt ?focus)
      (s::inReplyTo ?resource ?parent)
      (dct::isVersionOf ?resource ?current)
      (s::rating ?stmt ?rating)
LITERAL ?rating > 0 AND ?parent IS NULL AND ?current IS NULL
GROUP BY ?focus
ORDER BY count(?resource) DESC}, limit_page
      ).collect {|f, c| [f, c] }   # unwrap DBI::Row
    end
    list.collect {|focus, usage| yield focus, usage }
  end

  # read and cache rating from SamizdatRDF
  #
  def rating
    return nil unless @related.kind_of? Resource
    if @rating.nil?
      @rating, = rdf.select_one %{
SELECT ?rating
WHERE (rdf::subject ?stmt #{@related.id})
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{@id})
      (s::rating ?stmt ?rating)}

      @rating = @rating ? @rating.to_f : _('none')
    end
    @rating
  end

  # update rating in SamizdatRDF and in memory
  #
  def rating=(value)
    return nil unless @related.kind_of? Resource
    return nil unless value = Focus.validate_rating(value)

    raise UserError, _("You can't relate resource to itself") if
      @id == @related.id
    raise UserError, _('Translations should be published as replies to the original message') if
      'focus::Translation' == @uriref and
      rdf.get_property(@related.id, 's::inReplyTo').nil?

    # always make sure @id and @uriref are SQL-safe
    rdf.assert( %{
UPDATE ?rating = :rating
WHERE (rdf::subject ?stmt :related)
      (rdf::predicate ?stmt dc::relation)
      (rdf::object ?stmt #{@id or @uriref})
      (s::voteProposition ?vote ?stmt)
      (s::voteMember ?vote :member)
      (s::voteRating ?vote ?rating)},
      { :rating => value, :related => @related.id, :member => @session.id }
    )

    @rating = nil   # invalidate rating cache
    cache.flush
  end

  # order by rating, unrated after rated
  #
  def sort_index
    rating.kind_of?(Numeric) ? rating : -100
  end

  # print focus rating
  #
  def print_rating
    case rating
    when Numeric then "%4.2f" % rating
    when String then rating
    else rating.to_s
    end
  end
end
