/***************************************************************************
 * RT2400 SourceForge Project - http://rt2400.sourceforge.net              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 *                                                                         *
 *   Licensed under the GNU GPL                                            *
 *   Original code supplied under license from RaLink Inc, 2003.           *
 ***************************************************************************/

 /***************************************************************************
 *      Module Name: rtmp_info.c
 *
 *      Abstract: IOCTL related subroutines
 *
 *      Revision History:
 *      Who             When            What
 *      --------        -----------     -----------------------------
 *      RoryC           3rd  Jan 03     Created
 *      PaulW           29th Jul 03     Support iwpriv
 *      MarkW           9th  Feb 04     Baseline of code
 *      defekt          13th Feb 04     ATOMIC kmalloc fix
 *      Flavio		21st Jun 04	Fixed SIOCSIWRATE (MaxTxRate)
 *      Ivo             6th  Jul 04     Fix for essid set timing issue
 *      MarkW           8th  Dec 04     Fix setting of BSSID (982890)
 *	RobinC(RT2500)��16th Dec 04 � � support ifpreup scripts
 *      RobinC(RT2500)  21st Dec 04     RFMON Support
 *      MarkW(RT2500)   21st Dec 04     iwconfig mode monitor support
 *      LuisCorreia     9th  Jan 05     Fix ESSID+memory junk (1035107)
 *      MarkW           21st Feb 05     ESSID output fix
 * 		MarkW			9th  Jun 05		Fix channel change for ADHOC mode
 ***************************************************************************/


#include	"rt_config.h"

#ifndef IW_ESSID_MAX_SIZE
/* Maximum size of the ESSID and NICKN strings */
#define IW_ESSID_MAX_SIZE	32
#endif

#define NR_WEP_KEYS 4
#define WEP_SMALL_KEY_LEN (40/8)
#define WEP_LARGE_KEY_LEN (104/8)

/* The frequency of each channel in MHz */
const long channel_frequency[] = {
        2412, 2417, 2422, 2427, 2432, 2437, 2442,
        2447, 2452, 2457, 2462, 2467, 2472, 2484
};
#define NUM_CHANNELS ( sizeof(channel_frequency) / sizeof(channel_frequency[0]) )

const char NULL_AP[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
const char BROADCAST_AP[] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

int rt_ioctl_commit(PRTMP_ADAPTER pAd)
{
	u8		cntlstate = pAd->Mlme.CntlMachine.CurrState - CNTL_IDLE;

	if (cntlstate){
		printk(KERN_NOTICE "***RT2400***: !!! MLME busy, NIC will be reset!!! \n");
		NICDisableInterrupt(pAd);
		MlmeRadioOff(pAd);
	}
	MlmeHandler(pAd);
	if (cntlstate){
		MlmeRadioOn(pAd);
		NICEnableInterrupt(pAd);
	}

	return 0;

}

INT RT2400_ioctl(
	IN	struct net_device	*net_dev,
	IN	OUT	struct ifreq	*rq,
	IN	INT					cmd)
{
	PRTMP_ADAPTER						pAd= netdev_priv(net_dev);
	struct iwreq						*wrq = (struct iwreq *) rq;
	UCHAR								TxValue,RxValue;
	USHORT								subcmd;
	BOOLEAN								StateMachineTouched = FALSE;
	ULONG								BssLen, ulInfo, KeyIdx, Rssi, Preamble, FcsValue;
	ULONG								Now;
	INT									Status = NDIS_STATUS_SUCCESS;
	UINT								i;
	NDIS_802_11_RTS_THRESHOLD			RtsThresh;
	NDIS_802_11_FRAGMENTATION_THRESHOLD	FragThresh;
	NDIS_802_11_BSSID_LIST				*pBssidList = NULL;
	NDIS_802_11_MAC_ADDRESS				Bssid;
	RT_802_11_LINK_STATUS				LinkStatus;
	NDIS_802_11_CONFIGURATION			Configuration;
	NDIS_802_11_SSID                    Ssid;
	NDIS_802_11_NETWORK_INFRASTRUCTURE	BssType;
	NDIS_802_11_ANTENNA					Antenna;
	NDIS_802_11_TX_POWER_LEVEL			TxPowerLevel;
	NDIS_802_11_AUTHENTICATION_MODE		AuthMode;
	NDIS_802_11_WEP_STATUS				WepStatus;
	NDIS_802_11_WEP						WepKey;
	NDIS_802_11_CONFIGURATION			Config;
	NDIS_802_11_POWER_MODE				PowerMode;
	NDIS_802_11_STATISTICS				Statistics;
	RT_VERSION_INFO						DriverVersionInfo;
	BOOLEAN								RadioState;
#ifdef RT2400_DBG
	RT_802_11_HARDWARE_REGISTER			HardwareRegister;
#endif

	subcmd=wrq->u.data.flags;

	switch(cmd) {
		case SIOCGIWNAME:
			DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCGIWNAME\n");
			strcpy(wrq->u.name, "RT2400PCI");
			break;

		case SIOCGIWESSID:  //Get ESSID
			{
				char *essid = NULL;
				struct iw_point *erq = &wrq->u.essid;
				if(pAd->PortCfg.SsidLen)
				{
					erq->flags=1;
					erq->length = pAd->PortCfg.SsidLen;
					essid = pAd->PortCfg.Ssid;
				}
				else if(pAd->SsidLen)
				{// not the ANY ssid
					erq->flags = 1;
					erq->length = pAd->SsidLen;
					essid = pAd->Ssid;
				}
				else
				{//the ANY ssid was specified
				/* report ANY back */
					erq->flags=0;
					erq->length=0;
				}

				if(erq->pointer)
				{
					if(copy_to_user(erq->pointer, essid, erq->length))
					{
						Status = -EFAULT;
						break;
					}
				}
				DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCGIWESSID (Len=%d, ssid=%s...)\n", erq->length, essid);
			}
			break;

		case SIOCSIWESSID:  //Set ESSID
			{
				struct iw_point *erq = &wrq->u.essid;
				ULONG Length;

				memset(&Ssid, 0x00, sizeof(NDIS_802_11_SSID));

				if (erq->flags)
				{
					if (erq->length > IW_ESSID_MAX_SIZE)
					{
						Status = -E2BIG;
						break;
					}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,19)
					Length = erq->length - 1;
#else
					Length = erq->length;
#endif

					if (copy_from_user(Ssid.Ssid, erq->pointer, Length))
					{
						Status = -EFAULT;
						break;
					}
					Ssid.SsidLength = Length;
				}
				else
					Ssid.SsidLength = 0;  // ANY ssid

				if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_IN_USE))
					StateMachineTouched = TRUE;
				// tell CNTL state machine to call NdisMSetInformationComplete() after completing
				// this request, because this request is initiated by NDIS.
				pAd->Mlme.CntlAux.CurrReqIsFromNdis = TRUE;
				MlmeEnqueue(&pAd->Mlme.Queue,
				        MLME_CNTL_STATE_MACHINE,
				        OID_802_11_SSID,
						sizeof(NDIS_802_11_SSID),
						(VOID *)&Ssid
						);
				memcpy(pAd->Ssid, Ssid.Ssid, Ssid.SsidLength);
				pAd->SsidLen = Ssid.SsidLength;
 				pAd->PortCfg.AutoReconnect = TRUE;

				DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCSIWESSID (Len=%d,Ssid=%s)\n", Ssid.SsidLength, Ssid.Ssid);
				break;
			}
		case SIOCGIWNWID:
			Status = -EOPNOTSUPP;
			break;

		case SIOCSIWNWID:
			Status = -EOPNOTSUPP;
			break;

		case SIOCGIWRTS:  // Get the current RTS threshold
			wrq->u.rts.value = (INT) pAd->PortCfg.RtsThreshold;
			wrq->u.rts.disabled = (wrq->u.rts.value == MAX_RTS_THRESHOLD);
			wrq->u.rts.fixed = 1;

			DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCGIWRTS (=%d)\n",wrq->u.rts.value);
			break;

		case SIOCSIWRTS:
			RtsThresh = wrq->u.rts.value;
			if(wrq->u.rts.disabled)
				RtsThresh = MAX_RTS_THRESHOLD;
			if((RtsThresh < 0) || (RtsThresh > MAX_RTS_THRESHOLD))
			{
				Status = -EINVAL;
				break;
			}
			else
				pAd->PortCfg.RtsThreshold = (USHORT) RtsThresh;
			DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCSIWRTS (=%d)\n",RtsThresh);
			break;

		case SIOCGIWFRAG: // Get the current fragmentation threshold
			wrq->u.frag.value = pAd->PortCfg.FragmentThreshold;
			wrq->u.frag.disabled = (wrq->u.frag.value >= MAX_FRAG_THRESHOLD);
			wrq->u.frag.fixed = 1;
			break;

		case SIOCSIWFRAG:  // Set the desired fragmentation threshold
			FragThresh = wrq->u.frag.value;

			if(wrq->u.frag.disabled)
				FragThresh = MAX_FRAG_THRESHOLD;

			if((FragThresh < MIN_FRAG_THRESHOLD) || (FragThresh > MAX_FRAG_THRESHOLD))
			{
				Status = -EINVAL;
				break;
			}

			pAd->PortCfg.FragmentThreshold = (USHORT)FragThresh;
			DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCSIWFRAG (=%d)\n", FragThresh);
			break;
#if WIRELESS_EXT > 11
		case SIOCGIWSTATS:
			// TODO: Get /proc/net/wireless stats
			Status = -EOPNOTSUPP;
			break;

		case SIOCSIWSTATS:
			Status = -EOPNOTSUPP;
			break;
#endif
		case SIOCGIWAP:
			if (INFRA_ON(pAd) || ADHOC_ON(pAd))
                        {
				wrq->u.ap_addr.sa_family = ARPHRD_ETHER;
				memcpy(wrq->u.ap_addr.sa_data, &pAd->PortCfg.Bssid, ETH_ALEN);
                                DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCGIWAP(=%02x:%02x:%02x:%02x:%02x:%02x)\n",
                                pAd->PortCfg.Bssid.Octet[0],pAd->PortCfg.Bssid.Octet[1],pAd->PortCfg.Bssid.Octet[2],
                                pAd->PortCfg.Bssid.Octet[3],pAd->PortCfg.Bssid.Octet[4],pAd->PortCfg.Bssid.Octet[5]);
                        }
                        else
                        {
                                DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCGIWAP(=EMPTY)\n");
                                Status = -ENOTCONN;
                        }
			break;
		case SIOCSIWNICKN:
			{
				struct iw_point *erq = &wrq->u.data;
				char nickn[IW_ESSID_MAX_SIZE+1];

				memset(nickn, 0, sizeof(nickn));
				if (erq->flags)
				{
					if (erq->length > IW_ESSID_MAX_SIZE)
					{
						Status = -E2BIG;
						break;
					}
					if (copy_from_user(nickn, erq->pointer, erq->length))
					{
						Status = -EFAULT;
			break;
					}
					strcpy(pAd->nickn, nickn);
					DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCSIWNICKN(=%s)\n",pAd->nickn);
				}
			}
			break;
		case SIOCGIWNICKN:
			{
				struct iw_point *erq = &wrq->u.data;

				erq->length = strlen(pAd->nickn);
				if(copy_to_user(erq->pointer, pAd->nickn, erq->length))
					Status = -EFAULT;
			}
			break;
		case SIOCSIWAP:
			{
				PUCHAR ptr=NULL;
				ptr = (PUCHAR) &(wrq->u.ap_addr.sa_data);

				//memmove(&Bssid, ptr+2, sizeof(NDIS_802_11_MAC_ADDRESS));
				memmove(&Bssid, ptr, sizeof(NDIS_802_11_MAC_ADDRESS));

				if((memcmp(NULL_AP, &Bssid, sizeof(NDIS_802_11_MAC_ADDRESS)) == 0) ||
					(memcmp(BROADCAST_AP, &Bssid, sizeof(NDIS_802_11_MAC_ADDRESS)) == 0))
				{ //Any
					// tell CNTL state machine to call NdisMSetInformationComplete() after completing
					// this request, because this request is initiated by NDIS.
					memset(&Ssid, 0x00, sizeof(NDIS_802_11_SSID));
					pAd->Mlme.CntlAux.CurrReqIsFromNdis = TRUE;
					MlmeEnqueue(&pAd->Mlme.Queue,
								MLME_CNTL_STATE_MACHINE,
								OID_802_11_SSID,
								sizeof(NDIS_802_11_SSID),
								(VOID *)&Ssid
								);
					memcpy(pAd->Ssid, Ssid.Ssid, Ssid.SsidLength);
					pAd->SsidLen = Ssid.SsidLength;
 					pAd->PortCfg.AutoReconnect = TRUE;
					if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_IN_USE))
						StateMachineTouched = TRUE;
					DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCSIWAP (Len=%d,Ssid=%s)\n", Ssid.SsidLength, Ssid.Ssid);
				}
				else
				{
					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTPRIV_IOCTL_SET_BSSID\n");
					pAd->Mlme.CntlAux.CurrReqIsFromNdis = TRUE;

					MlmeEnqueue(&pAd->Mlme.Queue,
								MLME_CNTL_STATE_MACHINE,
								RT_OID_802_11_BSSID,
								sizeof(NDIS_802_11_MAC_ADDRESS),
								&Bssid);
					if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_INTERRUPT_IN_USE))
						StateMachineTouched = TRUE;
					DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCSIWAP Bssid[=%02x:%02x:%02x:%02x:%02x:%02x]\n",
								Bssid[0], Bssid[1], Bssid[2], Bssid[3], Bssid[4], Bssid[5]);
				}
			}
			break;

		case SIOCGIWFREQ:	//Get frequency / channel
			if (INFRA_ON(pAd) || ADHOC_ON(pAd))
				wrq->u.freq.m = pAd->PortCfg.Channel;
			else
				wrq->u.freq.m = pAd->PortCfg.IbssConfig.Channel;
			wrq->u.freq.e = 0;
			wrq->u.freq.i = 0;
			break;

		case SIOCSIWFREQ:  //Set frequency / channel
			{
				struct iw_freq *frq = &wrq->u.freq;
				int chan = -1;

				if((frq->e == 0) && (frq->m <= 1000))
				{
					/* Setting by channel number */
					chan = frq->m;
				}
				else
				{
					/* Setting by frequency - search the table */
					int mult = 1;

					for(i = 0; i < (6 - frq->e); i++)
						mult *= 10;

					for(i = 0; i < NUM_CHANNELS; i++)
						if(frq->m == (channel_frequency[i] * mult))
							chan = i+1;
				}

				if(!ChannelSanity(pAd, chan))
				{
					Status = -EINVAL;
					break;
				}
				pAd->PortCfg.IbssConfig.Channel = chan;
				DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCSIWFREQ (Channel=%d)\n", pAd->PortCfg.IbssConfig.Channel);
                                if (pAd->PortCfg.BssType == BSS_MONITOR || pAd->PortCfg.BssType == BSS_INDEP)
                                {
                                  pAd->PortCfg.Channel = chan;
                                  AsicSwitchChannel(pAd, pAd->PortCfg.Channel);
                                  AsicLockChannel(pAd, pAd->PortCfg.Channel);
                                }
			}
			break;
		case SIOCGIWRATE:  //Set Bit-Rate
			if (pAd->PortCfg.TxRate == RATE_1)
				wrq->u.bitrate.value = 1000000;
			else if (pAd->PortCfg.TxRate == RATE_2)
				wrq->u.bitrate.value = 2000000;
			else if (pAd->PortCfg.TxRate == RATE_5_5)
				wrq->u.bitrate.value = 5500000;
			else
				wrq->u.bitrate.value = 11000000;
			wrq->u.bitrate.fixed = (pAd->PortCfg.TxRate != RATE_11);
			wrq->u.bitrate.disabled = 0;
			break;

		case SIOCSIWRATE: //Set Bit-Rate
			switch (wrq->u.bitrate.value)
			{
				case -1:
					pAd->PortCfg.MaxTxRate = RATE_11;
					break; /* auto rate */
				case 1000000:
					pAd->PortCfg.MaxTxRate = RATE_1;
			break;
				case 2000000:
					pAd->PortCfg.MaxTxRate = RATE_2;
					break;
				case 5500000:
					pAd->PortCfg.MaxTxRate = RATE_5_5;
					break;
				case 11000000:
					pAd->PortCfg.MaxTxRate = RATE_11;
					break;
				default:
					Status = -EINVAL;
					break;
			}

			pAd->PortCfg.TxRateFixed = wrq->u.bitrate.fixed;

			if (pAd->PortCfg.TxRate > pAd->PortCfg.MaxTxRate || pAd->PortCfg.TxRateFixed)
				pAd->PortCfg.TxRate = pAd->PortCfg.MaxTxRate;

			DBGPRINT(RT_DEBUG_TRACE, "INFO::SIOCSIWRATE (MaxTxRate=%d)\n", pAd->PortCfg.MaxTxRate);
			break;
		case SIOCSIWENCODE:  //Set encryption stuff
			{
				int index = (wrq->u.encoding.flags & IW_ENCODE_INDEX) - 1;
					/* take the old default key if index is invalid */
				if((index < 0) || (index >= NR_WEP_KEYS))
					index = pAd->PortCfg.DefaultKeyId;  	// Default key for tx (shared key)
				if(wrq->u.encoding.pointer)
				{
					int len = wrq->u.encoding.length;

					if(len > WEP_LARGE_KEY_LEN)
						len = WEP_LARGE_KEY_LEN;

					memset(pAd->PortCfg.SharedKey[index].Key, 0, MAX_LEN_OF_KEY);
					if(copy_from_user(pAd->PortCfg.SharedKey[index].Key,
						wrq->u.encoding.pointer, len))
					{
						pAd->PortCfg.SharedKey[index].KeyLen = 0;
						Status = -EFAULT;
						break;
					}
					else
					{
						pAd->PortCfg.SharedKey[index].KeyLen =
							len <= WEP_SMALL_KEY_LEN ?
							WEP_SMALL_KEY_LEN : WEP_LARGE_KEY_LEN;
					}
				}
				pAd->PortCfg.DefaultKeyId = (UCHAR) index;
				pAd->PortCfg.PrivacyInvoked = ((wrq->u.encoding.flags & IW_ENCODE_DISABLED) == 0);
				if (wrq->u.encoding.flags & IW_ENCODE_RESTRICTED)
					pAd->PortCfg.AuthAlgorithm = AUTH_MODE_KEY;
				if (wrq->u.encoding.flags & IW_ENCODE_OPEN)
					pAd->PortCfg.AuthAlgorithm = AUTH_MODE_OPEN;

				if(pAd->PortCfg.PrivacyInvoked == FALSE)
					pAd->PortCfg.AuthAlgorithm = AUTH_MODE_OPEN;
			}
			break;
		case SIOCGIWENCODE: // Get the WEP keys and mode
			{
				int index = (wrq->u.encoding.flags & IW_ENCODE_INDEX) - 1;
				if ((index < 0) || (index >= NR_WEP_KEYS))
					index = pAd->PortCfg.DefaultKeyId;  	// Default key for tx (shared key)

				wrq->u.encoding.flags = pAd->PortCfg.PrivacyInvoked ? IW_ENCODE_RESTRICTED : IW_ENCODE_OPEN;
				if(!pAd->PortCfg.PrivacyInvoked)
					wrq->u.encoding.flags |= IW_ENCODE_DISABLED;
				if(wrq->u.encoding.pointer)
				{
					wrq->u.encoding.length = pAd->PortCfg.SharedKey[index].KeyLen;
					if (copy_to_user(wrq->u.encoding.pointer,
						pAd->PortCfg.SharedKey[index].Key,
						pAd->PortCfg.SharedKey[index].KeyLen))
					{
						Status = -EFAULT;
						break;
					}
					wrq->u.encoding.flags |= (index + 1);
				}
			}
			break;
		case SIOCGIWMODE: // Get operation mode
                        {
			        if (pAd->PortCfg.BssType == BSS_INDEP)
					wrq->u.mode = IW_MODE_ADHOC;
				else if (pAd->PortCfg.BssType == BSS_INFRA)
					wrq->u.mode = IW_MODE_INFRA;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20))
                                else if (pAd->PortCfg.BssType == BSS_MONITOR)
                                        wrq->u.mode = IW_MODE_MONITOR;
#endif
				else
					wrq->u.mode = IW_MODE_AUTO;
			}
			break;

		case SIOCSIWMODE:  //Set operation mode
			if(wrq->u.mode == IW_MODE_ADHOC)
			{
				pAd->PortCfg.BssType = BSS_INDEP;
				DBGPRINT(RT_DEBUG_TRACE, "INFO::RTPRIV_IOCTL_SET_INFRASTRUCTURE_MODE (AD-HOC)\n");
			}
			else if(wrq->u.mode == IW_MODE_INFRA)
			{
				pAd->PortCfg.BssType = BSS_INFRA;
				DBGPRINT(RT_DEBUG_TRACE, "INFO::RTPRIV_IOCTL_SET_INFRASTRUCTURE_MODE (INFRA)\n");
			}
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,20))
                        else if (wrq->u.mode == IW_MODE_MONITOR)
                        {
                                pAd->PortCfg.BssType = BSS_MONITOR;
                                DBGPRINT(RT_DEBUG_TRACE, "INFO:RTPRIV_IOCTL_SET_MONITOR_MODE (MONITOR)\n");
                        }
#endif
			else
                        {
                                DBGPRINT(RT_DEBUG_ERROR, "INVALID iwconfig Mode %d\n", wrq->u.mode);
				Status = -EINVAL;
                        }

                        if (pAd->PortCfg.BssType == BSS_MONITOR)
                        {
                              pAd->net_dev->type = 801;
                              RTMP_IO_WRITE32(pAd, RXCSR0, 0x46);
                        }
			else if (pAd->bAcceptPromiscuous == TRUE)
			{
			      pAd->net_dev->type = 1;
			      RTMP_IO_WRITE32(pAd, RXCSR0, 0x6e);
			}
                        else
                        {
                              pAd->net_dev->type = 1;
                              RTMP_IO_WRITE32(pAd, RXCSR0, 0x7e);
                        }

			break;
		case SIOCGIWPOWER:
		case SIOCSIWPOWER:
			Status = -EOPNOTSUPP;
			break;
		case SIOCGIWTXPOW:
			wrq->u.txpower.value = pAd->PortCfg.TxPower;
			wrq->u.txpower.disabled = RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF);
			wrq->u.txpower.fixed = 0;
			wrq->u.txpower.flags = IW_TXPOW_MWATT;
		break;
		case SIOCSIWTXPOW:
			if(wrq->u.txpower.disabled){
				MlmeRadioOff(pAd);
				printk(KERN_INFO "***rt2x00***: Info - SIOCSIWTXPOW: Radio off.\n");
			} else if(RTMP_TEST_FLAG(pAd, fRTMP_ADAPTER_RADIO_OFF)){
				MlmeRadioOn(pAd);
				printk(KERN_INFO "***rt2x00***: Info - SIOCSIWTXPOW: Radio on.\n");
			}
			TxPowerLevel = wrq->u.txpower.value;
			if(wrq->u.txpower.flags == IW_TXPOW_DBM){
				/*
				 * Formula to convert types:
				 * dBm = 30+10*LOG(WATT)
				 * Soo the formula to go the other way:
				 * Watt = 10^((dBm - 30)/10)
				 * Converting Watt to mWatt: *1000
				 */
				TxPowerLevel = (10^((TxPowerLevel - 30)/10)) * 1000;
			}

			for(i = 0; i < pAd->PortCfg.NumOfTxPowerLevel; i++){
				if(pAd->PortCfg.TxPowerLevel[i] <= TxPowerLevel)
					pAd->PortCfg.TxPower = pAd->PortCfg.TxPowerLevel[i];
				else
					break;
			}
			pAd->PortCfg.CurrentTxPowerLevelIndex = i;
			RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Tx_Power, pAd->PortCfg.TxPower);

		break;
		case SIOCDEVPRIVATE:
			switch(subcmd)
			{
				case RTLINUX_SET_OID_802_11_BSSID_LIST_SCAN:
					Now = jiffies;
#if 0
					if ((Oid == OID_802_11_BSSID_LIST_SCAN) &&
						(pAd->MediaState == NdisMediaStateConnected) &&
						(pAd->PortCfg.IgnoredScanNumber < 5))
					{
						DBGPRINT(RT_DEBUG_TRACE, ("!!! Link UP, ignore this set::RTLINUX_SET_OID_802_11_BSSID_LIST_SCAN\n"));
						pAd->PortCfg.IgnoredScanNumber++;
					}
					else
#endif

					if ((pAd->PortCfg.LastScanTime + 2 * 1000) < Now)
						{
							// Last scan within 2 sec, ignore it and report OK
							Status = NDIS_STATUS_SUCCESS;
						}
					else // if ((pAd->PortCfg.LastScanTime + 10 * 1000) < Now)	// Force at least 10 sec between 2 scan request
					{
						DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_BSSID_LIST_SCAN\n");
						// tell CNTL state machine to call NdisMSetInformationComplete() after completing
						// this request, because this request is initiated by NDIS.
						pAd->Mlme.CntlAux.CurrReqIsFromNdis = TRUE;
						// Reset Missed scan number
						pAd->PortCfg.IgnoredScanNumber = 0;
						pAd->PortCfg.LastScanTime = Now;

						MlmeEnqueue(&pAd->Mlme.Queue,
									MLME_CNTL_STATE_MACHINE,
									OID_802_11_BSSID_LIST_SCAN,
									0,
									NULL);

						StateMachineTouched = TRUE;
					}
					break;

				case RTLINUX_GET_OID_802_11_BSSID_LIST:
					BssLen = pAd->PortCfg.BssTab.BssNr * sizeof(NDIS_WLAN_BSSID) + sizeof(ULONG);
					pBssidList = (PNDIS_802_11_BSSID_LIST) kmalloc(BssLen, GFP_ATOMIC);
					if(pBssidList == NULL)
					{
		                Status = -ENOMEM;
		                break;
		            }
		            memset(pBssidList, 0, BssLen);

					pBssidList->NumberOfItems = pAd->PortCfg.BssTab.BssNr;

					for (i = 0; i < pAd->PortCfg.BssTab.BssNr; i++)
					{
						pBssidList->Bssid[i].Length = sizeof(NDIS_WLAN_BSSID);
		                memmove(&pBssidList->Bssid[i].MacAddress, &pAd->PortCfg.BssTab.BssEntry[i].Bssid, MAC_ADDR_LEN);
		                pBssidList->Bssid[i].Ssid.SsidLength = pAd->PortCfg.BssTab.BssEntry[i].SsidLen;
		                memmove(pBssidList->Bssid[i].Ssid.Ssid, pAd->PortCfg.BssTab.BssEntry[i].Ssid, pAd->PortCfg.BssTab.BssEntry[i].SsidLen);
		                pBssidList->Bssid[i].Privacy = pAd->PortCfg.BssTab.BssEntry[i].Privacy;
		                pBssidList->Bssid[i].Rssi = pAd->PortCfg.BssTab.BssEntry[i].Rssi - RSSI_TO_DBM_OFFSET;
		                pBssidList->Bssid[i].NetworkTypeInUse = Ndis802_11DS;
		                pBssidList->Bssid[i].Configuration.Length = sizeof(NDIS_802_11_CONFIGURATION);
		                pBssidList->Bssid[i].Configuration.BeaconPeriod = pAd->PortCfg.BssTab.BssEntry[i].BeaconPeriod;
		                pBssidList->Bssid[i].Configuration.ATIMWindow = pAd->PortCfg.BssTab.BssEntry[i].AtimWin;
						if (pAd->PortCfg.BssTab.BssEntry[i].Channel == 14)
							pBssidList->Bssid[i].Configuration.DSConfig = 2484000;
						else
							pBssidList->Bssid[i].Configuration.DSConfig = 2412000 + (pAd->PortCfg.BssTab.BssEntry[i].Channel - 1) * 5000;

		                if(pAd->PortCfg.BssTab.BssEntry[i].BssType == BSS_INFRA)
		                    pBssidList->Bssid[i].InfrastructureMode = Ndis802_11Infrastructure;
		                else
		                    pBssidList->Bssid[i].InfrastructureMode = Ndis802_11IBSS;

		                memmove(pBssidList->Bssid[i].SupportedRates, pAd->PortCfg.BssTab.BssEntry[i].Rates, pAd->PortCfg.BssTab.BssEntry[i].RatesLen);

		                DBGPRINT(RT_DEBUG_INFO,"INFO::RTLINUX_GET_OID_802_11_BSSID_LIST (BSS[%d], MAC = %02x:%02x:%02x:%02x:%02x:%02x, Ssid:%s)\n",
							i,
							pBssidList->Bssid[i].MacAddress[0],
                            pBssidList->Bssid[i].MacAddress[1],
                            pBssidList->Bssid[i].MacAddress[2],
                            pBssidList->Bssid[i].MacAddress[3],
                            pBssidList->Bssid[i].MacAddress[4],
                            pBssidList->Bssid[i].MacAddress[5],
                            pBssidList->Bssid[i].Ssid.Ssid);
					}

					wrq->u.data.length = BssLen;
					if (copy_to_user(wrq->u.data.pointer, pBssidList, wrq->u.data.length));
					break;

				case RTLINUX_GET_MEDIA_CONNECT_STATUS:
		            DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_GET_MEDIA_CONNECT_STATUS \n");
					wrq->u.data.length=sizeof(NDIS_MEDIA_STATE);
					if (copy_to_user(wrq->u.data.pointer, &pAd->MediaState, wrq->u.data.length));
					break;

				case RTLINUX_GET_OID_802_11_BSSID:
					if (INFRA_ON(pAd) || ADHOC_ON(pAd))
		            {
		                COPY_MAC_ADDR(&Bssid, &pAd->PortCfg.Bssid);
		                DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_BSSID(=%02x:%02x:%02x:%02x:%02x:%02x)\n",
		                         Bssid[0],Bssid[1],Bssid[2],Bssid[3],Bssid[4],Bssid[5]);
		                wrq->u.data.length = MAC_ADDR_LEN;
		                if (copy_to_user(wrq->u.data.pointer, &Bssid, wrq->u.data.length));
		            }
		            else
		            {
		                DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_BSSID(=EMPTY)\n");
		                Status = -ENOTCONN;
		            }
		            break;

				case RTLINUX_SET_OID_802_11_BSSID:
					if (wrq->u.data.length != sizeof(NDIS_802_11_MAC_ADDRESS))
			            Status = -EINVAL;
		                 else
			        {
						DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_BSSID\n");
						pAd->Mlme.CntlAux.CurrReqIsFromNdis = TRUE;

			            MlmeEnqueue(&pAd->Mlme.Queue,
			                        MLME_CNTL_STATE_MACHINE,
			                        RT_OID_802_11_BSSID,
			                        wrq->u.data.length,
			                        wrq->u.data.pointer);

			            StateMachineTouched = TRUE;
			        }
		            break;

				case RTLINUX_GET_OID_802_11_SSID:
					Ssid.SsidLength = pAd->PortCfg.SsidLen;
        		    memmove(Ssid.Ssid, pAd->PortCfg.Ssid, Ssid.SsidLength);
                            memset(Ssid.Ssid+Ssid.SsidLength,0,1);
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_SSID (Len=%d, ssid=%s...)\n", Ssid.SsidLength,Ssid.Ssid);

					wrq->u.data.length=sizeof(NDIS_802_11_SSID);
					if (copy_to_user(wrq->u.data.pointer, &Ssid, wrq->u.data.length));
					break;

				case RTLINUX_GET_OID_802_11_CONFIGURATION:
					Configuration.Length = sizeof(NDIS_802_11_CONFIGURATION);
					Configuration.BeaconPeriod = pAd->PortCfg.BeaconPeriod;
					Configuration.ATIMWindow = pAd->PortCfg.AtimWin;
					if (pAd->PortCfg.Channel == 14)
						Configuration.DSConfig = 2484000;
					else
						Configuration.DSConfig = 2412000 + (pAd->PortCfg.Channel - 1) * 5000;

					DBGPRINT(RT_DEBUG_TRACE, "INFO::OID_802_11_CONFIGURATION(BeaconPeriod=%d,AtimW=%d,Channel=%d) \n",
						Configuration.BeaconPeriod, Configuration.ATIMWindow, pAd->PortCfg.Channel);
					wrq->u.data.length = sizeof(NDIS_802_11_CONFIGURATION);
					if (copy_to_user(wrq->u.data.pointer, &Configuration, wrq->u.data.length));
					break;

				case RTLINUX_GET_RT_OID_802_11_QUERY_LINK_STATUS:
					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_RT_OID_802_11_QUERY_LINK_STATUS\n");
					if (pAd->PortCfg.TxRate == RATE_1)
		                LinkStatus.CurrTxRate = 2;  // unit: 500 kbps
		            else if (pAd->PortCfg.TxRate == RATE_2)
		                LinkStatus.CurrTxRate = 4;  // unit: 500 kbps
		            else if (pAd->PortCfg.TxRate == RATE_5_5)
		                LinkStatus.CurrTxRate = 11; // unit: 500 kbps
		            else
		                LinkStatus.CurrTxRate = 22; // unit: 500 kbps
		            LinkStatus.ChannelQuality = pAd->Mlme.RoamCqi;
		            LinkStatus.RxByteCount = pAd->RalinkCounters.ReceivedByteCount;
		            LinkStatus.TxByteCount = pAd->RalinkCounters.TransmittedByteCount;

					wrq->u.data.length = sizeof(RT_802_11_LINK_STATUS);
					if (copy_to_user(wrq->u.data.pointer, &LinkStatus, wrq->u.data.length));
					break;

				case RTLINUX_GET_OID_802_11_RSSI:
					Rssi = pAd->PortCfg.LastRssi - RSSI_TO_DBM_OFFSET;
					wrq->u.data.length = sizeof(NDIS_802_11_RSSI);
					if (copy_to_user(wrq->u.data.pointer, &Rssi, wrq->u.data.length));
					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_RSSI(=%d)\n", Rssi);
					break;

				case RTLINUX_GET_OID_802_11_INFRASTRUCTURE_MODE:
					if (ADHOC_ON(pAd))
		                ulInfo = Ndis802_11IBSS;
		            else if (INFRA_ON(pAd))
		                ulInfo = Ndis802_11Infrastructure;
		            else
		                ulInfo = Ndis802_11AutoUnknown;

					wrq->u.data.length = sizeof(NDIS_802_11_NETWORK_INFRASTRUCTURE);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));

					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_INFRASTRUCTURE_MODE(=%d)\n", ulInfo);
					break;

				case RTLINUX_SET_OID_802_11_INFRASTRUCTURE_MODE:
					if (wrq->u.data.length != sizeof(NDIS_802_11_NETWORK_INFRASTRUCTURE))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&BssType, wrq->u.data.pointer, wrq->u.data.length));
		                if (BssType == Ndis802_11IBSS)
		                {
		                    pAd->PortCfg.BssType = BSS_INDEP;
		                    DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_INFRASTRUCTURE_MODE (AD-HOC)\n");
		                }
		                else if (BssType == Ndis802_11Infrastructure)
		                {
		                    pAd->PortCfg.BssType = BSS_INFRA;
		                    DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_INFRASTRUCTURE_MODE (INFRA)\n");
		                }
		                else
		                {
		                    //if(BssType != Ndis802_11AutoUnknown)  // not supported???????????????
		                    Status = -EINVAL;
		                    DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_INFRASTRUCTURE_MODE (unknown)\n");
		                }
		            }
					break;

				case RTLINUX_GET_OID_802_11_TX_ANTENNA_SELECTED:
					if (pAd->PortCfg.CurrentTxAntenna == 0xff)
						ulInfo = 0xffffffff;
					else
						ulInfo = pAd->PortCfg.CurrentTxAntenna;

					wrq->u.data.length = sizeof(ulInfo);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));

					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_TX_ANTENNA_SELECTED(=%d)\n", ulInfo);
					break;

				case RTLINUX_SET_OID_802_11_TX_ANTENNA_SELECTED:
					if (wrq->u.data.length != sizeof(NDIS_802_11_ANTENNA))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&Antenna, wrq->u.data.pointer, wrq->u.data.length));
		                if (Antenna == 0xFFFFFFFF)
						{// Diversity
							pAd->PortCfg.CurrentTxAntenna = (UCHAR)Antenna;
							RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Tx_Configure, &TxValue);
                			TxValue = (TxValue & 0xFC) | 0x01;
                			RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Tx_Configure, TxValue);
						}
						else if (Antenna >= pAd->PortCfg.NumberOfAntenna)
							Status = -EINVAL;
						else
						{
							pAd->PortCfg.CurrentTxAntenna = (UCHAR)Antenna;

							RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Tx_Configure, &TxValue);
							if(Antenna == 0)
							{// Antenna A
								TxValue = (TxValue & 0xFC) | 0x00;
							}
							else if(Antenna == 1)
							{// Antenna B
								TxValue = (TxValue & 0xFC) | 0x02;
							}
                			RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Tx_Configure, TxValue);
						}
		            }
		            DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_SET_OID_802_11_TX_ANTENNA_SELECTED (=%d) \n",Antenna);
		            break;

				case RTLINUX_GET_OID_802_11_RX_ANTENNA_SELECTED:
					if (pAd->PortCfg.CurrentRxAntenna == 0xff)
						ulInfo = 0xffffffff;
					else
						ulInfo = pAd->PortCfg.CurrentRxAntenna;

					wrq->u.data.length = sizeof(ulInfo);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));

					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_RX_ANTENNA_SELECTED(=%d)\n", ulInfo);
            		break;

            	case RTLINUX_SET_OID_802_11_RX_ANTENNA_SELECTED:
					if (wrq->u.data.length != sizeof(NDIS_802_11_ANTENNA))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&Antenna, wrq->u.data.pointer, wrq->u.data.length));
		                if(Antenna == 0xFFFFFFFF)
						{// Diversity
							pAd->PortCfg.CurrentRxAntenna = (UCHAR)Antenna;
                			RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Rx_Configure, &RxValue);
                			RxValue = (RxValue & 0xF9) | 0x02;
                			RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Rx_Configure, RxValue);
						}
						else if(Antenna >= pAd->PortCfg.NumberOfAntenna)
							Status = -EINVAL;
						else
						{
							pAd->PortCfg.CurrentRxAntenna = (UCHAR)Antenna;

							RTMP_BBP_IO_READ32_BY_REG_ID(pAd, BBP_Rx_Configure, &RxValue);
							if(Antenna == 0)
							{// Antenna A
								RxValue = (RxValue & 0xF9) | 0x00;
							}
							else if(Antenna == 1)
							{// Antenna B
								RxValue = (RxValue & 0xF9) | 0x04;
							}
                			RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, BBP_Rx_Configure, RxValue);
						}
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_RX_ANTENNA_SELECTED (=%d)\n",Antenna);
		            break;

				case RTLINUX_GET_OID_802_11_DESIRED_RATES:
					wrq->u.data.length = sizeof(NDIS_802_11_RATES);
					if (copy_to_user(wrq->u.data.pointer, &pAd->PortCfg.DesiredRates, wrq->u.data.length));
		            DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_GET_OID_802_11_DESIRED_RATES (%d,%d,%d,%d,%d,%d,%d,%d)\n",
		                pAd->PortCfg.DesiredRates[0],pAd->PortCfg.DesiredRates[1],pAd->PortCfg.DesiredRates[2],pAd->PortCfg.DesiredRates[3],pAd->PortCfg.DesiredRates[4],pAd->PortCfg.DesiredRates[5],pAd->PortCfg.DesiredRates[6],pAd->PortCfg.DesiredRates[7]);
					break;

				case RTLINUX_SET_OID_802_11_DESIRED_RATES:
					if (wrq->u.data.length != sizeof(NDIS_802_11_RATES))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&pAd->PortCfg.DesiredRates, wrq->u.data.pointer, wrq->u.data.length));
		                // Changing DesiredRate may affect the MAX TX rate we used to TX frames out
		                MlmeUpdateTxRates(pAd);
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_DESIRED_RATES (%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x)\n",
		                pAd->PortCfg.DesiredRates[0],pAd->PortCfg.DesiredRates[1],
		                pAd->PortCfg.DesiredRates[2],pAd->PortCfg.DesiredRates[3],
		                pAd->PortCfg.DesiredRates[4],pAd->PortCfg.DesiredRates[5],
		                pAd->PortCfg.DesiredRates[6],pAd->PortCfg.DesiredRates[7] );
					break;

				case RTLINUX_GET_OID_802_11_QUERY_PREAMBLE:
					ulInfo = pAd->PortCfg.TxPreamble;
					wrq->u.data.length = sizeof(RT_802_11_PREAMBLE);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));
            		DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_QUERY_PREAMBLE(=%d)\n", ulInfo);
            		break;

				case RTLINUX_SET_OID_802_11_SET_PREAMBLE:
					if (wrq->u.data.length != sizeof(ULONG))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&Preamble, wrq->u.data.pointer, wrq->u.data.length));
		                if (Preamble == Rt802_11PreambleShort)
		                {
		                    pAd->PortCfg.WindowsTxPreamble = Preamble;
		                    MlmeSetTxPreamble(pAd, Rt802_11PreambleShort);
		                }
		                else if ((Preamble == Rt802_11PreambleLong) || (Preamble == Rt802_11PreambleAuto))
		                {
		                    // if user wants AUTO, initialize to LONG here, then change according to AP's
		                    // capability upon association.
		                    pAd->PortCfg.WindowsTxPreamble = Preamble;
		                    MlmeSetTxPreamble(pAd, Rt802_11PreambleLong);
		                }
		                else
		                    Status = -EINVAL;
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_SET_PREAMBLE (=%d)\n", Preamble);
					break;

				case RTLINUX_GET_OID_802_11_TX_POWER_LEVEL:
					ulInfo = (ULONG)pAd->PortCfg.TxPower;
					wrq->u.data.length = sizeof(NDIS_802_11_TX_POWER_LEVEL);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));
            		DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_TX_POWER_LEVEL(=%d)\n", ulInfo);
            		break;

            	case RTLINUX_SET_OID_802_11_TX_POWER_LEVEL:
            		if (wrq->u.data.length != sizeof(NDIS_802_11_TX_POWER_LEVEL))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&TxPowerLevel, wrq->u.data.pointer, wrq->u.data.length));
		                if (TxPowerLevel > MAX_TX_POWER_LEVEL)
		                    Status = -EINVAL;
		                else
		                    pAd->PortCfg.TxPower = (UCHAR)TxPowerLevel;
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_TX_POWER_LEVEL (=%d) \n",TxPowerLevel);
		            break;

				case RTLINUX_GET_OID_802_11_RTS_THRESHOLD:
					ulInfo = pAd->PortCfg.RtsThreshold;
					wrq->u.data.length = sizeof(NDIS_802_11_RTS_THRESHOLD);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_RTS_THRESHOLD(=%d)\n", ulInfo);
					break;

				case RTLINUX_SET_OID_802_11_RTS_THRESHOLD:
					if (wrq->u.data.length != sizeof(NDIS_802_11_RTS_THRESHOLD))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&RtsThresh, wrq->u.data.pointer, wrq->u.data.length));
						if (RtsThresh > MAX_RTS_THRESHOLD)
		                    Status = -EINVAL;
		                else
		                    pAd->PortCfg.RtsThreshold = (USHORT)RtsThresh;
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_RTS_THRESHOLD (=%d)\n",RtsThresh);
		            break;

            	case RTLINUX_GET_OID_802_11_FRAGMENTATION_THRESHOLD:
            		ulInfo = pAd->PortCfg.FragmentThreshold;
		            if (pAd->PortCfg.bFragmentZeroDisable == TRUE)
		            	ulInfo = 0;
		            wrq->u.data.length = sizeof(NDIS_802_11_FRAGMENTATION_THRESHOLD);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_FRAGMENTATION_THRESHOLD(=%d)\n", ulInfo);
		            break;

				case RTLINUX_SET_OID_802_11_FRAGMENTATION_THRESHOLD:
					if (wrq->u.data.length != sizeof(NDIS_802_11_FRAGMENTATION_THRESHOLD))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&FragThresh, wrq->u.data.pointer, wrq->u.data.length));
		                pAd->PortCfg.bFragmentZeroDisable = FALSE;
		                if (FragThresh > MAX_FRAG_THRESHOLD || FragThresh < MIN_FRAG_THRESHOLD)
		                {
		                	if (FragThresh == 0)
		                	{
		                		pAd->PortCfg.FragmentThreshold = MAX_FRAG_THRESHOLD;
		                		pAd->PortCfg.bFragmentZeroDisable = TRUE;
		                	}
		                	else
		                    	Status = -EINVAL;
		                }
		                else
		                    pAd->PortCfg.FragmentThreshold = (USHORT)FragThresh;
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_FRAGMENTATION_THRESHOLD (=%d) \n",FragThresh);
		            break;

				case RTLINUX_GET_OID_802_11_AUTHENTICATION_MODE:
					if (pAd->PortCfg.AuthAlgorithm == AUTH_MODE_OPEN)
		                ulInfo = Ndis802_11AuthModeOpen;
		            else if (pAd->PortCfg.AuthAlgorithm == AUTH_MODE_KEY)
		                ulInfo = Ndis802_11AuthModeShared;
		            else
		                ulInfo = Ndis802_11AuthModeAutoSwitch;
		            wrq->u.data.length = sizeof(NDIS_802_11_AUTHENTICATION_MODE);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));

		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_AUTHENTICATION_MODE(=%d)\n", ulInfo);
		            break;

				case RTLINUX_SET_OID_802_11_AUTHENTICATION_MODE:
					if (wrq->u.data.length != sizeof(NDIS_802_11_AUTHENTICATION_MODE))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&AuthMode, wrq->u.data.pointer, wrq->u.data.length));
		                if (AuthMode == Ndis802_11AuthModeOpen)
		            	    pAd->PortCfg.AuthAlgorithm = AUTH_MODE_OPEN;
		                else if (AuthMode == Ndis802_11AuthModeShared)
		                    pAd->PortCfg.AuthAlgorithm = AUTH_MODE_KEY;
		                else if (AuthMode == Ndis802_11AuthModeAutoSwitch)
		                    pAd->PortCfg.AuthAlgorithm = AUTH_MODE_AUTO_SWITCH;
		                else
		                    Status = -EINVAL;
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_AUTHENTICATION_MODE (=%d) \n",AuthMode);
		            break;

				case RTLINUX_GET_OID_802_11_WEP_STATUS:
					if (!pAd->PortCfg.PrivacyInvoked)
		                ulInfo = Ndis802_11WEPDisabled;
		            else
		            {
		                ulInfo = Ndis802_11WEPKeyAbsent;
						if (pAd->PortCfg.SharedKey[pAd->PortCfg.DefaultKeyId].KeyLen != 0)
						{
		         			for (i = 0; i < SHARE_KEY_NO; i++)
		            		{
		                		if(pAd->PortCfg.SharedKey[i].KeyLen != 0)
		                    		ulInfo = Ndis802_11WEPEnabled;
		            		}
						}
		            }
		            wrq->u.data.length = sizeof(NDIS_802_11_WEP_STATUS);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));

		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_WEP_STATUS(=%d)\n", ulInfo);
		            break;

				case RTLINUX_SET_OID_802_11_WEP_STATUS:
					if (wrq->u.data.length != sizeof(NDIS_802_11_WEP_STATUS))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&WepStatus, wrq->u.data.pointer, wrq->u.data.length));
		                if (WepStatus == Ndis802_11WEPEnabled)
		                    pAd->PortCfg.PrivacyInvoked = TRUE;
		                else if (WepStatus == Ndis802_11WEPDisabled)
		                    pAd->PortCfg.PrivacyInvoked = FALSE;
		                else
		                    Status = -EINVAL;
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_WEP_STATUS (=%d)\n",WepStatus);
		            break;

				case RTLINUX_SET_OID_802_11_REMOVE_WEP:
					if (wrq->u.data.length != sizeof(NDIS_802_11_KEY_INDEX))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&KeyIdx, wrq->u.data.pointer, wrq->u.data.length));
						if (KeyIdx & 0x80000000)
						{
                			// Should never set default bit when remove key
							Status = -EINVAL;
						}
						else
		                {
		                	KeyIdx = KeyIdx & 0x0fffffff;
		                    if (KeyIdx > 4)
		                        Status = -EINVAL;
		                    else
		                        pAd->PortCfg.SharedKey[KeyIdx].KeyLen = 0;
		                }
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_REMOVE_WEP (id=%d)\n",KeyIdx);
		            break;

				case RTLINUX_SET_OID_802_11_ADD_WEP:
					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_ADD_WEP\n");
					if (copy_from_user(&WepKey, wrq->u.data.pointer, wrq->u.data.length));
					KeyIdx = WepKey.KeyIndex & 0x0fffffff;

					// it is a shared key
					if (KeyIdx > 4)
						Status = -EINVAL;
					else
					{
						pAd->PortCfg.SharedKey[KeyIdx].KeyLen = (UCHAR) WepKey.KeyLength;
						memcpy(pAd->PortCfg.SharedKey[KeyIdx].Key, &WepKey.KeyMaterial, WepKey.KeyLength);
            			if (WepKey.KeyIndex & 0x80000000)
           				{
             				// Default key for tx (shared key)
               				pAd->PortCfg.DefaultKeyId = (UCHAR) KeyIdx;
            			}
				   }

					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_ADD_WEP (id=%d, Len=%d-byte)\n",
						KeyIdx, WepKey.KeyLength);
					break;

				case RTLINUX_SET_OID_802_11_CONFIGURATION:
					if (wrq->u.data.length != sizeof(NDIS_802_11_CONFIGURATION))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&Config, wrq->u.data.pointer, wrq->u.data.length));
		                pAd->PortCfg.IbssConfig.BeaconPeriod = (USHORT) Config.BeaconPeriod;
		                pAd->PortCfg.IbssConfig.AtimWin = (USHORT) Config.ATIMWindow;
		                if (Config.DSConfig == 2484000)
                			pAd->PortCfg.IbssConfig.Channel = 14;
						else if (Config.DSConfig >= 2412000 && Config.DSConfig < 2484000)
                			pAd->PortCfg.IbssConfig.Channel = (UCHAR)((Config.DSConfig - 2412000) / 5000) + 1;
						else
		                    Status = -EINVAL;
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_CONFIGURATION (BeacnPeriod=%d,AtimW=%d,Ch=%d)\n",
		                Config.BeaconPeriod, Config.ATIMWindow, pAd->PortCfg.IbssConfig.Channel);
		            break;

				case RTLINUX_GET_OID_802_11_POWER_MODE:
					ulInfo = pAd->PortCfg.WindowsPowerMode;
					wrq->u.data.length = sizeof(NDIS_802_11_POWER_MODE);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_POWER_MODE(=%d)\n", ulInfo);
		            break;

				case RTLINUX_SET_OID_802_11_POWER_MODE:
					if (wrq->u.data.length != sizeof(NDIS_802_11_POWER_MODE))
		                Status = -EINVAL;
		            else
		            {
		            	if (copy_from_user(&PowerMode, wrq->u.data.pointer, wrq->u.data.length));

		                // save user's policy here, but not change PortCfg.Psm immediately
		                if (PowerMode == Ndis802_11PowerModeCAM)
		                {
		                    // clear PSM bit immediately
		                    MlmeSetPsmBit(pAd, PWR_ACTIVE);

		                    pAd->PortCfg.RecvDtim = TRUE;
							if (pAd->PortCfg.WindowsACCAMEnable == FALSE)
			                    pAd->PortCfg.WindowsPowerMode = PowerMode;
			                pAd->PortCfg.WindowsBatteryPowerMode = PowerMode;
		                }
		                else if (PowerMode == Ndis802_11PowerModeMAX_PSP)
		                {
		                    // do NOT turn on PSM bit here, wait until MlmeCheckForPsmChange()
		                    // to exclude certain situations.
		                    //     MlmeSetPsmBit(pAd, PWR_SAVE);
							if (pAd->PortCfg.WindowsACCAMEnable == FALSE)
			                    pAd->PortCfg.WindowsPowerMode = PowerMode;
			                pAd->PortCfg.WindowsBatteryPowerMode = PowerMode;
		                    pAd->PortCfg.RecvDtim = FALSE;
		                    pAd->PortCfg.DefaultListenCount = 5;
		                }
		                else if (PowerMode == Ndis802_11PowerModeFast_PSP)
		                {
		                    // do NOT turn on PSM bit here, wait until MlmeCheckForPsmChange()
		                    // to exclude certain situations.
		                    //     MlmeSetPsmBit(pAd, PWR_SAVE);
		                    pAd->PortCfg.RecvDtim = TRUE;
							if (pAd->PortCfg.WindowsACCAMEnable == FALSE)
			                    pAd->PortCfg.WindowsPowerMode = PowerMode;
			                pAd->PortCfg.WindowsBatteryPowerMode = PowerMode;
		                    pAd->PortCfg.DefaultListenCount = 3;
		                }
		                else
		                    Status = -EINVAL;
		            }
		            DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_POWER_MODE (=%d)\n",PowerMode);
		            break;

				case RTLINUX_SET_OID_802_11_SSID:
					if (wrq->u.data.length > MAX_LEN_OF_SSID + sizeof(ULONG))
			            Status = -EINVAL;
		            else
		            {
		                if (copy_from_user(&Ssid, wrq->u.data.pointer, wrq->u.data.length));

		    	        if(Ssid.SsidLength > IW_ESSID_MAX_SIZE)
				            Status = -EINVAL;
			            else
			            {
		                    // tell CNTL state machine to call NdisMSetInformationComplete() after completing
		                    // this request, because this request is initiated by NDIS.
		                    pAd->Mlme.CntlAux.CurrReqIsFromNdis = TRUE;

				            MlmeEnqueue(&pAd->Mlme.Queue,
				                        MLME_CNTL_STATE_MACHINE,
				                        OID_802_11_SSID,
				                        wrq->u.data.length,
				                        wrq->u.data.pointer);
							memcpy(pAd->Ssid, Ssid.Ssid, Ssid.SsidLength);
							pAd->SsidLen = Ssid.SsidLength;
				            pAd->PortCfg.AutoReconnect = TRUE;
				            StateMachineTouched = TRUE;
			            }
		                DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_OID_802_11_SSID (Len=%d,Ssid=%s)\n", Ssid.SsidLength, Ssid.Ssid);
			        }
			        break;

				case RTLINUX_GET_OID_802_11_STATISTICS:
					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_GET_OID_802_11_STATISTICS \n");
		            // Update FCS counters
		            RTMP_IO_READ32(pAd, CNT0, &FcsValue);
		            pAd->WlanCounters.FCSErrorCount += FcsValue;

		            Statistics.TransmittedFragmentCount = pAd->WlanCounters.TransmittedFragmentCount;
		            Statistics.MulticastTransmittedFrameCount = pAd->WlanCounters.MulticastTransmittedFrameCount;
		            Statistics.FailedCount = pAd->WlanCounters.FailedCount;
		            Statistics.RetryCount = pAd->WlanCounters.RetryCount;
		            Statistics.MultipleRetryCount = pAd->WlanCounters.MultipleRetryCount;
		            Statistics.RTSSuccessCount = pAd->WlanCounters.RTSSuccessCount;
		            Statistics.RTSFailureCount = pAd->WlanCounters.RTSFailureCount;
		            Statistics.ACKFailureCount = pAd->WlanCounters.ACKFailureCount;
		            Statistics.FrameDuplicateCount = pAd->WlanCounters.FrameDuplicateCount;
		            Statistics.ReceivedFragmentCount = pAd->WlanCounters.ReceivedFragmentCount;
		            Statistics.MulticastReceivedFrameCount = pAd->WlanCounters.MulticastReceivedFrameCount;
		            Statistics.FCSErrorCount = pAd->WlanCounters.FCSErrorCount;
		            wrq->u.data.length = sizeof(NDIS_802_11_STATISTICS);
					if (copy_to_user(wrq->u.data.pointer, &Statistics, wrq->u.data.length));
		            break;

				case RTLINUX_GET_OID_GEN_RCV_OK:
					DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_GET_OID_GEN_RCV_OK \n");
					ulInfo = pAd->Counters.GoodReceives;
					wrq->u.data.length = sizeof(ulInfo);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));
					break;

				case RTLINUX_GET_OID_GEN_RCV_NO_BUFFER:
					DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_GET_OID_GEN_RCV_NO_BUFFER \n");
					ulInfo = pAd->Counters.RxNoBuffer;
					wrq->u.data.length = sizeof(ulInfo);
					if (copy_to_user(wrq->u.data.pointer, &ulInfo, wrq->u.data.length));
					break;

				case RTLINUX_SET_RT_OID_802_11_RESET_COUNTERS:
					DBGPRINT(RT_DEBUG_TRACE, "INFO::RTLINUX_SET_RT_OID_802_11_RESET_COUNTERS!!!\n");
					memset(&pAd->WlanCounters, 0, sizeof(COUNTER_802_11));
        		    memset(&pAd->Counters, 0, sizeof(COUNTER_802_3));
    	        	memset(&pAd->RalinkCounters, 0 ,sizeof(COUNTER_RALINK));
		            memset(&pAd->Mlme.PrevWlanCounters, 0, sizeof(COUNTER_802_11));
					break;

				case RTLINUX_GET_OID_802_3_CURRENT_ADDRESS:
					DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_GET_OID_802_3_CURRENT_ADDRESS \n");
					wrq->u.data.length = MAC_ADDR_LEN;
					if (copy_to_user(wrq->u.data.pointer, &pAd->CurrentAddress, wrq->u.data.length));
					break;

				case RTLINUX_GET_VERSION_INFO:
					DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_GET_VERSION_INFO \n");
					DriverVersionInfo.DriverMajorVersion = DRV_VERSION_MAJOR;
					DriverVersionInfo.DriverMinorVersion = DRV_VERSION_MINOR;
					DriverVersionInfo.DriverSubVersion   = DRV_VERSION_SUB;
					DriverVersionInfo.DriverBuildYear    = DRV_BUILD_YEAR;
					DriverVersionInfo.DriverBuildMonth   = DRV_BUILD_MONTH;
					DriverVersionInfo.DriverBuildDay     = DRV_BUILD_DAY;
					wrq->u.data.length = sizeof(RT_VERSION_INFO);
					if (copy_to_user(wrq->u.data.pointer, &DriverVersionInfo, wrq->u.data.length));
					break;

#ifdef RT2400_DBG
				case RTLINUX_GET_RT_OID_802_11_QUERY_HARDWARE_REGISTER:
		        	if (wrq->u.data.length != sizeof(RT_802_11_HARDWARE_REGISTER))
		                Status = -EINVAL;
		            else
		            {
			        	copy_from_user(&HardwareRegister, wrq->u.data.pointer, sizeof(RT_802_11_HARDWARE_REGISTER));
		                if (HardwareRegister.HardwareType == 0)
		                {
		                    // MAC register read procedure
		                    RTMP_IO_READ32(pAd, HardwareRegister.Offset, &HardwareRegister.Data);
		                    wrq->u.data.length = sizeof(RT_802_11_HARDWARE_REGISTER);
							copy_to_user(wrq->u.data.pointer, &HardwareRegister, wrq->u.data.length);
		                }
		                else if (HardwareRegister.HardwareType == 1)
		                {
		                    // BBP register read procedure
		                    RTMP_BBP_IO_READ32_BY_REG_ID(pAd, HardwareRegister.Offset, &HardwareRegister.Data);
		                    wrq->u.data.length = sizeof(RT_802_11_HARDWARE_REGISTER);
		                    copy_to_user(wrq->u.data.pointer, &HardwareRegister, wrq->u.data.length);
		                }
		                else
		                {
			                wrq->u.data.length = 0;
		                    Status = -EINVAL;
		                }
		                DBGPRINT(RT_DEBUG_TRACE, "Query::RT_OID_802_11_QUERY_HARDWARE_REGISTER (type=%d, offset=%d, data=%d)\n", HardwareRegister.HardwareType, HardwareRegister.Offset, HardwareRegister.Data);
		            }
		            break;

		        case RTLINUX_SET_RT_OID_802_11_SET_HARDWARE_REGISTER:
		        	if (wrq->u.data.length != sizeof(RT_802_11_HARDWARE_REGISTER))
		                Status = -EINVAL;
		            else
		            {
		            	copy_from_user(&HardwareRegister, wrq->u.data.pointer, sizeof(RT_802_11_HARDWARE_REGISTER));
		                if (HardwareRegister.HardwareType == 0)
		                {
		                    // MAC register write procedure
		                    RTMP_IO_WRITE32(pAd, HardwareRegister.Offset, HardwareRegister.Data);
		                }
		                else if (HardwareRegister.HardwareType == 1)
		                {
		                    // BBP register write procedure
		                    RTMP_BBP_IO_WRITE32_BY_REG_ID(pAd, HardwareRegister.Offset, HardwareRegister.Data);
		                }
		                else if (HardwareRegister.HardwareType == 2)
		                {
		                    // RF register write procedure (haven't been verified, like bit order)
#if 0
		                    RfCsr.word = 0;
		                    RfCsr.field.Busy = 1;
		                    RfCsr.field.RFRegValue = pHardwareRegister->Data;
		                    RfCsr.field.NumberOfBits= 20;
		                    RfCsr.field.IFSelect= 0;
		                    RfCsr.field.PLL_LD= 0;
		                    RTMP_RF_IO_WRITE32(pAd, RfCsr.word);
#endif
		                    Status = -EOPNOTSUPP;
		                }
		                DBGPRINT(RT_DEBUG_TRACE, "Set::RT_OID_802_11_QUERY_HARDWARE_REGISTER (type=%d, offset=%d, data=%d)\n", HardwareRegister.HardwareType, HardwareRegister.Offset, HardwareRegister.Data);
		            }
		        	break;
#endif
				case RTLINUX_SET_RT_OID_802_11_SET_RADIO:
					DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_SET_RT_OID_802_11_SET_RADIO \n");
					if (copy_from_user(&RadioState, wrq->u.data.pointer, wrq->u.data.length));
					if (pAd->PortCfg.bRadio != RadioState)
					{
						pAd->PortCfg.bRadio = RadioState;
						if (RadioState == TRUE)
							MlmeRadioOn(pAd);
						else
							MlmeRadioOff(pAd);
					}
					break;

				case RTLINUX_GET_RT_OID_802_11_QUERY_RADIO:
					DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_GET_RT_OID_802_11_QUERY_RADIO \n");
					wrq->u.data.length = sizeof(pAd->PortCfg.bRadio);
					if (copy_to_user(wrq->u.data.pointer, &pAd->PortCfg.bRadio, wrq->u.data.length));
					break;

				case RTLINUX_RT_OID_802_11_SET_COUNTRY_REGION:
					DBGPRINT(RT_DEBUG_INFO, "INFO::RTLINUX_RT_OID_802_11_SET_COUNTRY_REGION \n");
					if (copy_from_user(&pAd->PortCfg.CountryRegion, wrq->u.data.pointer, wrq->u.data.length));
					break;

				default:
					DBGPRINT(RT_DEBUG_TRACE, "INFO::unknown IOCTL's subcmd = 0x%08x\n", subcmd);
					Status = -EOPNOTSUPP;
					break;
			}
			break;

		default:
			DBGPRINT(RT_DEBUG_TRACE, "INFO::unknown IOCTL's cmd = 0x%08x\n", cmd);
			Status = -EOPNOTSUPP;
			break;
	}

	if(Status == NDIS_STATUS_SUCCESS)
	{
		if((cmd == SIOCDEVPRIVATE) && (subcmd == RTLINUX_GET_OID_802_11_BSSID_LIST))
            kfree(pBssidList);
	}
	if(StateMachineTouched) // Upper layer sent a MLME-related operations
	    rt_ioctl_commit(pAd);

	return Status;
}
