/*************************************************** */
/* Rule Set Based Access Control                     */
/*                                                   */
/* Author and (c) 1999-2005: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 31/May/2005                        */
/*************************************************** */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <rsbac/types.h>
#include <rsbac/syscalls.h>
#include <rsbac/error.h>
#include "nls.h"
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

char * progname;
int verbose = 0;
int delhome = 0;
rsbac_list_ta_number_t ta_number = 0;

void use(void)
    {
      printf(gettext("%s (RSBAC %s)\n***\n"), progname, VERSION);
      printf(gettext("Use: %s [flags] user [user2 ...]\n"), progname);
      printf(gettext(" -v = verbose,\n"));
      printf(gettext(" -r = remove user's home dir\n"));
      printf(gettext(" -N ta = transaction number (default = value of RSBAC_TA, if set, or 0)\n"));
    }

int process(char * name)
  {
      int res = 0;
      int dh_tmp = delhome;
      rsbac_uid_t user = RSBAC_NO_USER;
      union rsbac_um_mod_data_t data;

      if(rsbac_um_get_uid(ta_number, name, &user))
        {
          user = strtoul(name,0,0);
          if(!user && strcmp(name,"0"))
            {
              fprintf(stderr, gettext("%s: Unknown user %s\n"), progname, name);
              return 1;
            }
        }
      if(verbose)
        printf("Deleting user %s, uid %u\n", name, user);
      if(dh_tmp)
        {
          res = rsbac_um_get_user_item(ta_number, user, UM_homedir, &data);
          if(res < 0)
            {
              fprintf(stderr, "Getting user %s homedir failed with error", name);
              show_error(res);
              fprintf(stderr, ", homedir will not be deleted!\n");
              dh_tmp = 0;
            }
        }
      res = rsbac_um_remove_user(ta_number, user);
      if(res)
        {
          fprintf(stderr, "%s: ", name);
          show_error(res);
          return res;
        }
      if(dh_tmp)
        {
          char command[RSBAC_MAXNAMELEN];
          FILE * pfile;

          snprintf(command, RSBAC_MAXNAMELEN, "/bin/rm -rf \"%s\"",
                   data.string);
          pfile = popen(command, "w");
          if(!pfile)
            {
              fprintf(stderr, "Removing user %s homedir %s failed with error", name, data.string);
              show_error(res);
              fprintf(stderr, "\n");
            }
          else
            {
              pclose(pfile);
            }
        }
      return 0;
  }


int main(int argc, char ** argv)
{
  int res = 0;

  locale_init();

  progname = argv[0];

  {
    char * env = getenv("RSBAC_TA");

    if(env)
      ta_number = strtoul(env,0,0);
  }
  while((argc > 1) && (argv[1][0] == '-'))
    {
      char * pos = argv[1];
      pos++;
      while(*pos)
        {
          switch(*pos)
            {
              case 'h':
                use();
                return 0;
              case 'v':
                verbose++;
                break;
              case 'r':
                delhome=1;
                break;
              case 'N':
                if(argc > 2)
                  {
                    ta_number = strtoul(argv[2], 0, 10);
                    argc--;
                    argv++;
                  }
                else
                  {
                    fprintf(stderr, gettext("%s: missing transaction number value for parameter %c\n"), progname, *pos);
                    exit(1);
                  }
                break;

              default:
                fprintf(stderr, gettext("%s: unknown parameter %c\n"), progname, *pos);
                exit(1);
            }
          pos++;
        }
      argv++;
      argc--;
    }

  if (argc > 1)
    {
      int i;

      for(i=1; i< argc; i++)
        process(argv[i]);
      exit(0);
    }
  else
    {
      use();
      return 1;
    }
  return (0);
}
