//playlist.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2009-2010
 *
 *  This file is part of RoarAudio PlayList Daemon,
 *  a playlist management daemon for RoarAudio.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include "rpld.h"

struct rpld_playlist_entry * rpld_ple_new     (void) {
 struct rpld_playlist_entry * plent = roar_mm_malloc(sizeof(struct rpld_playlist_entry));
 static uint64_t counter = 0;
 unsigned char   rand_data = roar_random_uint16() & 0xFF;

 if ( plent == NULL )
  return NULL;

 memset(plent, 0, sizeof(struct rpld_playlist_entry));

 plent->global_tracknum        = (((ei_t      )getpid() & 0xFF) << 56) |
                                 (((uint64_t)RPLD_PL_MAGIC) << 48)     |
                                 (counter << 8)                        |
                                 rand_data;
 plent->global_short_tracknum  = (((ei_short_t)getpid() & 0xFF) << 24) ^ (counter & 0x00FF0000) >> 16;
 plent->global_short_tracknum |= (counter & 0xFFFF) << 8;
 plent->global_short_tracknum |= rand_data ^ RPLD_PL_MAGIC;

 plent->codec                  = -1;
 plent->meta.genre             = -1;

 plent->likeness               =  1;

#ifdef HAVE_LIB_UUID
 uuid_generate_random(plent->uuid);
#endif

 counter++;

 return plent;
}

void                         rpld_ple_free    (struct rpld_playlist_entry * plent) {
 if ( plent == NULL )
  return;

 roar_mm_free(plent);
}

struct rpld_playlist_entry * rpld_ple_copy    (struct rpld_playlist_entry * plent) {
 struct rpld_playlist_entry * ret = NULL;
 ei_t trknl;
 ei_short_t trkns;

 if ( plent == NULL )
  return NULL;

 if ( (ret = rpld_ple_new()) == NULL )
  return NULL;

 trknl = ret->global_tracknum;
 trkns = ret->global_short_tracknum;

 memcpy(ret, plent, sizeof(struct rpld_playlist_entry));
 memset(&(ret->list), 0, sizeof(ret->list));

 ret->global_tracknum       = trknl;
 ret->global_short_tracknum = trkns;

 return ret;
}

struct rpld_playlist_entry * rpld_ple_cut_out_next (struct rpld_playlist_entry * plent) {
 struct rpld_playlist_entry * next;

 if ( plent == NULL )
  return NULL;

 // test if there is a next element in chain:
 if ( (next = plent->list.next) == NULL )
  return NULL;

 plent->list.next = next->list.next;

 memset(&(next->list), 0, sizeof(next->list));

 return next;
}

const char                 * rpld_ple_time_hr (struct rpld_playlist_entry * plent) {
 static char buf[32];
 struct tm tm;
 time_t    length = plent->length;

 tm.tm_hour = length / 3600;
 length = length % 3600;

 tm.tm_min = length / 60;
 length = length % 60;

 tm.tm_sec = length; 

 snprintf(buf, 31, "%.2u:%.2u:%.2u", tm.tm_hour, tm.tm_min, tm.tm_sec);

 buf[31] = 0;

 return buf;
}

int                          rpld_ple_set_time_hr (struct rpld_playlist_entry * plent, char * time) {
 char * cur;
 char * next;
 time_t length = 0;

 if ( plent == NULL || time == NULL )
  return -1;

 cur  = time;

 while ( cur != NULL ) {
  next = strstr(cur, ":");
  if ( next != NULL ) {
   *next = 0;
    next++;
  }

  length *= 60;

  length += atol(cur);

  cur = next;
 }

 plent->length = length;

 return 0;
}

struct rpld_playlist       * rpld_pl_new      (void) {
 struct rpld_playlist * pl = roar_mm_malloc(sizeof(struct rpld_playlist));

 if ( pl == NULL )
  return NULL;

 memset(pl, 0, sizeof(struct rpld_playlist));

 return pl;
}

void                         rpld_pl_free     (struct rpld_playlist * pl) {
 if ( pl == NULL )
  return;

 rpld_pl_flush(pl);

 if ( pl->pointer_counter ) {
  pl->flags |= RPLD_PL_FLAG_UNLINKED;
  return;
 }

 if ( pl->id )
  rpld_pl_unregister(pl);

 if ( pl->name )
  roar_mm_free(pl->name);

 roar_mm_free(pl);
}

void                         rpld_pl_push     (struct rpld_playlist * pl, struct rpld_playlist_entry * plent) {
 struct rpld_playlist_entry * next = pl->first;
 struct rpld_playlist_entry * last;

 if ( rpld_pl_transaction(pl, RPLD_PL_TRANS_ASK) ) {
  plent->policy = pl->policy;
 }

 if ( next == NULL ) {
  pl->first = plent;
 } else {
  while (next != NULL) {
   last = next;
   next = next->list.next;
  }

  last->list.next = plent;
 }

 pl->likeness_hint += plent->likeness;
 pl->size_hint++;
}

void                         rpld_pl_add      (struct rpld_playlist * pl, struct rpld_playlist_entry * plent, ssize_t pos) {
 struct rpld_playlist_entry * next = pl->first;
 struct rpld_playlist_entry * last;

 if ( rpld_pl_transaction(pl, RPLD_PL_TRANS_ASK) ) {
  plent->policy = pl->policy;
 }

 if ( pos == -1 ) {
  plent->list.next = pl->first;
  pl->first        = plent;

  pl->likeness_hint += plent->likeness;
  pl->size_hint++;

  return;
 }

 if ( pos < 0 )
  return;

 pl->likeness_hint += plent->likeness;
 pl->size_hint++;

 if ( next == NULL ) {
  pl->first = plent;
 } else {
  while (next != NULL) {
//   printf("pos=%lu, next=%p\n", (unsigned long)pos, next);
   if ( pos == 0 ) {
    plent->list.next = next->list.next;
    next->list.next  = plent;
    return;
   }

   pos--;

   last = next;
   next = next->list.next;
  }

  last->list.next = plent;
 }
}

struct rpld_playlist_entry * rpld_pl_get_first(struct rpld_playlist * pl) {
 if ( pl == NULL )
  return NULL;

 if ( pl->flags & RPLD_PL_FLAG_VIRTUAL ) {
  return pl->first->list.next;
 } else {
  return pl->first;
 }
}

struct rpld_playlist_entry * rpld_pl_shift    (struct rpld_playlist * pl) {
 struct rpld_playlist_entry * first = pl->first;

 if ( pl->first == NULL ) {
  return NULL;
 } else {
  if ( pl->flags & RPLD_PL_FLAG_VIRTUAL ) {
   first = first->list.next;
   pl->first->list.next = first->list.next;
  } else {
   pl->first = first->list.next;
  }

  pl->likeness_hint -= first->likeness;
  pl->size_hint--;

  return first;
 }
}

struct rpld_playlist_entry * rpld_pl_pop      (struct rpld_playlist * pl) {
 struct rpld_playlist_entry * cur = pl->first;
 struct rpld_playlist_entry * old = NULL;
 struct rpld_playlist_entry * next;

 if ( pl->flags & RPLD_PL_FLAG_VIRTUAL ) {
  old = cur;
  cur = cur->list.next;

  if ( cur == NULL )
   return NULL;
 }

 while ((next = cur->list.next) != NULL) {
  old = cur;
  cur = next;
 }

 if ( old == NULL ) {
  pl->first = NULL;
 } else {
  old->list.next = NULL;
 }

 pl->likeness_hint -= cur->likeness;
 pl->size_hint--;

 return cur;
}

void                         rpld_pl_flush    (struct rpld_playlist * pl) {
 struct rpld_playlist_entry * next = pl->first;
 struct rpld_playlist_entry * cur;

 while (next != NULL) {
  cur  = next;
  next = next->list.next;

  rpld_ple_free(cur);
 }

 pl->first = NULL;

 if ( pl->flags & RPLD_PL_FLAG_VIRTUAL )
  pl->flags -= RPLD_PL_FLAG_VIRTUAL;

 pl->size_hint     = 0;
 pl->likeness_hint = 0;
}

int                          rpld_pl_delent   (struct rpld_playlist * pl, uint64_t tracknum) {
 struct rpld_playlist_entry * next = pl->first;
 struct rpld_playlist_entry * priv;

 if ( pl->flags & RPLD_PL_FLAG_VIRTUAL )
  next = next->list.next;

 if ( next == NULL ) {
  return -1;
 } else if ( next->global_tracknum == tracknum ) {
  pl->first = next->list.next;
  pl->likeness_hint -= next->likeness;
  pl->size_hint--;
  rpld_ple_free(next);
  return 0;
 }

 priv = next;
 next = next->list.next;

 while (next != NULL) {
  if ( next->global_tracknum == tracknum ) {
   priv->list.next = next->list.next;
   pl->likeness_hint -= next->likeness;
   pl->size_hint--;
   rpld_ple_free(next);
   return 0;
  }
  next = next->list.next;
 }

 return -1;
}

static float rand_float(float max);
static size_t rand_size_t(size_t max) {
/*
 size_t ret = rand() * max;

 ROAR_DBG("rand_size_t(max=%zu) = ?", max);

 return ret / RAND_MAX;
*/
 return rand_float((float)max);
}

static float rand_float(float max) {
 const float range = 65535.;
 float ret = (float)roar_random_uint16();

 ret *= max;

 ROAR_DBG("rand_size_t(max=%f) = ?", (double)max);

 return ret / range;
}

struct rpld_playlist_entry * rpld_pl_search   (struct rpld_playlist        * pl,
                                               struct rpld_playlist_search * search,
                                               struct rpld_playlist_entry  * pmatch) {
 struct rpld_playlist_entry * cur  = NULL;
 struct rpld_playlist_entry * prev = NULL;
 size_t counter                    = 0;
 float likeness                    = 0.0;
 size_t target_num                 = 0;
 float target_likeness             = 0.0;

 if ( pl == NULL || search == NULL )
  return NULL;

 ROAR_DBG("rpld_pl_search(*) = ?");

 ROAR_DBG("rpld_pl_search(*): pl ID=%i", rpld_pl_get_id(pl));

 switch (search->type) {
  case RPLD_PL_ST_NUM:
    target_num = search->subject.num;
   break;
  case RPLD_PL_ST_LIKENESS:
    target_likeness = search->subject.likeness;
   break;
  case RPLD_PL_ST_RANDOM:
    target_num = rand_size_t(pl->size_hint);
   break;
  case RPLD_PL_ST_RANDOM_LIKENESS:
    target_likeness = rand_float(pl->likeness_hint);
   break;
 }

 if ( search->options & RPLD_PL_SO_PREVIOUS ) {
  if ( rpld_pl_virtual(pl, RPLD_PL_VIRT_CREATE) != 0 )
   return NULL;
 }

 ROAR_DBG("rpld_pl_search(*) = ?");

 cur = pl->first;

 if ( pl->flags & RPLD_PL_FLAG_VIRTUAL ) {
  prev = cur;
  cur  = cur->list.next;
 }

 if ( pmatch != NULL ) {
  prev = pmatch;
  cur  = prev->list.next;
 }

 ROAR_DBG("rpld_pl_search(*) = ?");

#define _match() \
      if ( search->options & RPLD_PL_SO_PREVIOUS ) { \
       return prev; \
      } else { \
       return cur; \
      }

 while (cur != NULL) {
  likeness += cur->likeness;

  switch (search->type) {
   case RPLD_PL_ST_TRACKNUM_LONG:
     if ( cur->global_tracknum == search->subject.long_tracknum ) {
      _match();
     }
    break;
   case RPLD_PL_ST_TRACKNUM_SHORT:
     if ( cur->global_short_tracknum == search->subject.short_tracknum ) {
      _match();
     }
    break;
#ifdef HAVE_LIB_UUID
    case RPLD_PL_ST_UUID:
     if ( !uuid_compare(cur->uuid, search->subject.uuid) ) {
      _match();
     }
    break;
#endif
   case RPLD_PL_ST_NUM:
   case RPLD_PL_ST_RANDOM:
     if ( counter == target_num ) {
      _match();
     }
    break;
   case RPLD_PL_ST_LIKENESS:
   case RPLD_PL_ST_RANDOM_LIKENESS:
     if ( likeness >= target_likeness ) {
      _match();
     }
    break;
   default: return NULL;
  }

  prev = cur;
  cur  = cur->list.next;
  counter++;
 }

#undef _match

 // update _hints:
 if ( pmatch == NULL ) {
  pl->size_hint = counter;
  pl->likeness_hint = likeness;
 }

 // if we did not found any entry, but on random search we need to re-search with updated hints:
 if ( pmatch == NULL && counter > 0 ) {
  if ( search->type == RPLD_PL_ST_RANDOM || search->type == RPLD_PL_ST_RANDOM_LIKENESS ) {
   return rpld_pl_search(pl, search, pmatch);
  }
 }

 ROAR_DBG("rpld_pl_search(*) = ?");

 return NULL;
}

int                          rpld_pl_virtual  (struct rpld_playlist * pl, int virt) {
 struct rpld_playlist_entry * plent;

 if ( pl == NULL )
  return -1;

 switch (virt) {
  case RPLD_PL_VIRT_KEEP:
    return 0;
   break;
  case RPLD_PL_VIRT_DELETE:
    if ( !(pl->flags & RPLD_PL_FLAG_VIRTUAL) )
     return 0;

    plent = pl->first;
    pl->first = plent->list.next;
    rpld_ple_free(plent);

    pl->flags -= RPLD_PL_FLAG_VIRTUAL;

    return 0;
   break;
  case RPLD_PL_VIRT_CREATE:
    if ( pl->flags & RPLD_PL_FLAG_VIRTUAL )
     return 0;

    plent = rpld_ple_new();
    if ( plent == NULL )
     return -1;

    plent->list.next = pl->first;
    pl->first        = plent;

    pl->flags |= RPLD_PL_FLAG_VIRTUAL;

    return 0;
   break;
 }

 return -1;
}

int                          rpld_pl_merge    (struct rpld_playlist * pl, struct rpld_playlist * tm) {
 struct rpld_playlist_entry * ple;

 ROAR_DBG("rpld_pl_merge(pl=%p, tm=%p) = ?", pl, tm);

 if ( pl == NULL || tm == NULL )
  return -1;

 rpld_pl_unregister(tm);

 if ( rpld_pl_virtual(tm, RPLD_PL_VIRT_DELETE) == -1 ) {
  return -1;
 }

 ple = tm->first;
 tm->first = NULL;

 if ( ple != NULL ) {
  rpld_pl_push(pl, ple);
  pl->size_hint += tm->size_hint - 1;
  pl->likeness_hint += tm->likeness_hint - ple->likeness;
 }

 rpld_pl_free(tm);

 return 0;
}

int                          rpld_pl_transaction(struct rpld_playlist * pl, int what) {
 if ( pl == NULL )
  return -1;

 switch (what) {
  case RPLD_PL_TRANS_END:
    if ( pl->transactions != 1 ) {
     return -1;
    }
    pl->transactions--;
    if ( pl->transactions != 0 ) {
     pl->transactions++;
     return -1;
    }
    return 0;
   break;
  case RPLD_PL_TRANS_ASK:
    return pl->transactions;
   break;
  case RPLD_PL_TRANS_BEGIN:
    pl->transactions++;
    if ( pl->transactions > 1 ) {
     pl->transactions--;
     return -1;
    }
    return 0;
   break;
  default:
    return -1;
 }
}

int                          rpld_pl_policy     (struct rpld_playlist * pl, int opolicy, int npolicy) {
 struct rpld_playlist_entry * next;

 if ( pl == NULL )
  return -1;

 pl->policy = npolicy;

 next = pl->first;

 while (next != NULL) {
  next->policy = opolicy;
  next = next->list.next;
 }

 return 0;
}

int                          rpld_pl_begin      (struct rpld_playlist * pl, int opolicy, int npolicy) {
 int r = 0;

 if ( pl == NULL )
  return -1;

 if ( (r = rpld_pl_transaction(pl, RPLD_PL_TRANS_BEGIN)) != 0 )
  return r;

 if ( (r = rpld_pl_policy(pl, opolicy, npolicy)) != 0 ) {
  rpld_pl_transaction(pl, RPLD_PL_TRANS_END);
  return r;
 }

 return r;
}

int                          rpld_pl_commit     (struct rpld_playlist * pl) {
 struct rpld_playlist_entry * cur, * last;

 if ( pl == NULL )
  return -1;

 last = NULL;
 cur = pl->first;

 if ( pl->flags & RPLD_PL_FLAG_VIRTUAL ) {
  last = cur;
  cur  = cur->list.next;
 }

 while (cur != NULL) {
  switch (cur->policy) {
   case RPLD_PL_POLICY_KEEP:
     // we ignore this entry, keep it in playlist
    break;
   case RPLD_PL_POLICY_DELETE:
     if ( last == NULL ) {
      // specal case: we are at begining of playlist
      pl->first = cur->list.next;
      rpld_ple_free(cur);
     } else {
      // normal case: we are somewhere not the first element
      last->list.next = cur->list.next;
      rpld_ple_free(cur);
     }
    break;
   default:
     ROAR_WARN("rpld_pl_commit(*): Invalid policy type on playlist element. Keeping playlist entry");
     ROAR_ERR("rpld_pl_commit(*): Incomplet commit on playlist transaction! Check your data!");
  }

  last = cur;
  cur  = cur->list.next;
 }

 return rpld_pl_transaction(pl, RPLD_PL_TRANS_END);
}

int                          rpld_pl_set_name (struct rpld_playlist * pl, const char * name) {
 if ( pl == NULL )
  return -1;

 if ( pl->name != NULL )
  roar_mm_free(pl->name);

 if ( name == NULL ) {
  pl->name = NULL;
  return 0;
 }

 pl->name = roar_mm_strdup(name);

 if ( pl->name != NULL )
  return 0;

 return -1;
}

char *                       rpld_pl_get_name (struct rpld_playlist * pl) {
 if ( pl == NULL )
  return NULL;

 return pl->name;
}

int                          rpld_pl_set_id   (struct rpld_playlist * pl, pli_t id) {
 if ( pl == NULL )
  return -1;

 pl->id = id;

 return 0;
}

pli_t                        rpld_pl_get_id   (struct rpld_playlist * pl) {
 if ( pl == NULL )
  return 0;

 return pl->id;
}

int                          rpld_pl_set_parent(struct rpld_playlist * pl, pli_t parent) {
 if ( pl == NULL )
  return -1;

 pl->parent = parent;

 return 0;
}

pli_t                        rpld_pl_get_parent(struct rpld_playlist * pl) {
 if ( pl == NULL )
  return 0;

 return pl->parent;
}

int                          rpld_pl_register (struct rpld_playlist * pl) {
 static pli_t counter = 0;
 int i;

 if ( pl == NULL )
  return -1;

 for (i = 0; i < MAX_PLAYLISTS; i++) {
  if ( g_playlists[i] == NULL ) {
   counter++;
   g_playlists[i] = pl;
   // TODO: support tables sizes > 256 entrys
   return rpld_pl_set_id(pl, (counter << 8) | i);
  }
 }

 return -1;
}

int                          rpld_pl_unregister(struct rpld_playlist * pl) {
 if ( pl == NULL )
  return -1;

 if ( pl->id ) {
  if ( g_playlists[pl->id % MAX_PLAYLISTS] != pl )
   return -1;

  g_playlists[pl->id % MAX_PLAYLISTS] = NULL;
  return 0;
 }

 return -1;
}

struct rpld_playlist       * rpld_pl_get_by_id (pli_t id) {
 return g_playlists[id % MAX_PLAYLISTS];
}

struct rpld_playlist       * rpld_pl_get_by_name(const char * name) {
 int i;

 if ( name == NULL )
  return NULL;

 for (i = 0; i < MAX_PLAYLISTS; i++) {
  if ( g_playlists[i] != NULL ) {
   if ( !strcmp(g_playlists[i]->name, name) ) {
    return g_playlists[i];
   }
  }
 }

 return NULL;
}

size_t                       rpld_pl_num      (struct rpld_playlist * pl) {
 struct rpld_playlist_entry * cur;
 size_t count = 0;

 if ( pl == NULL )
  return 0;

 cur = rpld_pl_get_first(pl);

 while ( cur != NULL ) {
  count++;
  cur = cur->list.next;
 }

 return count;
}

struct rpld_playlist_pointer * rpld_plp_init    (struct rpld_playlist         * pl,
                                                 struct rpld_playlist_search  * pls) {
 struct rpld_playlist_pointer * plp = roar_mm_malloc(sizeof(struct rpld_playlist_pointer));

 if ( plp == NULL )
  return NULL;

 memset(plp, 0, sizeof(struct rpld_playlist_pointer));

 if ( pl != NULL ) {
  plp->hint.pl = pl;
  pl->pointer_counter++;
 }

 if ( pls != NULL ) {
  memcpy(&(plp->pls), pls, sizeof(struct rpld_playlist_search));
 }

 return plp;
}

int                          rpld_plp_uninit  (struct rpld_playlist_pointer * plp) {
 if ( plp == NULL )
  return -1;

 if ( plp->hint.pl != NULL ) {
  plp->hint.pl->pointer_counter--;

  if ( plp->hint.pl->pointer_counter == 0 ) {
   if ( plp->hint.pl->flags & RPLD_PL_FLAG_UNLINKED ) {
    rpld_pl_free(plp->hint.pl);
   }
  }
 }

 roar_mm_free(plp);

 return 0;
}

struct rpld_playlist_entry * rpld_plp_search  (struct rpld_playlist_pointer * plp) {
 struct rpld_playlist_entry * ple;
 int i;

 ROAR_DBG("rpld_plp_search(*) = ?");

 if ( plp == NULL )
  return NULL;

 if ( plp->hint.pl != NULL ) {
  ROAR_DBG("rpld_plp_search(*): We have a playlist hint");
  return rpld_pl_search(plp->hint.pl, &(plp->pls), NULL);
 }

 if ( plp->pls.type == RPLD_PL_ST_RANDOM || plp->pls.type == RPLD_PL_ST_RANDOM_LIKENESS ) {
  ROAR_DBG("rpld_plp_search(*): We have a random search.");
  return rpld_pl_search(rpld_pl_get_by_id(plp->pls.subject.pl), &(plp->pls), NULL);
 }

 ROAR_DBG("rpld_plp_search(*): No Playlist hint, searching...");
 // add some way to search multiple playlists
 for (i = 0; i < MAX_PLAYLISTS; i++) {
  if ( g_playlists[i] == g_history )
   continue;

  if ( g_playlists[i] != NULL ) {
   if ( (ple = rpld_pl_search(g_playlists[i], &(plp->pls), NULL)) != NULL )
    return ple;
  }
 }

 ROAR_DBG("rpld_plp_search(*): No matches found.");

 ROAR_DBG("rpld_plp_search(*) = NULL");
 return NULL;
}

int                            rpld_plp_copy    (struct rpld_playlist_pointer ** dst,
                                                 struct rpld_playlist_pointer * src) {
 if ( dst == NULL || src == NULL )
  return -1;

 if ( *dst != NULL )
  if ( rpld_plp_uninit(*dst) == -1 )
   return -1;

 if ( (*dst = rpld_plp_init(src->hint.pl, &(src->pls))) == NULL )
  return -1;

 return 0;
}

//ll
