///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// mayavi vtk visualization
//
// author: Pierre.Saramito@imag.fr
//
// date: 12 may 1997  update: 23 oct 2011
//
#include "rheolef/geo.h"
#include "rheolef/piola.h"
#include "rheolef/rheostream.h"
#include "rheolef/iorheo.h"
#include "rheolef/iofem.h"
using namespace std;
namespace rheolef { 


// ----------------------------------------------------------------------------
// python utils
// ----------------------------------------------------------------------------
template<class T>
static
std::string
python (const point_basic<T>& x, size_t d=3)
{
    std::ostringstream os;
    os << "[" << x[0];
    for (size_t i = 1; i < d; i++)
      os << ", " << x[i];
    os << "]" << flush;
    string buffer = os.str();
    return buffer;
}
// ----------------------------------------------------------------------------
// geo puts
// ----------------------------------------------------------------------------
// extern:
template <class T> odiststream& geo_put_vtk (odiststream&, const geo_basic<T,sequential>&);

template <class T>
odiststream&
visu_vtk_mayavi (odiststream& ops, const geo_basic<T,sequential>& omega)
{
  //
  // 1) prerequises
  //
  using namespace std;
  typedef typename geo_basic<T,sequential>::size_type size_type;
  typedef point_basic<size_type>                      ilat;
  ostream& os = ops.os();
  bool verbose  = iorheo::getverbose(os);
  bool clean    = iorheo::getclean(os);
  bool execute  = iorheo::getexecute(os);
  bool fill      = iorheo::getfill(os);    // isocontours or color fill
  bool elevation = iorheo::getelevation(os);
  bool color   = iorheo::getcolor(os);
  bool gray    = iorheo::getgray(os);
  bool black_and_white = iorheo::getblack_and_white(os);
  bool stereo    = iorheo::getstereo(os);
  bool full      = iorheo::getfull(os);
  bool iso       = iorheo::getiso(os);
  bool cut       = iorheo::getcut(os);
  bool volume    = iorheo::getvolume(os);
  bool grid       = iorheo::getgrid(os);
  bool ball      = iorheo::getball(os);
  bool lattice   = iorheo::getlattice(os);
  bool shrink   = iorheo::getshrink(os);
  bool tube     = iorheo::gettube(os);
  string format   = iorheo::getimage_format(os);
  string basename = iorheo::getbasename(os);
  point_basic<T> origin  = iofem::getorigin(os);
  point_basic<T> normal  = iofem::getnormal(os);
  string outfile_fmt = "";
  string tmp = get_tmpdir() + "/";
  if (!clean) tmp = "";

  size_type dim     = omega.dimension();
  size_type map_dim = omega.map_dimension();
  size_type nv      = omega.sizes().ownership_by_dimension[0].size();
  size_type nedg    = omega.sizes().ownership_by_dimension[1].size();
  size_type nfac    = omega.sizes().ownership_by_dimension[2].size();
  size_type nvol    = omega.sizes().ownership_by_dimension[3].size();
  size_type ne      = omega.sizes().ownership_by_dimension[map_dim].size();
  //
  // 2) output mesh data
  //
  string filelist;
  string filename = tmp+basename + ".vtk";
  filelist = filelist + " " + filename;
  ofstream vtk_os (filename.c_str());
  odiststream vtk (vtk_os);
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  geo_put_vtk (vtk, omega);
  vtk.close();
  //
  // 3) output domains data
  //
  for (size_type idom = 0, ndom = omega.n_domain(); idom < ndom; ++idom) {
      const geo_basic<T,sequential>& dom = omega.get_domain(idom);
      string filename = tmp+basename + "." + dom.name() + ".vtk";
      filelist = filelist + " " + filename;
      ofstream vtk_os_dom (filename.c_str());
      odiststream vtk_dom (vtk_os_dom);
      if (verbose) clog << "! file \"" << filename << "\" created.\n";
      geo_put_vtk (vtk_dom, dom);
      vtk_dom.close();
  }
  //
  // 3) create python data file
  //
  std::string py_name = filename = tmp+basename + ".py";
  filelist = filelist + " " + filename;
  ofstream py (filename.c_str());
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  py << "#!/usr/bin/env mayavi2" << endl
     << "# This is a mayavi script for the visualization of " << basename << ".vtk" << endl
     << "# automatically generated by rheolef."  << endl
     << endl
     ;
  // mayavi2 python library
  py << "from sys import version_info" << endl
     << "# mayavi2.py file has changed from directory since python 2.7:" << endl
     << "if version_info[0] >= 2 and version_info[1] >= 7 and version_info[2] >= 2:" << endl
     << "    from mayavi.scripts import mayavi2" << endl
     << "else:" << endl
     << "    from enthought.mayavi.scripts import mayavi2" << endl
     << "mayavi2.standalone(globals())     # start mayvi2 graphic interface" << endl
     << "from mayavi2_rheolef import *     # load rheolef specific functions" << endl
     << endl
     ;
  py << "d = [ '" << tmp+basename << "'";
  for (size_type idom = 0, ndom = omega.n_domain(); idom < ndom; ++idom) {
      const geo_basic<T,sequential>& dom = omega.get_domain(idom);;
      py << ", '" << dom.name() << "'";
  }
  py << "]" << endl;

  bool has_origin = (origin[0] != numeric_limits<Float>::max());
  bool view_2d = (omega.xmax()[2] - omega.xmin()[2] == 0);
  py << "opt = {                        \\" << endl
     << "    'label'   : 1," << endl
     << "    'axis'    : 1," << endl
     << "    'view_1d' : 0," << endl
     << "    'view_2d' : " << view_2d << "," << endl
     << "    'color'   : '" << (color ? "color" : (gray ? "gray" : "black_and_white")) << "'," << endl
     << "    'stereo'  : " << stereo << ",          \\" << endl
     << "    'ball'    : " << ball << ",            \\" << endl
     << "    'cut'     : " << cut  << ",            \\" << endl
     << "    'fill'    : " << fill << ",            \\" << endl
     << "    'full'    : " << full << ",            \\" << endl
     << "    'lattice' : " << lattice << ",                 \\" << endl
     << "    'shrink'  : " << shrink << ",          \\" << endl
     << "    'tube'    : " << tube << ",            \\" << endl
     << "    'has_origin' : " << has_origin  << ",              \\" << endl
     << "    'origin'     : " << python(origin) << ",               \\" << endl
     << "    'normal'     : " << python(normal) << ",              \\" << endl
     << "    'elevation'  : 0   \\" << endl
     << "  }" << endl
     << endl
     ;
  py << "mayavi2_geo(mayavi, opt, d)" << endl
     << endl
     ;
  //
  // 3) run pyton
  //
  int status = 0;
  string command;
  if (execute) {
      command = "LANG=C PYTHONPATH=" + string(_RHEOLEF_PKGDATADIR) + " mayavi2 -x " + py_name;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  //
  // 4) clear vtk data
  //
  if (clean) {
      command = "/bin/rm -f " + filelist;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  return ops;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template odiststream& visu_vtk_mayavi<Float>  (odiststream&, const geo_basic<Float,sequential>&);

}// namespace rheolef
