///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/form_element.h"
#include "rheolef/tensor.h"
#include "rheolef/geo_domain.h"
#include "rheolef/piola.h"
#include "rheolef/ublas_matrix_range.h"

namespace rheolef {
using namespace std;

// ==========================================================================
// part 1) allocator
// ==========================================================================
template<class T, class M>
void
form_element_rep<T,M>::initialize_all () const
{
  initialize();
#ifdef TODO
  check_macro (_X.get_global_geo() == _Y.get_global_geo(),
	"form assembly: global spaces based on different meshes: unsupported");
#endif // TODO
  if (_qopt.get_order() == 0 ||
      _qopt.get_order() == std::numeric_limits<quadrature_option_type::size_type>::max()) {
    size_t k1 = get_first_basis().degree();
    size_t k2 = get_second_basis().degree();
    size_type quad_order = k1 + k2 - n_derivative() + 1;
    if (is_weighted()) {
      size_type k3 = get_weight().get_space().get_numbering().get_basis().degree();
      quad_order += k3;
    } else {
    }
    if (coordinate_system() != space_constant::cartesian) quad_order++;
    if (_omega.get_background_geo().sizes().ownership_by_variant[reference_element::q].dis_size() != 0 ||
        _omega.get_background_geo().sizes().ownership_by_variant[reference_element::P].dis_size() != 0 ||
        _omega.get_background_geo().sizes().ownership_by_variant[reference_element::H].dis_size() != 0) {
          // integrate exactly ??
          quad_order += 2;
    }
    _qopt.set_order (quad_order);
  } else {
  }
  if (_qopt.get_family() == quadrature_option_type::max_family) {
    _qopt.set_family (quadrature_option_type::gauss);
  }
  _quad.set_order  (_qopt.get_order());
  _quad.set_family (_qopt.get_family());
trace_macro ("quadrature : " << _quad.get_family_name() << " order " << _quad.get_order());
  _bx_table.set    (_quad, get_first_basis());
  _by_table.set    (_quad, get_second_basis());
  _piola_table.set (_quad, get_piola_basis());
#ifdef TODO
  if (_is_weighted) {
    check_macro (get_weight().get_space().get_global_geo() == get_global_geo(),
	"weighted form assembly: weight space based on a different mesh: not supported");
    _bw_table.set (_quad, get_weight_basis());
  }
#endif // TODO
  _initialized = true;
}
// ==========================================================================
// part 2) axisymmetric coord system handler
// ==========================================================================
// TODO: simplify using with piola.h functions
template<class T, class M>
T
form_element_rep<T,M>::weight_coordinate_system (
    const geo_element&            K,
    const std::vector<size_type>& dis_inod,
    size_type                     q) const
{
  if (coordinate_system() == space_constant::cartesian || ! use_coordinate_system_weight()) {
    return 1;
  }
  point_basic<T> xq = rheolef::piola_transformation (_omega, _piola_table, K, dis_inod, q);
  switch (coordinate_system()) {
    case space_constant::axisymmetric_rz: 
	  return use_coordinate_system_dual_weight() ? 1/xq[0] : xq[0];
    case space_constant::axisymmetric_zr:
	  return use_coordinate_system_dual_weight() ? 1/xq[1] : xq[1];
    default: {
	fatal_macro ("unsupported coordinate system `" 
	<<  space_constant::coordinate_system_name (coordinate_system()) << "'");
	return 0;
    }
  }
}
// ==========================================================================
// part 3) some few algebra on elementary matrices
// ==========================================================================

// ----------------------------------------------------------
// t(u,v) = (t*u,v) with map_d <= 3
// ----------------------------------------------------------
template<class T>
static
T
tensor_apply (
  const tensor_basic<T>& t,
  const point_basic<T>& u,
  const point_basic<T>& v,
  typename tensor_basic<T>::size_type map_d)
{
  typedef typename tensor_basic<T>::size_type size_type;
  T sum = 0;
  for (size_type k = 0; k < map_d; k++) {
    for (size_type l = 0; l < map_d; l++) {
      sum += v[k]*t(k,l)*u[l];
    }
  }
  return sum;
}
// ----------------------------------------------------------
// add to elementary matrix : M += (phi otimes psi) * w
// ----------------------------------------------------------
template<class T, class M>
void
form_element_rep<T,M>::cumul_otimes (
    ublas::matrix<T>& m, 
    const T& w,
    quad_const_iterator phi,
    quad_const_iterator last_phi,
    quad_const_iterator first_psi,
    quad_const_iterator last_psi)
{
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      quad_const_iterator psi = first_psi;
      T phi_w = (*phi)*w;
      for (size_type j = 0; psi != last_psi; psi++, j++)
        m(i,j) += (*psi)*phi_w;
    }
}
// --------------------------------------------------------------------
// Mij += trans(grad_phi_i) * Dw * grad_psi_j
// --------------------------------------------------------------------
template<class T, class M>
void
form_element_rep<T,M>::cumul_otimes (
    ublas::matrix<T>& m, 
    const tensor_basic<T>& Dw,
    quad_const_iterator_grad grad_phi,
    quad_const_iterator_grad last_grad_phi,
    quad_const_iterator_grad first_grad_psi,
    quad_const_iterator_grad last_grad_psi,
    size_type map_d)
{
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      quad_const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	m(i,j) += tensor_apply (Dw, *grad_psi, *grad_phi, map_d);
      }
    }
}
template<class T>
static void my_cumul_otimes (tensor_basic<T>& t, const point_basic<T>& a, const point_basic<T>& b, size_t na, size_t nb) {
  cumul_otimes (t, a, b, na, nb);
}
// ==========================================================================
// part 4) predefined forms
// ==========================================================================

// ----------------------------------------------------------
// elementary mass form:
// ----------------------------------------------------------
/*
 Transformation of integration points over the surface.
 Let
   K in Omega  : a side of the surface Omega
   L in Lambda : a element of the bounding box Lambda, and containing K
 as K = L inter {phi(x)=0}.
 Let the two Piola transformations from the reference elements:
   F : hat_K --> K
       hat_x --> x = F(hat_x)
   G : tilde_L --> L
       tilde_x --> x = G(tilde_x)
 For computing the matrix on K, we want to compute an integral over K that is
 transformed on a integral over hat_K and then to a discrete sum:

   int_K f(x) dx = int_hat_K f(F(hat_x))  |DF(hat_x)|  d hat_x
                 = sum_q     f(F(hat_xq)) |DF(hat_xq)| hat_wq

 where (hat_xq, hat_wq) are given by a quadrature formulae over hat_K
 and DF(hat_K) = is the jacobian of the Piola transformation F.

 Here f is a product of basis functions defined over tilde_L:
 it is known via tilde_f as:
   f(x) = tilde_f (G^{-1}(x))

 Thus, the integration point hat_xq may be transformed via G^{-1} o F :

   int_K f(x) dx = sum_q tilde_f(G^{-1}(F(hat_xq))) |DF(hat_xq)| hat_wq

*/
template<class T, class M>
void
form_element_rep<T,M>::build_scalar_mass (const geo_element& K, ublas::matrix<T>& m) const
{
  std::vector<size_type> dis_inod_K;
  std::vector<size_type> dis_inod_L;
  _omega.dis_inod (K, dis_inod_K);
  reference_element hat_K   = K.variant();
  reference_element tilde_L = hat_K;
  if (is_on_band()) {
    size_type first_dis_ie = _omega.sizes().ownership_by_dimension [K.dimension()].first_index();
    size_type K_ie = K.dis_ie() - first_dis_ie;
    size_type L_ie = get_band().sid_ie2bnd_ie (K_ie);
    const geo_element& L = get_band().band() [L_ie];
    get_band().band().dis_inod (L, dis_inod_L);
    tilde_L = L.variant();
  }
  m.resize (get_second_basis().size(tilde_L), get_first_basis().size(tilde_L));
  m.clear();
  typename quadrature<T>::const_iterator
    first = _quad.begin(hat_K),
    last  = _quad.end  (hat_K);
  tensor_basic<T> DF;
  for (size_type q = 0; first != last; first++, q++) {
    rheolef::jacobian_piola_transformation (_omega, _piola_table, K, dis_inod_K, q, DF);
    T det_DF = rheolef::det_jacobian_piola_transformation (DF, coordinate_dimension(), K.dimension());
    if (!is_on_band()) {
      _bx_table.evaluate (hat_K, q);
      _by_table.evaluate (hat_K, q);
    } else {
      point_basic<T> xq = rheolef::piola_transformation (_omega, _piola_table, hat_K, dis_inod_K, q);
      point_basic<T> tilde_xq = rheolef::inverse_piola_transformation (get_band().band(), tilde_L, dis_inod_L, xq);
      _bx_table.evaluate (tilde_L, tilde_xq);
      _by_table.evaluate (tilde_L, tilde_xq);
    }
    T wq = 1;
    wq *= is_weighted() ? weight(K,q) : T(1);
    wq *= weight_coordinate_system (K, dis_inod_K, q);
    wq *= det_DF;
    wq *= (*first).w;
    cumul_otimes (m, wq,
        _by_table.begin(), _by_table.end(),
        _bx_table.begin(), _bx_table.end());
  }
}
// --------------------------------------------------------------------
// elementary grad_grad form:
// --------------------------------------------------------------------
/*
 With the same notations as for the mass form.
 For computing the matrix on K, we want to compute an integral over K that is
 transformed on a integral over hat_K and then to a discrete sum:

   int_K df_dx(x) dx = sum_q  df_dx(F(hat_xq)) |DF(hat_xq)| hat_wq

 Here f is a product of basis functions defined over tilde_L:
 it is known via tilde_f as:
   f(x) = tilde_f (G^{-1}(x))
 Its derivative expresses with a product of the jacobian DG^-T(tilde_x) :

   int_K f(x) dx = sum_q DG^-T (G^{-1}(F(hat_xq)))
                               d tilde_f/d tilde_x (G^{-1}(F(hat_xq)))
                               |DF(hat_xq)| hat_wq

*/
template<class T, class M>
void 
form_element_rep<T,M>::build_scalar_grad_grad (const geo_element& K, ublas::matrix<T>& a) const
{
  bool scalar_diffusion = true;
  std::vector<size_type> dis_inod_K;
  std::vector<size_type> dis_inod_L;
  _omega.dis_inod (K, dis_inod_K);
  reference_element hat_K   = K.variant();
  reference_element tilde_L = hat_K;
  if (is_on_band()) {
    size_type first_dis_ie = _omega.sizes().ownership_by_dimension [K.dimension()].first_index();
    size_type K_ie = K.dis_ie() - first_dis_ie;
    size_type L_ie = get_band().sid_ie2bnd_ie (K_ie);
    const geo_element& L = get_band().band() [L_ie];
    get_band().band().dis_inod (L, dis_inod_L);
    tilde_L = L.variant();
  }
  a.resize (get_second_basis().size(tilde_L), get_first_basis().size(tilde_L));
  a.clear();
  typename quadrature<T>::const_iterator
    first = _quad.begin(hat_K),
    last  = _quad.end  (hat_K);
  tensor_basic<T> DF, DG, invDG, invDG_P, P, Dw, W;
  size_type K_map_d = hat_K.dimension();
  size_type L_map_d = tilde_L.dimension();
  size_type d = coordinate_dimension();
  for (size_type q = 0; first != last; first++, q++) {
    rheolef::jacobian_piola_transformation (_omega, _piola_table, K, dis_inod_K, q, DF);
    T det_DF = rheolef::det_jacobian_piola_transformation (DF, d, K_map_d);
    if (!is_on_band()) {
      invDG_P = pseudo_inverse_jacobian_piola_transformation (DF, d, K_map_d);
      _bx_table.evaluate_grad (hat_K, q);
      _by_table.evaluate_grad (hat_K, q);
    } else {
      point_basic<T> xq = rheolef::piola_transformation (_omega, _piola_table, hat_K, dis_inod_K, q);
      point_basic<T> tilde_xq = rheolef::inverse_piola_transformation (get_band().band(), tilde_L, dis_inod_L, xq);
      rheolef::jacobian_piola_transformation (get_band().band(), _piola_table, tilde_L, dis_inod_L, tilde_xq, DG);
      invDG = pseudo_inverse_jacobian_piola_transformation (DG, d, L_map_d);
      // apply also the tangential projection P=(I-n*n)
      map_projector (DF, d, K_map_d, P);
      prod (invDG, P, invDG_P, L_map_d, d, d);
      _bx_table.evaluate_grad (tilde_L, tilde_xq);
      _by_table.evaluate_grad (tilde_L, tilde_xq);
    }
    T wq = 1;
    wq *= weight_coordinate_system (K, dis_inod_K, q);
    wq *= det_DF;
    wq *= (*first).w;
    if (scalar_diffusion) {
       wq *= (is_weighted() ? weight(K,q) : T(1));
       Dw = wq*invDG_P*trans(invDG_P);
    } else { // tensorial diffusion
       error_macro ("tensorial diffusion: not yet");
#ifdef TODO
       weight (K, q, W);
       Dw = wq*invDG_P*W*trans(invDG_P);
#endif // TODO
    }
    cumul_otimes (a, Dw,
      _by_table.begin_grad(), _by_table.end_grad(),
      _bx_table.begin_grad(), _bx_table.end_grad(),
      L_map_d);
  }
}
// --------------------------------------------------------------------
// general local mass matrix : multi-component spaces and weight
// --------------------------------------------------------------------
template<class T, class M>
void
form_element_rep<T,M>::build_general_mass (const geo_element& K, ublas::matrix<T>& m) const
{
  space_constant::valued_type space_valued  = get_first_space().valued_tag();
  if (space_valued == space_constant::scalar) {
    build_scalar_mass (K, m);
    return;
  }
  // space is vectorial or tensorial: weight could be scalar, tensor or tensor4
  size_type n_comp = get_first_space().size();
  size_type loc_n1dof = get_second_basis().size(K);
  size_type loc_n2dof = get_first_basis().size(K);
  m.resize (n_comp*loc_n1dof, n_comp*loc_n2dof);
  m.clear();
  space_constant::valued_type weight_valued = space_constant::scalar;
  if (is_weighted()) {
    weight_valued = _wh.get_space().get_constitution().valued_tag();
  }
  if (weight_valued == space_constant::scalar) {
    // --------------------------------------------------------------------------
    // all components have the same weight and the local matrix is block-diagonal
    // --------------------------------------------------------------------------
    ublas::matrix<T> m_ij;
    build_scalar_mass (K, m_ij);
    switch (space_valued) {
      case space_constant::scalar:
      case space_constant::vector: {
        for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
          mr_set (m, ublas::range(i_comp*loc_n1dof,(i_comp+1)*loc_n1dof), ublas::range(i_comp*loc_n2dof,(i_comp+1)*loc_n2dof), m_ij);
	}
        break;
      }
      case space_constant::tensor: {
        // symmetric tensor => 2 factor for non-diagonal subscripts
        for (size_type ij_comp = 0; ij_comp < n_comp; ij_comp++) {
	  space_constant::coordinate_type sys_coord = coordinate_system();
          std::pair<size_type,size_type> ij = space_constant::tensor_subscript (space_valued, sys_coord, ij_comp);
          size_t i_comp = ij.first;
          size_t j_comp = ij.second;
          T factor_ij = ((i_comp == j_comp) ? 1 : 2); // symmetry => multiplicity factor: when i_comp != j_comp : 2*w_ij
  	  mr_set (m, ublas::range(ij_comp*loc_n1dof,(ij_comp+1)*loc_n1dof), ublas::range(ij_comp*loc_n2dof,(ij_comp+1)*loc_n2dof), factor_ij*m_ij);
        }
        break;
      }
      default: {
        error_macro ("unexpected `" << space_constant::valued_name(space_valued) << "'-valued space for the `mass' form.");
      }
    }
  } else { // weight_valued != fem_helper::scalar
    // ------------------------------------------------------------------------------
    // components have different weight : the local matrix is no more block-diagonal
    // ------------------------------------------------------------------------------
#ifdef TODO
    switch (space_valued) {
      case fem_helper::vectorial: {
        // first case : vector space and tensor weight
        //     m(u,v) = int_Omega w_ij u_j v_i dx
	//     and w_ij = w_ji
        ublas::matrix<T> m_ij;
        for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
          for (size_type j_comp = 0; j_comp < n_comp; j_comp++) {
	    size_type ij_comp = fem_helper::tensor_index (weight_valued, sys_coord, i_comp, j_comp);
            build_scalar_mass (K, m_ij, ij_comp);
            mr_set (m, ublas::range(i_comp*loc_n1dof,(i_comp+1)*loc_n1dof), ublas::range(j_comp*loc_n2dof,(j_comp+1)*loc_n2dof), m_ij);
          }
        }
	break;
      }
      case fem_helper::tensorial: {
        // second case : tensor space and tensor4 weight
        //     m(tau,gamma) = int_Omega w_ijkl tau_kj gamma_ij dx
	//     and w_ijkl = w_jikl, w_ijkl = w_ijlk and w_ijkl = w_klij
	// symmetry => multiplicity factor: 4 when i!=j and k!=l or 2 when i!=j xor k!=l
        ublas::matrix<T> m_ijkl;
        for (size_type ij_comp = 0; ij_comp < n_comp; ij_comp++) {
          fem_helper::pair_size_type ij = fem_helper::tensor_subscript (space_valued, sys_coord, ij_comp);
          size_t i = ij.first;
          size_t j = ij.second;
          T factor_ij = ((i == j) ? 1 : 2);
          for (size_type kl_comp = 0; kl_comp < n_comp; kl_comp++) {
            fem_helper::pair_size_type kl = fem_helper::tensor_subscript (space_valued, sys_coord, kl_comp);
            size_t k = kl.first;
            size_t l = kl.second;
            T factor_kl = ((k == l) ? 1 : 2);
	    T factor_ijkl = factor_ij*factor_kl;
	    size_type ijkl_comp = fem_helper::tensor4_index (weight_valued, sys_coord, i, j, k, l);
            build_scalar_mass (K, m_ijkl, ijkl_comp);
            mr_set (m, ublas::range(ij_comp*loc_n1dof,(ij_comp+1)*loc_n1dof), ublas::range(kl_comp*loc_n2dof,(kl_comp+1)*loc_n2dof), factor_ijkl*m_ijkl);
          }
        }
	break;
      }
      default: {
        error_macro ("unexpected " << fem_helper::valued_field_name(space_valued) << "-valued space and `"
		<< fem_helper::valued_field_name(weight_valued)
		<< "'-valued weight for the `mass' form.");
      }
    }
#endif// TODO
  error_macro ("mass: multi-component with non-scalar weight: not yet");
  }
}
// --------------------------------------------------------------------
// (d_dxi phi, psi)
// Mij += trans(grad_phi_i) * Dw * grad_psi_j
// --------------------------------------------------------------------
template<class T, class M>
void 
form_element_rep<T,M>::build_d_dx (const geo_element& K, ublas::matrix<T>& m, size_type i_comp) const
{
  size_type map_d = K.dimension();
  check_macro (i_comp < map_d, "unexpected `d_dx" << i_comp << "' on element type "
	<< "`" << K.name() << "'");
  check_macro (coordinate_dimension() == K.dimension(),
    "unsupported `d_dx" << i_comp << "' form on element type `" << K.name()
    << " in " << coordinate_dimension() << "D geometry");
  reference_element hat_K = K;
  m.resize (get_second_basis().size(hat_K), get_first_basis().size(hat_K));
  m.clear();
  std::vector<size_type> dis_inod;
  _omega.dis_inod (K, dis_inod);
  typename quadrature<T>::const_iterator
    first = _quad.begin(hat_K),
    last  = _quad.end  (hat_K);
  tensor_basic<T> DF, invDF;
  for (size_type q = 0; first != last; first++, q++) {
    rheolef::jacobian_piola_transformation (_omega, _piola_table, K, dis_inod, q, DF);
    invDF = pseudo_inverse_jacobian_piola_transformation (DF, coordinate_dimension(), map_d);
    invDF = inv(DF,map_d);
    T wq = is_weighted() ? weight(K,q) : T(1);
    wq *= weight_coordinate_system (K, dis_inod, q);
    wq *= rheolef::det_jacobian_piola_transformation (DF, coordinate_dimension(), K.dimension());
    wq *= (*first).w;
    point_basic<T> invDFt_icomp_wq = invDF.col(i_comp) * wq;
  
    // put in a subroutine:
    quad_const_iterator      phi      = _by_table.begin(hat_K, q),
                             last_phi = _by_table.end  (hat_K, q);
    quad_const_iterator_grad first_grad_psi = _bx_table.begin_grad (hat_K, q),
                             last_grad_psi  = _bx_table.end_grad   (hat_K, q);
    for (size_type i = 0; phi != last_phi; phi++, i++) {
      quad_const_iterator_grad grad_psi = first_grad_psi;
      T phi_i = *phi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	T sum = 0;
	const point_basic<T>& grad_psi_j = *grad_psi;
        for (size_type k = 0; k < map_d; k++) {
          sum += phi_i * invDFt_icomp_wq[k] * grad_psi_j[k];
	}
	m(i,j) += sum;
      }
    }
  }
}
// --------------------------------------------------------------------
// int_K (gradt u : grad v) w dK = int_K [(M U).V] w dK,
//		with u = Sum_j,l U_{nx*l+j} psi_j
//			 v = Sum_i,k V_{ny*k+i} phi_i
// so M_{ny*k+i,nx*l+j} = int_K d psi_j/dk * d phi_i/dl w dK 
// --------------------------------------------------------------------
template<class T, class M>
void 
form_element_rep<T,M>::build_gradt_grad (const geo_element& K, ublas::matrix<T>& m) const
{
  size_type map_d = K.dimension();
  check_macro (coordinate_dimension() == K.dimension(),
    "unsupported `gradt_grad' form on element type `" << K.name()
    << " in " << coordinate_dimension() << "D geometry");
  size_type nx = get_first_basis().size(K);
  size_type ny = get_second_basis().size(K);
  m.resize (map_d*ny, map_d*nx);
  m.clear();
  std::vector<size_type> dis_inod;
  _omega.dis_inod (K, dis_inod);
  typename quadrature<T>::const_iterator
    first = _quad.begin(K),
    last  = _quad.end  (K);
  tensor_basic<T> DF, invDF, invDFt, Dw;
  for (size_type q = 0; first != last; first++, q++) {
    rheolef::jacobian_piola_transformation (_omega, _piola_table, K, dis_inod, q, DF);
    invDF = inv(DF,map_d);
    invDFt = trans(invDF);
    T wq = is_weighted() ? weight(K,q) : T(1);
    wq *= weight_coordinate_system (K, dis_inod, q);
    wq *= rheolef::det_jacobian_piola_transformation (DF, coordinate_dimension(), K.dimension());
    wq *= (*first).w;
    
    quad_const_iterator_grad grad_phi       = _by_table.begin_grad (K, q),
                             last_grad_phi  = _by_table.end_grad   (K, q),
                             first_grad_psi = _bx_table.begin_grad (K, q),
                             last_grad_psi  = _bx_table.end_grad   (K, q);
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      const point_basic<T>& hat_grad_phi_i = *grad_phi;
      point_basic<T> grad_phi_i = invDFt*hat_grad_phi_i;
      quad_const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	const point_basic<T>& hat_grad_psi_j = *grad_psi;
        point_basic<T> grad_psi_j = invDFt*hat_grad_psi_j;
        for (size_type k = 0; k < map_d; k++) {
          for (size_type l = 0; l < map_d; l++) {
	    m(i+ny*k,j+nx*l) += grad_psi_j[k]*grad_phi_i[l]*wq;
	  }
	}
      }
    }
  }
}
// --------------------------------------------------------------------
// vector-valued : <div_div phi, psi> = (div phi, div psi)
// Mij += trans(grad_phi_i) * Dw * grad_psi_j
// --------------------------------------------------------------------
template<class T, class M>
void 
form_element_rep<T,M>::build_div_div (const geo_element& K, ublas::matrix<T>& m) const
{
  size_type map_d = K.dimension();
  reference_element hat_K = K;
  check_macro (coordinate_dimension() == K.dimension(),
    "unsupported `gradt_grad' form on element type `" << K.name()
    << "' in " << coordinate_dimension() << "D geometry");
  size_type nx = get_first_basis().size(hat_K);
  size_type ny = get_second_basis().size(hat_K);
  m.resize (map_d*ny, map_d*nx);
  m.clear();
  std::vector<size_type> dis_inod;
  _omega.dis_inod (K, dis_inod);
  typename quadrature<T>::const_iterator
    first = _quad.begin(hat_K),
    last  = _quad.end  (hat_K);
  tensor_basic<T> DF, invDF, invDFt, Dw;
  for (size_type q = 0; first != last; first++, q++) {
    rheolef::jacobian_piola_transformation (_omega, _piola_table, K, dis_inod, q, DF);
    invDF = inv(DF,map_d);
    invDFt = trans(invDF);
    T wq = is_weighted() ? weight(K,q) : T(1);
    wq *= weight_coordinate_system (K, dis_inod, q);
    wq *= rheolef::det_jacobian_piola_transformation (DF, coordinate_dimension(), K.dimension());
    wq *= (*first).w;
    
    quad_const_iterator_grad grad_phi       = _by_table.begin_grad(hat_K, q),
                             last_grad_phi  = _by_table.end_grad  (hat_K, q),
                             first_grad_psi = _bx_table.begin_grad(hat_K, q),
                             last_grad_psi  = _bx_table.end_grad  (hat_K, q);
    for (size_type i = 0; grad_phi != last_grad_phi; grad_phi++, i++) {
      const point_basic<T>& hat_grad_phi_i = *grad_phi;
      point_basic<T> grad_phi_i = invDFt*hat_grad_phi_i;
      quad_const_iterator_grad grad_psi = first_grad_psi;
      for (size_type j = 0; grad_psi != last_grad_psi; grad_psi++, j++) {
	const point_basic<T>& hat_grad_psi_j = *grad_psi;
        point_basic<T> grad_psi_j = invDFt*hat_grad_psi_j;
        for (size_type k = 0; k < map_d; k++) {
          for (size_type l = 0; l < map_d; l++) {
	    m(i+ny*k,j+nx*l) += grad_psi_j[l]*grad_phi_i[k]*wq;
	  }
	}
      }
    }
  }
}
// ----------------------------------------------------------
// elementary grad_grad form:
// --------------------------------------------------------------------
// weighted forms: i-th component on node x_q in K
template<class T, class M>
void
form_element_rep<T,M>::set_weight (const field_basic<T,M>& wh) const
{
  _wh = wh;
  _wh.dis_dof_update();
  _bw_table.set (_quad, _wh.get_space().get_numbering().get_basis());
  _is_weighted = true;
  _qopt.set_order (0); // will be re-computed at re-initailization
  initialize_all ();
}
template<class T, class M>
T
form_element_rep<T,M>::weight (const geo_element& K, size_type q) const
{
    // get wh value at idof elements
    const basis_basic<T>& bw = _wh.get_space().get_numbering().get_basis();
    size_t loc_ndof = bw.size (K.variant()) ;
    std::vector<size_type> dis_idof1 (loc_ndof);
    _wh.get_space().dis_idof (K, dis_idof1);

    std::vector<T> dof (loc_ndof);
    for (size_type loc_idof = 0; loc_idof < loc_ndof; loc_idof++) {
      dof [loc_idof] = _wh.dis_dof (dis_idof1 [loc_idof]);
    }
    quad_const_iterator iter_b = _bw_table.begin (K, q);
    T wq = 0;
    for (size_type loc_idof = 0; loc_idof < loc_ndof; loc_idof++, iter_b++) {
      wq += dof [loc_idof] * (*iter_b); // sum_i w_coef(i)*hat_phi(x_i)
    }
    return wq;
}
// ==========================================================================
// instanciation in library
// ==========================================================================
template class form_element_rep<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class form_element_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
