# ifndef _RHEO_DAMPED_NEWTON_H
# define _RHEO_DAMPED_NEWTON_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/damped-newton-generic.h"

namespace rheolef { 

/*Class:damped_newton
NAME: @code{damped_newton} -- damped Newton nonlinear algorithm
@findex damped\_newton
@cindex nonlinear problem
@cindex Newton method
DESCRIPTION: 
  @noindent
  Nonlinear damped Newton algorithm for the resolution of the following problem:
  @example
       F(u) = 0
  @end example
  A simple call to the algorithm writes:
  @example
    my_problem P;
    field uh (Vh);
    damped_newton (P, uh, tol, max_iter);
  @end example
  The @code{my_problem} class may contains methods for the evaluation
  of F (aka residue) and its derivative:
  @example
    class my_problem @{
    public:
      my_problem();
      field residue (const field& uh) const;
      void update_derivative (const field& uh) const;
      field derivative_trans_mult (const field& mrh) const;
      field derivative_solve (const field& mrh) const;
      Float norm (const field& uh) const;
      Float dual_norm (const field& Muh) const;
    @};
  @end example
  See the example @code{p-laplacian.h} in the user's documentation
  for more.
AUTHOR: 
   | Pierre.Saramito@imag.fr
    LJK, 38041 Grenoble cedex 9, France
DATE:   14 oct 2009
METHODS: @newton
End:
*/
struct newton_identity_preconditioner {
  template <class Problem>
  Float operator() (const Problem& P, const typename Problem::value_type& MFv) const {
    return 0.5*P.dual_dot (MFv,MFv);
  }
  template <class Problem>
  Float operator() (const Problem& P, const typename Problem::value_type& MFu, const typename Problem::value_type& delta_u) const {
    return operator() (P, MFu);
  }
  template <class Problem>
  typename Problem::value_type grad (const Problem& P, const typename Problem::value_type& MFu) const { 
    return P.derivative_trans_mult (MFu);
  }
  template <class Problem>
  Float slope (const Problem& P, const typename Problem::value_type& MFu, const typename Problem::value_type& delta_u) const { 
    return -P.dual_dot (MFu,MFu);
  }
};
//<damped_newton:
template <class Problem, class Field, class Real, class Size>
int damped_newton (Problem P, Field& u, Real& tol, Size& max_iter, odiststream* p_derr=0) {
  return damped_newton(P, newton_identity_preconditioner(), u, tol, max_iter, p_derr);
}
//>damped_newton:
}// namespace rheolef
# endif // _RHEO_DAMPED_NEWTON_H
